//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// TupleAssignment.cc --
//
// Author           : Rodolfo G. Esteves
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon May 18 15:02:53 2015
// Update Count     : 2
//

#include <iterator>
#include <iostream>
#include <cassert>
#include "Tuples.h"
#include "GenPoly/DeclMutator.h"
#include "SynTree/Mutator.h"
#include "SynTree/Statement.h"
#include "SynTree/Declaration.h"
#include "SynTree/Type.h"
#include "SymTab/Mangler.h"
#include "Common/ScopedMap.h"

namespace Tuples {
	class TupleAssignExpander : public Mutator {
	public:
		virtual Expression * mutate( TupleAssignExpr * tupleExpr );
	};

	class TupleTypeReplacer : public GenPoly::DeclMutator {
	  public:
		typedef GenPoly::DeclMutator Parent;

		virtual Type * mutate( TupleType * tupleType );

		virtual CompoundStmt * mutate( CompoundStmt * stmt ) {
			typeMap.beginScope();
			stmt = Parent::mutate( stmt );
			typeMap.endScope();
			return stmt;
		}
	  private:
		ScopedMap< std::string, StructDecl * > typeMap;
	};

	void expandTuples( std::list< Declaration * > & translationUnit ) {
		TupleAssignExpander expander;
		mutateAll( translationUnit, expander );

		TupleTypeReplacer replacer;
		replacer.mutateDeclarationList( translationUnit );
	}

	Expression * TupleAssignExpander::mutate( TupleAssignExpr * tupleExpr ) {
		CompoundStmt * compoundStmt = new CompoundStmt( noLabels );
		std::list< Statement * > & stmts = compoundStmt->get_kids();
		for ( ObjectDecl * obj : tupleExpr->get_tempDecls() ) {
			stmts.push_back( new DeclStmt( noLabels, obj ) );
		}
		for ( Expression * assign : tupleExpr->get_assigns() ) {
			stmts.push_back( new ExprStmt( noLabels, assign ) );
		}
		tupleExpr->get_tempDecls().clear();
		tupleExpr->get_assigns().clear();
		delete tupleExpr;
		return new StmtExpr( compoundStmt );
	}

	Type * TupleTypeReplacer::mutate( TupleType * tupleType ) {
		std::string mangleName = SymTab::Mangler::mangleType( tupleType );
		TupleType * newType = safe_dynamic_cast< TupleType * > ( Parent::mutate( tupleType ) );
		if ( ! typeMap.count( mangleName ) ) {
			// generate struct type to replace tuple type
			StructDecl * decl = new StructDecl( "_tuple_type_" + mangleName );
			decl->set_body( true );
			int cnt = 0;
			for ( Type * t : *newType ) {
				decl->get_members().push_back( new ObjectDecl( "field_"+std::to_string(++cnt), DeclarationNode::NoStorageClass, LinkageSpec::C, nullptr, t->clone(), nullptr ) );
			}
			typeMap[mangleName] = decl;
			addDeclaration( decl );
		}
		delete newType;
		return new StructInstType( newType->get_qualifiers(), typeMap[mangleName] );
	}

} // namespace Tuples

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //

