//
// Cforall Version 1.0.0 Copyright (C) 2016 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// invoke.c --
//
// Author           : Thierry Delisle
// Created On       : Tue Jan 17 12:27:26 2016
// Last Modified By : Peter A. Buhr
// Last Modified On : Fri Feb  9 16:37:42 2018
// Update Count     : 5
//

#include <stdbool.h>
#include <stdlib.h>
#include <stdio.h>

#include "invoke.h"

#define __CFA_INVOKE_PRIVATE__
#include "invoke.h"

// magically invoke the "main" of the most derived class
// Called from the kernel when starting a coroutine or task so must switch back to user mode.

extern void __suspend_internal(void);
extern void __leave_coroutine(void);
extern void __leave_thread_monitor( struct thread_desc * this );
extern void disable_interrupts();
extern void enable_interrupts( __cfaabi_dbg_ctx_param );

void CtxInvokeCoroutine(
	void (*main)(void *),
	struct coroutine_desc *(*get_coroutine)(void *),
	void *this
) {
	struct coroutine_desc* cor = get_coroutine( this );

	if(cor->state == Primed) {
		__suspend_internal();
	}

	cor->state = Active;

	main( this );

	cor->state = Halted;

	//Final suspend, should never return
	__leave_coroutine();
	__cabi_abort( "Resumed dead coroutine" );
}

void CtxInvokeThread(
	void (*dtor)(void *),
	void (*main)(void *),
	struct thread_desc *(*get_thread)(void *),
	void *this
) {
	// First suspend, once the thread arrives here,
	// the function pointer to main can be invalidated without risk
	__suspend_internal();

	// Fetch the thread handle from the user defined thread structure
	struct thread_desc* thrd = get_thread( this );

	// Officially start the thread by enabling preemption
	enable_interrupts( __cfaabi_dbg_ctx );

	// Call the main of the thread
	main( this );

	// To exit a thread we must :
	// 1 - Mark it as halted
	// 2 - Leave its monitor
	// 3 - Disable the interupts
	// 4 - Final suspend
	// The order of these 4 operations is very important
	//Final suspend, should never return
	__leave_thread_monitor( thrd );
	__cabi_abort( "Resumed dead thread" );
}


void CtxStart(
	void (*main)(void *),
	struct coroutine_desc *(*get_coroutine)(void *),
	void *this,
	void (*invoke)(void *)
) {
	struct coStack_t* stack = &get_coroutine( this )->stack;

#if defined( __i386 )

	struct FakeStack {
	    void *fixedRegisters[3];		  	// fixed registers ebx, edi, esi (popped on 1st uSwitch, values unimportant)
	    uint32_t mxcr;                        // SSE Status and Control bits (control bits are preserved across function calls)
	    uint16_t fcw;                         // X97 FPU control word (preserved across function calls)
	    void *rturn;                          // where to go on return from uSwitch
	    void *dummyReturn;				// fake return compiler would have pushed on call to uInvoke
	    void *argument[3];				// for 16-byte ABI, 16-byte alignment starts here
	    void *padding;				// padding to force 16-byte alignment, as "base" is 16-byte aligned
	};

	((struct machine_context_t *)stack->context)->SP = (char *)stack->base - sizeof( struct FakeStack );
	((struct machine_context_t *)stack->context)->FP = NULL;		// terminate stack with NULL fp

	((struct FakeStack *)(((struct machine_context_t *)stack->context)->SP))->dummyReturn = NULL;
	((struct FakeStack *)(((struct machine_context_t *)stack->context)->SP))->argument[0] = this;     // argument to invoke
	((struct FakeStack *)(((struct machine_context_t *)stack->context)->SP))->rturn = invoke;
	((struct FakeStack *)(((struct machine_context_t *)stack->context)->SP))->mxcr = 0x1F80; //Vol. 2A 3-520
	((struct FakeStack *)(((struct machine_context_t *)stack->context)->SP))->fcw = 0x037F;  //Vol. 1 8-7

#elif defined( __x86_64 )

	struct FakeStack {
		void *fixedRegisters[5];            // fixed registers rbx, r12, r13, r14, r15
		uint32_t mxcr;                      // SSE Status and Control bits (control bits are preserved across function calls)
		uint16_t fcw;                       // X97 FPU control word (preserved across function calls)
		void *rturn;                        // where to go on return from uSwitch
		void *dummyReturn;                  // NULL return address to provide proper alignment
	};

	((struct machine_context_t *)stack->context)->SP = (char *)stack->base - sizeof( struct FakeStack );
	((struct machine_context_t *)stack->context)->FP = NULL;		// terminate stack with NULL fp

	((struct FakeStack *)(((struct machine_context_t *)stack->context)->SP))->dummyReturn = NULL;
	((struct FakeStack *)(((struct machine_context_t *)stack->context)->SP))->rturn = CtxInvokeStub;
	((struct FakeStack *)(((struct machine_context_t *)stack->context)->SP))->fixedRegisters[0] = this;
	((struct FakeStack *)(((struct machine_context_t *)stack->context)->SP))->fixedRegisters[1] = invoke;
	((struct FakeStack *)(((struct machine_context_t *)stack->context)->SP))->mxcr = 0x1F80; //Vol. 2A 3-520
	((struct FakeStack *)(((struct machine_context_t *)stack->context)->SP))->fcw = 0x037F;  //Vol. 1 8-7

#elif defined( __ARM_ARCH )

	struct FakeStack {
		float fpRegs[16];			// floating point registers
		void *intRegs[9];			// integer/pointer registers
		void *arg[2];				// placeholder for this pointer
	};

	((struct machine_context_t *)stack->context)->SP = (char *)stack->base - sizeof( struct FakeStack );
	((struct machine_context_t *)stack->context)->FP = NULL;

	struct FakeStack *fs = (struct FakeStack *)((struct machine_context_t *)stack->context)->SP;

	fs->intRegs[8] = CtxInvokeStub;
	fs->arg[0] = this;
	fs->arg[1] = invoke;

#else
	#error uknown hardware architecture
#endif
}

// Local Variables: //
// mode: c //
// tab-width: 4 //
// End: //
