//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// AdjustExprType.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sat May 16 23:41:42 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Mar  2 17:34:53 2016
// Update Count     : 4
//

#include "typeops.h"
#include "SynTree/Type.h"
#include "TypeEnvironment.h"
#include "SymTab/Indexer.h"

namespace ResolvExpr {
	class AdjustExprType : public Mutator {
		typedef Mutator Parent;
		using Parent::mutate;
	  public:
		AdjustExprType( const TypeEnvironment &env, const SymTab::Indexer &indexer );
	  private:
		virtual Type* mutate( VoidType *voidType );
		virtual Type* mutate( BasicType *basicType );
		virtual Type* mutate( PointerType *pointerType );
		virtual Type* mutate( ArrayType *arrayType );
		virtual Type* mutate( FunctionType *functionType );
		virtual Type* mutate( StructInstType *aggregateUseType );
		virtual Type* mutate( UnionInstType *aggregateUseType );
		virtual Type* mutate( EnumInstType *aggregateUseType );
		virtual Type* mutate( TraitInstType *aggregateUseType );
		virtual Type* mutate( TypeInstType *aggregateUseType );
		virtual Type* mutate( TupleType *tupleType );
		virtual Type* mutate( VarArgsType *varArgsType );
		virtual Type* mutate( ZeroType *zeroType );
		virtual Type* mutate( OneType *oneType );

		const TypeEnvironment &env;
		const SymTab::Indexer &indexer;
	};

	void adjustExprType( Type *&type, const TypeEnvironment &env, const SymTab::Indexer &indexer ) {
		AdjustExprType adjuster( env, indexer );
		Type *newType = type->acceptMutator( adjuster );
		type = newType;
	}

	AdjustExprType::AdjustExprType( const TypeEnvironment &env, const SymTab::Indexer &indexer )
		: env( env ), indexer( indexer ) {
	}

	Type *AdjustExprType::mutate( VoidType *voidType ) {
		return voidType;
	}

	Type *AdjustExprType::mutate( BasicType *basicType ) {
		return basicType;
	}

	Type *AdjustExprType::mutate( PointerType *pointerType ) {
		return pointerType;
	}

	Type *AdjustExprType::mutate( ArrayType *arrayType ) {
		// need to recursively mutate the base type in order for multi-dimensional arrays to work.
		PointerType *pointerType = new PointerType( arrayType->get_qualifiers(), arrayType->get_base()->clone()->acceptMutator( *this ) );
		delete arrayType;
		return pointerType;
	}

	Type *AdjustExprType::mutate( FunctionType *functionType ) {
		PointerType *pointerType = new PointerType( Type::Qualifiers(), functionType );
		return pointerType;
	}

	Type *AdjustExprType::mutate( StructInstType *aggregateUseType ) {
		return aggregateUseType;
	}

	Type *AdjustExprType::mutate( UnionInstType *aggregateUseType ) {
		return aggregateUseType;
	}

	Type *AdjustExprType::mutate( EnumInstType *aggregateUseType ) {
		return aggregateUseType;
	}

	Type *AdjustExprType::mutate( TraitInstType *aggregateUseType ) {
		return aggregateUseType;
	}

	Type *AdjustExprType::mutate( TypeInstType *typeInst ) {
		EqvClass eqvClass;
		if ( env.lookup( typeInst->get_name(), eqvClass ) ) {
			if ( eqvClass.data.kind == TypeDecl::Ftype ) {
				PointerType *pointerType = new PointerType( Type::Qualifiers(), typeInst );
				return pointerType;
			}
		} else if ( NamedTypeDecl *ntDecl = indexer.lookupType( typeInst->get_name() ) ) {
			if ( TypeDecl *tyDecl = dynamic_cast< TypeDecl* >( ntDecl ) ) {
				if ( tyDecl->get_kind() == TypeDecl::Ftype ) {
					PointerType *pointerType = new PointerType( Type::Qualifiers(), typeInst );
					return pointerType;
				} // if
			} // if
		} // if
		return typeInst;
	}

	Type *AdjustExprType::mutate( TupleType *tupleType ) {
		return tupleType;
	}

	Type *AdjustExprType::mutate( VarArgsType *varArgsType ) {
		return varArgsType;
	}

	Type *AdjustExprType::mutate( ZeroType *zeroType ) {
		return zeroType;
	}

	Type *AdjustExprType::mutate( OneType *oneType ) {
		return oneType;
	}
} // namespace ResolvExpr

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
