//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Indexer.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 21:37:33 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Thu Aug 17 16:08:40 2017
// Update Count     : 20
//

#include "Indexer.h"

#include <cassert>                 // for assert, strict_dynamic_cast
#include <iostream>                // for operator<<, basic_ostream, ostream
#include <string>                  // for string, operator<<, operator!=
#include <unordered_map>           // for operator!=, unordered_map<>::const...
#include <unordered_set>           // for unordered_set
#include <utility>                 // for pair, make_pair, move

#include "CodeGen/OperatorTable.h" // for isCtorDtor, isCtorDtorAssign
#include "Common/SemanticError.h"  // for SemanticError
#include "Common/utility.h"        // for cloneAll
#include "InitTweak/InitTweak.h"   // for isConstructor, isCopyFunction, isC...
#include "Mangler.h"               // for Mangler
#include "Parser/LinkageSpec.h"    // for isMangled, isOverridable, Spec
#include "ResolvExpr/typeops.h"    // for typesCompatible
#include "SynTree/Constant.h"      // for Constant
#include "SynTree/Declaration.h"   // for DeclarationWithType, FunctionDecl
#include "SynTree/Expression.h"    // for Expression, ImplicitCopyCtorExpr
#include "SynTree/Initializer.h"   // for Initializer
#include "SynTree/Statement.h"     // for CompoundStmt, Statement, ForStmt (...
#include "SynTree/Type.h"          // for Type, StructInstType, UnionInstType

#define debugPrint(x) if ( doDebug ) { std::cerr << x; }

namespace SymTab {
	typedef std::unordered_map< std::string, DeclarationWithType* > MangleTable;
	typedef std::unordered_map< std::string, MangleTable > IdTable;
	typedef std::unordered_map< std::string, NamedTypeDecl* > TypeTable;
	typedef std::unordered_map< std::string, StructDecl* > StructTable;
	typedef std::unordered_map< std::string, EnumDecl* > EnumTable;
	typedef std::unordered_map< std::string, UnionDecl* > UnionTable;
	typedef std::unordered_map< std::string, TraitDecl* > TraitTable;

	void dump( const IdTable &table, std::ostream &os ) {
		for ( IdTable::const_iterator id = table.begin(); id != table.end(); ++id ) {
			for ( MangleTable::const_iterator mangle = id->second.begin(); mangle != id->second.end(); ++mangle ) {
				os << mangle->second << std::endl;
			}
		}
	}

	template< typename Decl >
	void dump( const std::unordered_map< std::string, Decl* > &table, std::ostream &os ) {
		for ( typename std::unordered_map< std::string, Decl* >::const_iterator it = table.begin(); it != table.end(); ++it ) {
			os << it->second << std::endl;
		} // for
	}

	struct Indexer::Impl {
		Impl( unsigned long _scope ) : refCount(1), scope( _scope ), size( 0 ), base(),
				idTable(), typeTable(), structTable(), enumTable(), unionTable(), traitTable() {}
		Impl( unsigned long _scope, Indexer &&_base ) : refCount(1), scope( _scope ), size( 0 ), base( _base ),
				idTable(), typeTable(), structTable(), enumTable(), unionTable(), traitTable() {}
		unsigned long refCount;   ///< Number of references to these tables
		unsigned long scope;      ///< Scope these tables are associated with
		unsigned long size;       ///< Number of elements stored in this table
		const Indexer base;       ///< Base indexer this extends

		IdTable idTable;          ///< Identifier namespace
		TypeTable typeTable;      ///< Type namespace
		StructTable structTable;  ///< Struct namespace
		EnumTable enumTable;      ///< Enum namespace
		UnionTable unionTable;    ///< Union namespace
		TraitTable traitTable;    ///< Trait namespace
	};

	Indexer::Impl *Indexer::newRef( Indexer::Impl *toClone ) {
		if ( ! toClone ) return 0;

		// shorten the search chain by skipping empty links
		Indexer::Impl *ret = toClone->size == 0 ? toClone->base.tables : toClone;
		if ( ret ) { ++ret->refCount; }

		return ret;
	}

	void Indexer::deleteRef( Indexer::Impl *toFree ) {
		if ( ! toFree ) return;

		if ( --toFree->refCount == 0 ) delete toFree;
	}

	void Indexer::removeSpecialOverrides( const std::string &id, std::list< DeclarationWithType * > & out ) const {
		// only need to perform this step for constructors, destructors, and assignment functions
		if ( ! CodeGen::isCtorDtorAssign( id ) ) return;

		// helpful data structure
		struct ValueType {
			struct DeclBall {
				FunctionDecl * decl;
				bool isUserDefinedFunc; // properties for this particular decl
				bool isDefaultCtor;
				bool isDtor;
				bool isCopyFunc;
			};
			// properties for this type
			bool existsUserDefinedFunc = false;    // any user-defined function found
			bool existsUserDefinedCtor = false;    // any user-defined constructor found
			bool existsUserDefinedDtor = false;    // any user-defined destructor found
			bool existsUserDefinedCopyFunc = false;    // user-defined copy ctor found
			bool existsUserDefinedDefaultCtor = false; // user-defined default ctor found
			std::list< DeclBall > decls;

			// another FunctionDecl for the current type was found - determine
			// if it has special properties and update data structure accordingly
			ValueType & operator+=( FunctionDecl * function ) {
				bool isUserDefinedFunc = ! LinkageSpec::isOverridable( function->get_linkage() );
				bool isDefaultCtor = InitTweak::isDefaultConstructor( function );
				bool isDtor = InitTweak::isDestructor( function );
				bool isCopyFunc = InitTweak::isCopyFunction( function, function->get_name() );
				decls.push_back( DeclBall{ function, isUserDefinedFunc, isDefaultCtor, isDtor, isCopyFunc } );
				existsUserDefinedFunc = existsUserDefinedFunc || isUserDefinedFunc;
				existsUserDefinedCtor = existsUserDefinedCtor || (isUserDefinedFunc && CodeGen::isConstructor( function->get_name() ) );
				existsUserDefinedDtor = existsUserDefinedDtor || (isUserDefinedFunc && isDtor);
				existsUserDefinedCopyFunc = existsUserDefinedCopyFunc || (isUserDefinedFunc && isCopyFunc);
				existsUserDefinedDefaultCtor = existsUserDefinedDefaultCtor || (isUserDefinedFunc && isDefaultCtor);
				return *this;
			}
		}; // ValueType

		std::list< DeclarationWithType * > copy;
		copy.splice( copy.end(), out );

		// organize discovered declarations by type
		std::unordered_map< std::string, ValueType > funcMap;
		for ( DeclarationWithType * decl : copy ) {
			if ( FunctionDecl * function = dynamic_cast< FunctionDecl * >( decl ) ) {
				std::list< DeclarationWithType * > & params = function->get_functionType()->get_parameters();
				assert( ! params.empty() );
				// use base type of pointer, so that qualifiers on the pointer type aren't considered.
				Type * base = InitTweak::getPointerBase( params.front()->get_type() );
				assert( base );
				funcMap[ Mangler::mangle( base ) ] += function;
			} else {
				out.push_back( decl );
			}
		}

		// if a type contains user defined ctor/dtor/assign, then special rules trigger, which determine
		// the set of ctor/dtor/assign that are seen by the requester. In particular, if the user defines
		// a default ctor, then the generated default ctor should never be seen, likewise for copy ctor
		// and dtor. If the user defines any ctor/dtor, then no generated field ctors should be seen.
		// If the user defines any ctor then the generated default ctor should not be seen (intrinsic default
		// ctor must be overridden exactly).
		for ( std::pair< const std::string, ValueType > & pair : funcMap ) {
			ValueType & val = pair.second;
			for ( ValueType::DeclBall ball : val.decls ) {
				bool noUserDefinedFunc = ! val.existsUserDefinedFunc;
				bool isUserDefinedFunc = ball.isUserDefinedFunc;
				bool isAcceptableDefaultCtor = (! val.existsUserDefinedCtor || (! val.existsUserDefinedDefaultCtor && ball.decl->get_linkage() == LinkageSpec::Intrinsic)) && ball.isDefaultCtor; // allow default constructors only when no user-defined constructors exist, except in the case of intrinsics, which require exact overrides
				bool isAcceptableCopyFunc = ! val.existsUserDefinedCopyFunc && ball.isCopyFunc; // handles copy ctor and assignment operator
				bool isAcceptableDtor = ! val.existsUserDefinedDtor && ball.isDtor;
				if ( noUserDefinedFunc || isUserDefinedFunc || isAcceptableDefaultCtor || isAcceptableCopyFunc || isAcceptableDtor ) {
					// decl conforms to the rules described above, so it should be seen by the requester
					out.push_back( ball.decl );
				}
			}
		}
	}

	void Indexer::makeWritable() {
		if ( ! tables ) {
			// create indexer if not yet set
			tables = new Indexer::Impl( scope );
		} else if ( tables->refCount > 1 || tables->scope != scope ) {
			// make this indexer the base of a fresh indexer at the current scope
			tables = new Indexer::Impl( scope, std::move( *this ) );
		}
	}

	Indexer::Indexer() : tables( 0 ), scope( 0 ) {}

	Indexer::Indexer( const Indexer &that ) : doDebug( that.doDebug ), tables( newRef( that.tables ) ), scope( that.scope ) {}

	Indexer::Indexer( Indexer &&that ) : doDebug( that.doDebug ), tables( that.tables ), scope( that.scope ) {
		that.tables = 0;
	}

	Indexer::~Indexer() {
		deleteRef( tables );
	}

	Indexer& Indexer::operator= ( const Indexer &that ) {
		deleteRef( tables );

		tables = newRef( that.tables );
		scope = that.scope;
		doDebug = that.doDebug;

		return *this;
	}

	Indexer& Indexer::operator= ( Indexer &&that ) {
		deleteRef( tables );

		tables = that.tables;
		scope = that.scope;
		doDebug = that.doDebug;

		that.tables = 0;

		return *this;
	}

	void Indexer::lookupId( const std::string &id, std::list< DeclarationWithType* > &out ) const {
		std::unordered_set< std::string > foundMangleNames;

		Indexer::Impl *searchTables = tables;
		while ( searchTables ) {

			IdTable::const_iterator decls = searchTables->idTable.find( id );
			if ( decls != searchTables->idTable.end() ) {
				const MangleTable &mangleTable = decls->second;
				for ( MangleTable::const_iterator decl = mangleTable.begin(); decl != mangleTable.end(); ++decl ) {
					// mark the mangled name as found, skipping this insertion if a declaration for that name has already been found
					if ( foundMangleNames.insert( decl->first ).second == false ) continue;

					out.push_back( decl->second );
				}
			}

			// get declarations from base indexers
			searchTables = searchTables->base.tables;
		}

		// some special functions, e.g. constructors and destructors
		// remove autogenerated functions when they are defined so that
		// they can never be matched
		removeSpecialOverrides( id, out );
	}

	NamedTypeDecl *Indexer::lookupType( const std::string &id ) const {
		if ( ! tables ) return 0;

		TypeTable::const_iterator ret = tables->typeTable.find( id );
		return ret != tables->typeTable.end() ? ret->second : tables->base.lookupType( id );
	}

	StructDecl *Indexer::lookupStruct( const std::string &id ) const {
		if ( ! tables ) return 0;

		StructTable::const_iterator ret = tables->structTable.find( id );
		return ret != tables->structTable.end() ? ret->second : tables->base.lookupStruct( id );
	}

	EnumDecl *Indexer::lookupEnum( const std::string &id ) const {
		if ( ! tables ) return 0;

		EnumTable::const_iterator ret = tables->enumTable.find( id );
		return ret != tables->enumTable.end() ? ret->second : tables->base.lookupEnum( id );
	}

	UnionDecl *Indexer::lookupUnion( const std::string &id ) const {
		if ( ! tables ) return 0;

		UnionTable::const_iterator ret = tables->unionTable.find( id );
		return ret != tables->unionTable.end() ? ret->second : tables->base.lookupUnion( id );
	}

	TraitDecl *Indexer::lookupTrait( const std::string &id ) const {
		if ( ! tables ) return 0;

		TraitTable::const_iterator ret = tables->traitTable.find( id );
		return ret != tables->traitTable.end() ? ret->second : tables->base.lookupTrait( id );
	}

	DeclarationWithType *Indexer::lookupIdAtScope( const std::string &id, const std::string &mangleName, unsigned long scope ) const {
		if ( ! tables ) return 0;
		if ( tables->scope < scope ) return 0;

		IdTable::const_iterator decls = tables->idTable.find( id );
		if ( decls != tables->idTable.end() ) {
			const MangleTable &mangleTable = decls->second;
			MangleTable::const_iterator decl = mangleTable.find( mangleName );
			if ( decl != mangleTable.end() ) return decl->second;
		}

		return tables->base.lookupIdAtScope( id, mangleName, scope );
	}

	bool Indexer::hasIncompatibleCDecl( const std::string &id, const std::string &mangleName, unsigned long scope ) const {
		if ( ! tables ) return false;
		if ( tables->scope < scope ) return false;

		IdTable::const_iterator decls = tables->idTable.find( id );
		if ( decls != tables->idTable.end() ) {
			const MangleTable &mangleTable = decls->second;
			for ( MangleTable::const_iterator decl = mangleTable.begin(); decl != mangleTable.end(); ++decl ) {
				// check for C decls with the same name, skipping those with a compatible type (by mangleName)
				if ( ! LinkageSpec::isMangled( decl->second->get_linkage() ) && decl->first != mangleName ) return true;
			}
		}

		return tables->base.hasIncompatibleCDecl( id, mangleName, scope );
	}

	bool Indexer::hasCompatibleCDecl( const std::string &id, const std::string &mangleName, unsigned long scope ) const {
		if ( ! tables ) return false;
		if ( tables->scope < scope ) return false;

		IdTable::const_iterator decls = tables->idTable.find( id );
		if ( decls != tables->idTable.end() ) {
			const MangleTable &mangleTable = decls->second;
			for ( MangleTable::const_iterator decl = mangleTable.begin(); decl != mangleTable.end(); ++decl ) {
				// check for C decls with the same name, skipping
				// those with an incompatible type (by mangleName)
				if ( ! LinkageSpec::isMangled( decl->second->get_linkage() ) && decl->first == mangleName ) return true;
			}
		}

		return tables->base.hasCompatibleCDecl( id, mangleName, scope );
	}

	NamedTypeDecl *Indexer::lookupTypeAtScope( const std::string &id, unsigned long scope ) const {
		if ( ! tables ) return 0;
		if ( tables->scope < scope ) return 0;

		TypeTable::const_iterator ret = tables->typeTable.find( id );
		return ret != tables->typeTable.end() ? ret->second : tables->base.lookupTypeAtScope( id, scope );
	}

	StructDecl *Indexer::lookupStructAtScope( const std::string &id, unsigned long scope ) const {
		if ( ! tables ) return 0;
		if ( tables->scope < scope ) return 0;

		StructTable::const_iterator ret = tables->structTable.find( id );
		return ret != tables->structTable.end() ? ret->second : tables->base.lookupStructAtScope( id, scope );
	}

	EnumDecl *Indexer::lookupEnumAtScope( const std::string &id, unsigned long scope ) const {
		if ( ! tables ) return 0;
		if ( tables->scope < scope ) return 0;

		EnumTable::const_iterator ret = tables->enumTable.find( id );
		return ret != tables->enumTable.end() ? ret->second : tables->base.lookupEnumAtScope( id, scope );
	}

	UnionDecl *Indexer::lookupUnionAtScope( const std::string &id, unsigned long scope ) const {
		if ( ! tables ) return 0;
		if ( tables->scope < scope ) return 0;

		UnionTable::const_iterator ret = tables->unionTable.find( id );
		return ret != tables->unionTable.end() ? ret->second : tables->base.lookupUnionAtScope( id, scope );
	}

	TraitDecl *Indexer::lookupTraitAtScope( const std::string &id, unsigned long scope ) const {
		if ( ! tables ) return 0;
		if ( tables->scope < scope ) return 0;

		TraitTable::const_iterator ret = tables->traitTable.find( id );
		return ret != tables->traitTable.end() ? ret->second : tables->base.lookupTraitAtScope( id, scope );
	}

	bool addedIdConflicts( DeclarationWithType *existing, DeclarationWithType *added ) {
		// if we're giving the same name mangling to things of different types then there is something wrong
		assert( (dynamic_cast<ObjectDecl*>( added ) && dynamic_cast<ObjectDecl*>( existing ) )
			|| (dynamic_cast<FunctionDecl*>( added ) && dynamic_cast<FunctionDecl*>( existing ) ) );

		if ( LinkageSpec::isOverridable( existing->get_linkage() ) ) {
			// new definition shadows the autogenerated one, even at the same scope
			return false;
		} else if ( LinkageSpec::isMangled( added->get_linkage() ) || ResolvExpr::typesCompatible( added->get_type(), existing->get_type(), Indexer() ) ) {
			// typesCompatible doesn't really do the right thing here. When checking compatibility of function types,
			// we should ignore outermost pointer qualifiers, except _Atomic?
			FunctionDecl *newentry = dynamic_cast< FunctionDecl* >( added );
			FunctionDecl *oldentry = dynamic_cast< FunctionDecl* >( existing );
			if ( newentry && oldentry ) {
				if ( newentry->get_statements() && oldentry->get_statements() ) {
					throw SemanticError( "duplicate function definition for ", added );
				} // if
			} else {
				// two objects with the same mangled name defined in the same scope.
				// both objects must be marked extern or both must be intrinsic for this to be okay
				// xxx - perhaps it's actually if either is intrinsic then this is okay?
				//       might also need to be same storage class?
				ObjectDecl *newobj = dynamic_cast< ObjectDecl* >( added );
				ObjectDecl *oldobj = dynamic_cast< ObjectDecl* >( existing );
				if ( ! newobj->get_storageClasses().is_extern && ! oldobj->get_storageClasses().is_extern ) {
					throw SemanticError( "duplicate object definition for ", added );
				} // if
			} // if
		} else {
			throw SemanticError( "duplicate definition for ", added );
		} // if

		return true;
	}

	void Indexer::addId( DeclarationWithType *decl ) {
		debugPrint( "Adding Id " << decl->name << std::endl );
		makeWritable();

		const std::string &name = decl->name;
		std::string mangleName;
		if ( LinkageSpec::isOverridable( decl->linkage ) ) {
			// mangle the name without including the appropriate suffix, so overridable routines are placed into the
			// same "bucket" as their user defined versions.
			mangleName = Mangler::mangle( decl, false );
		} else {
			mangleName = Mangler::mangle( decl );
		} // if

		// this ensures that no two declarations with the same unmangled name at the same scope both have C linkage
		if ( ! LinkageSpec::isMangled( decl->linkage ) ) {
			// NOTE this is broken in Richard's original code in such a way that it never triggers (it
			// doesn't check decls that have the same manglename, and all C-linkage decls are defined to
			// have their name as their manglename, hence the error can never trigger).
			// The code here is closer to correct, but name mangling would have to be completely
			// isomorphic to C type-compatibility, which it may not be.
			if ( hasIncompatibleCDecl( name, mangleName, scope ) ) {
				throw SemanticError( "conflicting overload of C function ", decl );
			}
		} else {
			// Check that a Cforall declaration doesn't overload any C declaration
			if ( hasCompatibleCDecl( name, mangleName, scope ) ) {
				throw SemanticError( "Cforall declaration hides C function ", decl );
			}
		}

		// Skip repeat declarations of the same identifier
		DeclarationWithType *existing = lookupIdAtScope( name, mangleName, scope );
		if ( existing && addedIdConflicts( existing, decl ) ) return;

		// add to indexer
		tables->idTable[ name ][ mangleName ] = decl;
		++tables->size;
	}

	bool addedTypeConflicts( NamedTypeDecl *existing, NamedTypeDecl *added ) {
		if ( existing->get_base() == 0 ) {
			return false;
		} else if ( added->get_base() == 0 ) {
			return true;
		} else {
			throw SemanticError( "redeclaration of ", added );
		}
	}

	void Indexer::addType( NamedTypeDecl *decl ) {
		debugPrint( "Adding type " << decl->name << std::endl );
		makeWritable();

		const std::string &id = decl->get_name();
		TypeTable::iterator existing = tables->typeTable.find( id );
		if ( existing == tables->typeTable.end() ) {
			NamedTypeDecl *parent = tables->base.lookupTypeAtScope( id, scope );
			if ( ! parent || ! addedTypeConflicts( parent, decl ) ) {
				tables->typeTable.insert( existing, std::make_pair( id, decl ) );
				++tables->size;
			}
		} else {
			if ( ! addedTypeConflicts( existing->second, decl ) ) {
				existing->second = decl;
			}
		}
	}

	bool addedDeclConflicts( AggregateDecl *existing, AggregateDecl *added ) {
		if ( existing->get_members().empty() ) {
			return false;
		} else if ( ! added->get_members().empty() ) {
			throw SemanticError( "redeclaration of ", added );
		} // if
		return true;
	}

	void Indexer::addStruct( const std::string &id ) {
		debugPrint( "Adding fwd decl for struct " << id << std::endl );
		addStruct( new StructDecl( id ) );
	}

	void Indexer::addStruct( StructDecl *decl ) {
		debugPrint( "Adding struct " << decl->name << std::endl );
		makeWritable();

		const std::string &id = decl->get_name();
		StructTable::iterator existing = tables->structTable.find( id );
		if ( existing == tables->structTable.end() ) {
			StructDecl *parent = tables->base.lookupStructAtScope( id, scope );
			if ( ! parent || ! addedDeclConflicts( parent, decl ) ) {
				tables->structTable.insert( existing, std::make_pair( id, decl ) );
				++tables->size;
			}
		} else {
			if ( ! addedDeclConflicts( existing->second, decl ) ) {
				existing->second = decl;
			}
		}
	}

	void Indexer::addEnum( EnumDecl *decl ) {
		debugPrint( "Adding enum " << decl->name << std::endl );
		makeWritable();

		const std::string &id = decl->get_name();
		EnumTable::iterator existing = tables->enumTable.find( id );
		if ( existing == tables->enumTable.end() ) {
			EnumDecl *parent = tables->base.lookupEnumAtScope( id, scope );
			if ( ! parent || ! addedDeclConflicts( parent, decl ) ) {
				tables->enumTable.insert( existing, std::make_pair( id, decl ) );
				++tables->size;
			}
		} else {
			if ( ! addedDeclConflicts( existing->second, decl ) ) {
				existing->second = decl;
			}
		}
	}

	void Indexer::addUnion( const std::string &id ) {
		debugPrint( "Adding fwd decl for union " << id << std::endl );
		addUnion( new UnionDecl( id ) );
	}

	void Indexer::addUnion( UnionDecl *decl ) {
		debugPrint( "Adding union " << decl->name << std::endl );
		makeWritable();

		const std::string &id = decl->get_name();
		UnionTable::iterator existing = tables->unionTable.find( id );
		if ( existing == tables->unionTable.end() ) {
			UnionDecl *parent = tables->base.lookupUnionAtScope( id, scope );
			if ( ! parent || ! addedDeclConflicts( parent, decl ) ) {
				tables->unionTable.insert( existing, std::make_pair( id, decl ) );
				++tables->size;
			}
		} else {
			if ( ! addedDeclConflicts( existing->second, decl ) ) {
				existing->second = decl;
			}
		}
	}

	void Indexer::addTrait( TraitDecl *decl ) {
		debugPrint( "Adding trait " << decl->name << std::endl );
		makeWritable();

		const std::string &id = decl->get_name();
		TraitTable::iterator existing = tables->traitTable.find( id );
		if ( existing == tables->traitTable.end() ) {
			TraitDecl *parent = tables->base.lookupTraitAtScope( id, scope );
			if ( ! parent || ! addedDeclConflicts( parent, decl ) ) {
				tables->traitTable.insert( existing, std::make_pair( id, decl ) );
				++tables->size;
			}
		} else {
			if ( ! addedDeclConflicts( existing->second, decl ) ) {
				existing->second = decl;
			}
		}
	}

	void Indexer::addIds( const std::list< DeclarationWithType * > & decls ) {
		for ( auto d : decls ) {
			addId( d );
		}
	}

	void Indexer::addTypes( const std::list< TypeDecl * > & tds ) {
		for ( auto td : tds ) {
			addType( td );
			addIds( td->assertions );
		}
	}

	void Indexer::addFunctionType( FunctionType * ftype ) {
		addTypes( ftype->forall );
		addIds( ftype->returnVals );
		addIds( ftype->parameters );
	}

	void Indexer::enterScope() {
		++scope;

		if ( doDebug ) {
			std::cerr << "--- Entering scope " << scope << std::endl;
		}
	}

	void Indexer::leaveScope() {
		using std::cerr;

		assert( scope > 0 && "cannot leave initial scope" );
		if ( doDebug ) {
			cerr << "--- Leaving scope " << scope << " containing" << std::endl;
		}
		--scope;

		while ( tables && tables->scope > scope ) {
			if ( doDebug ) {
				dump( tables->idTable, cerr );
				dump( tables->typeTable, cerr );
				dump( tables->structTable, cerr );
				dump( tables->enumTable, cerr );
				dump( tables->unionTable, cerr );
				dump( tables->traitTable, cerr );
			}

			// swap tables for base table until we find one at an appropriate scope
			Indexer::Impl *base = newRef( tables->base.tables );
			deleteRef( tables );
			tables = base;
		}
	}

	void Indexer::print( std::ostream &os, int indent ) const {
	    using std::cerr;

		if ( tables ) {
			os << "--- scope " << tables->scope << " ---" << std::endl;

			os << "===idTable===" << std::endl;
			dump( tables->idTable, os );
			os << "===typeTable===" << std::endl;
			dump( tables->typeTable, os );
			os << "===structTable===" << std::endl;
			dump( tables->structTable, os );
			os << "===enumTable===" << std::endl;
			dump( tables->enumTable, os );
			os << "===unionTable===" << std::endl;
			dump( tables->unionTable, os );
			os << "===contextTable===" << std::endl;
			dump( tables->traitTable, os );

			tables->base.print( os, indent );
		} else {
			os << "--- end ---" << std::endl;
		}

	}
} // namespace SymTab

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
