//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// CommonType.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 06:59:27 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon Sep 25 15:18:17 2017
// Update Count     : 9
//

#include <cassert>                       // for strict_dynamic_cast
#include <map>                           // for _Rb_tree_const_iterator
#include <utility>                       // for pair

#include "ResolvExpr/TypeEnvironment.h"  // for OpenVarSet, AssertionSet
#include "SymTab/Indexer.h"              // for Indexer
#include "SynTree/Declaration.h"         // for TypeDecl, NamedTypeDecl (ptr...
#include "SynTree/Type.h"                // for BasicType, BasicType::Kind::...
#include "SynTree/Visitor.h"             // for Visitor
#include "Unify.h"                       // for unifyExact, bindVar, WidenMode
#include "typeops.h"                     // for isFtype

// #define DEBUG

namespace ResolvExpr {
	class CommonType : public Visitor {
	  public:
		CommonType( Type *type2, bool widenFirst, bool widenSecond, const SymTab::Indexer &indexer, TypeEnvironment &env, const OpenVarSet &openVars );
		Type *get_result() const { return result; }
	  private:
		virtual void visit( VoidType *voidType );
		virtual void visit( BasicType *basicType );
		virtual void visit( PointerType *pointerType );
		virtual void visit( ArrayType *arrayType );
		virtual void visit( ReferenceType *refType );
		virtual void visit( FunctionType *functionType );
		virtual void visit( StructInstType *aggregateUseType );
		virtual void visit( UnionInstType *aggregateUseType );
		virtual void visit( EnumInstType *aggregateUseType );
		virtual void visit( TraitInstType *aggregateUseType );
		virtual void visit( TypeInstType *aggregateUseType );
		virtual void visit( TupleType *tupleType );
		virtual void visit( VarArgsType *varArgsType );
		virtual void visit( ZeroType *zeroType );
		virtual void visit( OneType *oneType );

		template< typename Pointer > void getCommonWithVoidPointer( Pointer* voidPointer, Pointer* otherPointer );
		template< typename RefType > void handleRefType( RefType *inst, Type *other );

		Type *result;
		Type *type2;				// inherited
		bool widenFirst, widenSecond;
		const SymTab::Indexer &indexer;
		TypeEnvironment &env;
		const OpenVarSet &openVars;
	};

	Type * handleReference( Type * t1, Type * t2, bool widenFirst, bool widenSecond, const SymTab::Indexer &indexer, TypeEnvironment & env, const OpenVarSet &openVars ) {
		Type * common = nullptr;
		AssertionSet have, need;
		OpenVarSet newOpen( openVars );
		// need unify to bind type variables
		if ( unify( t1, t2, env, have, need, newOpen, indexer, common ) ) {
			// std::cerr << "unify success: " << widenFirst << " " << widenSecond << std::endl;
			if ( (widenFirst || t2->get_qualifiers() <= t1->get_qualifiers()) && (widenSecond || t1->get_qualifiers() <= t2->get_qualifiers()) ) {
				// std::cerr << "widen okay" << std::endl;
				common->get_qualifiers() |= t1->get_qualifiers();
				common->get_qualifiers() |= t2->get_qualifiers();
				return common;
			}
		}
		// std::cerr << "exact unify failed: " << t1 << " " << t2 << std::endl;
		return nullptr;
	}

	Type *commonType( Type *type1, Type *type2, bool widenFirst, bool widenSecond, const SymTab::Indexer &indexer, TypeEnvironment &env, const OpenVarSet &openVars ) {
		CommonType visitor( type2, widenFirst, widenSecond, indexer, env, openVars );

		int depth1 = type1->referenceDepth();
		int depth2 = type2->referenceDepth();
		if ( depth1 > 0 || depth2 > 0 ) {
			int diff = depth1-depth2;
			// TODO: should it be possible for commonType to generate complicated conversions? I would argue no, only conversions that involve types of the same reference level or a difference of 1 should be allowed.
			if ( diff > 1 || diff < -1 ) return nullptr;

			// special case where one type has a reference depth of 1 larger than the other
			if ( diff > 0 || diff < 0 ) {
				// std::cerr << "reference depth diff: " << diff << std::endl;
				Type * result = nullptr;
				ReferenceType * ref1 = dynamic_cast< ReferenceType * >( type1 );
				ReferenceType * ref2 = dynamic_cast< ReferenceType * >( type2 );
				if ( diff > 0 ) {
					// deeper on the left
					assert( ref1 );
					result = handleReference( ref1->base, type2, widenFirst, widenSecond, indexer, env, openVars );
				} else {
					// deeper on the right
					assert( ref2 );
					result = handleReference( type1, ref2->base, widenFirst, widenSecond, indexer, env, openVars );
				}
				if ( result && ref1 ) {
					// formal is reference, so result should be reference
					// std::cerr << "formal is reference; result should be reference" << std::endl;
					result = new ReferenceType( ref1->get_qualifiers(), result );
				}
				// std::cerr << "common type of reference [" << type1 << "] and [" << type2 << "] is [" << result << "]" << std::endl;
				return result;
			}
			// otherwise, both are reference types of the same depth and this is handled by the CommonType visitor.
		}

		type1->accept( visitor );
		Type *result = visitor.get_result();
		if ( ! result ) {
			// this appears to be handling for opaque type declarations
			if ( widenSecond ) {
				if ( TypeInstType *inst = dynamic_cast< TypeInstType* >( type2 ) ) {
					if ( NamedTypeDecl *nt = indexer.lookupType( inst->get_name() ) ) {
						TypeDecl *type = strict_dynamic_cast< TypeDecl* >( nt );
						if ( type->get_base() ) {
							Type::Qualifiers tq1 = type1->get_qualifiers(), tq2 = type2->get_qualifiers();
							AssertionSet have, need;
							OpenVarSet newOpen( openVars );
							type1->get_qualifiers() = Type::Qualifiers();
							type->get_base()->get_qualifiers() = tq1;
							if ( unifyExact( type1, type->get_base(), env, have, need, newOpen, indexer ) ) {
								result = type1->clone();
								result->get_qualifiers() = tq1 | tq2;
							} // if
							type1->get_qualifiers() = tq1;
							type->get_base()->get_qualifiers() = Type::Qualifiers();
						} // if
					} // if
				} // if
			} // if
		} // if
#ifdef DEBUG
		std::cerr << "============= commonType" << std::endl << "type1 is ";
		type1->print( std::cerr );
		std::cerr << " type2 is ";
		type2->print( std::cerr );
		if ( result ) {
			std::cerr << " common type is ";
			result->print( std::cerr );
		} else {
			std::cerr << " no common type";
		} // if
		std::cerr << std::endl;
#endif
		return result;
	}

	static const BasicType::Kind combinedType[ BasicType::NUMBER_OF_BASIC_TYPES ][ BasicType::NUMBER_OF_BASIC_TYPES ] =
	{
/* 		Bool		Char	SignedChar	UnsignedChar	ShortSignedInt	ShortUnsignedInt	SignedInt	UnsignedInt	LongSignedInt	LongUnsignedInt	LongLongSignedInt	LongLongUnsignedInt	Float	Double	LongDouble	FloatComplex	DoubleComplex	LongDoubleComplex	FloatImaginary	DoubleImaginary	LongDoubleImaginary   SignedInt128   UnsignedInt128 */
		/* Bool */ 	{ BasicType::Bool,		BasicType::Char,	BasicType::SignedChar,	BasicType::UnsignedChar,	BasicType::ShortSignedInt,	BasicType::ShortUnsignedInt,	BasicType::SignedInt,	BasicType::UnsignedInt,	BasicType::LongSignedInt,	BasicType::LongUnsignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongUnsignedInt,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::SignedInt128,	BasicType::UnsignedInt128, },
		/* Char */ 	{ BasicType::Char,		BasicType::Char,	BasicType::UnsignedChar,	BasicType::UnsignedChar,	BasicType::ShortSignedInt,	BasicType::ShortUnsignedInt,	BasicType::SignedInt,	BasicType::UnsignedInt,	BasicType::LongSignedInt,	BasicType::LongUnsignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongUnsignedInt,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::SignedInt128,	BasicType::UnsignedInt128, },
		/* SignedChar */ 	{ BasicType::SignedChar,	BasicType::UnsignedChar,	BasicType::SignedChar,	BasicType::UnsignedChar,	BasicType::ShortSignedInt,	BasicType::ShortUnsignedInt,	BasicType::SignedInt,	BasicType::UnsignedInt,	BasicType::LongSignedInt,	BasicType::LongUnsignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongUnsignedInt,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::SignedInt128,	BasicType::UnsignedInt128, },
		/* UnsignedChar */ 	{ BasicType::UnsignedChar,	BasicType::UnsignedChar,	BasicType::UnsignedChar,	BasicType::UnsignedChar,	BasicType::ShortSignedInt,	BasicType::ShortUnsignedInt,	BasicType::SignedInt,	BasicType::UnsignedInt,	BasicType::LongSignedInt,	BasicType::LongUnsignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongUnsignedInt,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::SignedInt128,	BasicType::UnsignedInt128, },
		/* ShortSignedInt */ 	{ BasicType::ShortSignedInt,	BasicType::ShortSignedInt,	BasicType::ShortSignedInt,	BasicType::ShortSignedInt,	BasicType::ShortSignedInt,	BasicType::ShortUnsignedInt,	BasicType::SignedInt,	BasicType::UnsignedInt,	BasicType::LongSignedInt,	BasicType::LongUnsignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongUnsignedInt,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::SignedInt128,	BasicType::UnsignedInt128, },
		/* ShortUnsignedInt */ 	{ BasicType::ShortUnsignedInt,	BasicType::ShortUnsignedInt,	BasicType::ShortUnsignedInt,	BasicType::ShortUnsignedInt,	BasicType::ShortUnsignedInt,	BasicType::ShortUnsignedInt,	BasicType::SignedInt,	BasicType::UnsignedInt,	BasicType::LongSignedInt,	BasicType::LongUnsignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongUnsignedInt,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::SignedInt128,	BasicType::UnsignedInt128, },
		/* SignedInt */ 	{ BasicType::SignedInt,		BasicType::SignedInt,	BasicType::SignedInt,	BasicType::SignedInt,	BasicType::SignedInt,	BasicType::SignedInt,	BasicType::SignedInt,	BasicType::UnsignedInt,	BasicType::LongSignedInt,	BasicType::LongUnsignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongUnsignedInt,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::SignedInt128,	BasicType::UnsignedInt128, },
		/* UnsignedInt */ 	{ BasicType::UnsignedInt,		BasicType::UnsignedInt,	BasicType::UnsignedInt,	BasicType::UnsignedInt,	BasicType::UnsignedInt,	BasicType::UnsignedInt,	BasicType::UnsignedInt,	BasicType::UnsignedInt,	BasicType::LongUnsignedInt,	BasicType::LongUnsignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongUnsignedInt,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::SignedInt128,	BasicType::UnsignedInt128, },
		/* LongSignedInt */ 	{ BasicType::LongSignedInt,		BasicType::LongSignedInt,	BasicType::LongSignedInt,	BasicType::LongSignedInt,	BasicType::LongSignedInt,	BasicType::LongSignedInt,	BasicType::LongSignedInt,	BasicType::LongUnsignedInt,	BasicType::LongSignedInt,	BasicType::LongUnsignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongUnsignedInt,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::SignedInt128,	BasicType::UnsignedInt128, },
		/* LongUnsignedInt */ 	{ BasicType::LongUnsignedInt,	BasicType::LongUnsignedInt,	BasicType::LongUnsignedInt,	BasicType::LongUnsignedInt,	BasicType::LongUnsignedInt,	BasicType::LongUnsignedInt,	BasicType::LongUnsignedInt,	BasicType::LongUnsignedInt,	BasicType::LongUnsignedInt,	BasicType::LongUnsignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongUnsignedInt,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::SignedInt128,	BasicType::UnsignedInt128, },
		/* LongLongSignedInt */ 	{ BasicType::LongLongSignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongSignedInt,	BasicType::LongLongUnsignedInt,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::SignedInt128,	BasicType::UnsignedInt128, },
		/* LongLongUnsignedInt */ 	{ BasicType::LongLongUnsignedInt,	BasicType::LongLongUnsignedInt,	BasicType::LongLongUnsignedInt,	BasicType::LongLongUnsignedInt,	BasicType::LongLongUnsignedInt,	BasicType::LongLongUnsignedInt,	BasicType::LongLongUnsignedInt,	BasicType::LongLongUnsignedInt,	BasicType::LongLongUnsignedInt,	BasicType::LongLongUnsignedInt,	BasicType::LongLongUnsignedInt,	BasicType::LongLongUnsignedInt,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::SignedInt128,	BasicType::UnsignedInt128, },
		/* Float */ 	{ BasicType::Float,	BasicType::Float,	BasicType::Float,	BasicType::Float,	BasicType::Float,	BasicType::Float,	BasicType::Float,	BasicType::Float,	BasicType::Float,	BasicType::Float,	BasicType::Float,	BasicType::Float,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::Float,	BasicType::Float, },
		/* Double */ 	{ BasicType::Double,	BasicType::Double,	BasicType::Double,	BasicType::Double,	BasicType::Double,	BasicType::Double,	BasicType::Double,	BasicType::Double,	BasicType::Double,	BasicType::Double,	BasicType::Double,	BasicType::Double,	BasicType::Double,	BasicType::Double,	BasicType::LongDouble,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::Double,	BasicType::Double, },
		/* LongDouble */ 	{ BasicType::LongDouble,		BasicType::LongDouble,	BasicType::LongDouble,	BasicType::LongDouble,	BasicType::LongDouble,	BasicType::LongDouble,	BasicType::LongDouble,	BasicType::LongDouble,	BasicType::LongDouble,	BasicType::LongDouble,	BasicType::LongDouble,	BasicType::LongDouble,	BasicType::LongDouble,	BasicType::LongDouble,	BasicType::LongDouble,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDouble,	BasicType::LongDouble, },
		/* FloatComplex */ 	{ BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::FloatComplex, },
		/* DoubleComplex */ 	{ BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex, },
		/* LongDoubleComplex */ 	{ BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex, },
		/* FloatImaginary */ 	{ BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatImaginary,	BasicType::DoubleImaginary,	BasicType::LongDoubleImaginary,	BasicType::FloatImaginary,	BasicType::FloatImaginary, },
		/* DoubleImaginary */ 	{ BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::DoubleComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::DoubleImaginary,	BasicType::DoubleImaginary,	BasicType::LongDoubleImaginary,	BasicType::DoubleImaginary,	BasicType::DoubleImaginary, },
		/* LongDoubleImaginary */ 	{ BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleComplex,	BasicType::LongDoubleImaginary,	BasicType::LongDoubleImaginary,	BasicType::LongDoubleImaginary },
		/* SignedInt128 */ 	{ BasicType::SignedInt128,	BasicType::SignedInt128,	BasicType::SignedInt128,	BasicType::SignedInt128,	BasicType::SignedInt128,	BasicType::SignedInt128,	BasicType::SignedInt128,	BasicType::SignedInt128,	BasicType::SignedInt128,	BasicType::SignedInt128,	BasicType::SignedInt128,	BasicType::SignedInt128,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::SignedInt128,	BasicType::UnsignedInt128, },
		/* UnsignedInt128 */ 	{ BasicType::UnsignedInt128,	BasicType::UnsignedInt128,	BasicType::UnsignedInt128,	BasicType::UnsignedInt128,	BasicType::UnsignedInt128,	BasicType::UnsignedInt128,	BasicType::UnsignedInt128,	BasicType::UnsignedInt128,	BasicType::UnsignedInt128,	BasicType::UnsignedInt128,	BasicType::UnsignedInt128,	BasicType::UnsignedInt128,	BasicType::Float,	BasicType::Double,	BasicType::LongDouble,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::FloatComplex,	BasicType::DoubleComplex,	BasicType::LongDoubleComplex,	BasicType::UnsignedInt128,	BasicType::UnsignedInt128, },
	};

	CommonType::CommonType( Type *type2, bool widenFirst, bool widenSecond, const SymTab::Indexer &indexer, TypeEnvironment &env, const OpenVarSet &openVars )
		: result( 0 ), type2( type2 ), widenFirst( widenFirst ), widenSecond( widenSecond ), indexer( indexer ), env( env ), openVars( openVars ) {
	}

	void CommonType::visit( VoidType * ) {}

	void CommonType::visit( BasicType *basicType ) {
		if ( BasicType *otherBasic = dynamic_cast< BasicType* >( type2 ) ) {
			BasicType::Kind newType = combinedType[ basicType->get_kind() ][ otherBasic->get_kind() ];
			if ( ( ( newType == basicType->get_kind() && basicType->get_qualifiers() >= otherBasic->get_qualifiers() ) || widenFirst ) && ( ( newType == otherBasic->get_kind() && basicType->get_qualifiers() <= otherBasic->get_qualifiers() ) || widenSecond ) ) {
				result = new BasicType( basicType->get_qualifiers() | otherBasic->get_qualifiers(), newType );
			} // if
		} else if ( dynamic_cast< EnumInstType * > ( type2 ) || dynamic_cast< ZeroType* >( type2 ) || dynamic_cast< OneType* >( type2 ) ) {
			// use signed int in lieu of the enum/zero/one type
			BasicType::Kind newType = combinedType[ basicType->get_kind() ][ BasicType::SignedInt ];
			if ( ( ( newType == basicType->get_kind() && basicType->get_qualifiers() >= type2->get_qualifiers() ) || widenFirst ) && ( ( newType != basicType->get_kind() && basicType->get_qualifiers() <= type2->get_qualifiers() ) || widenSecond ) ) {
				result = new BasicType( basicType->get_qualifiers() | type2->get_qualifiers(), newType );
			} // if
		} // if
	}

	template< typename Pointer >
	void CommonType::getCommonWithVoidPointer( Pointer* voidPointer, Pointer* otherPointer ) {
		if ( TypeInstType* var = dynamic_cast< TypeInstType* >( otherPointer->get_base() ) ) {
			OpenVarSet::const_iterator entry = openVars.find( var->get_name() );
			if ( entry != openVars.end() ) {
				AssertionSet need, have;
				WidenMode widen( widenFirst, widenSecond );
				if ( entry != openVars.end() && ! bindVar(var, voidPointer->get_base(), entry->second, env, need, have, openVars, widen, indexer ) ) return;
			}
		}
		result = voidPointer->clone();
		result->get_qualifiers() |= otherPointer->get_qualifiers();
	}

	void CommonType::visit( PointerType *pointerType ) {
		if ( PointerType *otherPointer = dynamic_cast< PointerType* >( type2 ) ) {
			// std::cerr << "commonType: two pointers: " << pointerType << " / " << otherPointer << std::endl;
			if ( widenFirst && dynamic_cast< VoidType* >( otherPointer->get_base() ) && ! isFtype(pointerType->get_base()) ) {
				getCommonWithVoidPointer( otherPointer, pointerType );
			} else if ( widenSecond && dynamic_cast< VoidType* >( pointerType->get_base() ) && ! isFtype(otherPointer->get_base()) ) {
				getCommonWithVoidPointer( pointerType, otherPointer );
			} else if ( ( pointerType->get_base()->get_qualifiers() >= otherPointer->get_base()->get_qualifiers() || widenFirst )
					   && ( pointerType->get_base()->get_qualifiers() <= otherPointer->get_base()->get_qualifiers() || widenSecond ) ) {
				// std::cerr << "middle case" << std::endl;
				Type::Qualifiers tq1 = pointerType->get_base()->get_qualifiers(), tq2 = otherPointer->get_base()->get_qualifiers();
				pointerType->get_base()->get_qualifiers() = Type::Qualifiers();
				otherPointer->get_base()->get_qualifiers() = Type::Qualifiers();
				AssertionSet have, need;
				OpenVarSet newOpen( openVars );
				if ( unifyExact( pointerType->get_base(), otherPointer->get_base(), env, have, need, newOpen, indexer ) ) {
					// std::cerr << "unifyExact success" << std::endl;
					if ( tq1 < tq2 ) {
						result = pointerType->clone();
					} else {
						result = otherPointer->clone();
					} // if
					result->get_qualifiers() = tq1 | tq2;
				} else {
					/// std::cerr << "place for ptr-to-type" << std::endl;
				} // if
				pointerType->get_base()->get_qualifiers() = tq1;
				otherPointer->get_base()->get_qualifiers() = tq2;
			} // if
		} else if ( widenSecond && dynamic_cast< ZeroType* >( type2 ) ) {
			result = pointerType->clone();
			result->get_qualifiers() |= type2->get_qualifiers();
		} // if
	}

	void CommonType::visit( ArrayType * ) {}

	void CommonType::visit( ReferenceType *refType ) {
		if ( ReferenceType *otherRef = dynamic_cast< ReferenceType* >( type2 ) ) {
			// std::cerr << "commonType: both references: " << refType << " / " << otherRef << std::endl;
			// std::cerr << ( refType->get_base()->get_qualifiers() >= otherRef->get_base()->get_qualifiers() || widenFirst ) << (refType->get_base()->get_qualifiers() <= otherRef->get_base()->get_qualifiers() || widenSecond) << std::endl;
			if ( widenFirst && dynamic_cast< VoidType* >( otherRef->get_base() ) && ! isFtype(refType->get_base()) ) {
				getCommonWithVoidPointer( otherRef, refType );
			} else if ( widenSecond && dynamic_cast< VoidType* >( refType->get_base() ) && ! isFtype(otherRef->get_base()) ) {
				getCommonWithVoidPointer( refType, otherRef );
			} else if ( ( refType->get_base()->get_qualifiers() >= otherRef->get_base()->get_qualifiers() || widenFirst )
					   && ( refType->get_base()->get_qualifiers() <= otherRef->get_base()->get_qualifiers() || widenSecond ) ) {
				// std::cerr << "middle case" << std::endl;
				Type::Qualifiers tq1 = refType->get_base()->get_qualifiers(), tq2 = otherRef->get_base()->get_qualifiers();
				refType->get_base()->get_qualifiers() = Type::Qualifiers();
				otherRef->get_base()->get_qualifiers() = Type::Qualifiers();
				AssertionSet have, need;
				OpenVarSet newOpen( openVars );
				if ( unifyExact( refType->get_base(), otherRef->get_base(), env, have, need, newOpen, indexer ) ) {
					if ( tq1 < tq2 ) {
						result = refType->clone();
					} else {
						result = otherRef->clone();
					} // if
					result->get_qualifiers() = tq1 | tq2;
				} else {
					/// std::cerr << "place for ptr-to-type" << std::endl;
				} // if
				refType->get_base()->get_qualifiers() = tq1;
				otherRef->get_base()->get_qualifiers() = tq2;
			} // if
		} else if ( widenSecond && dynamic_cast< ZeroType* >( type2 ) ) {
			result = refType->clone();
			result->get_qualifiers() |= type2->get_qualifiers();
		} // if
	}

	void CommonType::visit( FunctionType * ) {}
	void CommonType::visit( StructInstType * ) {}
	void CommonType::visit( UnionInstType * ) {}

	void CommonType::visit( EnumInstType *enumInstType ) {
		if ( dynamic_cast< BasicType * >( type2 ) || dynamic_cast< ZeroType* >( type2 ) || dynamic_cast< OneType* >( type2 ) ) {
			// reuse BasicType, EnumInstType code by swapping type2 with enumInstType
			ValueGuard< Type * > temp( type2 );
			type2 = enumInstType;
			temp.old->accept( *this );
		} // if
	}

	void CommonType::visit( TraitInstType * ) {
	}

	void CommonType::visit( TypeInstType *inst ) {
		if ( widenFirst ) {
			NamedTypeDecl *nt = indexer.lookupType( inst->get_name() );
			if ( nt ) {
				TypeDecl *type = strict_dynamic_cast< TypeDecl* >( nt );
				if ( type->get_base() ) {
					Type::Qualifiers tq1 = inst->get_qualifiers(), tq2 = type2->get_qualifiers();
					AssertionSet have, need;
					OpenVarSet newOpen( openVars );
					type2->get_qualifiers() = Type::Qualifiers();
					type->get_base()->get_qualifiers() = tq1;
					if ( unifyExact( type->get_base(), type2, env, have, need, newOpen, indexer ) ) {
						result = type2->clone();
						result->get_qualifiers() = tq1 | tq2;
					} // if
					type2->get_qualifiers() = tq2;
					type->get_base()->get_qualifiers() = Type::Qualifiers();
				} // if
			} // if
		} // if
	}

	void CommonType::visit( TupleType * ) {}
	void CommonType::visit( VarArgsType * ) {}

	void CommonType::visit( ZeroType *zeroType ) {
		if ( widenFirst ) {
			if ( dynamic_cast< BasicType* >( type2 ) || dynamic_cast< PointerType* >( type2 ) || dynamic_cast< EnumInstType* >( type2 ) ) {
				if ( widenSecond || zeroType->get_qualifiers() <= type2->get_qualifiers() ) {
					result = type2->clone();
					result->get_qualifiers() |= zeroType->get_qualifiers();
				}
			} else if ( widenSecond && dynamic_cast< OneType* >( type2 ) ) {
				result = new BasicType( zeroType->get_qualifiers(), BasicType::SignedInt );
				result->get_qualifiers() |= type2->get_qualifiers();
			}
		}
	}

	void CommonType::visit( OneType *oneType ) {
		if ( widenFirst ) {
			if ( dynamic_cast< BasicType* >( type2 ) || dynamic_cast< EnumInstType* >( type2 ) ) {
				if ( widenSecond || oneType->get_qualifiers() <= type2->get_qualifiers() ) {
					result = type2->clone();
					result->get_qualifiers() |= oneType->get_qualifiers();
				}
			} else if ( widenSecond && dynamic_cast< ZeroType* >( type2 ) ) {
				result = new BasicType( oneType->get_qualifiers(), BasicType::SignedInt );
				result->get_qualifiers() |= type2->get_qualifiers();
			}
		}
	}
} // namespace ResolvExpr

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
