//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// AggregateDecl.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 23:56:39 2015
// Last Modified By : Andrew Beach
// Last Modified On : Fri Aug  4 14:22:00 2017
// Update Count     : 22
//

#include <list>                  // for list
#include <ostream>               // for operator<<, basic_ostream, ostream
#include <string>                // for operator<<, string, char_traits

#include "Attribute.h"           // for Attribute
#include "Common/utility.h"      // for printAll, cloneAll, deleteAll
#include "Declaration.h"         // for AggregateDecl, TypeDecl, Declaration
#include "Parser/LinkageSpec.h"  // for Spec, linkageName, Cforall
#include "Type.h"                // for Type, Type::StorageClasses


AggregateDecl::AggregateDecl( const std::string &name, const std::list< Attribute * > & attributes, LinkageSpec::Spec linkage ) : Parent( name, Type::StorageClasses(), linkage ), body( false ), attributes( attributes ) {
}

AggregateDecl::AggregateDecl( const AggregateDecl &other ) : Parent( other ) {
	cloneAll( other.members, members );
	cloneAll( other.parameters, parameters );
	cloneAll( other.attributes, attributes );
	body = other.body;
}

AggregateDecl::~AggregateDecl() {
	deleteAll( attributes );
	deleteAll( parameters );
	deleteAll( members );
}

void AggregateDecl::print( std::ostream &os, Indenter indent ) const {
	using std::string;
	using std::endl;

	os << typeString() << " " << name << ":";
	if ( get_linkage() != LinkageSpec::Cforall ) {
		os << " " << LinkageSpec::linkageName( linkage );
	} // if
	os << " with body " << has_body();

	if ( ! parameters.empty() ) {
		os << endl << indent << "... with parameters" << endl;
		printAll( parameters, os, indent+1 );
	} // if
	if ( ! members.empty() ) {
		os << endl << indent << "... with members" << endl;
		printAll( members, os, indent+1 );
	} // if
	if ( ! attributes.empty() ) {
		os << endl << indent << "... with attributes" << endl;
		printAll( attributes, os, indent+1 );
	} // if
	os << endl;
}

void AggregateDecl::printShort( std::ostream &os, Indenter indent ) const {
	using std::string;
	using std::endl;

	os << typeString() << " " << name << " with body " << has_body() << endl;

	if ( ! parameters.empty() ) {
		os << indent << "... with parameters" << endl;
		printAll( parameters, os, indent+1 );
	} // if
}

std::string StructDecl::typeString() const { return "struct"; }

std::string UnionDecl::typeString() const { return "union"; }

std::string EnumDecl::typeString() const { return "enum"; }

std::string TraitDecl::typeString() const { return "trait"; }

bool EnumDecl::valueOf( Declaration * enumerator, long long int & value ) {
	if ( enumValues.empty() ) {
		long long int currentValue = 0;
		for ( Declaration * member : members ) {
			ObjectDecl * field = strict_dynamic_cast< ObjectDecl * >( member );
			if ( field->init ) {
				SingleInit * init = strict_dynamic_cast< SingleInit * >( field->init );
				auto result = eval( init->value );
				if ( ! result.second ) SemanticError( init->location, toString( "Non-constexpr in initialization of enumerator: ", field ) );
				currentValue = result.first;
			}
			assertf( enumValues.count( field->name ) == 0, "Enum %s has multiple members with the name %s", name.c_str(), field->name.c_str() );
			enumValues[ field->name ] = currentValue;
			++currentValue;
		}
	}
	if ( enumValues.count( enumerator->name ) ) {
		value = enumValues[ enumerator->name ];
		return true;
	}
	return false;
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
