//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// OperatorTable.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon Feb 10 18:12:12 2020
// Update Count     : 17
//

#include <algorithm>  // for any_of
#include <map>        // for map, _Rb_tree_const_iterator, map<>::const_iterator
#include <utility>    // for pair

#include "OperatorTable.h"
#include "Common/utility.h"

namespace CodeGen {
	namespace {
		const OperatorInfo tableValues[] = {
			{	"?[?]",   "",     "_operator_index",             "Index",                      OT_INDEX          },
			{	"?{}",    "=",    "_constructor",                "Constructor",                OT_CTOR           },
			{	"^?{}",   "",     "_destructor",                 "Destructor",                 OT_DTOR           },
			{	"?()",    "",     "_operator_call",              "Call Operator",              OT_CALL           },
			{	"?++",    "++",   "_operator_postincr",          "Postfix Increment",          OT_POSTFIXASSIGN  },
			{	"?--",    "--",   "_operator_postdecr",          "Postfix Decrement",          OT_POSTFIXASSIGN  },
			{	"*?",     "*",    "_operator_deref",             "Dereference",                OT_PREFIX         },
			{	"+?",     "+",    "_operator_unaryplus",         "Plus",                       OT_PREFIX         },
			{	"-?",     "-",    "_operator_unaryminus",        "Minus",                      OT_PREFIX         },
			{	"~?",     "~",    "_operator_bitnot",            "Bitwise Not",                OT_PREFIX         },
			{	"!?",     "!",    "_operator_lognot",            "Logical Not",                OT_PREFIX         },
			{	"++?",    "++",   "_operator_preincr",           "Prefix Increment",           OT_PREFIXASSIGN   },
			{	"--?",    "--",   "_operator_predecr",           "Prefix Decrement",           OT_PREFIXASSIGN   },
			{	"?\\?",   "\\",   "_operator_exponential",       "Exponentiation",             OT_INFIX          },
			{	"?*?",    "*",    "_operator_multiply",          "Multiplication",             OT_INFIX          },
			{	"?/?",    "/",    "_operator_divide",            "Division",                   OT_INFIX          },
			{	"?%?",    "%",    "_operator_modulus",           "Modulo",                     OT_INFIX          },
			{	"?+?",    "+",    "_operator_add",               "Addition",                   OT_INFIX          },
			{	"?-?",    "-",    "_operator_subtract",          "Substraction",               OT_INFIX          },
			{	"?<<?",   "<<",   "_operator_shiftleft",         "Shift Left",                 OT_INFIX          },
			{	"?>>?",   ">>",   "_operator_shiftright",        "Shift Right",                OT_INFIX          },
			{	"?<?",    "<",    "_operator_less",              "Less-than",                  OT_INFIX          },
			{	"?>?",    ">",    "_operator_greater",           "Greater-than",               OT_INFIX          },
			{	"?<=?",   "<=",   "_operator_lessequal",         "Less-than-or-Equal",         OT_INFIX          },
			{	"?>=?",   ">=",   "_operator_greaterequal",      "Greater-than-or-Equal",      OT_INFIX          },
			{	"?==?",   "==",   "_operator_equal",             "Equality",                   OT_INFIX          },
			{	"?!=?",   "!=",   "_operator_notequal",          "Not-Equal",                  OT_INFIX          },
			{	"?&?",    "&",    "_operator_bitand",            "Bitwise And",                OT_INFIX          },
			{	"?^?",    "^",    "_operator_bitxor",            "Bitwise Xor",                OT_INFIX          },
			{	"?|?",    "|",    "_operator_bitor",             "Bitwise Or",                 OT_INFIX          },
			{	"?=?",    "=",    "_operator_assign",            "Assignment",                 OT_INFIXASSIGN    },
			{	"?\\=?",  "\\=",  "_operator_expassign",         "Exponentiation Assignment",  OT_INFIXASSIGN    },
			{	"?*=?",   "*=",   "_operator_multassign",        "Multiplication Assignment",  OT_INFIXASSIGN    },
			{	"?/=?",   "/=",   "_operator_divassign",         "Division Assignment",        OT_INFIXASSIGN    },
			{	"?%=?",   "%=",   "_operator_modassign",         "Modulo Assignment",          OT_INFIXASSIGN    },
			{	"?+=?",   "+=",   "_operator_addassign",         "Addition Assignment",        OT_INFIXASSIGN    },
			{	"?-=?",   "-=",   "_operator_subassign",         "Substrction Assignment",     OT_INFIXASSIGN    },
			{	"?<<=?",  "<<=",  "_operator_shiftleftassign",   "Shift Left Assignment",      OT_INFIXASSIGN    },
			{	"?>>=?",  ">>=",  "_operator_shiftrightassign",  "Shift Right Assignment",     OT_INFIXASSIGN    },
			{	"?&=?",   "&=",   "_operator_bitandassign",      "Bitwise And Assignment",     OT_INFIXASSIGN    },
			{	"?^=?",   "^=",   "_operator_bitxorassign",      "Bitwise Xor Assignment",     OT_INFIXASSIGN    },
			{	"?|=?",   "|=",   "_operator_bitorassign",       "Bitwise Or Assignment",      OT_INFIXASSIGN    },
		};

		const int numOps = sizeof( tableValues ) / sizeof( OperatorInfo );

		std::map< std::string, OperatorInfo > table;

		void initialize() {
			for ( int i = 0; i < numOps; ++i ) {
				table[ tableValues[i].inputName ] = tableValues[i];
			} // for
		}
	} // namespace

	bool operatorLookup( const std::string & funcName, OperatorInfo & info ) {
		static bool init = false;
		if ( ! init ) {
			initialize();
		} // if

		std::map< std::string, OperatorInfo >::const_iterator i = table.find( funcName );
		if ( i == table.end() ) {
			if ( isPrefix( funcName, "?`" ) ) {			// user-defined postfix operator ?
				info.inputName = funcName;
				info.symbol = funcName.substr(2);
				info.outputName = toString( "__postfix_call_", info.symbol );
				info.type = OT_POSTFIX;
				return true;
			}
			return false;
		} else {
			info = i->second;
			return true;
		} // if
	}

	bool isOperator( const std::string & funcName ) {
		OperatorInfo info;
		return operatorLookup( funcName, info );
	}

	std::string operatorFriendlyName( const std::string & funcName ) {
		OperatorInfo info;
		if( operatorLookup( funcName, info ) ) {
			return info.friendlyName;
		}
		return "";
	}

	/// determines if a given function name is one of the operator types between [begin, end)
	template<typename Iterator>
	bool isOperatorType( const std::string & funcName, Iterator begin, Iterator end ) {
		OperatorInfo info;
		if ( operatorLookup( funcName, info ) ) {
			return std::find( begin, end, info.type ) != end;
		}
		return false;
	}

	bool isConstructor( const std::string & funcName ) {
		static OperatorType types[] = { OT_CTOR };
		return isOperatorType( funcName, std::begin(types), std::end(types) );
	}

	bool isDestructor( const std::string & funcName ) {
		static OperatorType types[] = { OT_DTOR };
		return isOperatorType( funcName, std::begin(types), std::end(types) );
	}

	bool isAssignment( const std::string & funcName ) {
		static OperatorType types[] = { OT_PREFIXASSIGN, OT_POSTFIXASSIGN, OT_INFIXASSIGN };
		return isOperatorType( funcName, std::begin(types), std::end(types) );
	}

	bool isCtorDtor( const std::string & funcName ) {
		static OperatorType types[] = { OT_CTOR, OT_DTOR };
		return isOperatorType( funcName, std::begin(types), std::end(types) );
	}

	bool isCtorDtorAssign( const std::string & funcName ) {
		static OperatorType types[] = { OT_CTOR, OT_DTOR, OT_PREFIXASSIGN, OT_POSTFIXASSIGN, OT_INFIXASSIGN };
		return isOperatorType( funcName, std::begin(types), std::end(types) );
	}
} // namespace CodeGen

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
