//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// DeclStats.cc --
//
// Author           : Aaron Moss
// Created On       : Wed Jan 31 16:40:00 2016
// Last Modified By : Aaron Moss
// Last Modified On : Wed Jan 31 16:40:00 2016
// Update Count     : 1
//

#include "DeclStats.h"

#include <iostream>
#include <map>
#include <string>
#include <unordered_map>
#include <unordered_set>

#include "Common/VectorMap.h"
#include "GenPoly/GenPoly.h"
#include "Parser/LinkageSpec.h"
#include "SynTree/Declaration.h"
#include "SynTree/Visitor.h"

namespace CodeTools {
	
	class DeclStats : public Visitor {
		template<typename T>
		static void sum(T& a, const T& b) { a += b; }

		static void sum(VectorMap<unsigned>& a, const VectorMap<unsigned>& b) {
			a.reserve( b.size() );
			for ( unsigned i = 0; i < b.size(); ++i ) {
				a[i] += b[i];
			}
		}

		template<typename K>
		static void sum(std::map<K, unsigned>& a, const std::map<K, unsigned>& b) {
			for ( const auto& entry : b ) {
				a[ entry.first ] += entry.second;
			}
		}

		template<typename K>
		static void sum(std::unordered_map<K, unsigned>& a, const std::unordered_map<K, unsigned>& b) {
			for ( const auto& entry : b ) {
				a[ entry.first ] += entry.second;
			}
		}

		struct ArgPackStats {
			VectorMap<unsigned> n;                 ///< Count of decls with each number of elements
			VectorMap<unsigned> n_basic;           ///< Count of decls with each number of basic type elements
			VectorMap<unsigned> n_poly;            ///< Count of decls with each number of polymorphic elements
			std::map<unsigned, unsigned> p_basic;  ///< Count of decls with each percentage of basic type elements
			std::map<unsigned, unsigned> p_poly;   ///< Count of decls with each percentage of polymorphic elements

			ArgPackStats& operator+= (const ArgPackStats& o) {
				sum(n, o.n);
				sum(n_basic, o.n_basic);
				sum(n_poly, o.n_poly);
				sum(p_basic, o.p_basic);
				sum(p_poly, o.p_poly);
				
				return *this;
			}

			/// Update based on a declaration list
			ArgPackStats& operator+= ( std::list<DeclarationWithType*>& decls ) {
				unsigned nn = 0;
				unsigned nn_basic = 0;
				unsigned nn_poly = 0;
				for ( auto decl : decls ) {
					nn += decl->get_type()->size();
					if ( dynamic_cast<BasicType*>( decl->get_type() ) ) ++nn_basic;
					else if ( GenPoly::hasPolyBase( decl->get_type() ) ) ++nn_poly;
				}
				++n.at( nn );
				++n_basic.at( nn_basic );
				++n_poly.at( nn_poly );
				if ( nn > 0 ) {
					++p_basic[ nn_basic*100/nn ];
					++p_poly[ nn_poly*100/nn ];
				}
				
				return *this;
			}
		};
		
		struct Stats {
			unsigned n_decls;     ///< Total number of declarations
			/// Count of declarations with each number of assertion parameters
			VectorMap<unsigned> n_type_params;
			/// Count of declarations with each name
			std::unordered_map<std::string, unsigned> by_name;
			/// Stats for the parameter list
			ArgPackStats params;
			/// Stats for the return list
			ArgPackStats returns;
			
			/// Count of declarations with each number of assertions
			std::map<unsigned, unsigned> n_assns;
			/// Stats for the assertions' parameters
			ArgPackStats assn_params;
			/// Stats for the assertions' return types
			ArgPackStats assn_returns;
			
			Stats() : n_decls(0), n_type_params(), by_name(), params(), returns(), n_assns(), assn_params(), assn_returns() {}

		public:
			Stats& operator+= (const Stats& o) {
				sum( n_decls, o.n_decls );
				sum( n_type_params, o.n_type_params );
				sum( by_name, o.by_name );
				sum( params, o.params );
				sum( returns, o.returns );
				sum( n_assns, o.n_assns );
				sum( assn_params, o.assn_params );
				sum( assn_returns, o.assn_returns );
				
				return *this;
			}
		};

		Stats for_linkage[LinkageSpec::NoOfSpecs];   ///< Stores separate stats per linkage
		std::unordered_set<std::string> seen_names;  ///< Stores manglenames already seen to avoid double-counting
		Stats total;

		void analyzeFunc( FunctionType* fnTy, ArgPackStats& params, ArgPackStats& returns ) {
			params += fnTy->get_parameters();
			returns += fnTy->get_returnVals();
		}

	public:
		using Visitor::visit;

		virtual void visit( FunctionDecl *decl ) {
			// skip if already seen declaration for this function
			const std::string& mangleName = decl->get_mangleName().empty() ? decl->get_name() : decl->get_mangleName();
			if ( ! seen_names.insert( mangleName ).second ) return;
			
			Stats& stats = for_linkage[ decl->get_linkage() ];

			++stats.n_decls;
			FunctionType* fnTy = decl->get_functionType();
			const Type::ForallList& forall = fnTy->get_forall();
			++stats.n_type_params.at( forall.size() );
			unsigned n_assns = 0;
			for ( TypeDecl* fdecl : forall ) {
				n_assns += fdecl->get_assertions().size();
				for ( DeclarationWithType* assn : fdecl->get_assertions() ) {
					FunctionType *assnTy = 0;
					if ( ObjectDecl *assnObj = dynamic_cast<ObjectDecl*>(assn) ) {
						if ( PointerType *ptrTy = dynamic_cast<PointerType*>(assnObj->get_type()) ) {
							assnTy = dynamic_cast<FunctionType*>(ptrTy->get_base());
						} else assnTy = dynamic_cast<FunctionType*>(assnObj->get_type());
					} else if ( FunctionDecl *assnDecl = dynamic_cast<FunctionDecl*>(assn) ) {
						assnTy = assnDecl->get_functionType();
					}
					if ( assnTy ) analyzeFunc( assnTy, stats.assn_params, stats.assn_returns );
				}
			}
			++stats.n_assns[ n_assns ];

			++stats.by_name[ decl->get_name() ];

			analyzeFunc( fnTy, stats.params, stats.returns );
		}

	private:
		template<typename F>
		void printAll( const std::string& name, F extract ) {
			std::cout << "\"" << name << "\",";
			for ( const auto& stats : for_linkage ) {
				std::cout << "," << extract(stats);
			}
			std::cout << "," << extract(total) << std::endl;
		}

		template<typename F>
		void printAllMap( const std::string& name, F extract ) {
			for ( const auto& entry : extract(total) ) {
				const auto& key = entry.first;
				std::cout << "\"" << name << "\"," << key;
				for ( const auto& stats : for_linkage ) {
					const auto& map = extract(stats);
					auto it = map.find( key );
					if ( it == map.end() ) std::cout << ",0";
					else std::cout << "," << it->second;
				}
				std::cout  << "," << entry.second << std::endl;
			}
		}

		template<typename F>
		void printAllHisto( const std::string& name, F extract ) {
			VectorMap<unsigned> histos[LinkageSpec::NoOfSpecs];
			VectorMap<unsigned> thisto;

			for ( const auto& entry : extract(total) ) { ++thisto.at( entry.second ); }

			for ( unsigned i = 0; i < LinkageSpec::NoOfSpecs; ++i ) {
				// can't be a higher count in one of the sub-histograms than the total
				histos[i].reserve( thisto.size() );

				for ( const auto& entry : extract(for_linkage[i]) ) { ++histos[i][entry.second]; }
			}

			for ( unsigned i = 0; i < thisto.size(); ++i ) {
				std::cout << "\"" << name << "\"," << i;
				for ( const auto& histo : histos ) {
					std::cout << "," << histo[i];
				}
				std::cout << "," << thisto[i] << std::endl;
			}
		}

		template<typename F>
		void printAllSparseHisto( const std::string& name, F extract ) {
			std::map<unsigned, unsigned> histos[LinkageSpec::NoOfSpecs];
			std::map<unsigned, unsigned> thisto;

			for ( const auto& entry : extract(total) ) { ++thisto[ entry.second ]; }

			for ( unsigned i = 0; i < LinkageSpec::NoOfSpecs; ++i ) {
				for ( const auto& entry : extract(for_linkage[i]) ) { ++histos[i][entry.second]; }
			}

			for ( const auto& entry : thisto ) {
				const auto& key = entry.first;
				std::cout << "\"" << name << "\"," << key;
				for ( unsigned i = 0; i < LinkageSpec::NoOfSpecs; ++i ) {
					auto it = histos[i].find( key );
					if ( it == histos[i].end() ) std::cout << ",0";
					else std::cout << "," << it->second;
				}
				std::cout << "," << entry.second << std::endl;
			}
		}

		template<typename F>
		void printAllPack( const std::string& name, F extract ) {
			printAllMap("n_basic_" + name, [&extract](const Stats& stats) { return extract(stats).n_basic; });
			printAllMap("n_poly_" + name, [&extract](const Stats& stats) { return extract(stats).n_poly; });
			printAllMap("n_" + name, [&extract](const Stats& stats) { return extract(stats).n; });
			printAllMap("%_basic_" + name, [&extract](const Stats& stats) { return extract(stats).p_basic; });
			printAllMap("%_poly_" + name, [&extract](const Stats& stats) { return extract(stats).p_poly; });
		}
		
	public:
		void print() {
			for ( auto& stats : for_linkage ) {
				total += stats;
			}

			std::cout << ",,\"intrinsic\",\"Cforall\",\"C\",\"autogen\",\"builtin\",\"TOTAL\"" << std::endl;

			printAllMap("n_type_params", [](const Stats& stats) { return stats.n_type_params; });
			printAll("n_decls", [](const Stats& stats) { return stats.n_decls; });
			printAll("unique_names", [](const Stats& stats) { return stats.by_name.size(); });
			printAllSparseHisto("overloads", [](const Stats& stats) { return stats.by_name; });
			printAllPack("params", [](const Stats& stats) { return stats.params; });
			printAllPack("returns", [](const Stats& stats) { return stats.returns; });
			printAllMap("n_assns", [](const Stats& stats) { return stats.n_assns; });
			printAllPack("assn_params", [](const Stats& stats) { return stats.assn_params; });
			printAllPack("assn_returns", [](const Stats& stats) { return stats.assn_returns; });
		}
	};

	void printDeclStats( std::list< Declaration * > &translationUnit ) {
		DeclStats stats;
		acceptAll( translationUnit, stats );
		stats.print();
	}
	
} // namespace CodeTools

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
