//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// ReferenceToType.cc -- 
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Sun Jun  7 08:31:48 2015
// Update Count     : 4
//

#include <string>
#include <cassert>

#include "Type.h"
#include "Declaration.h"
#include "Expression.h"
#include "TypeSubstitution.h"
#include "utility.h"

ReferenceToType::ReferenceToType( const Type::Qualifiers &tq, const std::string &name ) : Type( tq ), name( name ) {
}

ReferenceToType::ReferenceToType( const ReferenceToType &other ) : Type( other ), name( other.name ) {
	cloneAll( other.parameters, parameters );
}

ReferenceToType::~ReferenceToType() {
	deleteAll( parameters );
}

void ReferenceToType::print( std::ostream &os, int indent ) const {
	using std::endl;
	
	Type::print( os, indent );
	os << "instance of " << typeString() << " " << name << " ";
	if ( ! parameters.empty() ) {
		os << endl << std::string( indent, ' ' ) << "with parameters" << endl;
		printAll( parameters, os, indent+2 );
	} // if
}

namespace {
	void doLookup( const std::list< Declaration* > &members, const std::list< TypeDecl* > &parms, const std::list< Expression* > &args, const std::string &name, std::list< Declaration* > &foundDecls ) {
		std::list< Declaration* > found;
		for ( std::list< Declaration* >::const_iterator i = members.begin(); i != members.end(); ++i ) {
			if ( (*i)->get_name() == name ) {
				found.push_back( *i );
			} // if
		} // for
		applySubstitution( parms.begin(), parms.end(), args.begin(), found.begin(), found.end(), back_inserter( foundDecls ) );
	}
} // namespace

std::string StructInstType::typeString() const { return "struct"; }

std::list<TypeDecl*>* StructInstType::get_baseParameters() {
	if ( ! baseStruct ) return NULL;
	return &baseStruct->get_parameters();
}

void StructInstType::lookup( const std::string &name, std::list< Declaration* > &foundDecls ) const {
	assert( baseStruct );
	doLookup( baseStruct->get_members(), baseStruct->get_parameters(), parameters, name, foundDecls );
}

std::string UnionInstType::typeString() const { return "union"; }

std::list<TypeDecl*>* UnionInstType::get_baseParameters() {
	if ( ! baseUnion ) return NULL;
	return &baseUnion->get_parameters();
}

void UnionInstType::lookup( const std::string &name, std::list< Declaration* > &foundDecls ) const {
	assert( baseUnion );
	doLookup( baseUnion->get_members(), baseUnion->get_parameters(), parameters, name, foundDecls );
}

std::string EnumInstType::typeString() const { return "enum"; }

std::string ContextInstType::typeString() const { return "context"; }

ContextInstType::ContextInstType( const ContextInstType &other ) : Parent( other ) {
	cloneAll( other.members, members );
}

ContextInstType::~ContextInstType() {
	deleteAll( members );
}

TypeInstType::TypeInstType( const Type::Qualifiers &tq, const std::string &name, TypeDecl *baseType ) : Parent( tq, name ) {
	set_baseType( baseType );
}

TypeInstType::TypeInstType( const Type::Qualifiers &tq, const std::string &name, bool isFtype ) : Parent( tq, name ), baseType( 0 ), isFtype( isFtype ) {
}

void TypeInstType::set_baseType( TypeDecl *newValue ) {
	baseType = newValue;
	isFtype = newValue->get_kind() == TypeDecl::Ftype;
}

std::string TypeInstType::typeString() const { return "type"; }

void TypeInstType::print( std::ostream &os, int indent ) const {
	using std::endl;
	
	Type::print( os, indent );
	os << "instance of " << typeString() << " " << get_name() << " (" << ( isFtype ? "" : "not" ) << " function type) ";
	if ( ! parameters.empty() ) {
		os << endl << std::string( indent, ' ' ) << "with parameters" << endl;
		printAll( parameters, os, indent+2 );
	} // if
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
