//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// ExpressionNode.cc --
//
// Author           : Peter A. Buhr
// Created On       : Sat May 16 13:17:07 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Thu Mar 22 11:57:39 2018
// Update Count     : 801
//

#include <cassert>                 // for assert
#include <stdio.h>                 // for sscanf, size_t
#include <climits>                 // for LLONG_MAX, LONG_MAX, INT_MAX, UINT...
#include <list>                    // for list
#include <sstream>                 // for basic_istream::operator>>, basic_i...
#include <string>                  // for string, operator+, operator==

#include "Common/SemanticError.h"  // for SemanticError
#include "Common/utility.h"        // for maybeMoveBuild, maybeBuild, CodeLo...
#include "ParseNode.h"             // for ExpressionNode, maybeMoveBuildType
#include "SynTree/Constant.h"      // for Constant
#include "SynTree/Declaration.h"   // for EnumDecl, StructDecl, UnionDecl
#include "SynTree/Expression.h"    // for Expression, ConstantExpr, NameExpr
#include "SynTree/Statement.h"     // for CompoundStmt, Statement
#include "SynTree/Type.h"          // for BasicType, Type, Type::Qualifiers
#include "parserutility.h"         // for notZeroExpr

class Initializer;

using namespace std;

//##############################################################################

// Difficult to separate extra parts of constants during lexing because actions are not allow in the middle of patterns:
//
//		prefix action constant action suffix
//
// Alternatively, breaking a pattern using BEGIN does not work if the following pattern can be empty:
//
//		constant BEGIN CONT ...
//		<CONT>(...)? BEGIN 0 ... // possible empty suffix
//
// because the CONT rule is NOT triggered if the pattern is empty. Hence, constants are reparsed here to determine their
// type.

extern const Type::Qualifiers noQualifiers;				// no qualifiers on constants

static inline bool checkH( char c ) { return c == 'h' || c == 'H'; }
static inline bool checkL( char c ) { return c == 'l' || c == 'L'; }
static inline bool checkZ( char c ) { return c == 'z' || c == 'Z'; }
static inline bool checkU( char c ) { return c == 'u' || c == 'U'; }
static inline bool checkF( char c ) { return c == 'f' || c == 'F'; }
static inline bool checkD( char c ) { return c == 'd' || c == 'D'; }
static inline bool checkI( char c ) { return c == 'i' || c == 'I'; }
static inline bool checkB( char c ) { return c == 'b' || c == 'B'; }
static inline bool checkX( char c ) { return c == 'x' || c == 'X'; }

static const char * lnthsInt[2][6] = {
	{ "int8_t", "int16_t", "int32_t", "int64_t", "size_t", },
	{ "uint8_t", "uint16_t", "uint32_t", "uint64_t", "size_t", }
}; // lnthsInt

static inline void checkLNInt( string & str, int & lnth, int & size ) {
	string::size_type posn = str.find_first_of( "lL" ), start = posn;
  if ( posn == string::npos ) return;
	size = 4;											// assume largest size
	posn += 1;											// advance to size
	if ( str[posn] == '8' ) {							// 8
		lnth = 0;
	} else if ( str[posn] == '1' ) {
		posn += 1;
		if ( str[posn] == '6' ) {						// 16
			lnth = 1;
		} else {										// 128
			posn += 1;
			lnth = 5;
		} // if
	} else {
		if ( str[posn] == '3' ) {						// 32
			lnth = 2;
		} else if ( str[posn] == '6' ) {				// 64
			lnth = 3;
		} else {
			assertf( false, "internal error, bad integral length %s", str.c_str() );
		} // if
		posn += 1;
	} // if
	str.erase( start, posn - start + 1 );				// remove length suffix
} // checkLNInt

Expression * build_constantInteger( string & str ) {
	static const BasicType::Kind kind[2][6] = {
		// short (h) must be before char (hh)
		{ BasicType::ShortSignedInt, BasicType::SignedChar, BasicType::SignedInt, BasicType::LongSignedInt, BasicType::LongLongSignedInt, BasicType::SignedInt128, },
		{ BasicType::ShortUnsignedInt, BasicType::UnsignedChar, BasicType::UnsignedInt, BasicType::LongUnsignedInt, BasicType::LongLongUnsignedInt, BasicType::UnsignedInt128, },
	};

	bool dec = true, Unsigned = false;					// decimal, unsigned constant
	int size;											// 0 => short, 1 => char, 2 => int, 3 => long int, 4 => long long int, 5 => int128
	int lnth = -1;										// literal length

	unsigned long long int v;							// converted integral value
	size_t last = str.length() - 1;						// last subscript of constant
	Expression * ret;

	// special constants
	if ( str == "0" ) {
		ret = new ConstantExpr( Constant( (Type *)new ZeroType( noQualifiers ), str, (unsigned long long int)0 ) );
		goto CLEANUP;
	} // if
	if ( str == "1" ) {
		ret = new ConstantExpr( Constant( (Type *)new OneType( noQualifiers ), str, (unsigned long long int)1 ) );
		goto CLEANUP;
	} // if

	// Cannot be "0"

	if ( str[0] == '0' ) {								// radix character ?
		dec = false;
		if ( checkX( str[1] ) ) {						// hex constant ?
			sscanf( (char *)str.c_str(), "%llx", &v );
			//printf( "%llx %llu\n", v, v );
		} else if ( checkB( str[1] ) ) {				// binary constant ?
			v = 0;
			for ( unsigned int i = 2;; i += 1 ) {		// compute value
				if ( str[i] == '1' ) v |= 1;
			  if ( i == last ) break;
				v <<= 1;
			} // for
			//printf( "%llx %llu\n", v, v );
		} else {										// octal constant
			sscanf( (char *)str.c_str(), "%llo", &v );
			//printf( "%llo %llu\n", v, v );
		} // if
	} else {											// decimal constant ?
		sscanf( (char *)str.c_str(), "%llu", &v );
		//printf( "%llu %llu\n", v, v );
	} // if

	if ( v <= INT_MAX ) {								// signed int
		size = 2;
	} else if ( v <= UINT_MAX && ! dec ) {				// unsigned int
		size = 2;
		Unsigned = true;								// unsigned
	} else if ( v <= LONG_MAX ) {						// signed long int
		size = 3;
	} else if ( v <= ULONG_MAX && ( ! dec || LONG_MAX == LLONG_MAX ) ) { // signed long int
		size = 3;
		Unsigned = true;								// unsigned long int
	} else if ( v <= LLONG_MAX ) {						// signed long long int
		size = 4;
	} else {											// unsigned long long int
		size = 4;
		Unsigned = true;								// unsigned long long int
	} // if

	// At least one digit in integer constant, so safe to backup while looking for suffix.

	if ( checkU( str[last] ) ) {						// suffix 'u' ?
		Unsigned = true;
		if ( checkL( str[last - 1] ) ) {				// suffix 'l' ?
			size = 3;
			if ( checkL( str[last - 2] ) ) {			// suffix "ll" ?
				size = 4;
			} // if
		} else if ( checkH( str[last - 1] ) ) {			// suffix 'h' ?
			size = 0;
			if ( checkH( str[last - 2] ) ) {			// suffix "hh" ?
				size = 1;
			} // if
			str.erase( last - size - 1, size + 1 );		// remove 'h'/"hh"
		} else {										// suffix "ln" ?
			checkLNInt( str, lnth, size );
		} // if
	} else if ( checkL( str[ last ] ) ) {				// suffix 'l' ?
		size = 3;
		if ( checkL( str[last - 1] ) ) {				// suffix 'll' ?
			size = 4;
			if ( checkU( str[last - 2] ) ) {			// suffix 'u' ?
				Unsigned = true;
			} // if
		} else if ( checkU( str[last - 1] ) ) {			// suffix 'u' ?
			Unsigned = true;
		} // if
	} else if ( checkH( str[ last ] ) ) {				// suffix 'h' ?
		size = 0;
		if ( checkH( str[last - 1] ) ) {				// suffix "hh" ?
			size = 1;
			if ( checkU( str[last - 2] ) ) {			// suffix 'u' ?
				Unsigned = true;
			} // if
		} else if ( checkU( str[last - 1] ) ) {			// suffix 'u' ?
			Unsigned = true;
		} // if
		str.erase( last - size, size + 1 );				// remove 'h'/"hh"
	} else if ( checkZ( str[last] ) ) {					// suffix 'z' ?
		lnth = 4;
		str.erase( last, 1 );							// remove 'z'
	} else {											// suffix "ln" ?
		checkLNInt( str, lnth, size );
	} // if

	assert( 0 <= size && size < 6 );
	// Constant type is correct for overload resolving.
	ret = new ConstantExpr( Constant( new BasicType( noQualifiers, kind[Unsigned][size] ), str, v ) );
	if ( Unsigned && size < 2 ) {						// hh or h, less than int ?
		// int i = -1uh => 65535 not -1, so cast is necessary for unsigned, which unfortunately eliminates warnings for large values.
		ret = new CastExpr( ret, new BasicType( Type::Qualifiers(), kind[Unsigned][size] ), false );
	} else if ( lnth != -1 ) {							// explicit length ?
		if ( lnth == 5 ) {								// int128 ?
			size = 5;
			ret = new CastExpr( ret, new BasicType( Type::Qualifiers(), kind[Unsigned][size] ), false );
		} else {
			ret = new CastExpr( ret, new TypeInstType( Type::Qualifiers(), lnthsInt[Unsigned][lnth], false ), false );
		} // if
	} // if
  CLEANUP:

	delete &str;										// created by lex
	return ret;
} // build_constantInteger


static inline void checkLNFloat( string & str, int & lnth, int & size ) {
	string::size_type posn = str.find_first_of( "lL" ), start = posn;
  if ( posn == string::npos ) return;
	size = 2;											// assume largest size
	lnth = 0;
	posn += 1;											// advance to size
	if ( str[posn] == '3' ) {							// 32
		size = 0;
	} else if ( str[posn] == '6' ) {					// 64
		size = 1;
	} else if ( str[posn] == '8' || str[posn] == '1' ) { // 80, 128
		size = 2;
		if ( str[posn] == '1' ) posn += 1;
	} else {
		assertf( false, "internal error, bad floating point length %s", str.c_str() );
	} // if
	posn += 1;
	str.erase( start, posn - start + 1 );				// remove length suffix
} // checkLNFloat


Expression * build_constantFloat( string & str ) {
	static const BasicType::Kind kind[2][3] = {
		{ BasicType::Float, BasicType::Double, BasicType::LongDouble },
		{ BasicType::FloatComplex, BasicType::DoubleComplex, BasicType::LongDoubleComplex },
	};

	bool complx = false;								// real, complex
	int size = 1;										// 0 => float, 1 => double, 2 => long double
	int lnth = -1;										// literal length
	// floating-point constant has minimum of 2 characters: 1. or .1
	size_t last = str.length() - 1;
	double v;

	sscanf( str.c_str(), "%lg", &v );

	if ( checkI( str[last] ) ) {						// imaginary ?
		complx = true;
		last -= 1;										// backup one character
	} // if

	if ( checkF( str[last] ) ) {						// float ?
		size = 0;
	} else if ( checkD( str[last] ) ) {					// double ?
		size = 1;
	} else if ( checkL( str[last] ) ) {					// long double ?
		size = 2;
	} else {
		size = 1;										// double (default)
		checkLNFloat( str, lnth, size );
	} // if
	if ( ! complx && checkI( str[last - 1] ) ) {		// imaginary ?
		complx = true;
	} // if

	assert( 0 <= size && size < 3 );
	Expression * ret = new ConstantExpr( Constant( new BasicType( noQualifiers, kind[complx][size] ), str, v ) );
	if ( lnth != -1 ) {									// explicit length ?
		ret = new CastExpr( ret, new BasicType( Type::Qualifiers(), kind[complx][size] ), false );
	} // if

	delete &str;										// created by lex
	return ret;
} // build_constantFloat

static void sepString( string & str, string & units, char delimit ) {
	string::size_type posn = str.find_last_of( delimit ) + 1;
	if ( posn != str.length() ) {
		units = "?" + str.substr( posn );				// extract units
		str.erase( posn );								// remove units
	} // if
} // sepString

Expression * build_constantChar( string & str ) {
	string units;										// units
	sepString( str, units, '\'' );						// separate constant from units

	Expression * ret = new ConstantExpr( Constant( new BasicType( noQualifiers, BasicType::Char ), str, (unsigned long long int)(unsigned char)str[1] ) );
	if ( units.length() != 0 ) {
		ret = new UntypedExpr( new NameExpr( units ), { ret } );
	} // if

	delete &str;										// created by lex
	return ret;
} // build_constantChar

Expression * build_constantStr( string & str ) {
	string units;										// units
	sepString( str, units, '"' );						// separate constant from units

	Type * strtype;
	switch ( str[0] ) {									// str has >= 2 characters, i.e, null string "" => safe to look at subscripts 0/1
	  case 'u':
		if ( str[1] == '8' ) goto Default;				// utf-8 characters => array of char
		// lookup type of associated typedef
		strtype = new TypeInstType( Type::Qualifiers( Type::Const ), "char16_t", false );
		break;
	  case 'U':
		strtype = new TypeInstType( Type::Qualifiers( Type::Const ), "char32_t", false );
		break;
	  case 'L':
		strtype = new TypeInstType( Type::Qualifiers( Type::Const ), "wchar_t", false );
		break;
	  Default:											// char default string type
	  default:
		strtype = new BasicType( Type::Qualifiers( Type::Const ), BasicType::Char );
	} // switch
	ArrayType * at = new ArrayType( noQualifiers, strtype,
									new ConstantExpr( Constant::from_ulong( str.size() + 1 - 2 ) ), // +1 for '\0' and -2 for '"'
									false, false );
	Expression * ret = new ConstantExpr( Constant( at, str, (unsigned long long int)0 ) ); // constant 0 is ignored for pure string value
	if ( units.length() != 0 ) {
		ret = new UntypedExpr( new NameExpr( units ), { ret } );
	} // if

	delete &str;										// created by lex
	return ret;
} // build_constantStr

Expression * build_field_name_FLOATING_FRACTIONconstant( const string & str ) {
	if ( str.find_first_not_of( "0123456789", 1 ) != string::npos ) SemanticError( yylloc, "invalid tuple index " + str );
	Expression * ret = build_constantInteger( *new string( str.substr(1) ) );
	delete &str;
	return ret;
} // build_field_name_FLOATING_FRACTIONconstant

Expression * build_field_name_FLOATING_DECIMALconstant( const string & str ) {
	if ( str[str.size()-1] != '.' ) SemanticError( yylloc, "invalid tuple index " + str );
	Expression * ret = build_constantInteger( *new string( str.substr( 0, str.size()-1 ) ) );
	delete &str;
	return ret;
} // build_field_name_FLOATING_DECIMALconstant

Expression * build_field_name_FLOATINGconstant( const string & str ) {
	// str is of the form A.B -> separate at the . and return member expression
	int a, b;
	char dot;
	stringstream ss( str );
	ss >> a >> dot >> b;
	UntypedMemberExpr * ret = new UntypedMemberExpr( new ConstantExpr( Constant::from_int( b ) ), new ConstantExpr( Constant::from_int( a ) ) );
	delete &str;
	return ret;
} // build_field_name_FLOATINGconstant

Expression * make_field_name_fraction_constants( Expression * fieldName, Expression * fracts ) {
	if ( fracts ) {
		if ( UntypedMemberExpr * memberExpr = dynamic_cast< UntypedMemberExpr * >( fracts ) ) {
			memberExpr->set_member( make_field_name_fraction_constants( fieldName, memberExpr->get_aggregate() ) );
			return memberExpr;
		} else {
			return new UntypedMemberExpr( fracts, fieldName );
		} // if
	} // if
	return fieldName;
} // make_field_name_fraction_constants

Expression * build_field_name_fraction_constants( Expression * fieldName, ExpressionNode * fracts ) {
	return make_field_name_fraction_constants( fieldName, maybeMoveBuild< Expression >( fracts ) );
} // build_field_name_fraction_constants

NameExpr * build_varref( const string * name ) {
	NameExpr * expr = new NameExpr( *name );
	delete name;
	return expr;
} // build_varref

// TODO: get rid of this and OperKinds and reuse code from OperatorTable
static const char * OperName[] = {						// must harmonize with OperKinds
	// diadic
	"SizeOf", "AlignOf", "OffsetOf", "?+?", "?-?", "?\\?", "?*?", "?/?", "?%?", "||", "&&",
	"?|?", "?&?", "?^?", "Cast", "?<<?", "?>>?", "?<?", "?>?", "?<=?", "?>=?", "?==?", "?!=?",
	"?=?", "?@=?", "?\\=?", "?*=?", "?/=?", "?%=?", "?+=?", "?-=?", "?<<=?", "?>>=?", "?&=?", "?^=?", "?|=?",
	"?[?]", "...",
	// monadic
	"+?", "-?", "AddressOf", "*?", "!?", "~?", "++?", "?++", "--?", "?--",
}; // OperName

Expression * build_cast( DeclarationNode * decl_node, ExpressionNode * expr_node ) {
	Type * targetType = maybeMoveBuildType( decl_node );
	if ( dynamic_cast< VoidType * >( targetType ) ) {
		return new CastExpr( maybeMoveBuild< Expression >(expr_node), false );
	} else {
		return new CastExpr( maybeMoveBuild< Expression >(expr_node), targetType, false );
	} // if
} // build_cast

Expression * build_keyword_cast( KeywordCastExpr::Target target, ExpressionNode * expr_node ) {
	return new KeywordCastExpr( maybeMoveBuild< Expression >(expr_node), target );
}

Expression * build_virtual_cast( DeclarationNode * decl_node, ExpressionNode * expr_node ) {
	return new VirtualCastExpr( maybeMoveBuild< Expression >( expr_node ), maybeMoveBuildType( decl_node ) );
} // build_virtual_cast

Expression * build_fieldSel( ExpressionNode * expr_node, Expression * member ) {
	return new UntypedMemberExpr( member, maybeMoveBuild< Expression >(expr_node) );
} // build_fieldSel

Expression * build_pfieldSel( ExpressionNode * expr_node, Expression * member ) {
	UntypedExpr * deref = new UntypedExpr( new NameExpr( "*?" ) );
	deref->location = expr_node->location;
	deref->get_args().push_back( maybeMoveBuild< Expression >(expr_node) );
	UntypedMemberExpr * ret = new UntypedMemberExpr( member, deref );
	return ret;
} // build_pfieldSel

Expression * build_offsetOf( DeclarationNode * decl_node, NameExpr * member ) {
	return new UntypedOffsetofExpr{ maybeMoveBuildType( decl_node ), member->get_name() };
} // build_offsetOf

Expression * build_and_or( ExpressionNode * expr_node1, ExpressionNode * expr_node2, bool kind ) {
	return new LogicalExpr( notZeroExpr( maybeMoveBuild< Expression >(expr_node1) ), notZeroExpr( maybeMoveBuild< Expression >(expr_node2) ), kind );
} // build_and_or

Expression * build_unary_val( OperKinds op, ExpressionNode * expr_node ) {
	list< Expression * > args;
	args.push_back( maybeMoveBuild< Expression >(expr_node) );
	return new UntypedExpr( new NameExpr( OperName[ (int)op ] ), args );
} // build_unary_val

Expression * build_unary_ptr( OperKinds op, ExpressionNode * expr_node ) {
	list< Expression * > args;
	args.push_back(  maybeMoveBuild< Expression >(expr_node) ); // xxx -- this is exactly the same as the val case now, refactor this code.
	return new UntypedExpr( new NameExpr( OperName[ (int)op ] ), args );
} // build_unary_ptr

Expression * build_binary_val( OperKinds op, ExpressionNode * expr_node1, ExpressionNode * expr_node2 ) {
	list< Expression * > args;
	args.push_back( maybeMoveBuild< Expression >(expr_node1) );
	args.push_back( maybeMoveBuild< Expression >(expr_node2) );
	return new UntypedExpr( new NameExpr( OperName[ (int)op ] ), args );
} // build_binary_val

Expression * build_binary_ptr( OperKinds op, ExpressionNode * expr_node1, ExpressionNode * expr_node2 ) {
	list< Expression * > args;
	args.push_back( maybeMoveBuild< Expression >(expr_node1) );
	args.push_back( maybeMoveBuild< Expression >(expr_node2) );
	return new UntypedExpr( new NameExpr( OperName[ (int)op ] ), args );
} // build_binary_ptr

Expression * build_cond( ExpressionNode * expr_node1, ExpressionNode * expr_node2, ExpressionNode * expr_node3 ) {
	return new ConditionalExpr( notZeroExpr( maybeMoveBuild< Expression >(expr_node1) ), maybeMoveBuild< Expression >(expr_node2), maybeMoveBuild< Expression >(expr_node3) );
} // build_cond

Expression * build_tuple( ExpressionNode * expr_node ) {
	list< Expression * > exprs;
	buildMoveList( expr_node, exprs );
	return new UntypedTupleExpr( exprs );;
} // build_tuple

Expression * build_func( ExpressionNode * function, ExpressionNode * expr_node ) {
	list< Expression * > args;
	buildMoveList( expr_node, args );
	return new UntypedExpr( maybeMoveBuild< Expression >(function), args );
} // build_func

Expression * build_compoundLiteral( DeclarationNode * decl_node, InitializerNode * kids ) {
	Declaration * newDecl = maybeBuild< Declaration >(decl_node); // compound literal type
	if ( DeclarationWithType * newDeclWithType = dynamic_cast< DeclarationWithType * >( newDecl ) ) { // non-sue compound-literal type
		return new CompoundLiteralExpr( newDeclWithType->get_type(), maybeMoveBuild< Initializer >(kids) );
	// these types do not have associated type information
	} else if ( StructDecl * newDeclStructDecl = dynamic_cast< StructDecl * >( newDecl )  ) {
		if ( newDeclStructDecl->has_body() ) {
			return new CompoundLiteralExpr( new StructInstType( Type::Qualifiers(), newDeclStructDecl ), maybeMoveBuild< Initializer >(kids) );
		} else {
			return new CompoundLiteralExpr( new StructInstType( Type::Qualifiers(), newDeclStructDecl->get_name() ), maybeMoveBuild< Initializer >(kids) );
		} // if
	} else if ( UnionDecl * newDeclUnionDecl = dynamic_cast< UnionDecl * >( newDecl )  ) {
		if ( newDeclUnionDecl->has_body() ) {
			return new CompoundLiteralExpr( new UnionInstType( Type::Qualifiers(), newDeclUnionDecl ), maybeMoveBuild< Initializer >(kids) );
		} else {
			return new CompoundLiteralExpr( new UnionInstType( Type::Qualifiers(), newDeclUnionDecl->get_name() ), maybeMoveBuild< Initializer >(kids) );
		} // if
	} else if ( EnumDecl * newDeclEnumDecl = dynamic_cast< EnumDecl * >( newDecl )  ) {
		if ( newDeclEnumDecl->has_body() ) {
			return new CompoundLiteralExpr( new EnumInstType( Type::Qualifiers(), newDeclEnumDecl ), maybeMoveBuild< Initializer >(kids) );
		} else {
			return new CompoundLiteralExpr( new EnumInstType( Type::Qualifiers(), newDeclEnumDecl->get_name() ), maybeMoveBuild< Initializer >(kids) );
		} // if
	} else {
		assert( false );
	} // if
} // build_compoundLiteral

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
