//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Tuples.h --
//
// Author           : Andrew Beach
// Created On       : Mon Jun 17 14:41:00 2019
// Last Modified By : Andrew Beach
// Last Modified On : Wed Jun 12 15:43:00 2019
// Update Count     : 0
//

#include "Tuples.h"

#include "AST/Pass.hpp"
#include "AST/LinkageSpec.hpp"
#include "InitTweak/InitTweak.h"

namespace Tuples {

namespace {
	/// Determines if impurity (read: side-effects) may exist in a piece of code. Currently gives
	/// a very crude approximation, wherein any function call expression means the code may be
	/// impure.
    struct ImpurityDetector : public ast::WithShortCircuiting {
		ImpurityDetector( bool ignoreUnique ) : ignoreUnique( ignoreUnique ) {}
		bool maybeImpure = false;
		bool ignoreUnique;

		void previsit( ast::ApplicationExpr const * appExpr ) {
			visit_children = false;
			if ( ast::DeclWithType const * function = InitTweak::getFunction( appExpr ) ) {
				if ( function->linkage == ast::Linkage::Intrinsic
						&& ( function->name == "*?" || function->name == "?[?]" ) ) {
					visit_children = true;
					return;
				}
			}
			maybeImpure = true;
		}
		void previsit( ast::UntypedExpr const * ) {
			maybeImpure = true; visit_children = false;
		}
		void previsit( ast::UniqueExpr const * ) {
			if ( ignoreUnique ) {
				visit_children = false;
			}
		}
	};

	bool detectImpurity( const ast::Expr * expr, bool ignoreUnique ) {
		ast::Pass<ImpurityDetector> detector( ignoreUnique );
		expr->accept( detector );
		return detector.pass.maybeImpure;
	}
} // namespace

bool maybeImpureIgnoreUnique( const ast::Expr * expr ) {
	return detectImpurity( expr, true );
}

} // namespace Tuples

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
