//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// FixInit.h --
//
// Author           : Rob Schluntz
// Created On       : Wed Jan 13 16:29:30 2016
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Jun 21 17:35:05 2017
// Update Count     : 74
//

#include <stack>
#include <list>
#include <iterator>
#include <algorithm>
#include <unordered_map>
#include <unordered_set>

#include "InitTweak.h"
#include "GenInit.h"
#include "FixInit.h"
#include "FixGlobalInit.h"
#include "CodeGen/GenType.h"  // for warning/error messages
#include "Common/PassVisitor.h"
#include "GenPoly/DeclMutator.h"
#include "GenPoly/PolyMutator.h"
#include "ResolvExpr/Resolver.h"
#include "ResolvExpr/typeops.h"
#include "SymTab/Autogen.h"
#include "SymTab/Indexer.h"
#include "SynTree/AddStmtVisitor.h"
#include "SynTree/Attribute.h"
#include "SynTree/Declaration.h"
#include "SynTree/Expression.h"
#include "SynTree/Initializer.h"
#include "SynTree/Mutator.h"
#include "SynTree/Statement.h"
#include "SynTree/Type.h"
#include "Tuples/Tuples.h"

bool ctordtorp = false; // print all debug
bool ctorp = false; // print ctor debug
bool cpctorp = false; // print copy ctor debug
bool dtorp = false; // print dtor debug
#define PRINT( text ) if ( ctordtorp ) { text }
#define CP_CTOR_PRINT( text ) if ( ctordtorp || cpctorp ) { text }
#define DTOR_PRINT( text ) if ( ctordtorp || dtorp ) { text }

namespace InitTweak {
	namespace {
		typedef std::unordered_map< Expression *, TypeSubstitution * > EnvMap;
		typedef std::unordered_map< int, int > UnqCount;

		class InsertImplicitCalls : public WithTypeSubstitution {
		public:
			/// wrap function application expressions as ImplicitCopyCtorExpr nodes so that it is easy to identify which
			/// function calls need their parameters to be copy constructed
			static void insert( std::list< Declaration * > & translationUnit, EnvMap & envMap );

			InsertImplicitCalls( EnvMap & envMap ) : envMap( envMap ) {}

			Expression * postmutate( ApplicationExpr * appExpr );
			void premutate( StmtExpr * stmtExpr );

			// collects environments for relevant nodes
			EnvMap & envMap;
		};

		class ResolveCopyCtors final : public SymTab::Indexer {
		public:
			/// generate temporary ObjectDecls for each argument and return value of each ImplicitCopyCtorExpr,
			/// generate/resolve copy construction expressions for each, and generate/resolve destructors for both
			/// arguments and return value temporaries
			static void resolveImplicitCalls( std::list< Declaration * > & translationUnit, const EnvMap & envMap, UnqCount & unqCount );

			typedef SymTab::Indexer Parent;
			using Parent::visit;

			ResolveCopyCtors( const EnvMap & envMap, UnqCount & unqCount ) : envMap( envMap ), unqCount( unqCount ) {}

			virtual void visit( ImplicitCopyCtorExpr * impCpCtorExpr ) override;
			virtual void visit( UniqueExpr * unqExpr ) override;
			virtual void visit( StmtExpr * stmtExpr ) override;

			/// create and resolve ctor/dtor expression: fname(var, [cpArg])
			Expression * makeCtorDtor( const std::string & fname, ObjectDecl * var, Expression * cpArg = NULL );
			/// true if type does not need to be copy constructed to ensure correctness
			bool skipCopyConstruct( Type * type );
			void copyConstructArg( Expression *& arg, ImplicitCopyCtorExpr * impCpCtorExpr );
			void destructRet( ObjectDecl * ret, ImplicitCopyCtorExpr * impCpCtorExpr );

			TypeSubstitution * env;
			const EnvMap & envMap;
			UnqCount & unqCount; // count the number of times each unique expr ID appears
		};

		/// collects constructed object decls - used as a base class
		class ObjDeclCollector : public AddStmtVisitor {
		  public:
			typedef AddStmtVisitor Parent;
			using Parent::visit;
			typedef std::set< ObjectDecl * > ObjectSet;
			virtual void visit( CompoundStmt *compoundStmt ) override;
			virtual void visit( DeclStmt *stmt ) override;

			// don't go into other functions
			virtual void visit( __attribute__((unused)) FunctionDecl *decl ) override {}

		  protected:
			ObjectSet curVars;
		};

		// debug
		struct printSet {
			typedef ObjDeclCollector::ObjectSet ObjectSet;
			printSet( const ObjectSet & objs ) : objs( objs ) {}
			const ObjectSet & objs;
		};
		std::ostream & operator<<( std::ostream & out, const printSet & set) {
			out << "{ ";
			for ( ObjectDecl * obj : set.objs ) {
				out << obj->get_name() << ", " ;
			} // for
			out << " }";
			return out;
		}

		class LabelFinder final : public ObjDeclCollector {
		  public:
			typedef ObjDeclCollector Parent;
			typedef std::map< Label, ObjectSet > LabelMap;
			// map of Label -> live variables at that label
			LabelMap vars;

			void handleStmt( Statement * stmt );

			// xxx - This needs to be done better.
			// allow some generalization among different kinds of nodes with with similar parentage (e.g. all
			// expressions, all statements, etc.)  important to have this to provide a single entry point so that as new
			// subclasses are added, there is only one place that the code has to be updated, rather than ensure that
			// every specialized class knows about every new kind of statement that might be added.
			using Parent::visit;
			virtual void visit( CompoundStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( ExprStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( AsmStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( IfStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( WhileStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( ForStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( SwitchStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( CaseStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( BranchStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( ReturnStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( TryStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( CatchStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( FinallyStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( NullStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( DeclStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
			virtual void visit( ImplicitCtorDtorStmt *stmt ) override { handleStmt( stmt ); return Parent::visit( stmt ); }
		};

		class InsertDtors final : public ObjDeclCollector {
		public:
			/// insert destructor calls at the appropriate places.  must happen before CtorInit nodes are removed
			/// (currently by FixInit)
			static void insert( std::list< Declaration * > & translationUnit );

			typedef ObjDeclCollector Parent;
			typedef std::list< ObjectDecl * > OrderedDecls;
			typedef std::list< OrderedDecls > OrderedDeclsStack;

			InsertDtors( LabelFinder & finder ) : finder( finder ), labelVars( finder.vars ) {}

			using Parent::visit;

			virtual void visit( ObjectDecl * objDecl ) override;
			virtual void visit( FunctionDecl * funcDecl ) override;

			virtual void visit( CompoundStmt * compoundStmt ) override;
			virtual void visit( ReturnStmt * returnStmt ) override;
			virtual void visit( BranchStmt * stmt ) override;
		private:
			void handleGoto( BranchStmt * stmt );

			LabelFinder & finder;
			LabelFinder::LabelMap & labelVars;
			OrderedDeclsStack reverseDeclOrder;
		};

		class FixInit : public WithStmtsToAdd {
		  public:
			/// expand each object declaration to use its constructor after it is declared.
			static void fixInitializers( std::list< Declaration * > &translationUnit );

			DeclarationWithType * postmutate( ObjectDecl *objDecl );

			std::list< Declaration * > staticDtorDecls;
		};

		class FixCopyCtors final : public GenPoly::PolyMutator {
		  public:
			FixCopyCtors( UnqCount & unqCount ) : unqCount( unqCount ){}
			/// expand ImplicitCopyCtorExpr nodes into the temporary declarations, copy constructors, call expression,
			/// and destructors
			static void fixCopyCtors( std::list< Declaration * > &translationUnit, UnqCount & unqCount );

			typedef GenPoly::PolyMutator Parent;
			using Parent::mutate;
			virtual Expression * mutate( ImplicitCopyCtorExpr * impCpCtorExpr ) override;
			virtual Expression * mutate( UniqueExpr * unqExpr ) override;
			virtual Expression * mutate( StmtExpr * stmtExpr ) override;

			UnqCount & unqCount;
		};

		class GenStructMemberCalls final : public SymTab::Indexer {
		  public:
			typedef Indexer Parent;
			/// generate default/copy ctor and dtor calls for user-defined struct ctor/dtors
			/// for any member that is missing a corresponding ctor/dtor call.
			/// error if a member is used before constructed
			static void generate( std::list< Declaration * > & translationUnit );

			using Parent::visit;

			virtual void visit( FunctionDecl * funcDecl ) override;

			virtual void visit( MemberExpr * memberExpr ) override;
			virtual void visit( ApplicationExpr * appExpr ) override;

			SemanticError errors;
		  private:
			void handleFirstParam( Expression * firstParam );
			template< typename... Params >
			void emit( CodeLocation, const Params &... params );

			FunctionDecl * function = 0;
			std::set< DeclarationWithType * > unhandled;
			std::map< DeclarationWithType *, CodeLocation > usedUninit;
			ObjectDecl * thisParam = 0;
			bool isCtor = false; // true if current function is a constructor
			StructDecl * structDecl = 0;
		};

		// very simple resolver-like mutator class - used to
		// resolve UntypedExprs that are found within newly
		// generated constructor/destructor calls
		class MutatingResolver final : public Mutator {
		  public:
			MutatingResolver( SymTab::Indexer & indexer ) : indexer( indexer ) {}

			using Mutator::mutate;
			virtual DeclarationWithType* mutate( ObjectDecl *objectDecl ) override;
			virtual Expression* mutate( UntypedExpr *untypedExpr ) override;

		  private:
			SymTab::Indexer & indexer;
		};

		class FixCtorExprs final : public GenPoly::DeclMutator {
		  public:
			/// expands ConstructorExpr nodes into comma expressions, using a temporary for the first argument
			static void fix( std::list< Declaration * > & translationUnit );

			using GenPoly::DeclMutator::mutate;
			virtual Expression * mutate( ConstructorExpr * ctorExpr ) override;
		};
	} // namespace

	void fix( std::list< Declaration * > & translationUnit, const std::string & filename, bool inLibrary ) {
		// fixes ConstructorInit for global variables. should happen before fixInitializers.
		InitTweak::fixGlobalInit( translationUnit, filename, inLibrary );

		EnvMap envMap;
		UnqCount unqCount;

		InsertImplicitCalls::insert( translationUnit, envMap );
		ResolveCopyCtors::resolveImplicitCalls( translationUnit, envMap, unqCount );
		InsertDtors::insert( translationUnit );
		FixInit::fixInitializers( translationUnit );

		// FixCopyCtors must happen after FixInit, so that destructors are placed correctly
		FixCopyCtors::fixCopyCtors( translationUnit, unqCount );

		GenStructMemberCalls::generate( translationUnit );
		// xxx - ctor expansion currently has to be after FixCopyCtors, because there is currently a
		// hack in the way untyped assignments are generated, where the first argument cannot have
		// its address taken because of the way codegeneration handles UntypedExpr vs. ApplicationExpr.
		// Thus such assignment exprs must never pushed through expression resolution (and thus should
		// not go through the FixCopyCtors pass), otherwise they will fail -- guaranteed.
		// Also needs to happen after GenStructMemberCalls, since otherwise member constructors exprs
		// don't look right, and a member can be constructed more than once.
		FixCtorExprs::fix( translationUnit );
	}

	namespace {
		void InsertImplicitCalls::insert( std::list< Declaration * > & translationUnit, EnvMap & envMap ) {
			PassVisitor<InsertImplicitCalls> inserter( envMap );
			mutateAll( translationUnit, inserter );
		}

		void ResolveCopyCtors::resolveImplicitCalls( std::list< Declaration * > & translationUnit, const EnvMap & envMap, UnqCount & unqCount ) {
			ResolveCopyCtors resolver( envMap, unqCount );
			acceptAll( translationUnit, resolver );
		}

		void FixInit::fixInitializers( std::list< Declaration * > & translationUnit ) {
			PassVisitor<FixInit> fixer;

			// can't use mutateAll, because need to insert declarations at top-level
			// can't use DeclMutator, because sometimes need to insert IfStmt, etc.
			SemanticError errors;
			for ( std::list< Declaration * >::iterator i = translationUnit.begin(); i != translationUnit.end(); ++i ) {
				try {
					*i = maybeMutate( *i, fixer );
					translationUnit.splice( i, fixer.pass.staticDtorDecls );
				} catch( SemanticError &e ) {
					e.set_location( (*i)->location );
					errors.append( e );
				} // try
			} // for
			if ( ! errors.isEmpty() ) {
				throw errors;
			} // if
		}

		void InsertDtors::insert( std::list< Declaration * > & translationUnit ) {
			LabelFinder finder;
			InsertDtors inserter( finder );
			acceptAll( translationUnit, inserter );
		}

		void FixCopyCtors::fixCopyCtors( std::list< Declaration * > & translationUnit, UnqCount & unqCount ) {
			FixCopyCtors fixer( unqCount );
			mutateAll( translationUnit, fixer );
		}

		void GenStructMemberCalls::generate( std::list< Declaration * > & translationUnit ) {
			GenStructMemberCalls warner;
			acceptAll( translationUnit, warner );
		}

		void FixCtorExprs::fix( std::list< Declaration * > & translationUnit ) {
			FixCtorExprs fixer;
			fixer.mutateDeclarationList( translationUnit );
		}

		Expression * InsertImplicitCalls::postmutate( ApplicationExpr * appExpr ) {
			assert( appExpr );

			if ( VariableExpr * function = dynamic_cast< VariableExpr * > ( appExpr->get_function() ) ) {
				if ( LinkageSpec::isBuiltin( function->get_var()->get_linkage() ) ) {
					// optimization: don't need to copy construct in order to call intrinsic functions
					return appExpr;
				} else if ( DeclarationWithType * funcDecl = dynamic_cast< DeclarationWithType * > ( function->get_var() ) ) {
					FunctionType * ftype = dynamic_cast< FunctionType * >( GenPoly::getFunctionType( funcDecl->get_type() ) );
					assert( ftype );
					if ( isConstructor( funcDecl->get_name() ) && ftype->get_parameters().size() == 2 ) {
						Type * t1 = ftype->get_parameters().front()->get_type();
						Type * t2 = ftype->get_parameters().back()->get_type();
						PointerType * ptrType = safe_dynamic_cast< PointerType * > ( t1 );

						if ( ResolvExpr::typesCompatible( ptrType->get_base(), t2, SymTab::Indexer() ) ) {
							// optimization: don't need to copy construct in order to call a copy constructor
							return appExpr;
						} // if
					} else if ( isDestructor( funcDecl->get_name() ) ) {
						// correctness: never copy construct arguments to a destructor
						return appExpr;
					} // if
				} // if
			} // if
			CP_CTOR_PRINT( std::cerr << "InsertImplicitCalls: adding a wrapper " << appExpr << std::endl; )

			// wrap each function call so that it is easy to identify nodes that have to be copy constructed
			ImplicitCopyCtorExpr * expr = new ImplicitCopyCtorExpr( appExpr );
			// save the type substitution into the envMap so that it is easy to find.
			// Ensure it is not deleted with the ImplicitCopyCtorExpr by removing it before deletion.
			// The substitution is needed to obtain the type of temporary variables so that copy constructor
			// calls can be resolved. Normally this is what PolyMutator is for, but the pass that resolves
			// copy constructor calls must be an Indexer. We could alternatively make a PolyIndexer which
			// saves the environment, or compute the types of temporaries here, but it's much simpler to
			// save the environment here, and more cohesive to compute temporary variables and resolve copy
			// constructor calls together.
			assert( env );
			envMap[expr] = env;
			return expr;
		}

		void InsertImplicitCalls::premutate( StmtExpr * stmtExpr ) {
			assert( env );
			envMap[stmtExpr] = env;
		}

		bool ResolveCopyCtors::skipCopyConstruct( Type * type ) {
			return dynamic_cast< VarArgsType * >( type ) || GenPoly::getFunctionType( type ) || Tuples::isTtype( type );
		}

		Expression * ResolveCopyCtors::makeCtorDtor( const std::string & fname, ObjectDecl * var, Expression * cpArg ) {
			assert( var );
			// arrays are not copy constructed, so this should always be an ExprStmt
			ImplicitCtorDtorStmt * stmt = genCtorDtor( fname, var, cpArg );
			ExprStmt * exprStmt = safe_dynamic_cast< ExprStmt * >( stmt->get_callStmt() );
			Expression * untyped = exprStmt->get_expr();

			// resolve copy constructor
			// should only be one alternative for copy ctor and dtor expressions, since all arguments are fixed
			// (VariableExpr and already resolved expression)
			CP_CTOR_PRINT( std::cerr << "ResolvingCtorDtor " << untyped << std::endl; )
			Expression * resolved = ResolvExpr::findVoidExpression( untyped, *this );
			assert( resolved );
			if ( resolved->get_env() ) {
				// Extract useful information and discard new environments. Keeping them causes problems in PolyMutator passes.
				env->add( *resolved->get_env() );
				delete resolved->get_env();
				resolved->set_env( nullptr );
			} // if

			delete stmt;
			return resolved;
		}

		void ResolveCopyCtors::copyConstructArg( Expression *& arg, ImplicitCopyCtorExpr * impCpCtorExpr ) {
			static UniqueName tempNamer("_tmp_cp");
			assert( env );
			CP_CTOR_PRINT( std::cerr << "Type Substitution: " << *env << std::endl; )
			assert( arg->has_result() );
			Type * result = arg->get_result();
			if ( skipCopyConstruct( result ) ) return; // skip certain non-copyable types

			// type may involve type variables, so apply type substitution to get temporary variable's actual type
			result = result->clone();
			env->apply( result );
			ObjectDecl * tmp = new ObjectDecl( tempNamer.newName(), Type::StorageClasses(), LinkageSpec::C, 0, result, 0 );
			tmp->get_type()->set_const( false );

			// create and resolve copy constructor
			CP_CTOR_PRINT( std::cerr << "makeCtorDtor for an argument" << std::endl; )
			Expression * cpCtor = makeCtorDtor( "?{}", tmp, arg );

			if ( ApplicationExpr * appExpr = dynamic_cast< ApplicationExpr * >( cpCtor ) ) {
				// if the chosen constructor is intrinsic, the copy is unnecessary, so
				// don't create the temporary and don't call the copy constructor
				VariableExpr * function = dynamic_cast< VariableExpr * >( appExpr->get_function() );
				assert( function );
				if ( function->get_var()->get_linkage() == LinkageSpec::Intrinsic ) return;
			}

			// replace argument to function call with temporary
			arg = new CommaExpr( cpCtor, new VariableExpr( tmp ) );
			impCpCtorExpr->get_tempDecls().push_back( tmp );
			impCpCtorExpr->get_dtors().push_front( makeCtorDtor( "^?{}", tmp ) );
		}

		void ResolveCopyCtors::destructRet( ObjectDecl * ret, ImplicitCopyCtorExpr * impCpCtorExpr ) {
			impCpCtorExpr->get_dtors().push_front( makeCtorDtor( "^?{}", ret ) );
		}

		void ResolveCopyCtors::visit( ImplicitCopyCtorExpr *impCpCtorExpr ) {
			CP_CTOR_PRINT( std::cerr << "ResolveCopyCtors: " << impCpCtorExpr << std::endl; )
			Parent::visit( impCpCtorExpr );
			env = envMap.at(impCpCtorExpr);
			assert( env );

			ApplicationExpr * appExpr = impCpCtorExpr->get_callExpr();

			// take each argument and attempt to copy construct it.
			for ( Expression * & arg : appExpr->get_args() ) {
				copyConstructArg( arg, impCpCtorExpr );
			} // for

			// each return value from the call needs to be connected with an ObjectDecl at the call site, which is
			// initialized with the return value and is destructed later
			// xxx - handle named return values?
			Type * result = appExpr->get_result();
			if ( ! result->isVoid() ) {
				static UniqueName retNamer("_tmp_cp_ret");
				result = result->clone();
				env->apply( result );
				ObjectDecl * ret = new ObjectDecl( retNamer.newName(), Type::StorageClasses(), LinkageSpec::C, 0, result, 0 );
				ret->get_type()->set_const( false );
				impCpCtorExpr->get_returnDecls().push_back( ret );
				CP_CTOR_PRINT( std::cerr << "makeCtorDtor for a return" << std::endl; )
				if ( ! result->get_lvalue() ) {
					// destructing lvalue returns is bad because it can cause multiple destructor calls to the same object - the returned object is not a temporary
					destructRet( ret, impCpCtorExpr );
				}
			} // for
			CP_CTOR_PRINT( std::cerr << "after Resolving: " << impCpCtorExpr << std::endl; )
		}

		void ResolveCopyCtors::visit( StmtExpr * stmtExpr ) {
			Parent::visit( stmtExpr );
			env = envMap.at(stmtExpr);
			assert( stmtExpr->get_result() );
			Type * result = stmtExpr->get_result();
			if ( ! result->isVoid() ) {
				static UniqueName retNamer("_tmp_stmtexpr_ret");

				// create variable that will hold the result of the stmt expr
				result = result->clone();
				env->apply( result );
				ObjectDecl * ret = new ObjectDecl( retNamer.newName(), Type::StorageClasses(), LinkageSpec::C, 0, result, 0 );
				ret->get_type()->set_const( false );
				stmtExpr->get_returnDecls().push_front( ret );

				// must have a non-empty body, otherwise it wouldn't have a result
				CompoundStmt * body = stmtExpr->get_statements();
				assert( ! body->get_kids().empty() );
				// must be an ExprStmt, otherwise it wouldn't have a result
				ExprStmt * last = safe_dynamic_cast< ExprStmt * >( body->get_kids().back() );
				last->set_expr( makeCtorDtor( "?{}", ret, last->get_expr() ) );

				stmtExpr->get_dtors().push_front( makeCtorDtor( "^?{}", ret ) );
			} // if

		}

		void ResolveCopyCtors::visit( UniqueExpr * unqExpr ) {
			static std::unordered_set< int > vars;
			unqCount[ unqExpr->get_id() ]++;  // count the number of unique expressions for each ID
			if ( vars.count( unqExpr->get_id() ) ) {
				// xxx - hack to prevent double-handling of unique exprs, otherwise too many temporary variables and destructors are generated
				return;
			}

			Parent::visit( unqExpr );
			// it should never be necessary to wrap a void-returning expression in a UniqueExpr - if this assumption changes, this needs to be rethought
			assert( unqExpr->get_result() );
			if ( ImplicitCopyCtorExpr * impCpCtorExpr = dynamic_cast<ImplicitCopyCtorExpr*>( unqExpr->get_expr() ) ) {
				// note the variable used as the result from the call
				assert( impCpCtorExpr->get_result() && impCpCtorExpr->get_returnDecls().size() == 1 );
				unqExpr->set_var( new VariableExpr( impCpCtorExpr->get_returnDecls().front() ) );
			} else {
				// expr isn't a call expr, so create a new temporary variable to use to hold the value of the unique expression
				unqExpr->set_object( new ObjectDecl( toString("_unq", unqExpr->get_id()), Type::StorageClasses(), LinkageSpec::C, nullptr, unqExpr->get_result()->clone(), nullptr ) );
				unqExpr->set_var( new VariableExpr( unqExpr->get_object() ) );
			}
			vars.insert( unqExpr->get_id() );
		}

		Expression * FixCopyCtors::mutate( ImplicitCopyCtorExpr * impCpCtorExpr ) {
			CP_CTOR_PRINT( std::cerr << "FixCopyCtors: " << impCpCtorExpr << std::endl; )

			impCpCtorExpr = safe_dynamic_cast< ImplicitCopyCtorExpr * >( Parent::mutate( impCpCtorExpr ) );
			std::list< ObjectDecl * > & tempDecls = impCpCtorExpr->get_tempDecls();
			std::list< ObjectDecl * > & returnDecls = impCpCtorExpr->get_returnDecls();
			std::list< Expression * > & dtors = impCpCtorExpr->get_dtors();

			// add all temporary declarations and their constructors
			for ( ObjectDecl * obj : tempDecls ) {
				stmtsToAdd.push_back( new DeclStmt( noLabels, obj ) );
			} // for
			for ( ObjectDecl * obj : returnDecls ) {
				stmtsToAdd.push_back( new DeclStmt( noLabels, obj ) );
			} // for

			// add destructors after current statement
			for ( Expression * dtor : dtors ) {
				stmtsToAddAfter.push_back( new ExprStmt( noLabels, dtor ) );
			} // for

			// xxx - update to work with multiple return values
			ObjectDecl * returnDecl = returnDecls.empty() ? NULL : returnDecls.front();
			Expression * callExpr = impCpCtorExpr->get_callExpr();

			CP_CTOR_PRINT( std::cerr << "Coming out the back..." << impCpCtorExpr << std::endl; )

			// detach fields from wrapper node so that it can be deleted without deleting too much
			dtors.clear();
			tempDecls.clear();
			returnDecls.clear();
			impCpCtorExpr->set_callExpr( NULL );
			impCpCtorExpr->set_env( NULL );
			delete impCpCtorExpr;

			if ( returnDecl ) {
				UntypedExpr * assign = new UntypedExpr( new NameExpr( "?=?" ) );
				assign->get_args().push_back( new VariableExpr( returnDecl ) );
				assign->get_args().push_back( callExpr );
				// know the result type of the assignment is the type of the LHS (minus the pointer), so
				// add that onto the assignment expression so that later steps have the necessary information
				assign->set_result( returnDecl->get_type()->clone() );

				Expression * retExpr = new CommaExpr( assign, new VariableExpr( returnDecl ) );
				if ( callExpr->get_result()->get_lvalue() ) {
					// lvalue returning functions are funny. Lvalue.cc inserts a *? in front of any lvalue returning
					// non-intrinsic function. Add an AddressExpr to the call to negate the derefence and change the
					// type of the return temporary from T to T* to properly capture the return value. Then dereference
					// the result of the comma expression, since the lvalue returning call was originally wrapped with
					// an AddressExpr.  Effectively, this turns
					//   lvalue T f();
					//   &*f();
					// into
					//   T * f();
					//   T * tmp_cp_retN;
					//   &*(tmp_cp_retN = &*f(), tmp_cp_retN);		// the first * and second & are generated here
					// which work out in terms of types, but is pretty messy. It would be nice to find a better way.
					assign->get_args().back() = new AddressExpr( assign->get_args().back() );

					returnDecl->set_type( new PointerType( Type::Qualifiers(), returnDecl->get_type() ) );
					retExpr->set_result( new PointerType( Type::Qualifiers(), retExpr->get_result() ) );
					retExpr = UntypedExpr::createDeref( retExpr );
				} // if
				// move env from callExpr to retExpr
				retExpr->set_env( callExpr->get_env() );
				callExpr->set_env( nullptr );
				return retExpr;
			} else {
				return callExpr;
			} // if
		}

		Expression * FixCopyCtors::mutate( StmtExpr * stmtExpr ) {
			// function call temporaries should be placed at statement-level, rather than nested inside of a new statement expression,
			// since temporaries can be shared across sub-expressions, e.g.
			//   [A, A] f();
			//   g([A] x, [A] y);
			//   f(g());
			// f is executed once, so the return temporary is shared across the tuple constructors for x and y.
			std::list< Statement * > & stmts = stmtExpr->get_statements()->get_kids();
			for ( Statement *& stmt : stmts ) {
				stmt = stmt->acceptMutator( *this );
			} // for
			// stmtExpr = safe_dynamic_cast< StmtExpr * >( Parent::mutate( stmtExpr ) );
			assert( stmtExpr->get_result() );
			Type * result = stmtExpr->get_result();
			if ( ! result->isVoid() ) {
				for ( ObjectDecl * obj : stmtExpr->get_returnDecls() ) {
					stmtsToAdd.push_back( new DeclStmt( noLabels, obj ) );
				} // for
				// add destructors after current statement
				for ( Expression * dtor : stmtExpr->get_dtors() ) {
					stmtsToAddAfter.push_back( new ExprStmt( noLabels, dtor ) );
				} // for
				// must have a non-empty body, otherwise it wouldn't have a result
				CompoundStmt * body = stmtExpr->get_statements();
				assert( ! body->get_kids().empty() );
				assert( ! stmtExpr->get_returnDecls().empty() );
				body->get_kids().push_back( new ExprStmt( noLabels, new VariableExpr( stmtExpr->get_returnDecls().front() ) ) );
				stmtExpr->get_returnDecls().clear();
				stmtExpr->get_dtors().clear();
			}
			assert( stmtExpr->get_returnDecls().empty() );
			assert( stmtExpr->get_dtors().empty() );
			return stmtExpr;
		}

		Expression * FixCopyCtors::mutate( UniqueExpr * unqExpr ) {
			unqCount[ unqExpr->get_id() ]--;
			static std::unordered_map< int, std::list< Statement * > > dtors;
			static std::unordered_map< int, UniqueExpr * > unqMap;
			static std::unordered_set< int > addDeref;
			// has to be done to clean up ImplicitCopyCtorExpr nodes, even when this node was skipped in previous passes
			if ( unqMap.count( unqExpr->get_id() ) ) {
				// take data from other UniqueExpr to ensure consistency
				delete unqExpr->get_expr();
				unqExpr->set_expr( unqMap[unqExpr->get_id()]->get_expr()->clone() );
				delete unqExpr->get_result();
				unqExpr->set_result( maybeClone( unqExpr->get_expr()->get_result() ) );
				if ( unqCount[ unqExpr->get_id() ] == 0 ) {  // insert destructor after the last use of the unique expression
					stmtsToAddAfter.splice( stmtsToAddAfter.end(), dtors[ unqExpr->get_id() ] );
				}
				if ( addDeref.count( unqExpr->get_id() ) ) {
					// other UniqueExpr was dereferenced because it was an lvalue return, so this one should be too
					return UntypedExpr::createDeref( unqExpr );
				}
				return unqExpr;
			}
			FixCopyCtors fixer( unqCount );
			unqExpr->set_expr( unqExpr->get_expr()->acceptMutator( fixer ) ); // stmtexprs contained should not be separately fixed, so this must occur after the lookup
			stmtsToAdd.splice( stmtsToAdd.end(), fixer.stmtsToAdd );
			unqMap[unqExpr->get_id()] = unqExpr;
			if ( unqCount[ unqExpr->get_id() ] == 0 ) {  // insert destructor after the last use of the unique expression
				stmtsToAddAfter.splice( stmtsToAddAfter.end(), dtors[ unqExpr->get_id() ] );
			} else { // remember dtors for last instance of unique expr
				dtors[ unqExpr->get_id() ] = fixer.stmtsToAddAfter;
			}
			if ( UntypedExpr * deref = dynamic_cast< UntypedExpr * >( unqExpr->get_expr() ) ) {
				// unique expression is now a dereference, because the inner expression is an lvalue returning function call.
				// Normalize the expression by dereferencing the unique expression, rather than the inner expression
				// (i.e. move the dereference out a level)
				assert( getFunctionName( deref ) == "*?" );
				unqExpr->set_expr( getCallArg( deref, 0 ) );
				getCallArg( deref, 0 ) = unqExpr;
				addDeref.insert( unqExpr->get_id() );
				return deref;
			}
			return unqExpr;
		}

		DeclarationWithType *FixInit::postmutate( ObjectDecl *objDecl ) {
			// since this removes the init field from objDecl, it must occur after children are mutated (i.e. postmutate)
			if ( ConstructorInit * ctorInit = dynamic_cast< ConstructorInit * >( objDecl->get_init() ) ) {
				// a decision should have been made by the resolver, so ctor and init are not both non-NULL
				assert( ! ctorInit->get_ctor() || ! ctorInit->get_init() );
				if ( Statement * ctor = ctorInit->get_ctor() ) {
					if ( objDecl->get_storageClasses().is_static ) {
						// originally wanted to take advantage of gcc nested functions, but
						// we get memory errors with this approach. To remedy this, the static
						// variable is hoisted when the destructor needs to be called.
						//
						// generate:
						// static T __objName_static_varN;
						// void __objName_dtor_atexitN() {
						//   __dtor__...;
						// }
						// int f(...) {
						//   ...
						//   static bool __objName_uninitialized = true;
						//   if (__objName_uninitialized) {
						//     __ctor(__objName);
						//     __objName_uninitialized = false;
						//     atexit(__objName_dtor_atexitN);
						//   }
						//   ...
						// }

						static UniqueName dtorCallerNamer( "_dtor_atexit" );

						// static bool __objName_uninitialized = true
						BasicType * boolType = new BasicType( Type::Qualifiers(), BasicType::Bool );
						SingleInit * boolInitExpr = new SingleInit( new ConstantExpr( Constant::from_int( 1 ) ), noDesignators );
						ObjectDecl * isUninitializedVar = new ObjectDecl( objDecl->get_mangleName() + "_uninitialized", Type::StorageClasses( Type::Static ), LinkageSpec::Cforall, 0, boolType, boolInitExpr );
						isUninitializedVar->fixUniqueId();

						// __objName_uninitialized = false;
						UntypedExpr * setTrue = new UntypedExpr( new NameExpr( "?=?" ) );
						setTrue->get_args().push_back( new VariableExpr( isUninitializedVar ) );
						setTrue->get_args().push_back( new ConstantExpr( Constant::from_int( 0 ) ) );

						// generate body of if
						CompoundStmt * initStmts = new CompoundStmt( noLabels );
						std::list< Statement * > & body = initStmts->get_kids();
						body.push_back( ctor );
						body.push_back( new ExprStmt( noLabels, setTrue ) );

						// put it all together
						IfStmt * ifStmt = new IfStmt( noLabels, new VariableExpr( isUninitializedVar ), initStmts, 0 );
						stmtsToAddAfter.push_back( new DeclStmt( noLabels, isUninitializedVar ) );
						stmtsToAddAfter.push_back( ifStmt );

						Statement * dtor = ctorInit->get_dtor();
						objDecl->set_init( NULL );
						ctorInit->set_ctor( NULL );
						ctorInit->set_dtor( nullptr );
						if ( dtor ) {
							// if the object has a non-trivial destructor, have to
							// hoist it and the object into the global space and
							// call the destructor function with atexit.

							Statement * dtorStmt = dtor->clone();

							// void __objName_dtor_atexitN(...) {...}
							FunctionDecl * dtorCaller = new FunctionDecl( objDecl->get_mangleName() + dtorCallerNamer.newName(), Type::StorageClasses( Type::Static ), LinkageSpec::C, new FunctionType( Type::Qualifiers(), false ), new CompoundStmt( noLabels ) );
							dtorCaller->fixUniqueId();
							dtorCaller->get_statements()->push_back( dtorStmt );

							// atexit(dtor_atexit);
							UntypedExpr * callAtexit = new UntypedExpr( new NameExpr( "atexit" ) );
							callAtexit->get_args().push_back( new VariableExpr( dtorCaller ) );

							body.push_back( new ExprStmt( noLabels, callAtexit ) );

							// hoist variable and dtor caller decls to list of decls that will be added into global scope
							staticDtorDecls.push_back( objDecl );
							staticDtorDecls.push_back( dtorCaller );

							// need to rename object uniquely since it now appears
							// at global scope and there could be multiple function-scoped
							// static variables with the same name in different functions.
							// Note: it isn't sufficient to modify only the mangleName, because
							// then subsequent Indexer passes can choke on seeing the object's name
							// if another object has the same name and type. An unfortunate side-effect
							// of renaming the object is that subsequent NameExprs may fail to resolve,
							// but there shouldn't be any remaining past this point.
							static UniqueName staticNamer( "_static_var" );
							objDecl->set_name( objDecl->get_name() + staticNamer.newName() );
							objDecl->set_mangleName( SymTab::Mangler::mangle( objDecl ) );

							// xxx - temporary hack: need to return a declaration, but want to hoist the current object out of this scope
							// create a new object which is never used
							static UniqueName dummyNamer( "_dummy" );
							ObjectDecl * dummy = new ObjectDecl( dummyNamer.newName(), Type::StorageClasses( Type::Static ), LinkageSpec::Cforall, 0, new PointerType( Type::Qualifiers(), new VoidType( Type::Qualifiers() ) ), 0, std::list< Attribute * >{ new Attribute("unused") } );
							delete ctorInit;
							return dummy;
						}
					} else {
						ImplicitCtorDtorStmt * implicit = safe_dynamic_cast< ImplicitCtorDtorStmt * > ( ctor );
						ExprStmt * ctorStmt = dynamic_cast< ExprStmt * >( implicit->get_callStmt() );
						ApplicationExpr * ctorCall = nullptr;
						if ( ctorStmt && (ctorCall = isIntrinsicCallExpr( ctorStmt->get_expr() )) && ctorCall->get_args().size() == 2 ) {
							// clean up intrinsic copy constructor calls by making them into SingleInits
							objDecl->set_init( new SingleInit( ctorCall->get_args().back() ) );
							ctorCall->get_args().pop_back();
						} else {
							stmtsToAddAfter.push_back( ctor );
							objDecl->set_init( NULL );
							ctorInit->set_ctor( NULL );
						}
					} // if
				} else if ( Initializer * init = ctorInit->get_init() ) {
					objDecl->set_init( init );
					ctorInit->set_init( NULL );
				} else {
					// no constructor and no initializer, which is okay
					objDecl->set_init( NULL );
				} // if
				delete ctorInit;
			} // if
			return objDecl;
		}

		void ObjDeclCollector::visit( CompoundStmt * compoundStmt ) {
			std::set< ObjectDecl * > prevVars = curVars;
			Parent::visit( compoundStmt );
			curVars = prevVars;
		}

		void ObjDeclCollector::visit( DeclStmt * stmt ) {
			// keep track of all variables currently in scope
			if ( ObjectDecl * objDecl = dynamic_cast< ObjectDecl * > ( stmt->get_decl() ) ) {
				curVars.insert( objDecl );
			} // if
			Parent::visit( stmt );
		}

		void LabelFinder::handleStmt( Statement * stmt ) {
			// for each label, remember the variables in scope at that label.
			for ( Label l : stmt->get_labels() ) {
				vars[l] = curVars;
			} // for
		}

		template<typename Iterator, typename OutputIterator>
		void insertDtors( Iterator begin, Iterator end, OutputIterator out ) {
			for ( Iterator it = begin ; it != end ; ++it ) {
				// extract destructor statement from the object decl and insert it into the output. Note that this is
				// only called on lists of non-static objects with implicit non-intrinsic dtors, so if the user manually
				// calls an intrinsic dtor then the call must (and will) still be generated since the argument may
				// contain side effects.
				ObjectDecl * objDecl = *it;
				ConstructorInit * ctorInit = dynamic_cast< ConstructorInit * >( objDecl->get_init() );
				assert( ctorInit && ctorInit->get_dtor() );
				*out++ = ctorInit->get_dtor()->clone();
			} // for
		}

		void InsertDtors::visit( ObjectDecl * objDecl ) {
			// remember non-static destructed objects so that their destructors can be inserted later
			if ( ! objDecl->get_storageClasses().is_static ) {
				if ( ConstructorInit * ctorInit = dynamic_cast< ConstructorInit * >( objDecl->get_init() ) ) {
					// a decision should have been made by the resolver, so ctor and init are not both non-NULL
					assert( ! ctorInit->get_ctor() || ! ctorInit->get_init() );
					Statement * dtor = ctorInit->get_dtor();
					if ( dtor && ! isIntrinsicSingleArgCallStmt( dtor ) ) {
						// don't need to call intrinsic dtor, because it does nothing, but
						// non-intrinsic dtors must be called
						reverseDeclOrder.front().push_front( objDecl );
					} // if
				} // if
			} // if
			Parent::visit( objDecl );
		}

		template< typename Visitor >
		void handleFuncDecl( FunctionDecl * funcDecl, Visitor & visitor ) {
			maybeAccept( funcDecl->get_functionType(), visitor );
			maybeAccept( funcDecl->get_statements(), visitor );
		}

		void InsertDtors::visit( FunctionDecl * funcDecl ) {
			// each function needs to have its own set of labels
			ValueGuard< LabelFinder::LabelMap > oldLabels( labelVars );
			labelVars.clear();
			handleFuncDecl( funcDecl, finder );

			// all labels for this function have been collected, insert destructors as appropriate.
			// can't be Parent::mutate, because ObjDeclCollector bottoms out on FunctionDecl
			handleFuncDecl( funcDecl, *this );
		}

		void InsertDtors::visit( CompoundStmt * compoundStmt ) {
			// visit statements - this will also populate reverseDeclOrder list.  don't want to dump all destructors
			// when block is left, just the destructors associated with variables defined in this block, so push a new
			// list to the top of the stack so that we can differentiate scopes
			reverseDeclOrder.push_front( OrderedDecls() );
			Parent::visit( compoundStmt );

			// add destructors for the current scope that we're exiting, unless the last statement is a return, which
			// causes unreachable code warnings
			std::list< Statement * > & statements = compoundStmt->get_kids();
			if ( ! statements.empty() && ! dynamic_cast< ReturnStmt * >( statements.back() ) ) {
				insertDtors( reverseDeclOrder.front().begin(), reverseDeclOrder.front().end(), back_inserter( statements ) );
			}
			reverseDeclOrder.pop_front();
		}

		void InsertDtors::visit( __attribute((unused)) ReturnStmt * returnStmt ) {
			// return exits all scopes, so dump destructors for all scopes
			for ( OrderedDecls & od : reverseDeclOrder ) {
				insertDtors( od.begin(), od.end(), back_inserter( stmtsToAdd ) );
			} // for
		}

		// Handle break/continue/goto in the same manner as C++.  Basic idea: any objects that are in scope at the
		// BranchStmt but not at the labelled (target) statement must be destructed.  If there are any objects in scope
		// at the target location but not at the BranchStmt then those objects would be uninitialized so notify the user
		// of the error.  See C++ Reference 6.6 Jump Statements for details.
		void InsertDtors::handleGoto( BranchStmt * stmt ) {
			assert( stmt->get_target() != "" && "BranchStmt missing a label" );
			// S_L = lvars = set of objects in scope at label definition
			// S_G = curVars = set of objects in scope at goto statement
			ObjectSet & lvars = labelVars[ stmt->get_target() ];

			DTOR_PRINT(
				std::cerr << "at goto label: " << stmt->get_target().get_name() << std::endl;
				std::cerr << "S_G = " << printSet( curVars ) << std::endl;
				std::cerr << "S_L = " << printSet( lvars ) << std::endl;
			)

			ObjectSet diff;
			// S_L-S_G results in set of objects whose construction is skipped - it's an error if this set is non-empty
			std::set_difference( lvars.begin(), lvars.end(), curVars.begin(), curVars.end(), std::inserter( diff, diff.begin() ) );
			DTOR_PRINT(
				std::cerr << "S_L-S_G = " << printSet( diff ) << std::endl;
			)
			if ( ! diff.empty() ) {
				throw SemanticError( std::string("jump to label '") + stmt->get_target().get_name() + "' crosses initialization of " + (*diff.begin())->get_name() + " ", stmt );
			} // if
			// S_G-S_L results in set of objects that must be destructed
			diff.clear();
			std::set_difference( curVars.begin(), curVars.end(), lvars.begin(), lvars.end(), std::inserter( diff, diff.end() ) );
			DTOR_PRINT(
				std::cerr << "S_G-S_L = " << printSet( diff ) << std::endl;
			)
			if ( ! diff.empty() ) {
				// go through decl ordered list of objectdecl. for each element that occurs in diff, output destructor
				OrderedDecls ordered;
				for ( OrderedDecls & rdo : reverseDeclOrder ) {
					// add elements from reverseDeclOrder into ordered if they occur in diff - it is key that this happens in reverse declaration order.
					copy_if( rdo.begin(), rdo.end(), back_inserter( ordered ), [&]( ObjectDecl * objDecl ) { return diff.count( objDecl ); } );
				} // for
				insertDtors( ordered.begin(), ordered.end(), back_inserter( stmtsToAdd ) );
			} // if
		}

		void InsertDtors::visit( BranchStmt * stmt ) {
			switch( stmt->get_type() ) {
			  case BranchStmt::Continue:
			  case BranchStmt::Break:
				// could optimize the break/continue case, because the S_L-S_G check is unnecessary (this set should
				// always be empty), but it serves as a small sanity check.
			  case BranchStmt::Goto:
				handleGoto( stmt );
				break;
			  default:
				assert( false );
			} // switch
		}

		bool checkWarnings( FunctionDecl * funcDecl ) {
			// only check for warnings if the current function is a user-defined
			// constructor or destructor
			if ( ! funcDecl ) return false;
			if ( ! funcDecl->get_statements() ) return false;
			return isCtorDtor( funcDecl->get_name() ) && ! LinkageSpec::isOverridable( funcDecl->get_linkage() );
		}

		void GenStructMemberCalls::visit( FunctionDecl * funcDecl ) {
			ValueGuard< FunctionDecl * > oldFunction( funcDecl );
			ValueGuard< std::set< DeclarationWithType * > > oldUnhandled( unhandled );
			ValueGuard< std::map< DeclarationWithType *, CodeLocation > > oldUsedUninit( usedUninit );
			ValueGuard< ObjectDecl * > oldThisParam( thisParam );
			ValueGuard< bool > oldIsCtor( isCtor );
			ValueGuard< StructDecl * > oldStructDecl( structDecl );
			errors = SemanticError();  // clear previous errors

			// need to start with fresh sets
			unhandled.clear();
			usedUninit.clear();

			function = funcDecl;
			isCtor = isConstructor( function->get_name() );
			if ( checkWarnings( function ) ) {
				FunctionType * type = function->get_functionType();
				assert( ! type->get_parameters().empty() );
				thisParam = safe_dynamic_cast< ObjectDecl * >( type->get_parameters().front() );
				PointerType * ptrType = safe_dynamic_cast< PointerType * > ( thisParam->get_type() );
				StructInstType * structType = dynamic_cast< StructInstType * >( ptrType->get_base() );
				if ( structType ) {
					structDecl = structType->get_baseStruct();
					for ( Declaration * member : structDecl->get_members() ) {
						if ( ObjectDecl * field = dynamic_cast< ObjectDecl * >( member ) ) {
							// record all of the struct type's members that need to be constructed or
							// destructed by the end of the function
							unhandled.insert( field );
						}
					}
				}
			}
			Parent::visit( function );

			// remove the unhandled objects from usedUninit, because a call is inserted
			// to handle them - only objects that are later constructed are used uninitialized.
			std::map< DeclarationWithType *, CodeLocation > diff;
			// need the comparator since usedUninit and unhandled have different types
			struct comp_t {
				typedef decltype(usedUninit)::value_type usedUninit_t;
				typedef decltype(unhandled)::value_type unhandled_t;
				bool operator()(usedUninit_t x, unhandled_t y) { return x.first < y; }
				bool operator()(unhandled_t x, usedUninit_t y) { return x < y.first; }
			} comp;
			std::set_difference( usedUninit.begin(), usedUninit.end(), unhandled.begin(), unhandled.end(), std::inserter( diff, diff.begin() ), comp );
			for ( auto p : diff ) {
				DeclarationWithType * member = p.first;
				CodeLocation loc = p.second;
				// xxx - make error message better by also tracking the location that the object is constructed at?
				emit( loc, "in ", CodeGen::genPrettyType( function->get_functionType(), function->get_name() ), ", field ", member->get_name(), " used before being constructed" );
			}

			if ( ! unhandled.empty() ) {
				// need to explicitly re-add function parameters in order to resolve copy constructors
				enterScope();
				maybeAccept( function->get_functionType(), *this );

				// need to iterate through members in reverse in order for
				// ctor/dtor statements to come out in the right order
				for ( Declaration * member : reverseIterate( structDecl->get_members() ) ) {
					DeclarationWithType * field = dynamic_cast< DeclarationWithType * >( member );
					// skip non-DWT members
					if ( ! field ) continue;
					// skip handled members
					if ( ! unhandled.count( field ) ) continue;

					// insert and resolve default/copy constructor call for each field that's unhandled
					std::list< Statement * > stmt;
					UntypedExpr * deref = UntypedExpr::createDeref( new VariableExpr( thisParam ) );

					Expression * arg2 = 0;
					if ( isCopyConstructor( function ) ) {
						// if copy ctor, need to pass second-param-of-this-function.field
						std::list< DeclarationWithType * > & params = function->get_functionType()->get_parameters();
						assert( params.size() == 2 );
						arg2 = new MemberExpr( field, new VariableExpr( params.back() ) );
					}
					InitExpander srcParam( arg2 );
					SymTab::genImplicitCall( srcParam, new MemberExpr( field, deref ), function->get_name(), back_inserter( stmt ), field, isCtor );

					assert( stmt.size() <= 1 );
					if ( stmt.size() == 1 ) {
						Statement * callStmt = stmt.front();

						MutatingResolver resolver( *this );
						try {
							callStmt->acceptMutator( resolver );
							if ( isCtor ) {
								function->get_statements()->push_front( callStmt );
							} else {
								// destructor statements should be added at the end
								function->get_statements()->push_back( callStmt );
							}
						} catch ( SemanticError & error ) {
							emit( funcDecl->location, "in ", CodeGen::genPrettyType( function->get_functionType(), function->get_name() ), ", field ", field->get_name(), " not explicitly ", isCtor ? "constructed" : "destructed",  " and no ", isCtor ? "default constructor" : "destructor", " found" );
						}
					}
				}
				leaveScope();
			}
			if (! errors.isEmpty()) {
				throw errors;
			}
		}

		void GenStructMemberCalls::visit( ApplicationExpr * appExpr ) {
			if ( ! checkWarnings( function ) ) return;

			std::string fname = getFunctionName( appExpr );
			if ( fname == function->get_name() ) {
				// call to same kind of function
				Expression * firstParam = appExpr->get_args().front();

				if ( VariableExpr * varExpr = dynamic_cast< VariableExpr * >( firstParam ) ) {
					// if calling another constructor on thisParam, assume that function handles
					// all members - if it doesn't a warning will appear in that function.
					if ( varExpr->get_var() == thisParam ) {
						unhandled.clear();
					}
				} else {
					// if first parameter is a member expression then
					// remove the member from unhandled set.
					handleFirstParam( firstParam );
				}
			}

			Parent::visit( appExpr );
		}

		void GenStructMemberCalls::handleFirstParam( Expression * firstParam ) {
			using namespace std;
			if ( AddressExpr * addrExpr = dynamic_cast< AddressExpr * >( firstParam ) ) {
				if ( MemberExpr * memberExpr = dynamic_cast< MemberExpr * >( addrExpr->get_arg() ) ) {
					if ( ApplicationExpr * deref = dynamic_cast< ApplicationExpr * >( memberExpr->get_aggregate() ) ) {
						if ( getFunctionName( deref ) == "*?" && deref->get_args().size() == 1 ) {
							if ( VariableExpr * varExpr = dynamic_cast< VariableExpr * >( deref->get_args().front() ) ) {
								if ( varExpr->get_var() == thisParam ) {
									unhandled.erase( memberExpr->get_member() );
								}
							}
						}
					}
				}
			}
		}

		void GenStructMemberCalls::visit( MemberExpr * memberExpr ) {
			if ( ! checkWarnings( function ) ) return;
			if ( ! isCtor ) return;

			if ( ApplicationExpr * deref = dynamic_cast< ApplicationExpr * >( memberExpr->get_aggregate() ) ) {
				if ( getFunctionName( deref ) == "*?" && deref->get_args().size() == 1 ) {
					if ( VariableExpr * varExpr = dynamic_cast< VariableExpr * >( deref->get_args().front() ) ) {
						if ( varExpr->get_var() == thisParam ) {
							if ( unhandled.count( memberExpr->get_member() ) ) {
								// emit a warning because a member was used before it was constructed
								usedUninit.insert( { memberExpr->get_member(), memberExpr->location } );
							}
						}
					}
				}
			}
			Parent::visit( memberExpr );
		}

		template< typename Visitor, typename... Params >
		void error( Visitor & v, CodeLocation loc, const Params &... params ) {
			SemanticError err( toString( params... ) );
			err.set_location( loc );
			v.errors.append( err );
		}

		template< typename... Params >
		void GenStructMemberCalls::emit( CodeLocation loc, const Params &... params ) {
			// toggle warnings vs. errors here.
			// warn( params... );
			error( *this, loc, params... );
		}

		DeclarationWithType * MutatingResolver::mutate( ObjectDecl *objectDecl ) {
			// add object to the indexer assumes that there will be no name collisions
			// in generated code. If this changes, add mutate methods for entities with
			// scope and call {enter,leave}Scope explicitly.
			objectDecl->accept( indexer );
			return objectDecl;
		}

		Expression* MutatingResolver::mutate( UntypedExpr *untypedExpr ) {
			return safe_dynamic_cast< ApplicationExpr * >( ResolvExpr::findVoidExpression( untypedExpr, indexer ) );
		}

		Expression * FixCtorExprs::mutate( ConstructorExpr * ctorExpr ) {
			static UniqueName tempNamer( "_tmp_ctor_expr" );
			// xxx - is the size check necessary?
			assert( ctorExpr->has_result() && ctorExpr->get_result()->size() == 1 );

			// xxx - ideally we would reuse the temporary generated from the copy constructor passes from within firstArg if it exists and not generate a temporary if it's unnecessary.
			ObjectDecl * tmp = new ObjectDecl( tempNamer.newName(), Type::StorageClasses(), LinkageSpec::C, nullptr, ctorExpr->get_result()->clone(), nullptr );
			addDeclaration( tmp );

			// xxx - this can be TupleAssignExpr now. Need to properly handle this case.
			ApplicationExpr * callExpr = safe_dynamic_cast< ApplicationExpr * > ( ctorExpr->get_callExpr() );
			TypeSubstitution * env = ctorExpr->get_env();
			ctorExpr->set_callExpr( nullptr );
			ctorExpr->set_env( nullptr );

			Expression *& firstArg = callExpr->get_args().front();
			UntypedExpr * assign = new UntypedExpr( new NameExpr( "?=?" ) );
			assign->get_args().push_back( new VariableExpr( tmp ) );
			assign->get_args().push_back( firstArg );
			assign->set_result( ctorExpr->get_result()->clone() );
			firstArg = assign;

			CommaExpr * commaExpr = new CommaExpr( callExpr, new VariableExpr( tmp ) );
			commaExpr->set_env( env );
			delete ctorExpr;
			return commaExpr;
		}
	} // namespace
} // namespace InitTweak

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
