//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// StatementNode.cc --
//
// Author           : Rodolfo G. Esteves
// Created On       : Sat May 16 14:59:41 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon Aug 15 14:40:05 2016
// Update Count     : 317
//

#include <list>
#include <algorithm>
#include <cassert>

#include "ParseNode.h"
#include "SynTree/Statement.h"
#include "SynTree/Expression.h"
#include "parseutility.h"
#include "Common/utility.h"

using namespace std;


StatementNode::StatementNode( DeclarationNode *decl ) {
	if ( decl ) {
		DeclarationNode *agg = decl->extractAggregate();
		if ( agg ) {
			StatementNode *nextStmt = new StatementNode( new DeclStmt( noLabels, maybeBuild< Declaration >( decl ) ) );
			next = nextStmt;
			if ( decl->get_next() ) {
				next->set_next( new StatementNode( dynamic_cast<DeclarationNode *>(decl->get_next()) ) );
				decl->set_next( 0 );
			} // if
		} else {
			if ( decl->get_next() ) {
				next = new StatementNode( dynamic_cast<DeclarationNode *>( decl->get_next() ) );
				decl->set_next( 0 );
			} // if
			agg = decl;
		} // if
		stmt = new DeclStmt( noLabels, maybeBuild<Declaration>(agg) );
	} else {
		assert( false );
	} // if
}

StatementNode *StatementNode::append_last_case( StatementNode *stmt ) {
	StatementNode *prev = this;
	// find end of list and maintain previous pointer
	for ( StatementNode * curr = prev; curr != nullptr; curr = (StatementNode *)curr->get_next() ) {
		StatementNode *node = dynamic_cast<StatementNode *>(curr);
		assert( node );
		assert( dynamic_cast<CaseStmt *>(node->stmt) );
		prev = curr;
	} // for
	// convert from StatementNode list to Statement list
	StatementNode *node = dynamic_cast<StatementNode *>(prev);
	std::list<Statement *> stmts;
	buildList( stmt, stmts );
	// splice any new Statements to end of current Statements
	CaseStmt * caseStmt = dynamic_cast<CaseStmt *>(node->stmt);
	caseStmt->get_statements().splice( caseStmt->get_statements().end(), stmts );
	return this;
}

Statement *build_expr( ExpressionNode *ctl ) {
	Expression *e = maybeBuild< Expression >( ctl );

	if ( e )
		return new ExprStmt( noLabels, e );
	else
		return new NullStmt( noLabels );
}

Statement *build_if( ExpressionNode *ctl, StatementNode *then_stmt, StatementNode *else_stmt ) {
	Statement *thenb, *elseb = 0;
	std::list<Statement *> branches;
	buildList<Statement, StatementNode>( then_stmt, branches );
	assert( branches.size() == 1 );
	thenb = branches.front();

	if ( else_stmt ) {
		std::list<Statement *> branches;
		buildList<Statement, StatementNode>( else_stmt, branches );
		assert( branches.size() == 1 );
		elseb = branches.front();
	} // if
	return new IfStmt( noLabels, notZeroExpr( maybeBuild<Expression>(ctl) ), thenb, elseb );
}

Statement *build_switch( ExpressionNode *ctl, StatementNode *stmt ) {
	std::list<Statement *> branches;
	buildList<Statement, StatementNode>( stmt, branches );
	assert( branches.size() >= 0 );						// size == 0 for switch (...) {}, i.e., no declaration or statements
	return new SwitchStmt( noLabels, maybeBuild<Expression>(ctl), branches );
}
Statement *build_case( ExpressionNode *ctl ) {
	std::list<Statement *> branches;
	return new CaseStmt( noLabels, maybeBuild<Expression>(ctl), branches );
}
Statement *build_default() {
	std::list<Statement *> branches;
	return new CaseStmt( noLabels, nullptr, branches, true );
}

Statement *build_while( ExpressionNode *ctl, StatementNode *stmt, bool kind ) {
	std::list<Statement *> branches;
	buildList<Statement, StatementNode>( stmt, branches );
	assert( branches.size() == 1 );
	return new WhileStmt( noLabels, notZeroExpr( maybeBuild<Expression>(ctl) ), branches.front(), kind );
}

Statement *build_for( ForCtl *forctl, StatementNode *stmt ) {
	std::list<Statement *> branches;
	buildList<Statement, StatementNode>( stmt, branches );
	assert( branches.size() == 1 );

	std::list<Statement *> init;
	if ( forctl->init != 0 ) {
		buildList( forctl->init, init );
	} // if

	Expression *cond = 0;
	if ( forctl->condition != 0 )
		cond = notZeroExpr( maybeBuild<Expression>(forctl->condition) );

	Expression *incr = 0;
	if ( forctl->change != 0 )
		incr = maybeBuild<Expression>(forctl->change);

	delete forctl;
	return new ForStmt( noLabels, init, cond, incr, branches.front() );
}

Statement *build_branch( std::string identifier, BranchStmt::Type kind ) {
	return new BranchStmt( noLabels, identifier, kind );
}
Statement *build_computedgoto( ExpressionNode *ctl ) {
	return new BranchStmt( noLabels, maybeBuild<Expression>(ctl), BranchStmt::Goto );
}

Statement *build_return( ExpressionNode *ctl ) {
	std::list<Expression *> exps;
	buildList( ctl, exps );
	return new ReturnStmt( noLabels, exps.size() > 0 ? exps.back() : nullptr );
}
Statement *build_throw( ExpressionNode *ctl ) {
	std::list<Expression *> exps;
	buildList( ctl, exps );
	return new ReturnStmt( noLabels, exps.size() > 0 ? exps.back() : nullptr, true );
}

Statement *build_try( StatementNode *try_stmt, StatementNode *catch_stmt, StatementNode *finally_stmt ) {
	std::list<Statement *> branches;
	buildList<Statement, StatementNode>( catch_stmt, branches );
	CompoundStmt *tryBlock = dynamic_cast<CompoundStmt *>(maybeBuild<Statement>(try_stmt));
	assert( tryBlock );
	FinallyStmt *finallyBlock = dynamic_cast<FinallyStmt *>(maybeBuild<Statement>(finally_stmt) );
	return new TryStmt( noLabels, tryBlock, branches, finallyBlock );
}
Statement *build_catch( DeclarationNode *decl, StatementNode *stmt, bool catchAny ) {
	std::list<Statement *> branches;
	buildList<Statement, StatementNode>( stmt, branches );
	assert( branches.size() == 1 );
	return new CatchStmt( noLabels, maybeBuild<Declaration>(decl), branches.front(), catchAny );
}
Statement *build_finally( StatementNode *stmt ) {
	std::list<Statement *> branches;
	buildList<Statement, StatementNode>( stmt, branches );
	assert( branches.size() == 1 );
	return new FinallyStmt( noLabels, dynamic_cast<CompoundStmt *>( branches.front() ) );
}

Statement *build_compound( StatementNode *first ) {
	CompoundStmt *cs = new CompoundStmt( noLabels );
	buildList( first, cs->get_kids() );
	return cs;
}

Statement *build_asmstmt( bool voltile, ConstantExpr *instruction, ExpressionNode *output, ExpressionNode *input, ExpressionNode *clobber, LabelNode *gotolabels ) {
	std::list< Expression * > out, in;
	std::list< ConstantExpr * > clob;

	buildList( output, out );
	buildList( input, in );
	buildList( clobber, clob );
	return new AsmStmt( noLabels, voltile, instruction, out, in, clob, gotolabels ? gotolabels->labels : noLabels );
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
