//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// CurrentObject.h --
//
// Author           : Rob Schluntz
// Created On       : Tue Jun 13 15:28:32 2017
// Last Modified By : Rob Schluntz
// Last Modified On : Tue Jun 13 15:28:44 2017
// Update Count     : 2
//

#include <stack>
#include <iostream>

#include "CurrentObject.h"

#include "SynTree/Declaration.h"
#include "SynTree/Initializer.h"
#include "SynTree/Type.h"

#if 0
#define PRINT(x) x
#else
#define PRINT(x)
#endif

namespace ResolvExpr {
	long long int getConstValue( ConstantExpr * constExpr ) {
		if ( BasicType * basicType = dynamic_cast< BasicType * >( constExpr->get_result() ) ) {
			if ( basicType->get_kind() == BasicType::Char || basicType->get_kind() == BasicType::SignedChar || basicType->get_kind() == BasicType::UnsignedChar ) {
				assertf( constExpr->get_constant()->get_value().size() == 3, "constant value with unusual length: %s", constExpr->get_constant()->get_value().c_str() );
				return constExpr->get_constant()->get_value()[1];
			} else {
				return stoll( constExpr->get_constant()->get_value() );
			}
		} else {
			assertf( false, "unhandled type on getConstValue %s", constExpr->get_result() );
		}
	}

	struct Indenter {
		static const int amt = 2;
		unsigned int indent = 0;

		Indenter & operator+=(int nlevels) { indent += amt*nlevels; return *this; }
		Indenter & operator-=(int nlevels) { indent -= amt*nlevels; return *this; }
		Indenter operator+(int nlevels) { Indenter indenter = *this; return indenter += nlevels; }
		Indenter operator-(int nlevels) { Indenter indenter = *this; return indenter -= nlevels; }
		Indenter & operator++() { return *this += 1; }
		Indenter & operator--() { return *this -= 1; }
	};
	std::ostream & operator<<( std::ostream & out, Indenter & indent ) {
		return out << std::string(indent.indent, ' ');
	}

	class MemberIterator {
	public:
		virtual ~MemberIterator() {}

		virtual void setPosition( std::list< Expression * > & designators ) = 0;
		virtual std::list<InitAlternative> operator*() const = 0;
		virtual operator bool() const = 0;
		virtual MemberIterator & bigStep() = 0;
		virtual MemberIterator & smallStep() = 0;
		virtual Type * getType() = 0;
		virtual Type * getNext() = 0;

		virtual void print( std::ostream & out, Indenter indent ) const = 0;

		virtual std::list<InitAlternative> first() const = 0; // should be protected
	};

	std::ostream & operator<<(std::ostream & out, const MemberIterator & it) {
		Indenter indenter;
		it.print( out, indenter );
		return out;
	}

	/// create a new MemberIterator that traverses a type correctly
	MemberIterator * createMemberIterator( Type * type );

	/// iterates "other" types, e.g. basic types, pointer types, etc. which do not change at list initializer entry
	class SimpleIterator : public MemberIterator {
	public:
		SimpleIterator( Type * type ) : type( type ) {}

		virtual void setPosition( std::list< Expression * > & designators ) {
			assertf( designators.empty(), "simple iterator given non-empty designator..." ); // xxx - might be semantic error
		}

		virtual std::list<InitAlternative> operator*() const { return first(); }
		virtual operator bool() const { return type; }

		// big step is the same as small step
		virtual MemberIterator & bigStep() { return smallStep(); }
		virtual MemberIterator & smallStep() {
			type = nullptr;  // type is nullified on increment since SimpleIterators do not have members
			return *this;
		}

		virtual void print( std::ostream & out, Indenter indent ) const {
			out << "SimpleIterator(" << type << ")";
		}

		virtual Type * getType() { return type; }
		virtual Type * getNext() { return type; }

	protected:
		virtual std::list<InitAlternative> first() const {
			if ( type ) return std::list<InitAlternative>{ { type->clone(), new Designation( {} ) } };
			else return std::list<InitAlternative>{};
		}
	private:
		Type * type = nullptr;
	};

	class ArrayIterator : public MemberIterator {
	public:
		ArrayIterator( ArrayType * at ) : array( at ) {
			PRINT( std::cerr << "Creating array iterator: " << at << std::endl; )
			base = at->get_base();
			memberIter = createMemberIterator( base );
			setSize( at->get_dimension() );
		}

		~ArrayIterator() {
			delete memberIter;
		}

	private:
		void setSize( Expression * expr ) {
			if ( ConstantExpr * constExpr = dynamic_cast< ConstantExpr * >( expr ) ) {
				size = getConstValue( constExpr ); // xxx - if not a constant expression, it's not simple to determine how long the array actually is, which is necessary for initialization to be done correctly -- fix this
				PRINT( std::cerr << "array type with size: " << size << std::endl; )
			}	else if ( CastExpr * castExpr = dynamic_cast< CastExpr * >( expr ) ) {
				setSize( castExpr->get_arg() );
			} else {
				assertf( false, "unhandled expression in setSize: %s", toString( expr ).c_str() );
			}
		}

	public:
		void setPosition( Expression * expr ) {
			// need to permit integer-constant-expressions, including: integer constants, enumeration constants, character constants, sizeof expressions, _Alignof expressions, cast expressions
			if ( ConstantExpr * constExpr = dynamic_cast< ConstantExpr * >( expr ) ) {
				index = getConstValue( constExpr );
			} else if ( CastExpr * castExpr = dynamic_cast< CastExpr * >( expr ) ) {
				setPosition( castExpr->get_arg() );
			} else if ( VariableExpr * varExpr = dynamic_cast< VariableExpr * >( expr ) ) {
				assertf( dynamic_cast<EnumInstType *> ( varExpr->get_result() ), "ArrayIterator given variable that isn't an enum constant", toString( expr ).c_str() );
				index = 0; // xxx - get actual value of enum constant
			} else if ( dynamic_cast< SizeofExpr * >( expr ) || dynamic_cast< AlignofExpr * >( expr ) ) {
				index = 0; // xxx - get actual sizeof/alignof value?
			} else {
				assertf( false, "bad designator given to ArrayIterator: %s", toString( expr ).c_str() );
			}
		}

		virtual void setPosition( std::list< Expression * > & designators ) {
			if ( ! designators.empty() ) {
				setPosition( designators.front() );
				designators.pop_front();
				memberIter->setPosition( designators );
			}
		}

		virtual std::list<InitAlternative> operator*() const {
			return first();
		}

		virtual operator bool() const { return index < size; }

		virtual MemberIterator & bigStep() {
			PRINT( std::cerr << "bigStep in ArrayIterator (" << index << "/" << size << ")" << std::endl; )
			++index;
			delete memberIter;
			if ( index < size ) memberIter = createMemberIterator( base );
			else memberIter = nullptr;
			return *this;
		}

		virtual MemberIterator & smallStep() {
			PRINT( std::cerr << "smallStep in ArrayIterator (" << index << "/" << size << ")" << std::endl; )
			if ( memberIter ) {
				PRINT( std::cerr << "has member iter: " << *memberIter << std::endl; )
				memberIter->smallStep();
				if ( *memberIter ) {
					PRINT( std::cerr << "has valid member iter" << std::endl; )
					return *this;
				}
			}
			return bigStep();
		}

		virtual Type * getType() { return array; }
		virtual Type * getNext() { return base; }

		virtual std::list<InitAlternative> first() const {
			PRINT( std::cerr << "first in ArrayIterator (" << index << "/" << size << ")" << std::endl; )
			if ( memberIter && *memberIter ) {
				std::list<InitAlternative> ret = memberIter->first();
				for ( InitAlternative & alt : ret ) {
					alt.designation->get_designators().push_front( new ConstantExpr( Constant::from_ulong( index ) ) );
				}
				return ret;
			}
			return std::list<InitAlternative>();
		}

		virtual void print( std::ostream & out, Indenter indent ) const {
			out << "ArrayIterator(Array of " << base << ")";
			if ( memberIter ) {
				Indenter childIndent = indent+1;
				out << std::endl << childIndent;
				memberIter->print( out, childIndent );
			}
		}

	private:
		ArrayType * array = nullptr;
		Type * base = nullptr;
		size_t index = 0;
		size_t size = 0;
		MemberIterator * memberIter = nullptr;
	};

	class AggregateIterator : public MemberIterator {
	public:
		typedef std::list<Declaration *>::iterator iterator;
		const char * kind = ""; // for debug
		ReferenceToType * inst = nullptr;
		AggregateDecl * decl = nullptr;
		iterator curMember;
		bool atbegin = true; // false at first {small,big}Step -- this struct type is only added to the possibilities at the beginning
		Type * curType = nullptr;
		MemberIterator * memberIter = nullptr;

		AggregateIterator( const char * kind, ReferenceToType * inst, AggregateDecl * decl ) : kind( kind ), inst( inst ), decl( decl ), curMember( decl->get_members().begin() ) {
			init();
		}

		virtual ~AggregateIterator() {
			delete memberIter;
		}

		bool init() {
			PRINT( std::cerr << "--init()--" << std::endl; )
			if ( curMember != decl->get_members().end() ) {
				if ( ObjectDecl * field = dynamic_cast< ObjectDecl * >( *curMember ) ) {
					PRINT( std::cerr << "incremented to field: " << field << std::endl; )
					curType = field->get_type();
					memberIter = createMemberIterator( curType );
					return true;
				}
			}
			return false;
		}

		virtual std::list<InitAlternative> operator*() const {
			if (memberIter && *memberIter) {
				std::list<InitAlternative> ret = memberIter->first();
				for ( InitAlternative & alt : ret ) {
					PRINT( std::cerr << "iterating and adding designators" << std::endl; )
					alt.designation->get_designators().push_front( new VariableExpr( safe_dynamic_cast< ObjectDecl * >( *curMember ) ) );
				}
				return ret;
			}
			return std::list<InitAlternative>();
		}

		virtual void setPosition( std::list< Expression * > & designators ) {
			if ( ! designators.empty() ) {
				if ( VariableExpr * varExpr = dynamic_cast< VariableExpr * >( designators.front() ) ) {
					for ( curMember = decl->get_members().begin(); curMember != decl->get_members().end(); ++curMember ) {
						if ( *curMember == varExpr->get_var() ) {
							designators.pop_front();
							delete memberIter;
							memberIter = createMemberIterator( varExpr->get_result() );
							curType = varExpr->get_result();
							atbegin = curMember == decl->get_members().begin() && designators.empty(); // xxx - is this right??
							memberIter->setPosition( designators );
							return;
						} // if
					} // for
					assertf( false, "could not find member in %s: %s", kind, toString( varExpr ).c_str() );
				} else {
					assertf( false, "bad designator given to %s: %s", kind, toString( designators.front() ).c_str() );
				} // if
			} // if
		}

		virtual MemberIterator & smallStep() {
			PRINT( std::cerr << "smallStep in " << kind << std::endl; )
			atbegin = false;
			if ( memberIter ) {
				PRINT( std::cerr << "has member iter, incrementing..." << std::endl; )
				memberIter->smallStep();
				if ( *memberIter ) {
					PRINT( std::cerr << "success!" << std::endl; )
					return *this;
				}
			}
			return bigStep();
		}

		virtual Type * getType() { return inst; }
		virtual Type * getNext() {
			if ( memberIter && *memberIter ) return memberIter->getType(); // xxx - ??? recursive call???
			return nullptr;
		}

		virtual std::list<InitAlternative> first() const {
			std::list<InitAlternative> ret;
			PRINT( std::cerr << "first " << kind << std::endl; )
			if ( memberIter && *memberIter ) { // might not need *memberIter??
				PRINT( std::cerr << "adding children" << std::endl; )
				ret = memberIter->first();
				for ( InitAlternative & alt : ret ) {
					PRINT( std::cerr << "iterating and adding designators" << std::endl; )
					alt.designation->get_designators().push_front( new VariableExpr( safe_dynamic_cast< ObjectDecl * >( *curMember ) ) );
				}
			}
			// if ( curMember == std::next( decl->get_members().begin(), 1 ) ) { // xxx - this never triggers because curMember is incremented immediately on construction
			if ( atbegin ) { // xxx - this never triggers because curMember is incremented immediately on construction
				// xxx - what about case of empty struct??
				// only add self if at the very beginning of the structure
				PRINT( std::cerr << "adding self" << std::endl; )
				ret.push_front( { inst->clone(), new Designation( {} ) } );
			}
			return ret;
		}

		virtual void print( std::ostream & out, Indenter indent ) const {
			out << kind << "(" << decl->get_name() << ")";
			if ( memberIter ) {
				Indenter childIndent = indent+1;
				out << std::endl << childIndent;
				memberIter->print( out, childIndent );
			}
		}
	};

	class UnionIterator : public AggregateIterator {
	public:
		UnionIterator( UnionInstType * inst ) : AggregateIterator( "UnionIterator", inst, inst->get_baseUnion() ) {}

		virtual operator bool() const { return (memberIter && *memberIter); }
		virtual MemberIterator & bigStep() {
			// unions only initialize one member
			PRINT( std::cerr << "bigStep in " << kind << std::endl; )
			atbegin = false;
			delete memberIter;
			memberIter = nullptr;
			curType = nullptr;
			curMember = decl->get_members().end();
			return *this;
		}
		virtual std::list<InitAlternative> first() const { return std::list<InitAlternative>{}; }
	};

	class StructIterator : public AggregateIterator {
	public:
		StructIterator( StructInstType * inst ) : AggregateIterator( "StructIterator", inst, inst->get_baseStruct() ) {}

		virtual operator bool() const { return curMember != decl->get_members().end() || (memberIter && *memberIter); }

		virtual MemberIterator & bigStep() {
			PRINT( std::cerr << "bigStep in " << kind << std::endl; )
			atbegin = false;
			delete memberIter;
			memberIter = nullptr;
			curType = nullptr;
			for ( ; curMember != decl->get_members().end(); ) {
				++curMember;
				if ( init() ) {
					return *this;
				}
			}
			return *this;
		}
	};

	MemberIterator * createMemberIterator( Type * type ) {
		if ( ReferenceToType * aggr = dynamic_cast< ReferenceToType * >( type ) ) {
			if ( StructInstType * sit = dynamic_cast< StructInstType * >( aggr ) ) {
				return new StructIterator( sit );
			} else if ( UnionInstType * uit = dynamic_cast< UnionInstType * >( aggr ) ) {
				return new UnionIterator( uit );
			} else {
				assertf( false, "some other reftotype" );
			}
		} else if ( ArrayType * at = dynamic_cast< ArrayType * >( type ) ) {
			return new ArrayIterator( at );
		} else {
			return new SimpleIterator( type );
		}
	}

	CurrentObject::CurrentObject() {}
	CurrentObject::CurrentObject( Type * type ) {
		objStack.push( new SimpleIterator( type ) );
	}


	void CurrentObject::setNext( Designation * designation ) {
		assertf( ! objStack.empty(), "obj stack empty in setNext" );
		PRINT( std::cerr << "____setNext" << designation << std::endl; )
		objStack.top()->setPosition( designation->get_designators() );
	}

	Designation * CurrentObject::findNext( Designation * designation ) {
		typedef std::list< Expression * > DesignatorChain;
		PRINT( std::cerr << "___findNext" << std::endl; )
		// find all the d's
		std::list<DesignatorChain> desigAlts{ { } }, newDesigAlts;
		std::list<Type *> curTypes { (objStack.top())->getType() }, newTypes;
		for ( Expression * expr : designation->get_designators() ) {
			PRINT( std::cerr << "____untyped: " << expr << std::endl; )
			std::list<DesignatorChain>::iterator dit = desigAlts.begin();
			if ( NameExpr * nexpr = dynamic_cast<NameExpr *>(expr) ) {
				for ( Type * t : curTypes ) {
					assert( dit != desigAlts.end() );
					DesignatorChain & d = *dit;
					PRINT( std::cerr << "____actual: " << t << std::endl; )
					ReferenceToType * refType = dynamic_cast<ReferenceToType *>(t);
					std::list<Declaration *> members;
					if ( refType ) {
						refType->lookup( nexpr->get_name(), members ); // concatenate identical field name
						// xxx - need to also include anonymous members in this somehow...
						for ( Declaration * mem: members ) {
							if ( ObjectDecl * field = dynamic_cast<ObjectDecl *>(mem) ) {
								PRINT( std::cerr << "____alt: " << field->get_type() << std::endl; )
								DesignatorChain newD = d;
								newD.push_back( new VariableExpr( field ) );
								newDesigAlts.push_back( newD );
								newTypes.push_back( field->get_type() );
							} // if
						} // for
					} // if
					++dit;
				} // for
			} else {
				for ( Type * t : curTypes ) {
					assert( dit != desigAlts.end() );
					DesignatorChain & d = *dit;
					if ( ArrayType * at = dynamic_cast< ArrayType * > ( t ) ) {
						PRINT( std::cerr << "____alt: " << at->get_base() << std::endl; )
						d.push_back( expr );
						newDesigAlts.push_back( d );
						newTypes.push_back( at->get_base() );
					}
					++dit;
				} // for
			} // if
			desigAlts = newDesigAlts;
			newDesigAlts.clear();
			curTypes = newTypes;
			newTypes.clear();
			assertf( desigAlts.size() == curTypes.size(), "Designator alternatives (%d) and current types (%d) out of sync", desigAlts.size(), curTypes.size() );
		} // for
		if ( desigAlts.size() > 1 ) {
			throw SemanticError( toString("Too many alternatives (", desigAlts.size(), ") for designation: "), designation );
		} else if ( desigAlts.size() == 0 ) {
			throw SemanticError( "No reasonable alternatives for designation: ", designation );
		}
		DesignatorChain & d = desigAlts.back();
		PRINT( for ( Expression * expr : d ) {
			std::cerr << "____desig: " << expr << std::endl;
		} ) // for
		assertf( ! curTypes.empty(), "empty designator chosen");

		// set new designators
		assertf( ! objStack.empty(), "empty object stack when setting designation" );
		Designation * actualDesignation = new Designation( d );
		objStack.top()->setPosition( d ); // destroys d
		return actualDesignation;
	}

	void CurrentObject::increment() {
		PRINT( std::cerr << "____increment" << std::endl; )
		if ( ! objStack.empty() ) {
			PRINT( std::cerr << *objStack.top() << std::endl; )
			objStack.top()->smallStep();
		}
	}

	void CurrentObject::enterListInit() {
		PRINT( std::cerr << "____entering list init" << std::endl; )
		assertf( ! objStack.empty(), "empty obj stack entering list init" );
		Type * type = objStack.top()->getNext();
		if ( type ) {
			// xxx - record types.front()?
			objStack.push( createMemberIterator( type ) );
		} else {
			assertf( false, "not sure about this case..." );
		}
	}

	void CurrentObject::exitListInit() {
		PRINT( std::cerr << "____exiting list init" << std::endl; )
		assertf( ! objStack.empty(), "objstack empty" );
		delete objStack.top();
		objStack.pop();
		if ( ! objStack.empty() ) {
			PRINT( std::cerr << *objStack.top() << std::endl; )
			objStack.top()->bigStep();
		}
	}

	std::list< InitAlternative > CurrentObject::getOptions() {
		PRINT( std::cerr << "____getting current options" << std::endl; )
		assertf( ! objStack.empty(), "objstack empty in getOptions" );
		return **objStack.top();
	}
} // namespace ResolvExpr

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
