//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// InitializerNode.cc --
//
// Author           : Rodolfo G. Esteves
// Created On       : Sat May 16 13:20:24 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Sat Oct  1 23:09:51 2016
// Update Count     : 21
//

#include <cassert>
#include <iostream>
using namespace std;

#include "ParseNode.h"
#include "SynTree/Expression.h"
#include "SynTree/Initializer.h"

InitializerNode::InitializerNode( ExpressionNode *_expr, bool aggrp, ExpressionNode *des )
		: expr( _expr ), aggregate( aggrp ), designator( des ), kids( 0 ), maybeConstructed( true ) {
	if ( aggrp )
		kids = dynamic_cast< InitializerNode * >( get_next() );

	if ( kids != 0 )
		set_last( 0 );
}

InitializerNode::InitializerNode( InitializerNode *init, bool aggrp, ExpressionNode *des )
		: expr( 0 ), aggregate( aggrp ), designator( des ), kids( 0 ), maybeConstructed( true ) {
	if ( init != 0 )
		set_last( init );

	if ( aggrp )
		kids = dynamic_cast< InitializerNode * >( get_next() );

	if ( kids != 0 )
		set_next( 0 );
}

InitializerNode::~InitializerNode() {
	delete expr;
	delete designator;
	delete kids;
}

void InitializerNode::print( std::ostream &os, int indent ) const {
	os << std::string( indent, ' ' ) << "Initializer expression" << std::endl;
}

void InitializerNode::printOneLine( std::ostream &os ) const {
	if ( ! aggregate ) {
		if ( designator != 0 ) {
			os << "designated by: (";
			ExpressionNode *curdes = designator;
			while ( curdes != 0) {
				curdes->printOneLine(os);
				curdes = (ExpressionNode *)(curdes->get_next());
				if ( curdes ) os << ", ";
			} // while
			os << ")";
		} // if
		if ( expr ) expr->printOneLine(os);
	} else {  // It's an aggregate
		os << "[--";
		if ( next_init() != 0 )
			next_init()->printOneLine(os);
		if (aggregate) os << "--]";
	} // if

	InitializerNode *moreInit;
	if ( (moreInit = dynamic_cast< InitializerNode * >( get_next() ) ) ) {
		moreInit->printOneLine( os );
	}
}

Initializer *InitializerNode::build() const {
	if ( aggregate ) {
		// steal designators from children
		std::list< Designation * > designlist;
		InitializerNode * child = next_init();
		for ( ; child != nullptr; child = dynamic_cast< InitializerNode * >( child->get_next() ) ) {
			std::list< Expression * > desList;
			buildList< Expression, ExpressionNode >( child->designator, desList );
			designlist.push_back( new Designation( desList ) );
		} // for
		std::list< Initializer * > initlist;
		buildList< Initializer, InitializerNode >( next_init(), initlist );
		return new ListInit( initlist, designlist, maybeConstructed );
	} else {
		if ( get_expression() != 0) {
			return new SingleInit( maybeBuild< Expression >( get_expression() ), maybeConstructed );
		}
	} // if
	return 0;
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
