//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Mutator.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Thu Mar 30 16:45:19 2017
// Update Count     : 22
//

#include <cassert>
#include "Mutator.h"
#include "Initializer.h"
#include "Statement.h"
#include "Type.h"
#include "Declaration.h"
#include "Expression.h"
#include "Constant.h"
#include "Common/utility.h"
#include "TypeSubstitution.h"

Mutator::Mutator() {}

Mutator::~Mutator() {}

DeclarationWithType *Mutator::mutate( ObjectDecl *objectDecl ) {
	objectDecl->set_type( maybeMutate( objectDecl->get_type(), *this ) );
	objectDecl->set_init( maybeMutate( objectDecl->get_init(), *this ) );
	objectDecl->set_bitfieldWidth( maybeMutate( objectDecl->get_bitfieldWidth(), *this ) );
	return objectDecl;
}

DeclarationWithType *Mutator::mutate( FunctionDecl *functionDecl ) {
	functionDecl->set_functionType( maybeMutate( functionDecl->get_functionType(), *this ) );
	functionDecl->set_statements( maybeMutate( functionDecl->get_statements(), *this ) );
	return functionDecl;
}

Declaration *Mutator::handleAggregateDecl( AggregateDecl *aggregateDecl ) {
	mutateAll( aggregateDecl->get_parameters(), *this );
	mutateAll( aggregateDecl->get_members(), *this );
	return aggregateDecl;
}

Declaration *Mutator::mutate( StructDecl *aggregateDecl ) {
	handleAggregateDecl( aggregateDecl );
	return aggregateDecl;
}

Declaration *Mutator::mutate( UnionDecl *aggregateDecl ) {
	handleAggregateDecl( aggregateDecl );
	return aggregateDecl;
}

Declaration *Mutator::mutate( EnumDecl *aggregateDecl ) {
	handleAggregateDecl( aggregateDecl );
	return aggregateDecl;
}

Declaration *Mutator::mutate( TraitDecl *aggregateDecl ) {
	handleAggregateDecl( aggregateDecl );
	return aggregateDecl;
}

Declaration *Mutator::handleNamedTypeDecl( NamedTypeDecl *typeDecl ) {
	mutateAll( typeDecl->get_parameters(), *this );
	mutateAll( typeDecl->get_assertions(), *this );
	typeDecl->set_base( maybeMutate( typeDecl->get_base(), *this ) );
	return typeDecl;
}

TypeDecl *Mutator::mutate( TypeDecl *typeDecl ) {
	handleNamedTypeDecl( typeDecl );
	typeDecl->set_init( maybeMutate( typeDecl->get_init(), *this ) );
	return typeDecl;
}

Declaration *Mutator::mutate( TypedefDecl *typeDecl ) {
	handleNamedTypeDecl( typeDecl );
	return typeDecl;
}

AsmDecl *Mutator::mutate( AsmDecl *asmDecl ) {
	asmDecl->set_stmt( maybeMutate( asmDecl->get_stmt(), *this ) );
	return asmDecl;
}


CompoundStmt *Mutator::mutate( CompoundStmt *compoundStmt ) {
	mutateAll( compoundStmt->get_kids(), *this );
	return compoundStmt;
}

Statement *Mutator::mutate( ExprStmt *exprStmt ) {
	exprStmt->set_expr( maybeMutate( exprStmt->get_expr(), *this ) );
	return exprStmt;
}

Statement *Mutator::mutate( AsmStmt *asmStmt ) {
	asmStmt->set_instruction( maybeMutate( asmStmt->get_instruction(), *this ) );
	mutateAll( asmStmt->get_output(), *this );
	mutateAll( asmStmt->get_input(), *this );
	mutateAll( asmStmt->get_clobber(), *this );
	return asmStmt;
}

Statement *Mutator::mutate( IfStmt *ifStmt ) {
	ifStmt->set_condition( maybeMutate( ifStmt->get_condition(), *this ) );
	ifStmt->set_thenPart( maybeMutate( ifStmt->get_thenPart(), *this ) );
	ifStmt->set_elsePart( maybeMutate( ifStmt->get_elsePart(), *this ) );
	return ifStmt;
}

Statement *Mutator::mutate( WhileStmt *whileStmt ) {
	whileStmt->set_condition( maybeMutate( whileStmt->get_condition(), *this ) );
	whileStmt->set_body( maybeMutate( whileStmt->get_body(), *this ) );
	return whileStmt;
}

Statement *Mutator::mutate( ForStmt *forStmt ) {
	mutateAll( forStmt->get_initialization(), *this );
	forStmt->set_condition( maybeMutate( forStmt->get_condition(), *this ) );
	forStmt->set_increment( maybeMutate( forStmt->get_increment(), *this ) );
	forStmt->set_body( maybeMutate( forStmt->get_body(), *this ) );
	return forStmt;
}

Statement *Mutator::mutate( SwitchStmt *switchStmt ) {
	switchStmt->set_condition( maybeMutate( switchStmt->get_condition(), *this ) );
	mutateAll( switchStmt->get_statements(), *this );
	return switchStmt;
}

Statement *Mutator::mutate( CaseStmt *caseStmt ) {
	caseStmt->set_condition( maybeMutate( caseStmt->get_condition(), *this ) );
	mutateAll (caseStmt->get_statements(), *this );

	return caseStmt;
}

Statement *Mutator::mutate( BranchStmt *branchStmt ) {
	return branchStmt;
}

Statement *Mutator::mutate( ReturnStmt *returnStmt ) {
	returnStmt->set_expr( maybeMutate( returnStmt->get_expr(), *this ) );
	return returnStmt;
}

Statement *Mutator::mutate( TryStmt *tryStmt ) {
	tryStmt->set_block( maybeMutate( tryStmt->get_block(), *this ) );
	mutateAll( tryStmt->get_catchers(), *this );
	return tryStmt;
}

Statement *Mutator::mutate( CatchStmt *catchStmt ) {
	catchStmt->set_decl( maybeMutate( catchStmt->get_decl(), *this ) );
	catchStmt->set_body( maybeMutate( catchStmt->get_body(), *this ) );
	return catchStmt;
}

Statement *Mutator::mutate( FinallyStmt *finalStmt ) {
	finalStmt->set_block( maybeMutate( finalStmt->get_block(), *this ) );
	return finalStmt;
}

NullStmt *Mutator::mutate( NullStmt *nullStmt ) {
	return nullStmt;
}

Statement *Mutator::mutate( DeclStmt *declStmt ) {
	declStmt->set_decl( maybeMutate( declStmt->get_decl(), *this ) );
	return declStmt;
}

Statement *Mutator::mutate( ImplicitCtorDtorStmt *impCtorDtorStmt ) {
	impCtorDtorStmt->set_callStmt( maybeMutate( impCtorDtorStmt->get_callStmt(), *this ) );
	return impCtorDtorStmt;
}


Expression *Mutator::mutate( ApplicationExpr *applicationExpr ) {
	applicationExpr->set_env( maybeMutate( applicationExpr->get_env(), *this ) );
	applicationExpr->set_result( maybeMutate( applicationExpr->get_result(), *this ) );
	applicationExpr->set_function( maybeMutate( applicationExpr->get_function(), *this ) );
	mutateAll( applicationExpr->get_args(), *this );
	return applicationExpr;
}

Expression *Mutator::mutate( UntypedExpr *untypedExpr ) {
	untypedExpr->set_env( maybeMutate( untypedExpr->get_env(), *this ) );
	untypedExpr->set_result( maybeMutate( untypedExpr->get_result(), *this ) );
	mutateAll( untypedExpr->get_args(), *this );
	return untypedExpr;
}

Expression *Mutator::mutate( NameExpr *nameExpr ) {
	nameExpr->set_env( maybeMutate( nameExpr->get_env(), *this ) );
	nameExpr->set_result( maybeMutate( nameExpr->get_result(), *this ) );
	return nameExpr;
}

Expression *Mutator::mutate( AddressExpr *addressExpr ) {
	addressExpr->set_env( maybeMutate( addressExpr->get_env(), *this ) );
	addressExpr->set_result( maybeMutate( addressExpr->get_result(), *this ) );
	addressExpr->set_arg( maybeMutate( addressExpr->get_arg(), *this ) );
	return addressExpr;
}

Expression *Mutator::mutate( LabelAddressExpr *labelAddressExpr ) {
	labelAddressExpr->set_env( maybeMutate( labelAddressExpr->get_env(), *this ) );
	labelAddressExpr->set_result( maybeMutate( labelAddressExpr->get_result(), *this ) );
	labelAddressExpr->set_arg( maybeMutate( labelAddressExpr->get_arg(), *this ) );
	return labelAddressExpr;
}

Expression *Mutator::mutate( CastExpr *castExpr ) {
	castExpr->set_env( maybeMutate( castExpr->get_env(), *this ) );
	castExpr->set_result( maybeMutate( castExpr->get_result(), *this ) );
	castExpr->set_arg( maybeMutate( castExpr->get_arg(), *this ) );
	return castExpr;
}

Expression *Mutator::mutate( UntypedMemberExpr *memberExpr ) {
	memberExpr->set_env( maybeMutate( memberExpr->get_env(), *this ) );
	memberExpr->set_result( maybeMutate( memberExpr->get_result(), *this ) );
	memberExpr->set_aggregate( maybeMutate( memberExpr->get_aggregate(), *this ) );
	memberExpr->set_member( maybeMutate( memberExpr->get_member(), *this ) );
	return memberExpr;
}

Expression *Mutator::mutate( MemberExpr *memberExpr ) {
	memberExpr->set_env( maybeMutate( memberExpr->get_env(), *this ) );
	memberExpr->set_result( maybeMutate( memberExpr->get_result(), *this ) );
	memberExpr->set_aggregate( maybeMutate( memberExpr->get_aggregate(), *this ) );
	return memberExpr;
}

Expression *Mutator::mutate( VariableExpr *variableExpr ) {
	variableExpr->set_env( maybeMutate( variableExpr->get_env(), *this ) );
	variableExpr->set_result( maybeMutate( variableExpr->get_result(), *this ) );
	return variableExpr;
}

Expression *Mutator::mutate( ConstantExpr *constantExpr ) {
	constantExpr->set_env( maybeMutate( constantExpr->get_env(), *this ) );
	constantExpr->set_result( maybeMutate( constantExpr->get_result(), *this ) );
//  maybeMutate( constantExpr->get_constant(), *this )
	return constantExpr;
}

Expression *Mutator::mutate( SizeofExpr *sizeofExpr ) {
	sizeofExpr->set_env( maybeMutate( sizeofExpr->get_env(), *this ) );
	sizeofExpr->set_result( maybeMutate( sizeofExpr->get_result(), *this ) );
	if ( sizeofExpr->get_isType() ) {
		sizeofExpr->set_type( maybeMutate( sizeofExpr->get_type(), *this ) );
	} else {
		sizeofExpr->set_expr( maybeMutate( sizeofExpr->get_expr(), *this ) );
	}
	return sizeofExpr;
}

Expression *Mutator::mutate( AlignofExpr *alignofExpr ) {
	alignofExpr->set_env( maybeMutate( alignofExpr->get_env(), *this ) );
	alignofExpr->set_result( maybeMutate( alignofExpr->get_result(), *this ) );
	if ( alignofExpr->get_isType() ) {
		alignofExpr->set_type( maybeMutate( alignofExpr->get_type(), *this ) );
	} else {
		alignofExpr->set_expr( maybeMutate( alignofExpr->get_expr(), *this ) );
	}
	return alignofExpr;
}

Expression *Mutator::mutate( UntypedOffsetofExpr *offsetofExpr ) {
	offsetofExpr->set_env( maybeMutate( offsetofExpr->get_env(), *this ) );
	offsetofExpr->set_result( maybeMutate( offsetofExpr->get_result(), *this ) );
	offsetofExpr->set_type( maybeMutate( offsetofExpr->get_type(), *this ) );
	return offsetofExpr;
}

Expression *Mutator::mutate( OffsetofExpr *offsetofExpr ) {
	offsetofExpr->set_env( maybeMutate( offsetofExpr->get_env(), *this ) );
	offsetofExpr->set_result( maybeMutate( offsetofExpr->get_result(), *this ) );
	offsetofExpr->set_type( maybeMutate( offsetofExpr->get_type(), *this ) );
	offsetofExpr->set_member( maybeMutate( offsetofExpr->get_member(), *this ) );
	return offsetofExpr;
}

Expression *Mutator::mutate( OffsetPackExpr *offsetPackExpr ) {
	offsetPackExpr->set_env( maybeMutate( offsetPackExpr->get_env(), *this ) );
	offsetPackExpr->set_result( maybeMutate( offsetPackExpr->get_result(), *this ) );
	offsetPackExpr->set_type( maybeMutate( offsetPackExpr->get_type(), *this ) );
	return offsetPackExpr;
}

Expression *Mutator::mutate( AttrExpr *attrExpr ) {
	attrExpr->set_env( maybeMutate( attrExpr->get_env(), *this ) );
	attrExpr->set_result( maybeMutate( attrExpr->get_result(), *this ) );
	if ( attrExpr->get_isType() ) {
		attrExpr->set_type( maybeMutate( attrExpr->get_type(), *this ) );
	} else {
		attrExpr->set_expr( maybeMutate( attrExpr->get_expr(), *this ) );
	}
	return attrExpr;
}

Expression *Mutator::mutate( LogicalExpr *logicalExpr ) {
	logicalExpr->set_env( maybeMutate( logicalExpr->get_env(), *this ) );
	logicalExpr->set_result( maybeMutate( logicalExpr->get_result(), *this ) );
	logicalExpr->set_arg1( maybeMutate( logicalExpr->get_arg1(), *this ) );
	logicalExpr->set_arg2( maybeMutate( logicalExpr->get_arg2(), *this ) );
	return logicalExpr;
}

Expression *Mutator::mutate( ConditionalExpr *conditionalExpr ) {
	conditionalExpr->set_env( maybeMutate( conditionalExpr->get_env(), *this ) );
	conditionalExpr->set_result( maybeMutate( conditionalExpr->get_result(), *this ) );
	conditionalExpr->set_arg1( maybeMutate( conditionalExpr->get_arg1(), *this ) );
	conditionalExpr->set_arg2( maybeMutate( conditionalExpr->get_arg2(), *this ) );
	conditionalExpr->set_arg3( maybeMutate( conditionalExpr->get_arg3(), *this ) );
	return conditionalExpr;
}

Expression *Mutator::mutate( CommaExpr *commaExpr ) {
	commaExpr->set_env( maybeMutate( commaExpr->get_env(), *this ) );
	commaExpr->set_result( maybeMutate( commaExpr->get_result(), *this ) );
	commaExpr->set_arg1( maybeMutate( commaExpr->get_arg1(), *this ) );
	commaExpr->set_arg2( maybeMutate( commaExpr->get_arg2(), *this ) );
	return commaExpr;
}

Expression *Mutator::mutate( TypeExpr *typeExpr ) {
	typeExpr->set_env( maybeMutate( typeExpr->get_env(), *this ) );
	typeExpr->set_result( maybeMutate( typeExpr->get_result(), *this ) );
	typeExpr->set_type( maybeMutate( typeExpr->get_type(), *this ) );
	return typeExpr;
}

Expression *Mutator::mutate( AsmExpr *asmExpr ) {
	asmExpr->set_env( maybeMutate( asmExpr->get_env(), *this ) );
	asmExpr->set_inout( maybeMutate( asmExpr->get_inout(), *this ) );
	asmExpr->set_constraint( maybeMutate( asmExpr->get_constraint(), *this ) );
	asmExpr->set_operand( maybeMutate( asmExpr->get_operand(), *this ) );
	return asmExpr;
}

Expression* Mutator::mutate( ImplicitCopyCtorExpr *impCpCtorExpr ) {
	impCpCtorExpr->set_env( maybeMutate( impCpCtorExpr->get_env(), *this ) );
	impCpCtorExpr->set_result( maybeMutate( impCpCtorExpr->get_result(), *this ) );
	impCpCtorExpr->set_callExpr( maybeMutate( impCpCtorExpr->get_callExpr(), *this ) );
	mutateAll( impCpCtorExpr->get_tempDecls(), *this );
	mutateAll( impCpCtorExpr->get_returnDecls(), *this );
	mutateAll( impCpCtorExpr->get_dtors(), *this );
	return impCpCtorExpr;
}

Expression* Mutator::mutate( ConstructorExpr *ctorExpr ) {
	ctorExpr->set_env( maybeMutate( ctorExpr->get_env(), *this ) );
	ctorExpr->set_result( maybeMutate( ctorExpr->get_result(), *this ) );
	ctorExpr->set_callExpr( maybeMutate( ctorExpr->get_callExpr(), *this ) );
	return ctorExpr;
}

Expression *Mutator::mutate( CompoundLiteralExpr *compLitExpr ) {
	compLitExpr->set_env( maybeMutate( compLitExpr->get_env(), *this ) );
	compLitExpr->set_result( maybeMutate( compLitExpr->get_result(), *this ) );
	compLitExpr->set_initializer( maybeMutate( compLitExpr->get_initializer(), *this ) );
	return compLitExpr;
}

Expression *Mutator::mutate( UntypedValofExpr *valofExpr ) {
	valofExpr->set_env( maybeMutate( valofExpr->get_env(), *this ) );
	valofExpr->set_result( maybeMutate( valofExpr->get_result(), *this ) );
	return valofExpr;
}

Expression *Mutator::mutate( RangeExpr *rangeExpr ) {
	rangeExpr->set_env( maybeMutate( rangeExpr->get_env(), *this ) );
	rangeExpr->set_low( maybeMutate( rangeExpr->get_low(), *this ) );
	rangeExpr->set_high( maybeMutate( rangeExpr->get_high(), *this ) );
	return rangeExpr;
}

Expression *Mutator::mutate( UntypedTupleExpr *tupleExpr ) {
	tupleExpr->set_env( maybeMutate( tupleExpr->get_env(), *this ) );
	tupleExpr->set_result( maybeMutate( tupleExpr->get_result(), *this ) );
	mutateAll( tupleExpr->get_exprs(), *this );
	return tupleExpr;
}

Expression *Mutator::mutate( TupleExpr *tupleExpr ) {
	tupleExpr->set_env( maybeMutate( tupleExpr->get_env(), *this ) );
	tupleExpr->set_result( maybeMutate( tupleExpr->get_result(), *this ) );
	mutateAll( tupleExpr->get_exprs(), *this );
	return tupleExpr;
}

Expression *Mutator::mutate( TupleIndexExpr *tupleExpr ) {
	tupleExpr->set_env( maybeMutate( tupleExpr->get_env(), *this ) );
	tupleExpr->set_result( maybeMutate( tupleExpr->get_result(), *this ) );
	tupleExpr->set_tuple( maybeMutate( tupleExpr->get_tuple(), *this ) );
	return tupleExpr;
}

Expression *Mutator::mutate( MemberTupleExpr *tupleExpr ) {
	tupleExpr->set_env( maybeMutate( tupleExpr->get_env(), *this ) );
	tupleExpr->set_result( maybeMutate( tupleExpr->get_result(), *this ) );
	tupleExpr->set_member( maybeMutate( tupleExpr->get_member(), *this ) );
	tupleExpr->set_aggregate( maybeMutate( tupleExpr->get_aggregate(), *this ) );
	return tupleExpr;
}

Expression *Mutator::mutate( TupleAssignExpr *assignExpr ) {
	assignExpr->set_env( maybeMutate( assignExpr->get_env(), *this ) );
	assignExpr->set_result( maybeMutate( assignExpr->get_result(), *this ) );
	assignExpr->set_stmtExpr( maybeMutate( assignExpr->get_stmtExpr(), *this ) );
	return assignExpr;
}

Expression *Mutator::mutate( StmtExpr *stmtExpr ) {
	stmtExpr->set_env( maybeMutate( stmtExpr->get_env(), *this ) );
	stmtExpr->set_result( maybeMutate( stmtExpr->get_result(), *this ) );
	stmtExpr->set_statements( maybeMutate( stmtExpr->get_statements(), *this ) );
	mutateAll( stmtExpr->get_returnDecls(), *this );
	mutateAll( stmtExpr->get_dtors(), *this );
	return stmtExpr;
}

Expression *Mutator::mutate( UniqueExpr *uniqueExpr ) {
	uniqueExpr->set_env( maybeMutate( uniqueExpr->get_env(), *this ) );
	uniqueExpr->set_result( maybeMutate( uniqueExpr->get_result(), *this ) );
	uniqueExpr->set_expr( maybeMutate( uniqueExpr->get_expr(), *this ) );
	return uniqueExpr;
}

Expression *Mutator::mutate( UntypedInitExpr * initExpr ) {
	initExpr->set_env( maybeMutate( initExpr->get_env(), *this ) );
	initExpr->set_result( maybeMutate( initExpr->get_result(), *this ) );
	initExpr->set_expr( maybeMutate( initExpr->get_expr(), *this ) );
	// not currently mutating initAlts, but this doesn't matter since this node is only used in the resolver.
	return initExpr;
}

Expression *Mutator::mutate( InitExpr * initExpr ) {
	initExpr->set_env( maybeMutate( initExpr->get_env(), *this ) );
	initExpr->set_result( maybeMutate( initExpr->get_result(), *this ) );
	initExpr->set_expr( maybeMutate( initExpr->get_expr(), *this ) );
	initExpr->set_designation( maybeMutate( initExpr->get_designation(), *this ) );
	return initExpr;
}


Type *Mutator::mutate( VoidType *voidType ) {
	mutateAll( voidType->get_forall(), *this );
	return voidType;
}

Type *Mutator::mutate( BasicType *basicType ) {
	mutateAll( basicType->get_forall(), *this );
	return basicType;
}

Type *Mutator::mutate( PointerType *pointerType ) {
	mutateAll( pointerType->get_forall(), *this );
	pointerType->set_base( maybeMutate( pointerType->get_base(), *this ) );
	return pointerType;
}

Type *Mutator::mutate( ArrayType *arrayType ) {
	mutateAll( arrayType->get_forall(), *this );
	arrayType->set_dimension( maybeMutate( arrayType->get_dimension(), *this ) );
	arrayType->set_base( maybeMutate( arrayType->get_base(), *this ) );
	return arrayType;
}

Type *Mutator::mutate( FunctionType *functionType ) {
	mutateAll( functionType->get_forall(), *this );
	mutateAll( functionType->get_returnVals(), *this );
	mutateAll( functionType->get_parameters(), *this );
	return functionType;
}

Type *Mutator::handleReferenceToType( ReferenceToType *aggregateUseType ) {
	mutateAll( aggregateUseType->get_forall(), *this );
	mutateAll( aggregateUseType->get_parameters(), *this );
	return aggregateUseType;
}

Type *Mutator::mutate( StructInstType *aggregateUseType ) {
	handleReferenceToType( aggregateUseType );
	return aggregateUseType;
}

Type *Mutator::mutate( UnionInstType *aggregateUseType ) {
	handleReferenceToType( aggregateUseType );
	return aggregateUseType;
}

Type *Mutator::mutate( EnumInstType *aggregateUseType ) {
	handleReferenceToType( aggregateUseType );
	return aggregateUseType;
}

Type *Mutator::mutate( TraitInstType *aggregateUseType ) {
	handleReferenceToType( aggregateUseType );
	mutateAll( aggregateUseType->get_members(), *this );
	return aggregateUseType;
}

Type *Mutator::mutate( TypeInstType *aggregateUseType ) {
	handleReferenceToType( aggregateUseType );
	return aggregateUseType;
}

Type *Mutator::mutate( TupleType *tupleType ) {
	mutateAll( tupleType->get_forall(), *this );
	mutateAll( tupleType->get_types(), *this );
	return tupleType;
}

Type *Mutator::mutate( TypeofType *typeofType ) {
	assert( typeofType->get_expr() );
	typeofType->set_expr( typeofType->get_expr()->acceptMutator( *this ) );
	return typeofType;
}

Type *Mutator::mutate( AttrType *attrType ) {
	if ( attrType->get_isType() ) {
		assert( attrType->get_type() );
		attrType->set_type( attrType->get_type()->acceptMutator( *this ) );
	} else {
		assert( attrType->get_expr() );
		attrType->set_expr( attrType->get_expr()->acceptMutator( *this ) );
	}
	return attrType;
}

Type *Mutator::mutate( VarArgsType *varArgsType ) {
	mutateAll( varArgsType->get_forall(), *this );
	return varArgsType;
}

Type *Mutator::mutate( ZeroType *zeroType ) {
	mutateAll( zeroType->get_forall(), *this );
	return zeroType;
}

Type *Mutator::mutate( OneType *oneType ) {
	mutateAll( oneType->get_forall(), *this );
	return oneType;
}


Designation *Mutator::mutate( Designation * designation ) {
	mutateAll( designation->get_designators(), *this );
	return designation;
}

Initializer *Mutator::mutate( SingleInit *singleInit ) {
	singleInit->set_value( singleInit->get_value()->acceptMutator( *this ) );
	return singleInit;
}

Initializer *Mutator::mutate( ListInit *listInit ) {
	mutateAll( listInit->get_designations(), *this );
	mutateAll( listInit->get_initializers(), *this );
	return listInit;
}

Initializer *Mutator::mutate( ConstructorInit *ctorInit ) {
	ctorInit->set_ctor( maybeMutate( ctorInit->get_ctor(), *this ) );
	ctorInit->set_dtor( maybeMutate( ctorInit->get_dtor(), *this ) );
	ctorInit->set_init( maybeMutate( ctorInit->get_init(), *this ) );
	return ctorInit;
}


Subrange *Mutator::mutate( Subrange *subrange ) {
	return subrange;
}


Constant *Mutator::mutate( Constant *constant ) {
	return constant;
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
