//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Validate.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 21:50:04 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Jul 12 17:49:21 2016
// Update Count     : 298
//

// The "validate" phase of translation is used to take a syntax tree and convert it into a standard form that aims to be
// as regular in structure as possible.  Some assumptions can be made regarding the state of the tree after this pass is
// complete, including:
//
// - No nested structure or union definitions; any in the input are "hoisted" to the level of the containing struct or
//   union.
//
// - All enumeration constants have type EnumInstType.
//
// - The type "void" never occurs in lists of function parameter or return types; neither do tuple types.  A function
//   taking no arguments has no argument types, and tuples are flattened.
//
// - No context instances exist; they are all replaced by the set of declarations signified by the context, instantiated
//   by the particular set of type arguments.
//
// - Every declaration is assigned a unique id.
//
// - No typedef declarations or instances exist; the actual type is substituted for each instance.
//
// - Each type, struct, and union definition is followed by an appropriate assignment operator.
//
// - Each use of a struct or union is connected to a complete definition of that struct or union, even if that
//   definition occurs later in the input.

#include <list>
#include <iterator>
#include "Common/ScopedMap.h"
#include "Common/utility.h"
#include "Common/UniqueName.h"
#include "Validate.h"
#include "SynTree/Visitor.h"
#include "SynTree/Mutator.h"
#include "SynTree/Type.h"
#include "SynTree/Expression.h"
#include "SynTree/Statement.h"
#include "SynTree/TypeSubstitution.h"
#include "Indexer.h"
#include "FixFunction.h"
// #include "ImplementationType.h"
#include "GenPoly/DeclMutator.h"
#include "AddVisit.h"
#include "MakeLibCfa.h"
#include "TypeEquality.h"
#include "Autogen.h"
#include "ResolvExpr/typeops.h"
#include <algorithm>

#define debugPrint( x ) if ( doDebug ) { std::cout << x; }

namespace SymTab {
	class HoistStruct : public Visitor {
	  public:
		/// Flattens nested struct types
		static void hoistStruct( std::list< Declaration * > &translationUnit );

		std::list< Declaration * > &get_declsToAdd() { return declsToAdd; }

		virtual void visit( StructDecl *aggregateDecl );
		virtual void visit( UnionDecl *aggregateDecl );

		virtual void visit( CompoundStmt *compoundStmt );
		virtual void visit( SwitchStmt *switchStmt );
	  private:
		HoistStruct();

		template< typename AggDecl > void handleAggregate( AggDecl *aggregateDecl );

		std::list< Declaration * > declsToAdd;
		bool inStruct;
	};

	/// Replaces enum types by int, and function or array types in function parameter and return lists by appropriate pointers.
	class EnumAndPointerDecayPass : public Visitor {
		typedef Visitor Parent;
		virtual void visit( EnumDecl *aggregateDecl );
		virtual void visit( FunctionType *func );
	};

	/// Associates forward declarations of aggregates with their definitions
	class Pass2 : public Indexer {
		typedef Indexer Parent;
	  public:
		Pass2( bool doDebug, const Indexer *indexer );
	  private:
		virtual void visit( StructInstType *structInst );
		virtual void visit( UnionInstType *unionInst );
		virtual void visit( TraitInstType *contextInst );
		virtual void visit( StructDecl *structDecl );
		virtual void visit( UnionDecl *unionDecl );
		virtual void visit( TypeInstType *typeInst );

		const Indexer *indexer;

		typedef std::map< std::string, std::list< StructInstType * > > ForwardStructsType;
		typedef std::map< std::string, std::list< UnionInstType * > > ForwardUnionsType;
		ForwardStructsType forwardStructs;
		ForwardUnionsType forwardUnions;
	};

	/// Replaces array and function types in forall lists by appropriate pointer type
	class Pass3 : public Indexer {
		typedef Indexer Parent;
	  public:
		Pass3( const Indexer *indexer );
	  private:
		virtual void visit( ObjectDecl *object );
		virtual void visit( FunctionDecl *func );

		const Indexer *indexer;
	};

	class ReturnChecker : public Visitor {
	  public:
		/// Checks that return statements return nothing if their return type is void
		/// and return something if the return type is non-void.
		static void checkFunctionReturns( std::list< Declaration * > & translationUnit );
	  private:
		virtual void visit( FunctionDecl * functionDecl );

		virtual void visit( ReturnStmt * returnStmt );

		std::list< DeclarationWithType * > returnVals;
	};

	class EliminateTypedef : public Mutator {
	  public:
		EliminateTypedef() : scopeLevel( 0 ) {}
		/// Replaces typedefs by forward declarations
		static void eliminateTypedef( std::list< Declaration * > &translationUnit );
	  private:
		virtual Declaration *mutate( TypedefDecl *typeDecl );
		virtual TypeDecl *mutate( TypeDecl *typeDecl );
		virtual DeclarationWithType *mutate( FunctionDecl *funcDecl );
		virtual DeclarationWithType *mutate( ObjectDecl *objDecl );
		virtual CompoundStmt *mutate( CompoundStmt *compoundStmt );
		virtual Type *mutate( TypeInstType *aggregateUseType );
		virtual Expression *mutate( CastExpr *castExpr );

		virtual Declaration *mutate( StructDecl * structDecl );
		virtual Declaration *mutate( UnionDecl * unionDecl );
		virtual Declaration *mutate( EnumDecl * enumDecl );
		virtual Declaration *mutate( TraitDecl * contextDecl );

		template<typename AggDecl>
		AggDecl *handleAggregate( AggDecl * aggDecl );

		template<typename AggDecl>
		void addImplicitTypedef( AggDecl * aggDecl );

		typedef std::unique_ptr<TypedefDecl> TypedefDeclPtr;
		typedef ScopedMap< std::string, std::pair< TypedefDeclPtr, int > > TypedefMap;
		typedef std::map< std::string, TypeDecl * > TypeDeclMap;
		TypedefMap typedefNames;
		TypeDeclMap typedeclNames;
		int scopeLevel;
	};

	class VerifyCtorDtor : public Visitor {
	public:
		/// ensure that constructors and destructors have at least one
		/// parameter, the first of which must be a pointer, and no
		/// return values.
		static void verify( std::list< Declaration * > &translationUnit );

		virtual void visit( FunctionDecl *funcDecl );
	};

	class CompoundLiteral : public GenPoly::DeclMutator {
		DeclarationNode::StorageClass storageclass = DeclarationNode::NoStorageClass;

		virtual DeclarationWithType * mutate( ObjectDecl *objectDecl );
		virtual Expression *mutate( CompoundLiteralExpr *compLitExpr );
	};

	void validate( std::list< Declaration * > &translationUnit, bool doDebug ) {
		EnumAndPointerDecayPass epc;
		Pass2 pass2( doDebug, 0 );
		Pass3 pass3( 0 );
		CompoundLiteral compoundliteral;

		EliminateTypedef::eliminateTypedef( translationUnit );
		HoistStruct::hoistStruct( translationUnit );
		autogenerateRoutines( translationUnit ); // moved up, used to be below compoundLiteral - currently needs EnumAndPointerDecayPass
		acceptAll( translationUnit, epc );
		acceptAll( translationUnit, pass2 );
		ReturnChecker::checkFunctionReturns( translationUnit );
		compoundliteral.mutateDeclarationList( translationUnit );
		acceptAll( translationUnit, pass3 );
		VerifyCtorDtor::verify( translationUnit );
	}

	void validateType( Type *type, const Indexer *indexer ) {
		EnumAndPointerDecayPass epc;
		Pass2 pass2( false, indexer );
		Pass3 pass3( indexer );
		type->accept( epc );
		type->accept( pass2 );
		type->accept( pass3 );
	}

	void HoistStruct::hoistStruct( std::list< Declaration * > &translationUnit ) {
		HoistStruct hoister;
		acceptAndAdd( translationUnit, hoister, true );
	}

	HoistStruct::HoistStruct() : inStruct( false ) {
	}

	void filter( std::list< Declaration * > &declList, bool (*pred)( Declaration * ), bool doDelete ) {
		std::list< Declaration * >::iterator i = declList.begin();
		while ( i != declList.end() ) {
			std::list< Declaration * >::iterator next = i;
			++next;
			if ( pred( *i ) ) {
				if ( doDelete ) {
					delete *i;
				} // if
				declList.erase( i );
			} // if
			i = next;
		} // while
	}

	bool isStructOrUnion( Declaration *decl ) {
		return dynamic_cast< StructDecl * >( decl ) || dynamic_cast< UnionDecl * >( decl );
	}

	template< typename AggDecl >
	void HoistStruct::handleAggregate( AggDecl *aggregateDecl ) {
		if ( inStruct ) {
			// Add elements in stack order corresponding to nesting structure.
			declsToAdd.push_front( aggregateDecl );
			Visitor::visit( aggregateDecl );
		} else {
			inStruct = true;
			Visitor::visit( aggregateDecl );
			inStruct = false;
		} // if
		// Always remove the hoisted aggregate from the inner structure.
		filter( aggregateDecl->get_members(), isStructOrUnion, false );
	}

	void HoistStruct::visit( StructDecl *aggregateDecl ) {
		handleAggregate( aggregateDecl );
	}

	void HoistStruct::visit( UnionDecl *aggregateDecl ) {
		handleAggregate( aggregateDecl );
	}

	void HoistStruct::visit( CompoundStmt *compoundStmt ) {
		addVisit( compoundStmt, *this );
	}

	void HoistStruct::visit( SwitchStmt *switchStmt ) {
		addVisit( switchStmt, *this );
	}

	void EnumAndPointerDecayPass::visit( EnumDecl *enumDecl ) {
		// Set the type of each member of the enumeration to be EnumConstant
		for ( std::list< Declaration * >::iterator i = enumDecl->get_members().begin(); i != enumDecl->get_members().end(); ++i ) {
			ObjectDecl * obj = dynamic_cast< ObjectDecl * >( *i );
			assert( obj );
			obj->set_type( new EnumInstType( Type::Qualifiers( true, false, false, false, false, false ), enumDecl->get_name() ) );
		} // for
		Parent::visit( enumDecl );
	}

	namespace {
		template< typename DWTList >
		void fixFunctionList( DWTList & dwts, FunctionType * func ) {
			// the only case in which "void" is valid is where it is the only one in the list; then it should be removed
			// entirely other fix ups are handled by the FixFunction class
			typedef typename DWTList::iterator DWTIterator;
			DWTIterator begin( dwts.begin() ), end( dwts.end() );
			if ( begin == end ) return;
			FixFunction fixer;
			DWTIterator i = begin;
			*i = (*i)->acceptMutator( fixer );
			if ( fixer.get_isVoid() ) {
				DWTIterator j = i;
				++i;
				delete *j;
				dwts.erase( j );
				if ( i != end ) {
					throw SemanticError( "invalid type void in function type ", func );
				} // if
			} else {
				++i;
				for ( ; i != end; ++i ) {
					FixFunction fixer;
					*i = (*i )->acceptMutator( fixer );
					if ( fixer.get_isVoid() ) {
						throw SemanticError( "invalid type void in function type ", func );
					} // if
				} // for
			} // if
		}
	}

	void EnumAndPointerDecayPass::visit( FunctionType *func ) {
		// Fix up parameters and return types
		fixFunctionList( func->get_parameters(), func );
		fixFunctionList( func->get_returnVals(), func );
		Visitor::visit( func );
	}

	Pass2::Pass2( bool doDebug, const Indexer *other_indexer ) : Indexer( doDebug ) {
		if ( other_indexer ) {
			indexer = other_indexer;
		} else {
			indexer = this;
		} // if
	}

	void Pass2::visit( StructInstType *structInst ) {
		Parent::visit( structInst );
		StructDecl *st = indexer->lookupStruct( structInst->get_name() );
		// it's not a semantic error if the struct is not found, just an implicit forward declaration
		if ( st ) {
			//assert( ! structInst->get_baseStruct() || structInst->get_baseStruct()->get_members().empty() || ! st->get_members().empty() );
			structInst->set_baseStruct( st );
		} // if
		if ( ! st || st->get_members().empty() ) {
			// use of forward declaration
			forwardStructs[ structInst->get_name() ].push_back( structInst );
		} // if
	}

	void Pass2::visit( UnionInstType *unionInst ) {
		Parent::visit( unionInst );
		UnionDecl *un = indexer->lookupUnion( unionInst->get_name() );
		// it's not a semantic error if the union is not found, just an implicit forward declaration
		if ( un ) {
			unionInst->set_baseUnion( un );
		} // if
		if ( ! un || un->get_members().empty() ) {
			// use of forward declaration
			forwardUnions[ unionInst->get_name() ].push_back( unionInst );
		} // if
	}

	void Pass2::visit( TraitInstType *contextInst ) {
		Parent::visit( contextInst );
		TraitDecl *ctx = indexer->lookupTrait( contextInst->get_name() );
		if ( ! ctx ) {
			throw SemanticError( "use of undeclared context " + contextInst->get_name() );
		} // if
		for ( std::list< TypeDecl * >::const_iterator i = ctx->get_parameters().begin(); i != ctx->get_parameters().end(); ++i ) {
			for ( std::list< DeclarationWithType * >::const_iterator assert = (*i )->get_assertions().begin(); assert != (*i )->get_assertions().end(); ++assert ) {
				if ( TraitInstType *otherCtx = dynamic_cast< TraitInstType * >(*assert ) ) {
					cloneAll( otherCtx->get_members(), contextInst->get_members() );
				} else {
					contextInst->get_members().push_back( (*assert )->clone() );
				} // if
			} // for
		} // for

		if ( ctx->get_parameters().size() != contextInst->get_parameters().size() ) {
			throw SemanticError( "incorrect number of context parameters: ", contextInst );
		} // if

		// need to clone members of the context for ownership purposes
		std::list< Declaration * > members;
		std::transform( ctx->get_members().begin(), ctx->get_members().end(), back_inserter( members ), [](Declaration * dwt) { return dwt->clone(); } );

		applySubstitution( ctx->get_parameters().begin(), ctx->get_parameters().end(), contextInst->get_parameters().begin(), members.begin(), members.end(), back_inserter( contextInst->get_members() ) );
	}

	void Pass2::visit( StructDecl *structDecl ) {
		// visit struct members first so that the types of self-referencing members are updated properly
		Parent::visit( structDecl );
		if ( ! structDecl->get_members().empty() ) {
			ForwardStructsType::iterator fwds = forwardStructs.find( structDecl->get_name() );
			if ( fwds != forwardStructs.end() ) {
				for ( std::list< StructInstType * >::iterator inst = fwds->second.begin(); inst != fwds->second.end(); ++inst ) {
					(*inst )->set_baseStruct( structDecl );
				} // for
				forwardStructs.erase( fwds );
			} // if
		} // if
	}

	void Pass2::visit( UnionDecl *unionDecl ) {
		Parent::visit( unionDecl );
		if ( ! unionDecl->get_members().empty() ) {
			ForwardUnionsType::iterator fwds = forwardUnions.find( unionDecl->get_name() );
			if ( fwds != forwardUnions.end() ) {
				for ( std::list< UnionInstType * >::iterator inst = fwds->second.begin(); inst != fwds->second.end(); ++inst ) {
					(*inst )->set_baseUnion( unionDecl );
				} // for
				forwardUnions.erase( fwds );
			} // if
		} // if
	}

	void Pass2::visit( TypeInstType *typeInst ) {
		if ( NamedTypeDecl *namedTypeDecl = lookupType( typeInst->get_name() ) ) {
			if ( TypeDecl *typeDecl = dynamic_cast< TypeDecl * >( namedTypeDecl ) ) {
				typeInst->set_isFtype( typeDecl->get_kind() == TypeDecl::Ftype );
			} // if
		} // if
	}

	Pass3::Pass3( const Indexer *other_indexer ) :  Indexer( false ) {
		if ( other_indexer ) {
			indexer = other_indexer;
		} else {
			indexer = this;
		} // if
	}

	/// Fix up assertions
	void forallFixer( Type *func ) {
		for ( std::list< TypeDecl * >::iterator type = func->get_forall().begin(); type != func->get_forall().end(); ++type ) {
			std::list< DeclarationWithType * > toBeDone, nextRound;
			toBeDone.splice( toBeDone.end(), (*type )->get_assertions() );
			while ( ! toBeDone.empty() ) {
				for ( std::list< DeclarationWithType * >::iterator assertion = toBeDone.begin(); assertion != toBeDone.end(); ++assertion ) {
					if ( TraitInstType *ctx = dynamic_cast< TraitInstType * >( (*assertion )->get_type() ) ) {
						for ( std::list< Declaration * >::const_iterator i = ctx->get_members().begin(); i != ctx->get_members().end(); ++i ) {
							DeclarationWithType *dwt = dynamic_cast< DeclarationWithType * >( *i );
							assert( dwt );
							nextRound.push_back( dwt->clone() );
						}
						delete ctx;
					} else {
						FixFunction fixer;
						*assertion = (*assertion )->acceptMutator( fixer );
						if ( fixer.get_isVoid() ) {
							throw SemanticError( "invalid type void in assertion of function ", func );
						}
						(*type )->get_assertions().push_back( *assertion );
					} // if
				} // for
				toBeDone.clear();
				toBeDone.splice( toBeDone.end(), nextRound );
			} // while
		} // for
	}

	void Pass3::visit( ObjectDecl *object ) {
		forallFixer( object->get_type() );
		if ( PointerType *pointer = dynamic_cast< PointerType * >( object->get_type() ) ) {
			forallFixer( pointer->get_base() );
		} // if
		Parent::visit( object );
		object->fixUniqueId();
	}

	void Pass3::visit( FunctionDecl *func ) {
		forallFixer( func->get_type() );
		Parent::visit( func );
		func->fixUniqueId();
	}

	void ReturnChecker::checkFunctionReturns( std::list< Declaration * > & translationUnit ) {
		ReturnChecker checker;
		acceptAll( translationUnit, checker );
	}

	void ReturnChecker::visit( FunctionDecl * functionDecl ) {
		std::list< DeclarationWithType * > oldReturnVals = returnVals;
		returnVals = functionDecl->get_functionType()->get_returnVals();
		Visitor::visit( functionDecl );
		returnVals = oldReturnVals;
	}

	void ReturnChecker::visit( ReturnStmt * returnStmt ) {
		// Previously this also checked for the existence of an expr paired with no return values on
		// the  function return type. This is incorrect, since you can have an expression attached to
		// a return statement in a void-returning function in C. The expression is treated as if it
		// were cast to void.
		if ( returnStmt->get_expr() == NULL && returnVals.size() != 0 ) {
			throw SemanticError( "Non-void function returns no values: " , returnStmt );
		}
	}


	bool isTypedef( Declaration *decl ) {
		return dynamic_cast< TypedefDecl * >( decl );
	}

	void EliminateTypedef::eliminateTypedef( std::list< Declaration * > &translationUnit ) {
		EliminateTypedef eliminator;
		mutateAll( translationUnit, eliminator );
		if ( eliminator.typedefNames.count( "size_t" ) ) {
			// grab and remember declaration of size_t
			SizeType = eliminator.typedefNames["size_t"].first->get_base()->clone();
		} else {
			// xxx - missing global typedef for size_t - default to long unsigned int, even though that may be wrong
			// eventually should have a warning for this case.
			SizeType = new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt );
		}
		filter( translationUnit, isTypedef, true );

	}

	Type *EliminateTypedef::mutate( TypeInstType * typeInst ) {
		// instances of typedef types will come here. If it is an instance
		// of a typdef type, link the instance to its actual type.
		TypedefMap::const_iterator def = typedefNames.find( typeInst->get_name() );
		if ( def != typedefNames.end() ) {
			Type *ret = def->second.first->get_base()->clone();
			ret->get_qualifiers() += typeInst->get_qualifiers();
			// place instance parameters on the typedef'd type
			if ( ! typeInst->get_parameters().empty() ) {
				ReferenceToType *rtt = dynamic_cast<ReferenceToType*>(ret);
				if ( ! rtt ) {
					throw SemanticError("cannot apply type parameters to base type of " + typeInst->get_name());
				}
				rtt->get_parameters().clear();
				cloneAll( typeInst->get_parameters(), rtt->get_parameters() );
				mutateAll( rtt->get_parameters(), *this );  // recursively fix typedefs on parameters
			} // if
			delete typeInst;
			return ret;
		} else {
			TypeDeclMap::const_iterator base = typedeclNames.find( typeInst->get_name() );
			assert( base != typedeclNames.end() );
			typeInst->set_baseType( base->second );
		} // if
		return typeInst;
	}

	Declaration *EliminateTypedef::mutate( TypedefDecl * tyDecl ) {
		Declaration *ret = Mutator::mutate( tyDecl );

		if ( typedefNames.count( tyDecl->get_name() ) == 1 && typedefNames[ tyDecl->get_name() ].second == scopeLevel ) {
			// typedef to the same name from the same scope
			// must be from the same type

			Type * t1 = tyDecl->get_base();
			Type * t2 = typedefNames[ tyDecl->get_name() ].first->get_base();
			if ( ! ResolvExpr::typesCompatible( t1, t2, Indexer() ) ) {
				throw SemanticError( "cannot redefine typedef: " + tyDecl->get_name() );
			}
		} else {
			typedefNames[ tyDecl->get_name() ] = std::make_pair( TypedefDeclPtr( tyDecl ), scopeLevel );
		} // if

		// When a typedef is a forward declaration:
		//    typedef struct screen SCREEN;
		// the declaration portion must be retained:
		//    struct screen;
		// because the expansion of the typedef is:
		//    void rtn( SCREEN *p ) => void rtn( struct screen *p )
		// hence the type-name "screen" must be defined.
		// Note, qualifiers on the typedef are superfluous for the forward declaration.
		if ( StructInstType *aggDecl = dynamic_cast< StructInstType * >( tyDecl->get_base() ) ) {
			return new StructDecl( aggDecl->get_name() );
		} else if ( UnionInstType *aggDecl = dynamic_cast< UnionInstType * >( tyDecl->get_base() ) ) {
			return new UnionDecl( aggDecl->get_name() );
		} else if ( EnumInstType *enumDecl = dynamic_cast< EnumInstType * >( tyDecl->get_base() ) ) {
			return new EnumDecl( enumDecl->get_name() );
		} else {
			return ret->clone();
		} // if
	}

	TypeDecl *EliminateTypedef::mutate( TypeDecl * typeDecl ) {
		TypedefMap::iterator i = typedefNames.find( typeDecl->get_name() );
		if ( i != typedefNames.end() ) {
			typedefNames.erase( i ) ;
		} // if

		typedeclNames[ typeDecl->get_name() ] = typeDecl;
		return typeDecl;
	}

	DeclarationWithType *EliminateTypedef::mutate( FunctionDecl * funcDecl ) {
		typedefNames.beginScope();
		DeclarationWithType *ret = Mutator::mutate( funcDecl );
		typedefNames.endScope();
		return ret;
	}

	DeclarationWithType *EliminateTypedef::mutate( ObjectDecl * objDecl ) {
		typedefNames.beginScope();
		DeclarationWithType *ret = Mutator::mutate( objDecl );
		typedefNames.endScope();
		// is the type a function?
		if ( FunctionType *funtype = dynamic_cast<FunctionType *>( ret->get_type() ) ) {
			// replace the current object declaration with a function declaration
			return new FunctionDecl( ret->get_name(), ret->get_storageClass(), ret->get_linkage(), funtype, 0, ret->get_isInline(), ret->get_isNoreturn() );
		} else if ( objDecl->get_isInline() || objDecl->get_isNoreturn() ) {
			throw SemanticError( "invalid inline or _Noreturn specification in declaration of ", objDecl );
		} // if
		return ret;
	}

	Expression *EliminateTypedef::mutate( CastExpr * castExpr ) {
		typedefNames.beginScope();
		Expression *ret = Mutator::mutate( castExpr );
		typedefNames.endScope();
		return ret;
	}

	CompoundStmt *EliminateTypedef::mutate( CompoundStmt * compoundStmt ) {
		typedefNames.beginScope();
		scopeLevel += 1;
		CompoundStmt *ret = Mutator::mutate( compoundStmt );
		scopeLevel -= 1;
		std::list< Statement * >::iterator i = compoundStmt->get_kids().begin();
		while ( i != compoundStmt->get_kids().end() ) {
			std::list< Statement * >::iterator next = i+1;
			if ( DeclStmt *declStmt = dynamic_cast< DeclStmt * >( *i ) ) {
				if ( dynamic_cast< TypedefDecl * >( declStmt->get_decl() ) ) {
					delete *i;
					compoundStmt->get_kids().erase( i );
				} // if
			} // if
			i = next;
		} // while
		typedefNames.endScope();
		return ret;
	}

	// there may be typedefs nested within aggregates in order for everything to work properly, these should be removed
	// as well
	template<typename AggDecl>
	AggDecl *EliminateTypedef::handleAggregate( AggDecl * aggDecl ) {
		std::list<Declaration *>::iterator it = aggDecl->get_members().begin();
		for ( ; it != aggDecl->get_members().end(); ) {
			std::list< Declaration * >::iterator next = it+1;
			if ( dynamic_cast< TypedefDecl * >( *it ) ) {
				delete *it;
				aggDecl->get_members().erase( it );
			} // if
			it = next;
		}
		return aggDecl;
	}

	template<typename AggDecl>
	void EliminateTypedef::addImplicitTypedef( AggDecl * aggDecl ) {
		if ( typedefNames.count( aggDecl->get_name() ) == 0 ) {
			Type *type;
			if ( StructDecl * newDeclStructDecl = dynamic_cast< StructDecl * >( aggDecl ) ) {
				type = new StructInstType( Type::Qualifiers(), newDeclStructDecl->get_name() );
			} else if ( UnionDecl * newDeclUnionDecl = dynamic_cast< UnionDecl * >( aggDecl ) ) {
				type = new UnionInstType( Type::Qualifiers(), newDeclUnionDecl->get_name() );
			} else if ( EnumDecl * newDeclEnumDecl = dynamic_cast< EnumDecl * >( aggDecl )  ) {
				type = new EnumInstType( Type::Qualifiers(), newDeclEnumDecl->get_name() );
			} // if
			TypedefDeclPtr tyDecl( new TypedefDecl( aggDecl->get_name(), DeclarationNode::NoStorageClass, type ) );
			typedefNames[ aggDecl->get_name() ] = std::make_pair( std::move( tyDecl ), scopeLevel );
		} // if
	}

	Declaration *EliminateTypedef::mutate( StructDecl * structDecl ) {
		addImplicitTypedef( structDecl );
		Mutator::mutate( structDecl );
		return handleAggregate( structDecl );
	}

	Declaration *EliminateTypedef::mutate( UnionDecl * unionDecl ) {
		addImplicitTypedef( unionDecl );
		Mutator::mutate( unionDecl );
		return handleAggregate( unionDecl );
	}

	Declaration *EliminateTypedef::mutate( EnumDecl * enumDecl ) {
		addImplicitTypedef( enumDecl );
		Mutator::mutate( enumDecl );
		return handleAggregate( enumDecl );
	}

	Declaration *EliminateTypedef::mutate( TraitDecl * contextDecl ) {
		Mutator::mutate( contextDecl );
		return handleAggregate( contextDecl );
	}

	void VerifyCtorDtor::verify( std::list< Declaration * > & translationUnit ) {
		VerifyCtorDtor verifier;
		acceptAll( translationUnit, verifier );
	}

	void VerifyCtorDtor::visit( FunctionDecl * funcDecl ) {
		FunctionType * funcType = funcDecl->get_functionType();
		std::list< DeclarationWithType * > &returnVals = funcType->get_returnVals();
		std::list< DeclarationWithType * > &params = funcType->get_parameters();

		if ( funcDecl->get_name() == "?{}" || funcDecl->get_name() == "^?{}" ) {
			if ( params.size() == 0 ) {
				throw SemanticError( "Constructors and destructors require at least one parameter ", funcDecl );
			}
			if ( ! dynamic_cast< PointerType * >( params.front()->get_type() ) ) {
				throw SemanticError( "First parameter of a constructor or destructor must be a pointer ", funcDecl );
			}
			if ( returnVals.size() != 0 ) {
				throw SemanticError( "Constructors and destructors cannot have explicit return values ", funcDecl );
			}
		}

		Visitor::visit( funcDecl );
	}

	DeclarationWithType * CompoundLiteral::mutate( ObjectDecl *objectDecl ) {
		storageclass = objectDecl->get_storageClass();
		DeclarationWithType * temp = Mutator::mutate( objectDecl );
		storageclass = DeclarationNode::NoStorageClass;
		return temp;
	}

	Expression *CompoundLiteral::mutate( CompoundLiteralExpr *compLitExpr ) {
		// transform [storage_class] ... (struct S){ 3, ... };
		// into [storage_class] struct S temp =  { 3, ... };
		static UniqueName indexName( "_compLit" );

		ObjectDecl *tempvar = new ObjectDecl( indexName.newName(), storageclass, LinkageSpec::C, 0, compLitExpr->get_type(), compLitExpr->get_initializer() );
		compLitExpr->set_type( 0 );
		compLitExpr->set_initializer( 0 );
		delete compLitExpr;
		DeclarationWithType * newtempvar = mutate( tempvar );
		addDeclaration( newtempvar );					// add modified temporary to current block
		return new VariableExpr( newtempvar );
	}
} // namespace SymTab

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
