//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Indexer.h -- 
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 21:38:55 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Mar  2 17:34:14 2016
// Update Count     : 6
//

#ifndef INDEXER_H
#define INDEXER_H

#include <list>
#include <string>

#include "SynTree/Visitor.h"

namespace SymTab {
	class Indexer : public Visitor {
	  public:
		Indexer( bool useDebug = false );

		Indexer( const Indexer &that );
		Indexer( Indexer &&that );
		virtual ~Indexer();
		Indexer& operator= ( const Indexer &that );
		Indexer& operator= ( Indexer &&that );

		//using Visitor::visit;
		virtual void visit( ObjectDecl *objectDecl );
		virtual void visit( FunctionDecl *functionDecl );
		virtual void visit( TypeDecl *typeDecl );
		virtual void visit( TypedefDecl *typeDecl );
		virtual void visit( StructDecl *aggregateDecl );
		virtual void visit( UnionDecl *aggregateDecl );
		virtual void visit( EnumDecl *aggregateDecl );
		virtual void visit( TraitDecl *aggregateDecl );

		virtual void visit( CompoundStmt *compoundStmt );

		virtual void visit( ApplicationExpr *applicationExpr );
		virtual void visit( UntypedExpr *untypedExpr );
		virtual void visit( NameExpr *nameExpr );
		virtual void visit( CastExpr *castExpr );
		virtual void visit( AddressExpr *addressExpr );
		virtual void visit( LabelAddressExpr *labAddressExpr );
		virtual void visit( UntypedMemberExpr *memberExpr );
		virtual void visit( MemberExpr *memberExpr );
		virtual void visit( VariableExpr *variableExpr );
		virtual void visit( ConstantExpr *constantExpr ); 
		virtual void visit( SizeofExpr *sizeofExpr );
		virtual void visit( AlignofExpr *alignofExpr );
		virtual void visit( UntypedOffsetofExpr *offsetofExpr );
		virtual void visit( OffsetofExpr *offsetofExpr );
		virtual void visit( OffsetPackExpr *offsetPackExpr );
		virtual void visit( AttrExpr *attrExpr );
		virtual void visit( LogicalExpr *logicalExpr );
		virtual void visit( ConditionalExpr *conditionalExpr );
		virtual void visit( CommaExpr *commaExpr );
		virtual void visit( TupleExpr *tupleExpr );
		virtual void visit( SolvedTupleExpr *tupleExpr );
		virtual void visit( TypeExpr *typeExpr );
		virtual void visit( AsmExpr *asmExpr );
		virtual void visit( UntypedValofExpr *valofExpr );

		virtual void visit( TraitInstType *contextInst );
		virtual void visit( StructInstType *contextInst );
		virtual void visit( UnionInstType *contextInst );

		virtual void visit( ForStmt *forStmt );

		// when using an indexer manually (e.g., within a mutator traversal), it is necessary to tell the indexer
		// explicitly when scopes begin and end
		void enterScope();
		void leaveScope();

		/// Gets all declarations with the given ID
		void lookupId( const std::string &id, std::list< DeclarationWithType* > &out ) const;
		/// Gets the top-most type declaration with the given ID
		NamedTypeDecl *lookupType( const std::string &id ) const;
		/// Gets the top-most struct declaration with the given ID
		StructDecl *lookupStruct( const std::string &id ) const;
		/// Gets the top-most enum declaration with the given ID
		EnumDecl *lookupEnum( const std::string &id ) const;
		/// Gets the top-most union declaration with the given ID
		UnionDecl *lookupUnion( const std::string &id ) const;
		/// Gets the top-most trait declaration with the given ID
		TraitDecl *lookupTrait( const std::string &id ) const;
  
		void print( std::ostream &os, int indent = 0 ) const;
	  private:
		/// looks up a specific mangled ID at the given scope
		DeclarationWithType *lookupIdAtScope( const std::string &id, const std::string &mangleName, unsigned long scope ) const;
		/// returns true if there exists a declaration with C linkage and the given name with a different mangled name
		bool hasIncompatibleCDecl( const std::string &id, const std::string &mangleName ) const;
		// equivalents to lookup functions that only look at tables at scope `scope` (which should be >= tables->scope)
		NamedTypeDecl *lookupTypeAtScope( const std::string &id, unsigned long scope ) const;
		StructDecl *lookupStructAtScope( const std::string &id, unsigned long scope ) const;
		EnumDecl *lookupEnumAtScope( const std::string &id, unsigned long scope ) const;
		UnionDecl *lookupUnionAtScope( const std::string &id, unsigned long scope ) const;
		TraitDecl *lookupTraitAtScope( const std::string &id, unsigned long scope ) const;
		
		void addId( DeclarationWithType *decl );
		void addType( NamedTypeDecl *decl );
		void addStruct( const std::string &id );
		void addStruct( StructDecl *decl );
		void addEnum( EnumDecl *decl );
		void addUnion( const std::string &id );
		void addUnion( UnionDecl *decl );
		void addTrait( TraitDecl *decl );
		
		struct Impl;
		Impl *tables;         ///< Copy-on-write instance of table data structure
		unsigned long scope;  ///< Scope index of this pointer
		bool doDebug;         ///< Display debugging trace?

		/// Takes a new ref to a table (returns null if null)
		static Impl *newRef( Impl *toClone );
		/// Clears a ref to a table (does nothing if null)
		static void deleteRef( Impl *toFree );

		/// Ensures that tables variable is writable (i.e. allocated, uniquely owned by this Indexer, and at the current scope)
		void makeWritable();
	};
} // namespace SymTab

#endif // INDEXER_H

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
