//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// CandidateFinder.cpp --
//
// Author           : Aaron B. Moss
// Created On       : Wed Jun 5 14:30:00 2019
// Last Modified By : Aaron B. Moss
// Last Modified On : Wed Jun 5 14:30:00 2019
// Update Count     : 1
//

#include "CandidateFinder.hpp"

#include <sstream>
#include <string>
#include <unordered_map>

#include "Candidate.hpp"
#include "CompilationState.h"
#include "Cost.h"
#include "Resolver.h"
#include "SatisfyAssertions.hpp"
#include "typeops.h"              // for adjustExprType
#include "AST/Expr.hpp"
#include "AST/Node.hpp"
#include "AST/Pass.hpp"
#include "AST/Print.hpp"
#include "SymTab/Mangler.h"

#define PRINT( text ) if ( resolvep ) { text }

namespace ResolvExpr {

namespace {

	/// Actually visits expressions to find their candidate interpretations
	struct Finder {
		CandidateFinder & candFinder;
		const ast::SymbolTable & symtab;
		CandidateList & candidates;
		const ast::TypeEnvironment & tenv;
		ast::ptr< ast::Type > & targetType;

		Finder( CandidateFinder & f )
		: candFinder( f ), symtab( f.symtab ), candidates( f.candidates ), tenv( f.env ), 
		  targetType( f.targetType ) {}
		
		#warning unimplemented
	};

	/// Prunes a list of candidates down to those that have the minimum conversion cost for a given 
	/// return type. Skips ambiguous candidates.
	CandidateList pruneCandidates( CandidateList & candidates ) {
		struct PruneStruct {
			CandidateRef candidate;
			bool ambiguous;

			PruneStruct() = default;
			PruneStruct( const CandidateRef & c ) : candidate( c ), ambiguous( false ) {}
		};

		// find lowest-cost candidate for each type
		std::unordered_map< std::string, PruneStruct > selected;
		for ( CandidateRef & candidate : candidates ) {
			std::string mangleName;
			{
				ast::ptr< ast::Type > newType = candidate->expr->result;
				candidate->env.apply( newType );
				mangleName = Mangle::mangle( newType );
			}

			auto found = selected.find( mangleName );
			if ( found != selected.end() ) {
				if ( candidate->cost < found->second.candidate->cost ) {
					PRINT(
						std::cerr << "cost " << candidate->cost << " beats " 
							<< found->second.candidate->cost << std::endl;
					)

					found->second = PruneStruct{ candidate };
				} else if ( candidate->cost == found->second.candidate->cost ) {
					// if one of the candidates contains a deleted identifier, can pick the other, 
					// since deleted expressions should not be ambiguous if there is another option 
					// that is at least as good
					if ( findDeletedExpr( candidate->expr ) ) {
						// do nothing
						PRINT( std::cerr << "candidate is deleted" << std::endl; )
					} else if ( findDeletedExpr( found->second.candidate->expr ) ) {
						PRINT( std::cerr << "current is deleted" << std::endl; )
						found->second = PruneStruct{ candidate };
					} else {
						PRINT( std::cerr << "marking ambiguous" << std::endl; )
						found->second.ambiguous = true;
					}
				} else {
					PRINT(
						std::cerr << "cost " << candidate->cost << " loses to " 
							<< found->second.candidate->cost << std::endl;
					)
				}
			} else {
				selected.emplace_hint( found, mangleName, candidate );
			}
		}

		// report unambiguous min-cost candidates
		CandidateList out;
		for ( auto & target : selected ) {
			if ( target.second.ambiguous ) continue;

			CandidateRef cand = target.second.candidate;
			
			ast::ptr< ast::Type > newResult = cand->expr->result;
			cand->env.applyFree( newResult );
			cand->expr = ast::mutate_field(
				cand->expr.get(), &ast::Expr::result, std::move( newResult ) );
			
			out.emplace_back( cand );
		}
		return out;
	}

	/// Returns a list of alternatives with the minimum cost in the given list
	CandidateList findMinCost( const CandidateList & candidates ) {
		CandidateList out;
		Cost minCost = Cost::infinity;
		for ( const CandidateRef & r : candidates ) {
			if ( r->cost < minCost ) {
				minCost = r->cost;
				out.clear();
				out.emplace_back( r );
			} else if ( r->cost == minCost ) {
				out.emplace_back( r );
			}
		}
		return out;
	}

} // anonymous namespace

void CandidateFinder::find( const ast::Expr * expr, ResolvMode mode ) {
	// Find alternatives for expression
	ast::Pass<Finder> finder{ *this };
	expr->accept( finder );

	if ( mode.failFast && candidates.empty() ) {
		SemanticError( expr, "No reasonable alternatives for expression " );
	}

	if ( mode.satisfyAssns || mode.prune ) {
		// trim candidates to just those where the assertions are satisfiable
		// - necessary pre-requisite to pruning
		CandidateList satisfied;
		std::vector< std::string > errors;
		for ( auto & candidate : candidates ) {
			satisfyAssertions( *candidate, symtab, satisfied, errors );
		}

		// fail early if none such
		if ( mode.failFast && satisfied.empty() ) {
			std::ostringstream stream;
			stream << "No alternatives with satisfiable assertions for " << expr << "\n";
			for ( const auto& err : errors ) {
				stream << err;
			}
			SemanticError( expr->location, stream.str() );
		}

		// reset candidates
		candidates = std::move( satisfied );
	}

	if ( mode.prune ) {
		// trim candidates to single best one
		PRINT(
			std::cerr << "alternatives before prune:" << std::endl;
			print( std::cerr, candidates );
		)

		CandidateList pruned = pruneCandidates( candidates );
		
		if ( mode.failFast && pruned.empty() ) {
			std::ostringstream stream;
			CandidateList winners = findMinCost( candidates );
			stream << "Cannot choose between " << winners.size() << " alternatives for "
				"expression\n";
			ast::print( stream, expr );
			stream << " Alternatives are:\n";
			print( stream, winners, 1 );
			SemanticError( expr->location, stream.str() );
		}

		auto oldsize = candidates.size();
		candidates = std::move( pruned );

		PRINT(
			std::cerr << "there are " << oldsize << " alternatives before elimination" << std::endl;
		)
		PRINT(
			std::cerr << "there are " << candidates.size() << " alternatives after elimination" 
				<< std::endl;
		)
	}

	// adjust types after pruning so that types substituted by pruneAlternatives are correctly 
	// adjusted
	if ( mode.adjust ) {
		for ( CandidateRef & r : candidates ) {
			r->expr = ast::mutate_field( 
				r->expr.get(), &ast::Expr::result, 
				adjustExprType( r->expr->result, r->env, symtab ) );
		}
	}

	// Central location to handle gcc extension keyword, etc. for all expressions
	for ( CandidateRef & r : candidates ) {
		if ( r->expr->extension != expr->extension ) {
			r->expr.get_and_mutate()->extension = expr->extension;
		}
	}
}

std::vector< CandidateFinder > CandidateFinder::findSubExprs( 
	const std::vector< ast::ptr< ast::Expr > > & xs 
) {
	std::vector< CandidateFinder > out;

	for ( const auto & x : xs ) {
		out.emplace_back( symtab, env );
		out.back().find( x, ResolvMode::withAdjustment() );
		
		PRINT(
			std::cerr << "findSubExprs" << std::endl;
			print( std::cerr, out.back().candidates );
		)
	}

	return out;
}

} // namespace ResolvExpr

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
