//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// SemanticError.h --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Aug 29 22:03:36 2017
// Update Count     : 17
//

#pragma once

#include <exception>									// for exception
#include <iostream>										// for ostream
#include <list>											// for list
#include <string>										// for string
#include <unistd.h>										// for isatty

#include "CodeLocation.h"								// for CodeLocation, toString

//-----------------------------------------------------------------------------
// Errors
struct error {
	CodeLocation location;
	std::string description;

	error() = default;
	error( CodeLocation loc, const std::string & str ) : location( loc ), description( str ) {}
};

class SemanticError : public std::exception {
  public:
  	SemanticError() = default;
	SemanticError( CodeLocation location, std::string error );
	~SemanticError() throw() {}

	// constructs an exception using the given message and the printed representation of the obj (T must have a print method)
	template< typename T > SemanticError(const T * obj, const std::string & error);
	template< typename T > SemanticError( CodeLocation location, const T * obj, const std::string & error);

	static inline const std::string & error_str() {
		static std::string str = isatty( STDERR_FILENO ) ? "\e[31merror:\e[39m " : "error: ";
		return str;
	}

	void append( SemanticError & other );
	void append( CodeLocation location, const std::string & );
	bool isEmpty() const;
	void print();
  private:
	std::list< error > errors;
};

template< typename T >
SemanticError::SemanticError( const T * obj, const std::string & error )
	: SemanticError( obj->location, toString( error, obj ) )
{}

template< typename T >
SemanticError::SemanticError( CodeLocation location, const T * obj, const std::string & error )
	: SemanticError( location, toString( error, obj ) )
{}

//-----------------------------------------------------------------------------
// Warnings
class SemanticWarning {
  public:
  	SemanticWarning( CodeLocation location, std::string error );
	~SemanticWarning() throw() {}

	// constructs an exception using the given message and the printed representation of the obj (T must have a print method)
	template< typename T > SemanticWarning(const T * obj, const std::string & error);
	template< typename T > SemanticWarning( CodeLocation location, const T * obj, const std::string & error);

	static inline const std::string & warning_str() {
		static std::string str = isatty( STDERR_FILENO ) ? "\e[95mwarning:\e[39m " : "warning: ";
		return str;
	}

  private:
};

template< typename T >
SemanticWarning::SemanticWarning( const T * obj, const std::string & error )
	: SemanticWarning( obj->location, toString( error, obj ) )
{}

template< typename T >
SemanticWarning::SemanticWarning( CodeLocation location, const T * obj, const std::string & error )
	: SemanticWarning( location, toString( error, obj ) )
{}

//-----------------------------------------------------------------------------
// Helpers
static inline const std::string & bold_ttycode() {
	static std::string str = isatty( STDERR_FILENO ) ? "\e[1m" : "";
	return str;
}

static inline const std::string & reset_font_ttycode() {
	static std::string str = isatty( STDERR_FILENO ) ? "\e[0m" : "";
	return str;
}

static inline std::string make_bold( const std::string & str ) {
	return bold_ttycode() + str + reset_font_ttycode();
}

struct bold {};
static inline std::ostream & operator<<(std::ostream & os, bold) {
	os << bold_ttycode();
	return os;
}

struct reset_font {};
static inline std::ostream & operator<<(std::ostream & os, reset_font) {
	os << reset_font_ttycode();
	return os;
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
