//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// FixInit.h --
//
// Author           : Rob Schluntz
// Created On       : Wed Jan 13 16:29:30 2016
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Jun 21 17:35:05 2017
// Update Count     : 74
//
#include "FixInit.h"

#include <stddef.h>                    // for NULL
#include <algorithm>                   // for set_difference, copy_if
#include <cassert>                     // for assert, strict_dynamic_cast
#include <iostream>                    // for operator<<, ostream, basic_ost...
#include <iterator>                    // for insert_iterator, back_inserter
#include <list>                        // for _List_iterator, list, list<>::...
#include <map>                         // for _Rb_tree_iterator, _Rb_tree_co...
#include <memory>                      // for allocator_traits<>::value_type
#include <set>                         // for set, set<>::value_type
#include <unordered_map>               // for unordered_map, unordered_map<>...
#include <unordered_set>               // for unordered_set
#include <utility>                     // for pair

#include "CodeGen/GenType.h"           // for genPrettyType
#include "CodeGen/OperatorTable.h"
#include "Common/PassVisitor.h"        // for PassVisitor, WithStmtsToAdd
#include "Common/SemanticError.h"      // for SemanticError
#include "Common/UniqueName.h"         // for UniqueName
#include "Common/utility.h"            // for CodeLocation, ValueGuard, toSt...
#include "FixGlobalInit.h"             // for fixGlobalInit
#include "GenInit.h"                   // for genCtorDtor
#include "GenPoly/GenPoly.h"           // for getFunctionType
#include "GenPoly/PolyMutator.h"       // for PolyMutator
#include "InitTweak.h"                 // for getFunctionName, getCallArg
#include "Parser/LinkageSpec.h"        // for C, Spec, Cforall, isBuiltin
#include "ResolvExpr/Resolver.h"       // for findVoidExpression
#include "ResolvExpr/typeops.h"        // for typesCompatible
#include "SymTab/Autogen.h"            // for genImplicitCall
#include "SymTab/Indexer.h"            // for Indexer
#include "SymTab/Mangler.h"            // for Mangler
#include "SynTree/AddStmtVisitor.h"    // for AddStmtVisitor
#include "SynTree/Attribute.h"         // for Attribute
#include "SynTree/Constant.h"          // for Constant
#include "SynTree/Declaration.h"       // for ObjectDecl, FunctionDecl, Decl...
#include "SynTree/Expression.h"        // for UniqueExpr, VariableExpr, Unty...
#include "SynTree/Initializer.h"       // for ConstructorInit, SingleInit
#include "SynTree/Label.h"             // for Label, noLabels, operator<
#include "SynTree/Mutator.h"           // for mutateAll, Mutator, maybeMutate
#include "SynTree/Statement.h"         // for ExprStmt, CompoundStmt, Branch...
#include "SynTree/Type.h"              // for Type, Type::StorageClasses
#include "SynTree/TypeSubstitution.h"  // for TypeSubstitution, operator<<
#include "SynTree/Visitor.h"           // for acceptAll, maybeAccept

bool ctordtorp = false; // print all debug
bool ctorp = false; // print ctor debug
bool cpctorp = false; // print copy ctor debug
bool dtorp = false; // print dtor debug
#define PRINT( text ) if ( ctordtorp ) { text }
#define CP_CTOR_PRINT( text ) if ( ctordtorp || cpctorp ) { text }
#define DTOR_PRINT( text ) if ( ctordtorp || dtorp ) { text }

namespace InitTweak {
	namespace {
		typedef std::unordered_map< int, int > UnqCount;

		struct InsertImplicitCalls : public WithTypeSubstitution {
			/// wrap function application expressions as ImplicitCopyCtorExpr nodes so that it is easy to identify which
			/// function calls need their parameters to be copy constructed
			static void insert( std::list< Declaration * > & translationUnit );

			Expression * postmutate( ApplicationExpr * appExpr );
		};

		struct ResolveCopyCtors final : public WithIndexer, public WithShortCircuiting, public WithTypeSubstitution {
			/// generate temporary ObjectDecls for each argument and return value of each ImplicitCopyCtorExpr,
			/// generate/resolve copy construction expressions for each, and generate/resolve destructors for both
			/// arguments and return value temporaries
			static void resolveImplicitCalls( std::list< Declaration * > & translationUnit, UnqCount & unqCount );

			ResolveCopyCtors( UnqCount & unqCount ) : unqCount( unqCount ) {}

			void postvisit( ImplicitCopyCtorExpr * impCpCtorExpr );
			void postvisit( StmtExpr * stmtExpr );
			void previsit( UniqueExpr * unqExpr );
			void postvisit( UniqueExpr * unqExpr );

			/// create and resolve ctor/dtor expression: fname(var, [cpArg])
			Expression * makeCtorDtor( const std::string & fname, ObjectDecl * var, Expression * cpArg = NULL );
			/// true if type does not need to be copy constructed to ensure correctness
			bool skipCopyConstruct( Type * type );
			void copyConstructArg( Expression *& arg, ImplicitCopyCtorExpr * impCpCtorExpr );
			void destructRet( ObjectDecl * ret, ImplicitCopyCtorExpr * impCpCtorExpr );

			UnqCount & unqCount; // count the number of times each unique expr ID appears
			std::unordered_set< int > vars;
		};

		/// collects constructed object decls - used as a base class
		struct ObjDeclCollector : public WithGuards, public WithShortCircuiting {
			// use ordered data structure to maintain ordering for set_difference and for consistent error messages
			typedef std::list< ObjectDecl * > ObjectSet;
			void previsit( CompoundStmt *compoundStmt );
			void previsit( DeclStmt *stmt );

			// don't go into other functions
			void previsit( FunctionDecl * ) { visit_children = false; }

		  protected:
			ObjectSet curVars;
		};

		// debug
		template<typename ObjectSet>
		struct PrintSet {
			PrintSet( const ObjectSet & objs ) : objs( objs ) {}
			const ObjectSet & objs;
		};
		template<typename ObjectSet>
		PrintSet<ObjectSet> printSet( const ObjectSet & objs ) { return PrintSet<ObjectSet>( objs ); }
		template<typename ObjectSet>
		std::ostream & operator<<( std::ostream & out, const PrintSet<ObjectSet> & set) {
			out << "{ ";
			for ( ObjectDecl * obj : set.objs ) {
				out << obj->get_name() << ", " ;
			} // for
			out << " }";
			return out;
		}

		struct LabelFinder final : public ObjDeclCollector {
			typedef std::map< Label, ObjectSet > LabelMap;
			// map of Label -> live variables at that label
			LabelMap vars;

			typedef ObjDeclCollector Parent;
			using Parent::previsit;
			void previsit( Statement * stmt );

			void previsit( CompoundStmt *compoundStmt );
			void previsit( DeclStmt *stmt );
		};

		struct InsertDtors final : public ObjDeclCollector, public WithStmtsToAdd {
			/// insert destructor calls at the appropriate places.  must happen before CtorInit nodes are removed
			/// (currently by FixInit)
			static void insert( std::list< Declaration * > & translationUnit );

			typedef std::list< ObjectDecl * > OrderedDecls;
			typedef std::list< OrderedDecls > OrderedDeclsStack;

			InsertDtors( PassVisitor<LabelFinder> & finder ) : finder( finder ), labelVars( finder.pass.vars ) {}

			typedef ObjDeclCollector Parent;
			using Parent::previsit;

			void previsit( ObjectDecl * objDecl );
			void previsit( FunctionDecl * funcDecl );

			void previsit( CompoundStmt * compoundStmt );
			void postvisit( CompoundStmt * compoundStmt );
			void previsit( ReturnStmt * returnStmt );
			void previsit( BranchStmt * stmt );
		private:
			void handleGoto( BranchStmt * stmt );

			PassVisitor<LabelFinder> & finder;
			LabelFinder::LabelMap & labelVars;
			OrderedDeclsStack reverseDeclOrder;
		};

		class FixInit : public WithStmtsToAdd {
		  public:
			/// expand each object declaration to use its constructor after it is declared.
			static void fixInitializers( std::list< Declaration * > &translationUnit );

			DeclarationWithType * postmutate( ObjectDecl *objDecl );

			std::list< Declaration * > staticDtorDecls;
		};

		class FixCopyCtors final : public WithStmtsToAdd, public WithShortCircuiting, public WithVisitorRef<FixCopyCtors> {
		  public:
			FixCopyCtors( UnqCount & unqCount ) : unqCount( unqCount ){}
			/// expand ImplicitCopyCtorExpr nodes into the temporary declarations, copy constructors, call expression,
			/// and destructors
			static void fixCopyCtors( std::list< Declaration * > &translationUnit, UnqCount & unqCount );

			Expression * postmutate( ImplicitCopyCtorExpr * impCpCtorExpr );
			void premutate( StmtExpr * stmtExpr );
			void premutate( UniqueExpr * unqExpr );

			UnqCount & unqCount;
		};

		struct GenStructMemberCalls final : public WithGuards, public WithShortCircuiting, public WithIndexer {
			/// generate default/copy ctor and dtor calls for user-defined struct ctor/dtors
			/// for any member that is missing a corresponding ctor/dtor call.
			/// error if a member is used before constructed
			static void generate( std::list< Declaration * > & translationUnit );

			void previsit( FunctionDecl * funcDecl );
			void postvisit( FunctionDecl * funcDecl );

			void previsit( MemberExpr * memberExpr );
			void previsit( ApplicationExpr * appExpr );

			SemanticError errors;
		  private:
			template< typename... Params >
			void emit( CodeLocation, const Params &... params );

			FunctionDecl * function = 0;
			std::set< DeclarationWithType * > unhandled;
			std::map< DeclarationWithType *, CodeLocation > usedUninit;
			ObjectDecl * thisParam = 0;
			bool isCtor = false; // true if current function is a constructor
			StructDecl * structDecl = 0;
		};

		// very simple resolver-like mutator class - used to
		// resolve UntypedExprs that are found within newly
		// generated constructor/destructor calls
		class MutatingResolver final : public Mutator {
		  public:
			MutatingResolver( SymTab::Indexer & indexer ) : indexer( indexer ) {}

			using Mutator::mutate;
			virtual DeclarationWithType* mutate( ObjectDecl *objectDecl ) override;
			virtual Expression* mutate( UntypedExpr *untypedExpr ) override;

		  private:
			SymTab::Indexer & indexer;
		};

		struct FixCtorExprs final : public WithDeclsToAdd, public WithIndexer {
			/// expands ConstructorExpr nodes into comma expressions, using a temporary for the first argument
			static void fix( std::list< Declaration * > & translationUnit );

			Expression * postmutate( ConstructorExpr * ctorExpr );
		};
	} // namespace

	void fix( std::list< Declaration * > & translationUnit, const std::string & filename, bool inLibrary ) {
		// fixes ConstructorInit for global variables. should happen before fixInitializers.
		InitTweak::fixGlobalInit( translationUnit, filename, inLibrary );

		UnqCount unqCount;

		InsertImplicitCalls::insert( translationUnit );
		ResolveCopyCtors::resolveImplicitCalls( translationUnit, unqCount );
		InsertDtors::insert( translationUnit );
		FixInit::fixInitializers( translationUnit );

		// FixCopyCtors must happen after FixInit, so that destructors are placed correctly
		FixCopyCtors::fixCopyCtors( translationUnit, unqCount );

		GenStructMemberCalls::generate( translationUnit );
		// xxx - ctor expansion currently has to be after FixCopyCtors, because there is currently a
		// hack in the way untyped assignments are generated, where the first argument cannot have
		// its address taken because of the way codegeneration handles UntypedExpr vs. ApplicationExpr.
		// Thus such assignment exprs must never pushed through expression resolution (and thus should
		// not go through the FixCopyCtors pass), otherwise they will fail -- guaranteed.
		// Also needs to happen after GenStructMemberCalls, since otherwise member constructors exprs
		// don't look right, and a member can be constructed more than once.
		FixCtorExprs::fix( translationUnit );
	}

	namespace {
		void InsertImplicitCalls::insert( std::list< Declaration * > & translationUnit ) {
			PassVisitor<InsertImplicitCalls> inserter;
			mutateAll( translationUnit, inserter );
		}

		void ResolveCopyCtors::resolveImplicitCalls( std::list< Declaration * > & translationUnit, UnqCount & unqCount ) {
			PassVisitor<ResolveCopyCtors> resolver( unqCount );
			acceptAll( translationUnit, resolver );
		}

		void FixInit::fixInitializers( std::list< Declaration * > & translationUnit ) {
			PassVisitor<FixInit> fixer;

			// can't use mutateAll, because need to insert declarations at top-level
			// can't use DeclMutator, because sometimes need to insert IfStmt, etc.
			SemanticError errors;
			for ( std::list< Declaration * >::iterator i = translationUnit.begin(); i != translationUnit.end(); ++i ) {
				try {
					*i = maybeMutate( *i, fixer );
					translationUnit.splice( i, fixer.pass.staticDtorDecls );
				} catch( SemanticError &e ) {
					e.set_location( (*i)->location );
					errors.append( e );
				} // try
			} // for
			if ( ! errors.isEmpty() ) {
				throw errors;
			} // if
		}

		void InsertDtors::insert( std::list< Declaration * > & translationUnit ) {
			PassVisitor<LabelFinder> finder;
			PassVisitor<InsertDtors> inserter( finder );
			acceptAll( translationUnit, inserter );
		}

		void FixCopyCtors::fixCopyCtors( std::list< Declaration * > & translationUnit, UnqCount & unqCount ) {
			PassVisitor<FixCopyCtors> fixer( unqCount );
			mutateAll( translationUnit, fixer );
		}

		void GenStructMemberCalls::generate( std::list< Declaration * > & translationUnit ) {
			PassVisitor<GenStructMemberCalls> warner;
			acceptAll( translationUnit, warner );
		}

		void FixCtorExprs::fix( std::list< Declaration * > & translationUnit ) {
			PassVisitor<FixCtorExprs> fixer;
			mutateAll( translationUnit, fixer );
		}

		Expression * InsertImplicitCalls::postmutate( ApplicationExpr * appExpr ) {
			assert( appExpr );

			if ( VariableExpr * function = dynamic_cast< VariableExpr * > ( appExpr->get_function() ) ) {
				if ( LinkageSpec::isBuiltin( function->get_var()->get_linkage() ) ) {
					// optimization: don't need to copy construct in order to call intrinsic functions
					return appExpr;
				} else if ( DeclarationWithType * funcDecl = dynamic_cast< DeclarationWithType * > ( function->get_var() ) ) {
					FunctionType * ftype = dynamic_cast< FunctionType * >( GenPoly::getFunctionType( funcDecl->get_type() ) );
					assertf( ftype, "Function call without function type: %s", toString( funcDecl ).c_str() );
					if ( CodeGen::isConstructor( funcDecl->get_name() ) && ftype->get_parameters().size() == 2 ) {
						Type * t1 = getPointerBase( ftype->get_parameters().front()->get_type() );
						Type * t2 = ftype->get_parameters().back()->get_type();
						assert( t1 );

						if ( ResolvExpr::typesCompatible( t1, t2, SymTab::Indexer() ) ) {
							// optimization: don't need to copy construct in order to call a copy constructor
							return appExpr;
						} // if
					} else if ( CodeGen::isDestructor( funcDecl->get_name() ) ) {
						// correctness: never copy construct arguments to a destructor
						return appExpr;
					} // if
				} // if
			} // if
			CP_CTOR_PRINT( std::cerr << "InsertImplicitCalls: adding a wrapper " << appExpr << std::endl; )

			// wrap each function call so that it is easy to identify nodes that have to be copy constructed
			ImplicitCopyCtorExpr * expr = new ImplicitCopyCtorExpr( appExpr );
			// Move the type substitution to the new top-level, if it is attached to the appExpr.
			// Ensure it is not deleted with the ImplicitCopyCtorExpr by removing it before deletion.
			// The substitution is needed to obtain the type of temporary variables so that copy constructor
			// calls can be resolved.
			assert( env );
			std::swap( expr->env, appExpr->env );
			return expr;
		}

		bool ResolveCopyCtors::skipCopyConstruct( Type * type ) { return ! isConstructable( type ); }

		Expression * ResolveCopyCtors::makeCtorDtor( const std::string & fname, ObjectDecl * var, Expression * cpArg ) {
			assert( var );
			// arrays are not copy constructed, so this should always be an ExprStmt
			ImplicitCtorDtorStmt * stmt = genCtorDtor( fname, var, cpArg );
			ExprStmt * exprStmt = strict_dynamic_cast< ExprStmt * >( stmt->get_callStmt() );
			Expression * untyped = exprStmt->get_expr();

			// resolve copy constructor
			// should only be one alternative for copy ctor and dtor expressions, since all arguments are fixed
			// (VariableExpr and already resolved expression)
			CP_CTOR_PRINT( std::cerr << "ResolvingCtorDtor " << untyped << std::endl; )
			Expression * resolved = ResolvExpr::findVoidExpression( untyped, indexer );
			assert( resolved );
			if ( resolved->get_env() ) {
				// Extract useful information and discard new environments. Keeping them causes problems in PolyMutator passes.
				env->add( *resolved->get_env() );
				delete resolved->get_env();
				resolved->set_env( nullptr );
			} // if

			delete stmt;
			return resolved;
		}

		void ResolveCopyCtors::copyConstructArg( Expression *& arg, ImplicitCopyCtorExpr * impCpCtorExpr ) {
			static UniqueName tempNamer("_tmp_cp");
			assert( env );
			CP_CTOR_PRINT( std::cerr << "Type Substitution: " << *env << std::endl; )
			assert( arg->has_result() );
			Type * result = arg->get_result();
			if ( skipCopyConstruct( result ) ) return; // skip certain non-copyable types

			// type may involve type variables, so apply type substitution to get temporary variable's actual type
			result = result->clone();
			env->apply( result );
			ObjectDecl * tmp = ObjectDecl::newObject( "__tmp", result, nullptr );
			tmp->get_type()->set_const( false );

			// create and resolve copy constructor
			CP_CTOR_PRINT( std::cerr << "makeCtorDtor for an argument" << std::endl; )
			Expression * cpCtor = makeCtorDtor( "?{}", tmp, arg );

			if ( ApplicationExpr * appExpr = dynamic_cast< ApplicationExpr * >( cpCtor ) ) {
				// if the chosen constructor is intrinsic, the copy is unnecessary, so
				// don't create the temporary and don't call the copy constructor
				VariableExpr * function = dynamic_cast< VariableExpr * >( appExpr->get_function() );
				assert( function );
				if ( function->get_var()->get_linkage() == LinkageSpec::Intrinsic ) return;
			}

			// set a unique name for the temporary once it's certain the call is necessary
			tmp->name = tempNamer.newName();

			// replace argument to function call with temporary
			arg = new CommaExpr( cpCtor, new VariableExpr( tmp ) );
			impCpCtorExpr->get_tempDecls().push_back( tmp );
			impCpCtorExpr->get_dtors().push_front( makeCtorDtor( "^?{}", tmp ) );
		}

		void ResolveCopyCtors::destructRet( ObjectDecl * ret, ImplicitCopyCtorExpr * impCpCtorExpr ) {
			impCpCtorExpr->get_dtors().push_front( makeCtorDtor( "^?{}", ret ) );
		}

		void ResolveCopyCtors::postvisit( ImplicitCopyCtorExpr *impCpCtorExpr ) {
			CP_CTOR_PRINT( std::cerr << "ResolveCopyCtors: " << impCpCtorExpr << std::endl; )

			ApplicationExpr * appExpr = impCpCtorExpr->get_callExpr();

			// take each argument and attempt to copy construct it.
			for ( Expression * & arg : appExpr->get_args() ) {
				copyConstructArg( arg, impCpCtorExpr );
			} // for

			// each return value from the call needs to be connected with an ObjectDecl at the call site, which is
			// initialized with the return value and is destructed later
			// xxx - handle named return values?
			Type * result = appExpr->get_result();
			if ( ! result->isVoid() ) {
				static UniqueName retNamer("_tmp_cp_ret");
				result = result->clone();
				env->apply( result );
				ObjectDecl * ret = ObjectDecl::newObject( retNamer.newName(), result, nullptr );
				ret->get_type()->set_const( false );
				impCpCtorExpr->get_returnDecls().push_back( ret );
				CP_CTOR_PRINT( std::cerr << "makeCtorDtor for a return" << std::endl; )
				if ( ! dynamic_cast< ReferenceType * >( result ) ) {
					// destructing reference returns is bad because it can cause multiple destructor calls to the same object - the returned object is not a temporary
					destructRet( ret, impCpCtorExpr );
				}
			} // for
			CP_CTOR_PRINT( std::cerr << "after Resolving: " << impCpCtorExpr << std::endl; )
		}

		void ResolveCopyCtors::postvisit( StmtExpr * stmtExpr ) {
			assert( env );
			assert( stmtExpr->get_result() );
			Type * result = stmtExpr->get_result();
			if ( ! result->isVoid() ) {
				static UniqueName retNamer("_tmp_stmtexpr_ret");

				// create variable that will hold the result of the stmt expr
				result = result->clone();
				env->apply( result );
				ObjectDecl * ret = ObjectDecl::newObject( retNamer.newName(), result, nullptr );
				ret->get_type()->set_const( false );
				stmtExpr->get_returnDecls().push_front( ret );

				// must have a non-empty body, otherwise it wouldn't have a result
				CompoundStmt * body = stmtExpr->get_statements();
				assert( ! body->get_kids().empty() );
				// must be an ExprStmt, otherwise it wouldn't have a result
				ExprStmt * last = strict_dynamic_cast< ExprStmt * >( body->get_kids().back() );
				last->set_expr( makeCtorDtor( "?{}", ret, last->get_expr() ) );

				stmtExpr->get_dtors().push_front( makeCtorDtor( "^?{}", ret ) );
			} // if
		}

		void ResolveCopyCtors::previsit( UniqueExpr * unqExpr ) {
			unqCount[ unqExpr->get_id() ]++;  // count the number of unique expressions for each ID
			if ( vars.count( unqExpr->get_id() ) ) {
				// xxx - hack to prevent double-handling of unique exprs, otherwise too many temporary variables and destructors are generated
				visit_children = false;
			}
		}

		void ResolveCopyCtors::postvisit( UniqueExpr * unqExpr ) {
			if ( vars.count( unqExpr->get_id() ) ) {
				// xxx - hack to prevent double-handling of unique exprs, otherwise too many temporary variables and destructors are generated
				return;
			}

			// it should never be necessary to wrap a void-returning expression in a UniqueExpr - if this assumption changes, this needs to be rethought
			assert( unqExpr->get_result() );
			if ( ImplicitCopyCtorExpr * impCpCtorExpr = dynamic_cast<ImplicitCopyCtorExpr*>( unqExpr->get_expr() ) ) {
				// note the variable used as the result from the call
				assert( impCpCtorExpr->get_result() && impCpCtorExpr->get_returnDecls().size() == 1 );
				unqExpr->set_var( new VariableExpr( impCpCtorExpr->get_returnDecls().front() ) );
			} else {
				// expr isn't a call expr, so create a new temporary variable to use to hold the value of the unique expression
				unqExpr->set_object( ObjectDecl::newObject( toString("_unq", unqExpr->get_id()), unqExpr->get_result()->clone(), nullptr ) );
				unqExpr->set_var( new VariableExpr( unqExpr->get_object() ) );
			}
			vars.insert( unqExpr->get_id() );
		}

		Expression * FixCopyCtors::postmutate( ImplicitCopyCtorExpr * impCpCtorExpr ) {
			CP_CTOR_PRINT( std::cerr << "FixCopyCtors: " << impCpCtorExpr << std::endl; )

			std::list< ObjectDecl * > & tempDecls = impCpCtorExpr->get_tempDecls();
			std::list< ObjectDecl * > & returnDecls = impCpCtorExpr->get_returnDecls();
			std::list< Expression * > & dtors = impCpCtorExpr->get_dtors();

			// add all temporary declarations and their constructors
			for ( ObjectDecl * obj : tempDecls ) {
				stmtsToAddBefore.push_back( new DeclStmt( noLabels, obj ) );
			} // for
			for ( ObjectDecl * obj : returnDecls ) {
				stmtsToAddBefore.push_back( new DeclStmt( noLabels, obj ) );
			} // for

			// add destructors after current statement
			for ( Expression * dtor : dtors ) {
				stmtsToAddAfter.push_back( new ExprStmt( noLabels, dtor ) );
			} // for

			ObjectDecl * returnDecl = returnDecls.empty() ? nullptr : returnDecls.front();
			Expression * callExpr = impCpCtorExpr->get_callExpr();

			CP_CTOR_PRINT( std::cerr << "Coming out the back..." << impCpCtorExpr << std::endl; )

			// detach fields from wrapper node so that it can be deleted without deleting too much
			dtors.clear();
			tempDecls.clear();
			returnDecls.clear();
			impCpCtorExpr->set_callExpr( nullptr );
			std::swap( impCpCtorExpr->env, callExpr->env );
			assert( impCpCtorExpr->env == nullptr );
			delete impCpCtorExpr;

			if ( returnDecl ) {
				UntypedExpr * assign = new UntypedExpr( new NameExpr( "?=?" ) );
				assign->get_args().push_back( new VariableExpr( returnDecl ) );
				assign->get_args().push_back( callExpr );
				// know the result type of the assignment is the type of the LHS (minus the pointer), so
				// add that onto the assignment expression so that later steps have the necessary information
				assign->set_result( returnDecl->get_type()->clone() );

				Expression * retExpr = new CommaExpr( assign, new VariableExpr( returnDecl ) );
				// move env from callExpr to retExpr
				retExpr->set_env( callExpr->get_env() );
				callExpr->set_env( nullptr );
				return retExpr;
			} else {
				return callExpr;
			} // if
		}

		void FixCopyCtors::premutate( StmtExpr * stmtExpr ) {
			// function call temporaries should be placed at statement-level, rather than nested inside of a new statement expression,
			// since temporaries can be shared across sub-expressions, e.g.
			//   [A, A] f();
			//   g([A] x, [A] y);
			//   g(f());
			// f is executed once, so the return temporary is shared across the tuple constructors for x and y.
			// Explicitly mutating children instead of mutating the inner compound statment forces the temporaries to be added
			// to the outer context, rather than inside of the statement expression.
			visit_children = false;
			std::list< Statement * > & stmts = stmtExpr->get_statements()->get_kids();
			for ( Statement *& stmt : stmts ) {
				stmt = stmt->acceptMutator( *visitor );
			} // for
			assert( stmtExpr->get_result() );
			Type * result = stmtExpr->get_result();
			if ( ! result->isVoid() ) {
				for ( ObjectDecl * obj : stmtExpr->get_returnDecls() ) {
					stmtsToAddBefore.push_back( new DeclStmt( noLabels, obj ) );
				} // for
				// add destructors after current statement
				for ( Expression * dtor : stmtExpr->get_dtors() ) {
					stmtsToAddAfter.push_back( new ExprStmt( noLabels, dtor ) );
				} // for
				// must have a non-empty body, otherwise it wouldn't have a result
				assert( ! stmts.empty() );
				assert( ! stmtExpr->get_returnDecls().empty() );
				stmts.push_back( new ExprStmt( noLabels, new VariableExpr( stmtExpr->get_returnDecls().front() ) ) );
				stmtExpr->get_returnDecls().clear();
				stmtExpr->get_dtors().clear();
			}
			assert( stmtExpr->get_returnDecls().empty() );
			assert( stmtExpr->get_dtors().empty() );
		}

		void FixCopyCtors::premutate( UniqueExpr * unqExpr ) {
			visit_children = false;
			unqCount[ unqExpr->get_id() ]--;
			static std::unordered_map< int, std::list< Statement * > > dtors;
			static std::unordered_map< int, UniqueExpr * > unqMap;
			// has to be done to clean up ImplicitCopyCtorExpr nodes, even when this node was skipped in previous passes
			if ( unqMap.count( unqExpr->get_id() ) ) {
				// take data from other UniqueExpr to ensure consistency
				delete unqExpr->get_expr();
				unqExpr->set_expr( unqMap[unqExpr->get_id()]->get_expr()->clone() );
				delete unqExpr->get_result();
				unqExpr->set_result( maybeClone( unqExpr->get_expr()->get_result() ) );
				if ( unqCount[ unqExpr->get_id() ] == 0 ) {  // insert destructor after the last use of the unique expression
					stmtsToAddAfter.splice( stmtsToAddAfter.end(), dtors[ unqExpr->get_id() ] );
				}
				return;
			}
			PassVisitor<FixCopyCtors> fixer( unqCount );
			unqExpr->set_expr( unqExpr->get_expr()->acceptMutator( fixer ) ); // stmtexprs contained should not be separately fixed, so this must occur after the lookup
			stmtsToAddBefore.splice( stmtsToAddBefore.end(), fixer.pass.stmtsToAddBefore );
			unqMap[unqExpr->get_id()] = unqExpr;
			if ( unqCount[ unqExpr->get_id() ] == 0 ) {  // insert destructor after the last use of the unique expression
				stmtsToAddAfter.splice( stmtsToAddAfter.end(), dtors[ unqExpr->get_id() ] );
			} else { // remember dtors for last instance of unique expr
				dtors[ unqExpr->get_id() ] = fixer.pass.stmtsToAddAfter;
			}
			return;
		}

		DeclarationWithType *FixInit::postmutate( ObjectDecl *objDecl ) {
			// since this removes the init field from objDecl, it must occur after children are mutated (i.e. postmutate)
			if ( ConstructorInit * ctorInit = dynamic_cast< ConstructorInit * >( objDecl->get_init() ) ) {
				// a decision should have been made by the resolver, so ctor and init are not both non-NULL
				assert( ! ctorInit->get_ctor() || ! ctorInit->get_init() );
				if ( Statement * ctor = ctorInit->get_ctor() ) {
					if ( objDecl->get_storageClasses().is_static ) {
						// originally wanted to take advantage of gcc nested functions, but
						// we get memory errors with this approach. To remedy this, the static
						// variable is hoisted when the destructor needs to be called.
						//
						// generate:
						// static T __objName_static_varN;
						// void __objName_dtor_atexitN() {
						//   __dtor__...;
						// }
						// int f(...) {
						//   ...
						//   static bool __objName_uninitialized = true;
						//   if (__objName_uninitialized) {
						//     __ctor(__objName);
						//     __objName_uninitialized = false;
						//     atexit(__objName_dtor_atexitN);
						//   }
						//   ...
						// }

						static UniqueName dtorCallerNamer( "_dtor_atexit" );

						// static bool __objName_uninitialized = true
						BasicType * boolType = new BasicType( Type::Qualifiers(), BasicType::Bool );
						SingleInit * boolInitExpr = new SingleInit( new ConstantExpr( Constant::from_int( 1 ) ) );
						ObjectDecl * isUninitializedVar = new ObjectDecl( objDecl->get_mangleName() + "_uninitialized", Type::StorageClasses( Type::Static ), LinkageSpec::Cforall, 0, boolType, boolInitExpr );
						isUninitializedVar->fixUniqueId();

						// __objName_uninitialized = false;
						UntypedExpr * setTrue = new UntypedExpr( new NameExpr( "?=?" ) );
						setTrue->get_args().push_back( new VariableExpr( isUninitializedVar ) );
						setTrue->get_args().push_back( new ConstantExpr( Constant::from_int( 0 ) ) );

						// generate body of if
						CompoundStmt * initStmts = new CompoundStmt( noLabels );
						std::list< Statement * > & body = initStmts->get_kids();
						body.push_back( ctor );
						body.push_back( new ExprStmt( noLabels, setTrue ) );

						// put it all together
						IfStmt * ifStmt = new IfStmt( noLabels, new VariableExpr( isUninitializedVar ), initStmts, 0 );
						stmtsToAddAfter.push_back( new DeclStmt( noLabels, isUninitializedVar ) );
						stmtsToAddAfter.push_back( ifStmt );

						Statement * dtor = ctorInit->get_dtor();
						objDecl->set_init( nullptr );
						ctorInit->set_ctor( nullptr );
						ctorInit->set_dtor( nullptr );
						if ( dtor ) {
							// if the object has a non-trivial destructor, have to
							// hoist it and the object into the global space and
							// call the destructor function with atexit.

							Statement * dtorStmt = dtor->clone();

							// void __objName_dtor_atexitN(...) {...}
							FunctionDecl * dtorCaller = new FunctionDecl( objDecl->get_mangleName() + dtorCallerNamer.newName(), Type::StorageClasses( Type::Static ), LinkageSpec::C, new FunctionType( Type::Qualifiers(), false ), new CompoundStmt( noLabels ) );
							dtorCaller->fixUniqueId();
							dtorCaller->get_statements()->push_back( dtorStmt );

							// atexit(dtor_atexit);
							UntypedExpr * callAtexit = new UntypedExpr( new NameExpr( "atexit" ) );
							callAtexit->get_args().push_back( new VariableExpr( dtorCaller ) );

							body.push_back( new ExprStmt( noLabels, callAtexit ) );

							// hoist variable and dtor caller decls to list of decls that will be added into global scope
							staticDtorDecls.push_back( objDecl );
							staticDtorDecls.push_back( dtorCaller );

							// need to rename object uniquely since it now appears
							// at global scope and there could be multiple function-scoped
							// static variables with the same name in different functions.
							// Note: it isn't sufficient to modify only the mangleName, because
							// then subsequent Indexer passes can choke on seeing the object's name
							// if another object has the same name and type. An unfortunate side-effect
							// of renaming the object is that subsequent NameExprs may fail to resolve,
							// but there shouldn't be any remaining past this point.
							static UniqueName staticNamer( "_static_var" );
							objDecl->set_name( objDecl->get_name() + staticNamer.newName() );
							objDecl->set_mangleName( SymTab::Mangler::mangle( objDecl ) );

							// xxx - temporary hack: need to return a declaration, but want to hoist the current object out of this scope
							// create a new object which is never used
							static UniqueName dummyNamer( "_dummy" );
							ObjectDecl * dummy = new ObjectDecl( dummyNamer.newName(), Type::StorageClasses( Type::Static ), LinkageSpec::Cforall, 0, new PointerType( Type::Qualifiers(), new VoidType( Type::Qualifiers() ) ), 0, std::list< Attribute * >{ new Attribute("unused") } );
							delete ctorInit;
							return dummy;
						}
					} else {
						ImplicitCtorDtorStmt * implicit = strict_dynamic_cast< ImplicitCtorDtorStmt * > ( ctor );
						ExprStmt * ctorStmt = dynamic_cast< ExprStmt * >( implicit->get_callStmt() );
						ApplicationExpr * ctorCall = nullptr;
						if ( ctorStmt && (ctorCall = isIntrinsicCallExpr( ctorStmt->get_expr() )) && ctorCall->get_args().size() == 2 ) {
							// clean up intrinsic copy constructor calls by making them into SingleInits
							objDecl->set_init( new SingleInit( ctorCall->get_args().back() ) );
							ctorCall->get_args().pop_back();
						} else {
							stmtsToAddAfter.push_back( ctor );
							objDecl->set_init( nullptr );
							ctorInit->set_ctor( nullptr );
						}
					} // if
				} else if ( Initializer * init = ctorInit->get_init() ) {
					objDecl->set_init( init );
					ctorInit->set_init( nullptr );
				} else {
					// no constructor and no initializer, which is okay
					objDecl->set_init( nullptr );
				} // if
				delete ctorInit;
			} // if
			return objDecl;
		}

		void ObjDeclCollector::previsit( CompoundStmt * ) {
			GuardValue( curVars );
		}

		void ObjDeclCollector::previsit( DeclStmt * stmt ) {
			// keep track of all variables currently in scope
			if ( ObjectDecl * objDecl = dynamic_cast< ObjectDecl * > ( stmt->get_decl() ) ) {
				curVars.push_back( objDecl );
			} // if
		}

		void LabelFinder::previsit( Statement * stmt ) {
			// for each label, remember the variables in scope at that label.
			for ( Label l : stmt->get_labels() ) {
				vars[l] = curVars;
			} // for
		}

		void LabelFinder::previsit( CompoundStmt * stmt ) {
			previsit( (Statement *)stmt );
			Parent::previsit( stmt );
		}

		void LabelFinder::previsit( DeclStmt * stmt ) {
			previsit( (Statement *)stmt );
			Parent::previsit( stmt );
		}


		template<typename Iterator, typename OutputIterator>
		void insertDtors( Iterator begin, Iterator end, OutputIterator out ) {
			for ( Iterator it = begin ; it != end ; ++it ) {
				// extract destructor statement from the object decl and insert it into the output. Note that this is
				// only called on lists of non-static objects with implicit non-intrinsic dtors, so if the user manually
				// calls an intrinsic dtor then the call must (and will) still be generated since the argument may
				// contain side effects.
				ObjectDecl * objDecl = *it;
				ConstructorInit * ctorInit = dynamic_cast< ConstructorInit * >( objDecl->get_init() );
				assert( ctorInit && ctorInit->get_dtor() );
				*out++ = ctorInit->get_dtor()->clone();
			} // for
		}

		void InsertDtors::previsit( ObjectDecl * objDecl ) {
			// remember non-static destructed objects so that their destructors can be inserted later
			if ( ! objDecl->get_storageClasses().is_static ) {
				if ( ConstructorInit * ctorInit = dynamic_cast< ConstructorInit * >( objDecl->get_init() ) ) {
					// a decision should have been made by the resolver, so ctor and init are not both non-NULL
					assert( ! ctorInit->get_ctor() || ! ctorInit->get_init() );
					Statement * dtor = ctorInit->get_dtor();
					// don't need to call intrinsic dtor, because it does nothing, but
					// non-intrinsic dtors must be called
					if ( dtor && ! isIntrinsicSingleArgCallStmt( dtor ) ) {
						// set dtor location to the object's location for error messages
						ctorInit->dtor->location = objDecl->location;
						reverseDeclOrder.front().push_front( objDecl );
					} // if
				} // if
			} // if
		}

		void InsertDtors::previsit( FunctionDecl * funcDecl ) {
			// each function needs to have its own set of labels
			GuardValue( labelVars );
			labelVars.clear();
			maybeAccept( funcDecl->type, finder );
			maybeAccept( funcDecl->statements, finder );

			// all labels for this function have been collected, insert destructors as appropriate via implicit recursion.
		}

		void InsertDtors::previsit( CompoundStmt * compoundStmt ) {
			// visit statements - this will also populate reverseDeclOrder list.  don't want to dump all destructors
			// when block is left, just the destructors associated with variables defined in this block, so push a new
			// list to the top of the stack so that we can differentiate scopes
			reverseDeclOrder.push_front( OrderedDecls() );
			Parent::previsit( compoundStmt );
		}

		void InsertDtors::postvisit( CompoundStmt * compoundStmt ) {
			// add destructors for the current scope that we're exiting, unless the last statement is a return, which
			// causes unreachable code warnings
			std::list< Statement * > & statements = compoundStmt->get_kids();
			if ( ! statements.empty() && ! dynamic_cast< ReturnStmt * >( statements.back() ) ) {
				insertDtors( reverseDeclOrder.front().begin(), reverseDeclOrder.front().end(), back_inserter( statements ) );
			}
			reverseDeclOrder.pop_front();
		}

		void InsertDtors::previsit( ReturnStmt * ) {
			// return exits all scopes, so dump destructors for all scopes
			for ( OrderedDecls & od : reverseDeclOrder ) {
				insertDtors( od.begin(), od.end(), back_inserter( stmtsToAddBefore ) );
			} // for
		}

		// Handle break/continue/goto in the same manner as C++.  Basic idea: any objects that are in scope at the
		// BranchStmt but not at the labelled (target) statement must be destructed.  If there are any objects in scope
		// at the target location but not at the BranchStmt then those objects would be uninitialized so notify the user
		// of the error.  See C++ Reference 6.6 Jump Statements for details.
		void InsertDtors::handleGoto( BranchStmt * stmt ) {
			// can't do anything for computed goto
			if ( stmt->computedTarget ) return;

			assertf( stmt->get_target() != "", "BranchStmt missing a label: %s", toString( stmt ).c_str() );
			// S_L = lvars = set of objects in scope at label definition
			// S_G = curVars = set of objects in scope at goto statement
			ObjectSet & lvars = labelVars[ stmt->get_target() ];

			DTOR_PRINT(
				std::cerr << "at goto label: " << stmt->get_target().get_name() << std::endl;
				std::cerr << "S_G = " << printSet( curVars ) << std::endl;
				std::cerr << "S_L = " << printSet( lvars ) << std::endl;
			)

			ObjectSet diff;
			// S_L-S_G results in set of objects whose construction is skipped - it's an error if this set is non-empty
			std::set_difference( lvars.begin(), lvars.end(), curVars.begin(), curVars.end(), std::inserter( diff, diff.begin() ) );
			DTOR_PRINT(
				std::cerr << "S_L-S_G = " << printSet( diff ) << std::endl;
			)
			if ( ! diff.empty() ) {
				throw SemanticError( std::string("jump to label '") + stmt->get_target().get_name() + "' crosses initialization of " + (*diff.begin())->get_name() + " ", stmt );
			} // if
			// S_G-S_L results in set of objects that must be destructed
			diff.clear();
			std::set_difference( curVars.begin(), curVars.end(), lvars.begin(), lvars.end(), std::inserter( diff, diff.end() ) );
			DTOR_PRINT(
				std::cerr << "S_G-S_L = " << printSet( diff ) << std::endl;
			)
			if ( ! diff.empty() ) {
				// create an auxilliary set for fast lookup -- can't make diff a set, because diff ordering should be consistent for error messages.
				std::unordered_set<ObjectDecl *> needsDestructor( diff.begin(), diff.end() );

				// go through decl ordered list of objectdecl. for each element that occurs in diff, output destructor
				OrderedDecls ordered;
				for ( OrderedDecls & rdo : reverseDeclOrder ) {
					// add elements from reverseDeclOrder into ordered if they occur in diff - it is key that this happens in reverse declaration order.
					copy_if( rdo.begin(), rdo.end(), back_inserter( ordered ), [&]( ObjectDecl * objDecl ) { return needsDestructor.count( objDecl ); } );
				} // for
				insertDtors( ordered.begin(), ordered.end(), back_inserter( stmtsToAddBefore ) );
			} // if
		}

		void InsertDtors::previsit( BranchStmt * stmt ) {
			switch( stmt->get_type() ) {
			  case BranchStmt::Continue:
			  case BranchStmt::Break:
				// could optimize the break/continue case, because the S_L-S_G check is unnecessary (this set should
				// always be empty), but it serves as a small sanity check.
			  case BranchStmt::Goto:
				handleGoto( stmt );
				break;
			  default:
				assert( false );
			} // switch
		}

		bool checkWarnings( FunctionDecl * funcDecl ) {
			// only check for warnings if the current function is a user-defined
			// constructor or destructor
			if ( ! funcDecl ) return false;
			if ( ! funcDecl->get_statements() ) return false;
			return CodeGen::isCtorDtor( funcDecl->get_name() ) && ! LinkageSpec::isOverridable( funcDecl->get_linkage() );
		}

		void GenStructMemberCalls::previsit( FunctionDecl * funcDecl ) {
			GuardValue( funcDecl );
			GuardValue( unhandled );
			GuardValue( usedUninit );
			GuardValue( thisParam );
			GuardValue( isCtor );
			GuardValue( structDecl );
			errors = SemanticError();  // clear previous errors

			// need to start with fresh sets
			unhandled.clear();
			usedUninit.clear();

			function = funcDecl;
			isCtor = CodeGen::isConstructor( function->get_name() );
			if ( checkWarnings( function ) ) {
				FunctionType * type = function->get_functionType();
				assert( ! type->get_parameters().empty() );
				thisParam = strict_dynamic_cast< ObjectDecl * >( type->get_parameters().front() );
				Type * thisType = getPointerBase( thisParam->get_type() );
				StructInstType * structType = dynamic_cast< StructInstType * >( thisType );
				if ( structType ) {
					structDecl = structType->get_baseStruct();
					for ( Declaration * member : structDecl->get_members() ) {
						if ( ObjectDecl * field = dynamic_cast< ObjectDecl * >( member ) ) {
							// record all of the struct type's members that need to be constructed or
							// destructed by the end of the function
							unhandled.insert( field );
						}
					}
				}
			}
		}

		void addIds( SymTab::Indexer & indexer, const std::list< DeclarationWithType * > & decls ) {
			for ( auto d : decls ) {
				indexer.addId( d );
			}
		}

		void addTypes( SymTab::Indexer & indexer, const std::list< TypeDecl * > & tds ) {
			for ( auto td : tds ) {
				indexer.addType( td );
				addIds( indexer, td->assertions );
			}
		}

		void GenStructMemberCalls::postvisit( FunctionDecl * funcDecl ) {
			// remove the unhandled objects from usedUninit, because a call is inserted
			// to handle them - only objects that are later constructed are used uninitialized.
			std::map< DeclarationWithType *, CodeLocation > diff;
			// need the comparator since usedUninit and unhandled have different types
			struct comp_t {
				typedef decltype(usedUninit)::value_type usedUninit_t;
				typedef decltype(unhandled)::value_type unhandled_t;
				bool operator()(usedUninit_t x, unhandled_t y) { return x.first < y; }
				bool operator()(unhandled_t x, usedUninit_t y) { return x < y.first; }
			} comp;
			std::set_difference( usedUninit.begin(), usedUninit.end(), unhandled.begin(), unhandled.end(), std::inserter( diff, diff.begin() ), comp );
			for ( auto p : diff ) {
				DeclarationWithType * member = p.first;
				CodeLocation loc = p.second;
				// xxx - make error message better by also tracking the location that the object is constructed at?
				emit( loc, "in ", CodeGen::genPrettyType( function->get_functionType(), function->get_name() ), ", field ", member->get_name(), " used before being constructed" );
			}

			if ( ! unhandled.empty() ) {
				// need to explicitly re-add function parameters to the indexer in order to resolve copy constructors
				auto guard = makeFuncGuard( [this]() { indexer.enterScope(); }, [this]() { indexer.leaveScope(); } );
				addTypes( indexer, function->type->forall );
				addIds( indexer, function->type->returnVals );
				addIds( indexer, function->type->parameters );

				// need to iterate through members in reverse in order for
				// ctor/dtor statements to come out in the right order
				for ( Declaration * member : reverseIterate( structDecl->get_members() ) ) {
					DeclarationWithType * field = dynamic_cast< DeclarationWithType * >( member );
					// skip non-DWT members
					if ( ! field ) continue;
					// skip non-constructable members
					if ( ! tryConstruct( field ) ) continue;
					// skip handled members
					if ( ! unhandled.count( field ) ) continue;

					// insert and resolve default/copy constructor call for each field that's unhandled
					std::list< Statement * > stmt;
					Expression * arg2 = 0;
					if ( isCopyConstructor( function ) ) {
						// if copy ctor, need to pass second-param-of-this-function.field
						std::list< DeclarationWithType * > & params = function->get_functionType()->get_parameters();
						assert( params.size() == 2 );
						arg2 = new MemberExpr( field, new VariableExpr( params.back() ) );
					}
					InitExpander srcParam( arg2 );
					// cast away reference type and construct field.
					Expression * thisExpr = new CastExpr( new VariableExpr( thisParam ), thisParam->get_type()->stripReferences()->clone() );
					Expression * memberDest = new MemberExpr( field, thisExpr );
					SymTab::genImplicitCall( srcParam, memberDest, function->get_name(), back_inserter( stmt ), field, isCtor );

					assert( stmt.size() <= 1 );
					if ( stmt.size() == 1 ) {
						Statement * callStmt = stmt.front();

						MutatingResolver resolver( indexer );
						try {
							callStmt->acceptMutator( resolver );
							if ( isCtor ) {
								function->get_statements()->push_front( callStmt );
							} else {
								// destructor statements should be added at the end
								function->get_statements()->push_back( callStmt );
							}
						} catch ( SemanticError & error ) {
							emit( funcDecl->location, "in ", CodeGen::genPrettyType( function->get_functionType(), function->get_name() ), ", field ", field->get_name(), " not explicitly ", isCtor ? "constructed" : "destructed",  " and no ", isCtor ? "default constructor" : "destructor", " found" );
						}
					}
				}
			}
			if (! errors.isEmpty()) {
				throw errors;
			}
		}

		/// true if expr is effectively just the 'this' parameter
		bool isThisExpression( Expression * expr, DeclarationWithType * thisParam ) {
			// TODO: there are more complicated ways to pass 'this' to a constructor, e.g. &*, *&, etc.
			if ( VariableExpr * varExpr = dynamic_cast< VariableExpr * >( expr ) ) {
				return varExpr->get_var() == thisParam;
			} else if ( CastExpr * castExpr = dynamic_cast< CastExpr * > ( expr ) ) {
				return isThisExpression( castExpr->get_arg(), thisParam );
			}
			return false;
		}

		/// returns a MemberExpr if expr is effectively just member access on the 'this' parameter, else nullptr
		MemberExpr * isThisMemberExpr( Expression * expr, DeclarationWithType * thisParam ) {
			if ( MemberExpr * memberExpr = dynamic_cast< MemberExpr * >( expr ) ) {
				if ( isThisExpression( memberExpr->get_aggregate(), thisParam ) ) {
					return memberExpr;
				}
			} else if ( CastExpr * castExpr = dynamic_cast< CastExpr * >( expr ) ) {
				return isThisMemberExpr( castExpr->get_arg(), thisParam );
			}
			return nullptr;
		}

		void GenStructMemberCalls::previsit( ApplicationExpr * appExpr ) {
			if ( ! checkWarnings( function ) ) {
				visit_children = false;
				return;
			}

			std::string fname = getFunctionName( appExpr );
			if ( fname == function->get_name() ) {
				// call to same kind of function
				Expression * firstParam = appExpr->get_args().front();

				if ( isThisExpression( firstParam, thisParam ) ) {
					// if calling another constructor on thisParam, assume that function handles
					// all members - if it doesn't a warning will appear in that function.
					unhandled.clear();
				} else if ( MemberExpr * memberExpr = isThisMemberExpr( firstParam, thisParam ) ) {
					// if first parameter is a member expression on the this parameter,
					// then remove the member from unhandled set.
					if ( isThisExpression( memberExpr->get_aggregate(), thisParam ) ) {
						unhandled.erase( memberExpr->get_member() );
					}
				}
			}
		}

		void GenStructMemberCalls::previsit( MemberExpr * memberExpr ) {
			if ( ! checkWarnings( function ) || ! isCtor ) {
				visit_children = false;
				return;
			}

			if ( isThisExpression( memberExpr->get_aggregate(), thisParam ) ) {
				if ( unhandled.count( memberExpr->get_member() ) ) {
					// emit a warning because a member was used before it was constructed
					usedUninit.insert( { memberExpr->get_member(), memberExpr->location } );
				}
			}
		}

		template< typename Visitor, typename... Params >
		void error( Visitor & v, CodeLocation loc, const Params &... params ) {
			SemanticError err( toString( params... ) );
			err.set_location( loc );
			v.errors.append( err );
		}

		template< typename... Params >
		void GenStructMemberCalls::emit( CodeLocation loc, const Params &... params ) {
			// toggle warnings vs. errors here.
			// warn( params... );
			error( *this, loc, params... );
		}

		DeclarationWithType * MutatingResolver::mutate( ObjectDecl *objectDecl ) {
			// add object to the indexer assumes that there will be no name collisions
			// in generated code. If this changes, add mutate methods for entities with
			// scope and call {enter,leave}Scope explicitly.
			indexer.addId( objectDecl );
			return objectDecl;
		}

		Expression* MutatingResolver::mutate( UntypedExpr *untypedExpr ) {
			return strict_dynamic_cast< ApplicationExpr * >( ResolvExpr::findVoidExpression( untypedExpr, indexer ) );
		}

		Expression * FixCtorExprs::postmutate( ConstructorExpr * ctorExpr ) {
			static UniqueName tempNamer( "_tmp_ctor_expr" );
			// xxx - is the size check necessary?
			assert( ctorExpr->has_result() && ctorExpr->get_result()->size() == 1 );

			// xxx - ideally we would reuse the temporary generated from the copy constructor passes from within firstArg if it exists and not generate a temporary if it's unnecessary.
			ObjectDecl * tmp = ObjectDecl::newObject( tempNamer.newName(), ctorExpr->get_result()->clone(), nullptr );
			declsToAddBefore.push_back( tmp );

			// xxx - this can be TupleAssignExpr now. Need to properly handle this case.
			ApplicationExpr * callExpr = strict_dynamic_cast< ApplicationExpr * > ( ctorExpr->get_callExpr() );
			TypeSubstitution * env = ctorExpr->get_env();
			ctorExpr->set_callExpr( nullptr );
			ctorExpr->set_env( nullptr );
			delete ctorExpr;

			// build assignment and replace constructor's first argument with new temporary
			Expression *& firstArg = callExpr->get_args().front();
			Expression * assign = new UntypedExpr( new NameExpr( "?=?" ), { new AddressExpr( new VariableExpr( tmp ) ), new AddressExpr( firstArg ) } );
			firstArg = new VariableExpr( tmp );

			// resolve assignment and dispose of new env
			Expression * resolvedAssign = ResolvExpr::findVoidExpression( assign, indexer );
			delete resolvedAssign->env;
			resolvedAssign->env = nullptr;
			delete assign;

			// for constructor expr:
			//   T x;
			//   x{};
			// results in:
			//   T x;
			//   T & tmp;
			//   &tmp = &x, ?{}(tmp), tmp
			CommaExpr * commaExpr = new CommaExpr( resolvedAssign, new CommaExpr( callExpr, new VariableExpr( tmp ) ) );
			commaExpr->set_env( env );
			return commaExpr;
		}
	} // namespace
} // namespace InitTweak

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
