//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Validate.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 21:50:04 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon Aug 28 13:47:23 2017
// Update Count     : 359
//

// The "validate" phase of translation is used to take a syntax tree and convert it into a standard form that aims to be
// as regular in structure as possible.  Some assumptions can be made regarding the state of the tree after this pass is
// complete, including:
//
// - No nested structure or union definitions; any in the input are "hoisted" to the level of the containing struct or
//   union.
//
// - All enumeration constants have type EnumInstType.
//
// - The type "void" never occurs in lists of function parameter or return types.  A function
//   taking no arguments has no argument types.
//
// - No context instances exist; they are all replaced by the set of declarations signified by the context, instantiated
//   by the particular set of type arguments.
//
// - Every declaration is assigned a unique id.
//
// - No typedef declarations or instances exist; the actual type is substituted for each instance.
//
// - Each type, struct, and union definition is followed by an appropriate assignment operator.
//
// - Each use of a struct or union is connected to a complete definition of that struct or union, even if that
//   definition occurs later in the input.

#include "Validate.h"

#include <cassert>                     // for assertf, assert
#include <cstddef>                     // for size_t
#include <list>                        // for list
#include <string>                      // for string
#include <utility>                     // for pair

#include "CodeGen/CodeGenerator.h"     // for genName
#include "CodeGen/OperatorTable.h"     // for isCtorDtor, isCtorDtorAssign
#include "ControlStruct/Mutate.h"      // for ForExprMutator
#include "Common/PassVisitor.h"        // for PassVisitor, WithDeclsToAdd
#include "Common/ScopedMap.h"          // for ScopedMap
#include "Common/SemanticError.h"      // for SemanticError
#include "Common/UniqueName.h"         // for UniqueName
#include "Common/utility.h"            // for operator+, cloneAll, deleteAll
#include "Concurrency/Keywords.h"      // for applyKeywords
#include "FixFunction.h"               // for FixFunction
#include "Indexer.h"                   // for Indexer
#include "InitTweak/GenInit.h"         // for fixReturnStatements
#include "InitTweak/InitTweak.h"       // for isCtorDtorAssign
#include "Parser/LinkageSpec.h"        // for C
#include "ResolvExpr/typeops.h"        // for typesCompatible
#include "ResolvExpr/Resolver.h"       // for findSingleExpression
#include "ResolvExpr/ResolveTypeof.h"  // for resolveTypeof
#include "SymTab/Autogen.h"            // for SizeType
#include "SynTree/Attribute.h"         // for noAttributes, Attribute
#include "SynTree/Constant.h"          // for Constant
#include "SynTree/Declaration.h"       // for ObjectDecl, DeclarationWithType
#include "SynTree/Expression.h"        // for CompoundLiteralExpr, Expressio...
#include "SynTree/Initializer.h"       // for ListInit, Initializer
#include "SynTree/Label.h"             // for operator==, Label
#include "SynTree/Mutator.h"           // for Mutator
#include "SynTree/Type.h"              // for Type, TypeInstType, EnumInstType
#include "SynTree/TypeSubstitution.h"  // for TypeSubstitution
#include "SynTree/Visitor.h"           // for Visitor
#include "Validate/HandleAttributes.h" // for handleAttributes
#include "Validate/FindSpecialDecls.h" // for FindSpecialDecls

class CompoundStmt;
class ReturnStmt;
class SwitchStmt;

#define debugPrint( x ) if ( doDebug ) x

namespace SymTab {
	/// hoists declarations that are difficult to hoist while parsing
	struct HoistTypeDecls final : public WithDeclsToAdd {
		void previsit( SizeofExpr * );
		void previsit( AlignofExpr * );
		void previsit( UntypedOffsetofExpr * );
		void previsit( CompoundLiteralExpr * );
		void handleType( Type * );
	};

	struct FixQualifiedTypes final : public WithIndexer {
		Type * postmutate( QualifiedType * );
	};

	struct HoistStruct final : public WithDeclsToAdd, public WithGuards {
		/// Flattens nested struct types
		static void hoistStruct( std::list< Declaration * > &translationUnit );

		void previsit( StructDecl * aggregateDecl );
		void previsit( UnionDecl * aggregateDecl );
		void previsit( StaticAssertDecl * assertDecl );
		void previsit( StructInstType * type );
		void previsit( UnionInstType * type );
		void previsit( EnumInstType * type );

	  private:
		template< typename AggDecl > void handleAggregate( AggDecl *aggregateDecl );

		AggregateDecl * parentAggr = nullptr;
	};

	/// Fix return types so that every function returns exactly one value
	struct ReturnTypeFixer {
		static void fix( std::list< Declaration * > &translationUnit );

		void postvisit( FunctionDecl * functionDecl );
		void postvisit( FunctionType * ftype );
	};

	/// Replaces enum types by int, and function or array types in function parameter and return lists by appropriate pointers.
	struct EnumAndPointerDecay {
		void previsit( EnumDecl *aggregateDecl );
		void previsit( FunctionType *func );
	};

	/// Associates forward declarations of aggregates with their definitions
	struct LinkReferenceToTypes final : public WithIndexer, public WithGuards, public WithVisitorRef<LinkReferenceToTypes>, public WithShortCircuiting {
		LinkReferenceToTypes( const Indexer *indexer );
		void postvisit( TypeInstType *typeInst );

		void postvisit( EnumInstType *enumInst );
		void postvisit( StructInstType *structInst );
		void postvisit( UnionInstType *unionInst );
		void postvisit( TraitInstType *traitInst );
		void previsit( QualifiedType * qualType );
		void postvisit( QualifiedType * qualType );

		void postvisit( EnumDecl *enumDecl );
		void postvisit( StructDecl *structDecl );
		void postvisit( UnionDecl *unionDecl );
		void postvisit( TraitDecl * traitDecl );

		void previsit( StructDecl *structDecl );
		void previsit( UnionDecl *unionDecl );

		void renameGenericParams( std::list< TypeDecl * > & params );

	  private:
		const Indexer *local_indexer;

		typedef std::map< std::string, std::list< EnumInstType * > > ForwardEnumsType;
		typedef std::map< std::string, std::list< StructInstType * > > ForwardStructsType;
		typedef std::map< std::string, std::list< UnionInstType * > > ForwardUnionsType;
		ForwardEnumsType forwardEnums;
		ForwardStructsType forwardStructs;
		ForwardUnionsType forwardUnions;
		/// true if currently in a generic type body, so that type parameter instances can be renamed appropriately
		bool inGeneric = false;
	};

	/// Replaces array and function types in forall lists by appropriate pointer type and assigns each Object and Function declaration a unique ID.
	struct ForallPointerDecay final {
		void previsit( ObjectDecl * object );
		void previsit( FunctionDecl * func );
		void previsit( FunctionType * ftype );
		void previsit( StructDecl * aggrDecl );
		void previsit( UnionDecl * aggrDecl );
	};

	struct ReturnChecker : public WithGuards {
		/// Checks that return statements return nothing if their return type is void
		/// and return something if the return type is non-void.
		static void checkFunctionReturns( std::list< Declaration * > & translationUnit );

		void previsit( FunctionDecl * functionDecl );
		void previsit( ReturnStmt * returnStmt );

		typedef std::list< DeclarationWithType * > ReturnVals;
		ReturnVals returnVals;
	};

	struct ReplaceTypedef final : public WithVisitorRef<ReplaceTypedef>, public WithGuards, public WithShortCircuiting, public WithDeclsToAdd {
		ReplaceTypedef() : scopeLevel( 0 ) {}
		/// Replaces typedefs by forward declarations
		static void replaceTypedef( std::list< Declaration * > &translationUnit );

		void premutate( QualifiedType * );
		Type * postmutate( QualifiedType * qualType );
		Type * postmutate( TypeInstType * aggregateUseType );
		Declaration * postmutate( TypedefDecl * typeDecl );
		void premutate( TypeDecl * typeDecl );
		void premutate( FunctionDecl * funcDecl );
		void premutate( ObjectDecl * objDecl );
		DeclarationWithType * postmutate( ObjectDecl * objDecl );

		void premutate( CastExpr * castExpr );

		void premutate( CompoundStmt * compoundStmt );

		void premutate( StructDecl * structDecl );
		void premutate( UnionDecl * unionDecl );
		void premutate( EnumDecl * enumDecl );
		void premutate( TraitDecl * );

		void premutate( FunctionType * ftype );

	  private:
		template<typename AggDecl>
		void addImplicitTypedef( AggDecl * aggDecl );
		template< typename AggDecl >
		void handleAggregate( AggDecl * aggr );

		typedef std::unique_ptr<TypedefDecl> TypedefDeclPtr;
		typedef ScopedMap< std::string, std::pair< TypedefDeclPtr, int > > TypedefMap;
		typedef ScopedMap< std::string, TypeDecl * > TypeDeclMap;
		TypedefMap typedefNames;
		TypeDeclMap typedeclNames;
		int scopeLevel;
		bool inFunctionType = false;
	};

	struct EliminateTypedef {
		/// removes TypedefDecls from the AST
		static void eliminateTypedef( std::list< Declaration * > &translationUnit );

		template<typename AggDecl>
		void handleAggregate( AggDecl *aggregateDecl );

		void previsit( StructDecl * aggregateDecl );
		void previsit( UnionDecl * aggregateDecl );
		void previsit( CompoundStmt * compoundStmt );
	};

	struct VerifyCtorDtorAssign {
		/// ensure that constructors, destructors, and assignment have at least one
		/// parameter, the first of which must be a pointer, and that ctor/dtors have no
		/// return values.
		static void verify( std::list< Declaration * > &translationUnit );

		void previsit( FunctionDecl *funcDecl );
	};

	/// ensure that generic types have the correct number of type arguments
	struct ValidateGenericParameters {
		void previsit( StructInstType * inst );
		void previsit( UnionInstType * inst );
	};

	struct FixObjectType : public WithIndexer {
		/// resolves typeof type in object, function, and type declarations
		static void fix( std::list< Declaration * > & translationUnit );

		void previsit( ObjectDecl * );
		void previsit( FunctionDecl * );
		void previsit( TypeDecl * );
	};

	struct ArrayLength : public WithIndexer {
		/// for array types without an explicit length, compute the length and store it so that it
		/// is known to the rest of the phases. For example,
		///   int x[] = { 1, 2, 3 };
		///   int y[][2] = { { 1, 2, 3 }, { 1, 2, 3 } };
		/// here x and y are known at compile-time to have length 3, so change this into
		///   int x[3] = { 1, 2, 3 };
		///   int y[3][2] = { { 1, 2, 3 }, { 1, 2, 3 } };
		static void computeLength( std::list< Declaration * > & translationUnit );

		void previsit( ObjectDecl * objDecl );
		void previsit( ArrayType * arrayType );
	};

	struct CompoundLiteral final : public WithDeclsToAdd, public WithVisitorRef<CompoundLiteral> {
		Type::StorageClasses storageClasses;

		void premutate( ObjectDecl *objectDecl );
		Expression * postmutate( CompoundLiteralExpr *compLitExpr );
	};

	struct LabelAddressFixer final : public WithGuards {
		std::set< Label > labels;

		void premutate( FunctionDecl * funcDecl );
		Expression * postmutate( AddressExpr * addrExpr );
	};

	void validate( std::list< Declaration * > &translationUnit, __attribute__((unused)) bool doDebug ) {
		PassVisitor<EnumAndPointerDecay> epc;
		PassVisitor<LinkReferenceToTypes> lrt( nullptr );
		PassVisitor<ForallPointerDecay> fpd;
		PassVisitor<CompoundLiteral> compoundliteral;
		PassVisitor<ValidateGenericParameters> genericParams;
		PassVisitor<LabelAddressFixer> labelAddrFixer;
		PassVisitor<HoistTypeDecls> hoistDecls;
		PassVisitor<FixQualifiedTypes> fixQual;

		acceptAll( translationUnit, hoistDecls );
		ReplaceTypedef::replaceTypedef( translationUnit );
		ReturnTypeFixer::fix( translationUnit ); // must happen before autogen
		acceptAll( translationUnit, epc ); // must happen before VerifyCtorDtorAssign, because void return objects should not exist; before LinkReferenceToTypes because it is an indexer and needs correct types for mangling
		acceptAll( translationUnit, lrt ); // must happen before autogen, because sized flag needs to propagate to generated functions
		mutateAll( translationUnit, fixQual ); // must happen after LinkReferenceToTypes, because aggregate members are accessed
		HoistStruct::hoistStruct( translationUnit ); // must happen after EliminateTypedef, so that aggregate typedefs occur in the correct order
		EliminateTypedef::eliminateTypedef( translationUnit ); //
		acceptAll( translationUnit, genericParams );  // check as early as possible - can't happen before LinkReferenceToTypes
		VerifyCtorDtorAssign::verify( translationUnit );  // must happen before autogen, because autogen examines existing ctor/dtors
		ReturnChecker::checkFunctionReturns( translationUnit );
		InitTweak::fixReturnStatements( translationUnit ); // must happen before autogen
		Concurrency::applyKeywords( translationUnit );
		acceptAll( translationUnit, fpd ); // must happen before autogenerateRoutines, after Concurrency::applyKeywords because uniqueIds must be set on declaration before resolution
		ControlStruct::hoistControlDecls( translationUnit );  // hoist initialization out of for statements; must happen before autogenerateRoutines
		autogenerateRoutines( translationUnit ); // moved up, used to be below compoundLiteral - currently needs EnumAndPointerDecay
		Concurrency::implementMutexFuncs( translationUnit );
		Concurrency::implementThreadStarter( translationUnit );
		mutateAll( translationUnit, compoundliteral );
		ResolvExpr::resolveWithExprs( translationUnit ); // must happen before FixObjectType because user-code is resolved and may contain with variables
		FixObjectType::fix( translationUnit );
		ArrayLength::computeLength( translationUnit );
		Validate::findSpecialDecls( translationUnit );
		mutateAll( translationUnit, labelAddrFixer );
		Validate::handleAttributes( translationUnit );
	}

	void validateType( Type *type, const Indexer *indexer ) {
		PassVisitor<EnumAndPointerDecay> epc;
		PassVisitor<LinkReferenceToTypes> lrt( indexer );
		PassVisitor<ForallPointerDecay> fpd;
		type->accept( epc );
		type->accept( lrt );
		type->accept( fpd );
	}


	void HoistTypeDecls::handleType( Type * type ) {
		// some type declarations are buried in expressions and not easy to hoist during parsing; hoist them here
		AggregateDecl * aggr = nullptr;
		if ( StructInstType * inst = dynamic_cast< StructInstType * >( type ) ) {
			aggr = inst->baseStruct;
		} else if ( UnionInstType * inst = dynamic_cast< UnionInstType * >( type ) ) {
			aggr = inst->baseUnion;
		} else if ( EnumInstType * inst = dynamic_cast< EnumInstType * >( type ) ) {
			aggr = inst->baseEnum;
		}
		if ( aggr && aggr->body ) {
			declsToAddBefore.push_front( aggr );
		}
	}

	void HoistTypeDecls::previsit( SizeofExpr * expr ) {
		handleType( expr->type );
	}

	void HoistTypeDecls::previsit( AlignofExpr * expr ) {
		handleType( expr->type );
	}

	void HoistTypeDecls::previsit( UntypedOffsetofExpr * expr ) {
		handleType( expr->type );
	}

	void HoistTypeDecls::previsit( CompoundLiteralExpr * expr ) {
		handleType( expr->result );
	}


	Type * FixQualifiedTypes::postmutate( QualifiedType * qualType ) {
		Type * parent = qualType->parent;
		Type * child = qualType->child;
		if ( dynamic_cast< GlobalScopeType * >( qualType->parent ) ) {
			// .T => lookup T at global scope
			if ( TypeInstType * inst = dynamic_cast< TypeInstType * >( child ) ) {
				auto td = indexer.globalLookupType( inst->name );
				if ( ! td ) {
					SemanticError( qualType->location, toString("Use of undefined global type ", inst->name) );
				}
				auto base = td->base;
				assert( base );
				Type * ret = base->clone();
				ret->get_qualifiers() = qualType->get_qualifiers();
				return ret;
			} else {
				// .T => T is not a type name
				assertf( false, "unhandled global qualified child type: %s", toCString(child) );
			}
		} else {
			// S.T => S must be an aggregate type, find the declaration for T in S.
			AggregateDecl * aggr = nullptr;
			if ( StructInstType * inst = dynamic_cast< StructInstType * >( parent ) ) {
				aggr = inst->baseStruct;
			} else if ( UnionInstType * inst = dynamic_cast< UnionInstType * > ( parent ) ) {
				aggr = inst->baseUnion;
			} else {
				SemanticError( qualType->location, toString("Qualified type requires an aggregate on the left, but has: ", parent) );
			}
			assert( aggr ); // TODO: need to handle forward declarations
			for ( Declaration * member : aggr->members ) {
				if ( TypeInstType * inst = dynamic_cast< TypeInstType * >( child ) ) {
					// name on the right is a typedef
					if ( NamedTypeDecl * aggr = dynamic_cast< NamedTypeDecl * > ( member ) ) {
						if ( aggr->name == inst->name ) {
							assert( aggr->base );
							Type * ret = aggr->base->clone();
							ret->get_qualifiers() = qualType->get_qualifiers();
							TypeSubstitution sub = parent->genericSubstitution();
							sub.apply(ret);
							return ret;
						}
					}
				} else {
					// S.T - S is not an aggregate => error
					assertf( false, "unhandled qualified child type: %s", toCString(qualType) );
				}
			}
			// failed to find a satisfying definition of type
			SemanticError( qualType->location, toString("Undefined type in qualified type: ", qualType) );
		}

		// ... may want to link canonical SUE definition to each forward decl so that it becomes easier to lookup?
	}


	void HoistStruct::hoistStruct( std::list< Declaration * > &translationUnit ) {
		PassVisitor<HoistStruct> hoister;
		acceptAll( translationUnit, hoister );
	}

	bool shouldHoist( Declaration *decl ) {
		return dynamic_cast< StructDecl * >( decl ) || dynamic_cast< UnionDecl * >( decl ) || dynamic_cast< StaticAssertDecl * >( decl );
	}

	namespace {
		void qualifiedName( AggregateDecl * aggr, std::ostringstream & ss ) {
			if ( aggr->parent ) qualifiedName( aggr->parent, ss );
			ss << "__" << aggr->name;
		}

		// mangle nested type names using entire parent chain
		std::string qualifiedName( AggregateDecl * aggr ) {
			std::ostringstream ss;
			qualifiedName( aggr, ss );
			return ss.str();
		}
	}

	template< typename AggDecl >
	void HoistStruct::handleAggregate( AggDecl *aggregateDecl ) {
		if ( parentAggr ) {
			aggregateDecl->parent = parentAggr;
			aggregateDecl->name = qualifiedName( aggregateDecl );
			// Add elements in stack order corresponding to nesting structure.
			declsToAddBefore.push_front( aggregateDecl );
		} else {
			GuardValue( parentAggr );
			parentAggr = aggregateDecl;
		} // if
		// Always remove the hoisted aggregate from the inner structure.
		GuardAction( [aggregateDecl]() { filter( aggregateDecl->members, shouldHoist, false ); } );
	}

	void HoistStruct::previsit( StaticAssertDecl * assertDecl ) {
		if ( parentAggr ) {
			declsToAddBefore.push_back( assertDecl );
		}
	}

	void HoistStruct::previsit( StructDecl * aggregateDecl ) {
		handleAggregate( aggregateDecl );
	}

	void HoistStruct::previsit( UnionDecl * aggregateDecl ) {
		handleAggregate( aggregateDecl );
	}

	void HoistStruct::previsit( StructInstType * type ) {
		// need to reset type name after expanding to qualified name
		assert( type->baseStruct );
		type->name = type->baseStruct->name;
	}

	void HoistStruct::previsit( UnionInstType * type ) {
		assert( type->baseUnion );
		type->name = type->baseUnion->name;
	}

	void HoistStruct::previsit( EnumInstType * type ) {
		assert( type->baseEnum );
		type->name = type->baseEnum->name;
	}


	bool isTypedef( Declaration *decl ) {
		return dynamic_cast< TypedefDecl * >( decl );
	}

	void EliminateTypedef::eliminateTypedef( std::list< Declaration * > &translationUnit ) {
		PassVisitor<EliminateTypedef> eliminator;
		acceptAll( translationUnit, eliminator );
		filter( translationUnit, isTypedef, true );
	}

	template< typename AggDecl >
	void EliminateTypedef::handleAggregate( AggDecl *aggregateDecl ) {
		filter( aggregateDecl->members, isTypedef, true );
	}

	void EliminateTypedef::previsit( StructDecl * aggregateDecl ) {
		handleAggregate( aggregateDecl );
	}

	void EliminateTypedef::previsit( UnionDecl * aggregateDecl ) {
		handleAggregate( aggregateDecl );
	}

	void EliminateTypedef::previsit( CompoundStmt * compoundStmt ) {
		// remove and delete decl stmts
		filter( compoundStmt->kids, [](Statement * stmt) {
			if ( DeclStmt *declStmt = dynamic_cast< DeclStmt * >( stmt ) ) {
				if ( dynamic_cast< TypedefDecl * >( declStmt->decl ) ) {
					return true;
				} // if
			} // if
			return false;
		}, true);
	}

	void EnumAndPointerDecay::previsit( EnumDecl *enumDecl ) {
		// Set the type of each member of the enumeration to be EnumConstant
		for ( std::list< Declaration * >::iterator i = enumDecl->members.begin(); i != enumDecl->members.end(); ++i ) {
			ObjectDecl * obj = dynamic_cast< ObjectDecl * >( *i );
			assert( obj );
			obj->set_type( new EnumInstType( Type::Qualifiers( Type::Const ), enumDecl->name ) );
		} // for
	}

	namespace {
		template< typename DWTList >
		void fixFunctionList( DWTList & dwts, bool isVarArgs, FunctionType * func ) {
			auto nvals = dwts.size();
			bool containsVoid = false;
			for ( auto & dwt : dwts ) {
				// fix each DWT and record whether a void was found
				containsVoid |= fixFunction( dwt );
			}

			// the only case in which "void" is valid is where it is the only one in the list
			if ( containsVoid && ( nvals > 1 || isVarArgs ) ) {
				SemanticError( func, "invalid type void in function type " );
			}

			// one void is the only thing in the list; remove it.
			if ( containsVoid ) {
				delete dwts.front();
				dwts.clear();
			}
		}
	}

	void EnumAndPointerDecay::previsit( FunctionType *func ) {
		// Fix up parameters and return types
		fixFunctionList( func->parameters, func->isVarArgs, func );
		fixFunctionList( func->returnVals, false, func );
	}

	LinkReferenceToTypes::LinkReferenceToTypes( const Indexer *other_indexer ) {
		if ( other_indexer ) {
			local_indexer = other_indexer;
		} else {
			local_indexer = &indexer;
		} // if
	}

	void LinkReferenceToTypes::postvisit( EnumInstType *enumInst ) {
		EnumDecl *st = local_indexer->lookupEnum( enumInst->name );
		// it's not a semantic error if the enum is not found, just an implicit forward declaration
		if ( st ) {
			enumInst->baseEnum = st;
		} // if
		if ( ! st || ! st->body ) {
			// use of forward declaration
			forwardEnums[ enumInst->name ].push_back( enumInst );
		} // if
	}

	void checkGenericParameters( ReferenceToType * inst ) {
		for ( Expression * param : inst->parameters ) {
			if ( ! dynamic_cast< TypeExpr * >( param ) ) {
				SemanticError( inst, "Expression parameters for generic types are currently unsupported: " );
			}
		}
	}

	void LinkReferenceToTypes::postvisit( StructInstType *structInst ) {
		StructDecl *st = local_indexer->lookupStruct( structInst->name );
		// it's not a semantic error if the struct is not found, just an implicit forward declaration
		if ( st ) {
			structInst->baseStruct = st;
		} // if
		if ( ! st || ! st->body ) {
			// use of forward declaration
			forwardStructs[ structInst->name ].push_back( structInst );
		} // if
		checkGenericParameters( structInst );
	}

	void LinkReferenceToTypes::postvisit( UnionInstType *unionInst ) {
		UnionDecl *un = local_indexer->lookupUnion( unionInst->name );
		// it's not a semantic error if the union is not found, just an implicit forward declaration
		if ( un ) {
			unionInst->baseUnion = un;
		} // if
		if ( ! un || ! un->body ) {
			// use of forward declaration
			forwardUnions[ unionInst->name ].push_back( unionInst );
		} // if
		checkGenericParameters( unionInst );
	}

	void LinkReferenceToTypes::previsit( QualifiedType * ) {
		visit_children = false;
	}

	void LinkReferenceToTypes::postvisit( QualifiedType * qualType ) {
		// linking only makes sense for the 'oldest ancestor' of the qualified type
		qualType->parent->accept( *visitor );
	}

	template< typename Decl >
	void normalizeAssertions( std::list< Decl * > & assertions ) {
		// ensure no duplicate trait members after the clone
		auto pred = [](Decl * d1, Decl * d2) {
			// only care if they're equal
			DeclarationWithType * dwt1 = dynamic_cast<DeclarationWithType *>( d1 );
			DeclarationWithType * dwt2 = dynamic_cast<DeclarationWithType *>( d2 );
			if ( dwt1 && dwt2 ) {
				if ( dwt1->name == dwt2->name && ResolvExpr::typesCompatible( dwt1->get_type(), dwt2->get_type(), SymTab::Indexer() ) ) {
					// std::cerr << "=========== equal:" << std::endl;
					// std::cerr << "d1: " << d1 << std::endl;
					// std::cerr << "d2: " << d2 << std::endl;
					return false;
				}
			}
			return d1 < d2;
		};
		std::set<Decl *, decltype(pred)> unique_members( assertions.begin(), assertions.end(), pred );
		// if ( unique_members.size() != assertions.size() ) {
		// 	std::cerr << "============different" << std::endl;
		// 	std::cerr << unique_members.size() << " " << assertions.size() << std::endl;
		// }

		std::list< Decl * > order;
		order.splice( order.end(), assertions );
		std::copy_if( order.begin(), order.end(), back_inserter( assertions ), [&]( Decl * decl ) {
			return unique_members.count( decl );
		});
	}

	// expand assertions from trait instance, performing the appropriate type variable substitutions
	template< typename Iterator >
	void expandAssertions( TraitInstType * inst, Iterator out ) {
		assertf( inst->baseTrait, "Trait instance not linked to base trait: %s", toCString( inst ) );
		std::list< DeclarationWithType * > asserts;
		for ( Declaration * decl : inst->baseTrait->members ) {
			asserts.push_back( strict_dynamic_cast<DeclarationWithType *>( decl->clone() ) );
		}
		// substitute trait decl parameters for instance parameters
		applySubstitution( inst->baseTrait->parameters.begin(), inst->baseTrait->parameters.end(), inst->parameters.begin(), asserts.begin(), asserts.end(), out );
	}

	void LinkReferenceToTypes::postvisit( TraitDecl * traitDecl ) {
		if ( traitDecl->name == "sized" ) {
			// "sized" is a special trait - flick the sized status on for the type variable
			assertf( traitDecl->parameters.size() == 1, "Built-in trait 'sized' has incorrect number of parameters: %zd", traitDecl->parameters.size() );
			TypeDecl * td = traitDecl->parameters.front();
			td->set_sized( true );
		}

		// move assertions from type parameters into the body of the trait
		for ( TypeDecl * td : traitDecl->parameters ) {
			for ( DeclarationWithType * assert : td->assertions ) {
				if ( TraitInstType * inst = dynamic_cast< TraitInstType * >( assert->get_type() ) ) {
					expandAssertions( inst, back_inserter( traitDecl->members ) );
				} else {
					traitDecl->members.push_back( assert->clone() );
				}
			}
			deleteAll( td->assertions );
			td->assertions.clear();
		} // for
	}

	void LinkReferenceToTypes::postvisit( TraitInstType * traitInst ) {
		// handle other traits
		TraitDecl *traitDecl = local_indexer->lookupTrait( traitInst->name );
		if ( ! traitDecl ) {
			SemanticError( traitInst->location, "use of undeclared trait " + traitInst->name );
		} // if
		if ( traitDecl->parameters.size() != traitInst->parameters.size() ) {
			SemanticError( traitInst, "incorrect number of trait parameters: " );
		} // if
		traitInst->baseTrait = traitDecl;

		// need to carry over the 'sized' status of each decl in the instance
		for ( auto p : group_iterate( traitDecl->parameters, traitInst->parameters ) ) {
			TypeExpr * expr = dynamic_cast< TypeExpr * >( std::get<1>(p) );
			if ( ! expr ) {
				SemanticError( std::get<1>(p), "Expression parameters for trait instances are currently unsupported: " );
			}
			if ( TypeInstType * inst = dynamic_cast< TypeInstType * >( expr->get_type() ) ) {
				TypeDecl * formalDecl = std::get<0>(p);
				TypeDecl * instDecl = inst->baseType;
				if ( formalDecl->get_sized() ) instDecl->set_sized( true );
			}
		}
		// normalizeAssertions( traitInst->members );
	}

	void LinkReferenceToTypes::postvisit( EnumDecl *enumDecl ) {
		// visit enum members first so that the types of self-referencing members are updated properly
		if ( enumDecl->body ) {
			ForwardEnumsType::iterator fwds = forwardEnums.find( enumDecl->name );
			if ( fwds != forwardEnums.end() ) {
				for ( std::list< EnumInstType * >::iterator inst = fwds->second.begin(); inst != fwds->second.end(); ++inst ) {
					(*inst)->baseEnum = enumDecl;
				} // for
				forwardEnums.erase( fwds );
			} // if

			for ( Declaration * member : enumDecl->members ) {
				ObjectDecl * field = strict_dynamic_cast<ObjectDecl *>( member );
				if ( field->init ) {
					// need to resolve enumerator initializers early so that other passes that determine if an expression is constexpr have the appropriate information.
					SingleInit * init = strict_dynamic_cast<SingleInit *>( field->init );
					ResolvExpr::findSingleExpression( init->value, new BasicType( Type::Qualifiers(), BasicType::SignedInt ), indexer );
				}
			}
		} // if
	}

	void LinkReferenceToTypes::renameGenericParams( std::list< TypeDecl * > & params ) {
		// rename generic type parameters uniquely so that they do not conflict with user-defined function forall parameters, e.g.
		//   forall(otype T)
		//   struct Box {
		//     T x;
		//   };
		//   forall(otype T)
		//   void f(Box(T) b) {
		//     ...
		//   }
		// The T in Box and the T in f are different, so internally the naming must reflect that.
		GuardValue( inGeneric );
		inGeneric = ! params.empty();
		for ( TypeDecl * td : params ) {
			td->name = "__" + td->name + "_generic_";
		}
	}

	void LinkReferenceToTypes::previsit( StructDecl * structDecl ) {
		renameGenericParams( structDecl->parameters );
	}

	void LinkReferenceToTypes::previsit( UnionDecl * unionDecl ) {
		renameGenericParams( unionDecl->parameters );
	}

	void LinkReferenceToTypes::postvisit( StructDecl *structDecl ) {
		// visit struct members first so that the types of self-referencing members are updated properly
		// xxx - need to ensure that type parameters match up between forward declarations and definition (most importantly, number of type parameters and their defaults)
		if ( structDecl->body ) {
			ForwardStructsType::iterator fwds = forwardStructs.find( structDecl->name );
			if ( fwds != forwardStructs.end() ) {
				for ( std::list< StructInstType * >::iterator inst = fwds->second.begin(); inst != fwds->second.end(); ++inst ) {
					(*inst)->baseStruct = structDecl;
				} // for
				forwardStructs.erase( fwds );
			} // if
		} // if
	}

	void LinkReferenceToTypes::postvisit( UnionDecl *unionDecl ) {
		if ( unionDecl->body ) {
			ForwardUnionsType::iterator fwds = forwardUnions.find( unionDecl->name );
			if ( fwds != forwardUnions.end() ) {
				for ( std::list< UnionInstType * >::iterator inst = fwds->second.begin(); inst != fwds->second.end(); ++inst ) {
					(*inst)->baseUnion = unionDecl;
				} // for
				forwardUnions.erase( fwds );
			} // if
		} // if
	}

	void LinkReferenceToTypes::postvisit( TypeInstType *typeInst ) {
		// ensure generic parameter instances are renamed like the base type
		if ( inGeneric && typeInst->baseType ) typeInst->name = typeInst->baseType->name;
		if ( NamedTypeDecl *namedTypeDecl = local_indexer->lookupType( typeInst->name ) ) {
			if ( TypeDecl *typeDecl = dynamic_cast< TypeDecl * >( namedTypeDecl ) ) {
				typeInst->set_isFtype( typeDecl->get_kind() == TypeDecl::Ftype );
			} // if
		} // if
	}

	/// Fix up assertions - flattens assertion lists, removing all trait instances
	void forallFixer( std::list< TypeDecl * > & forall, BaseSyntaxNode * node ) {
		for ( TypeDecl * type : forall ) {
			std::list< DeclarationWithType * > asserts;
			asserts.splice( asserts.end(), type->assertions );
			// expand trait instances into their members
			for ( DeclarationWithType * assertion : asserts ) {
				if ( TraitInstType *traitInst = dynamic_cast< TraitInstType * >( assertion->get_type() ) ) {
					// expand trait instance into all of its members
					expandAssertions( traitInst, back_inserter( type->assertions ) );
					delete traitInst;
				} else {
					// pass other assertions through
					type->assertions.push_back( assertion );
				} // if
			} // for
			// apply FixFunction to every assertion to check for invalid void type
			for ( DeclarationWithType *& assertion : type->assertions ) {
				bool isVoid = fixFunction( assertion );
				if ( isVoid ) {
					SemanticError( node, "invalid type void in assertion of function " );
				} // if
			} // for
			// normalizeAssertions( type->assertions );
		} // for
	}

	void ForallPointerDecay::previsit( ObjectDecl *object ) {
		// ensure that operator names only apply to functions or function pointers
		if ( CodeGen::isOperator( object->name ) && ! dynamic_cast< FunctionType * >( object->type->stripDeclarator() ) ) {
			SemanticError( object->location, toCString( "operator ", object->name.c_str(), " is not a function or function pointer." )  );
		}
		object->fixUniqueId();
	}

	void ForallPointerDecay::previsit( FunctionDecl *func ) {
		func->fixUniqueId();
	}

	void ForallPointerDecay::previsit( FunctionType * ftype ) {
		forallFixer( ftype->forall, ftype );
	}

	void ForallPointerDecay::previsit( StructDecl * aggrDecl ) {
		forallFixer( aggrDecl->parameters, aggrDecl );
	}

	void ForallPointerDecay::previsit( UnionDecl * aggrDecl ) {
		forallFixer( aggrDecl->parameters, aggrDecl );
	}

	void ReturnChecker::checkFunctionReturns( std::list< Declaration * > & translationUnit ) {
		PassVisitor<ReturnChecker> checker;
		acceptAll( translationUnit, checker );
	}

	void ReturnChecker::previsit( FunctionDecl * functionDecl ) {
		GuardValue( returnVals );
		returnVals = functionDecl->get_functionType()->get_returnVals();
	}

	void ReturnChecker::previsit( ReturnStmt * returnStmt ) {
		// Previously this also checked for the existence of an expr paired with no return values on
		// the  function return type. This is incorrect, since you can have an expression attached to
		// a return statement in a void-returning function in C. The expression is treated as if it
		// were cast to void.
		if ( ! returnStmt->get_expr() && returnVals.size() != 0 ) {
			SemanticError( returnStmt, "Non-void function returns no values: " );
		}
	}


	void ReplaceTypedef::replaceTypedef( std::list< Declaration * > &translationUnit ) {
		PassVisitor<ReplaceTypedef> eliminator;
		mutateAll( translationUnit, eliminator );
		if ( eliminator.pass.typedefNames.count( "size_t" ) ) {
			// grab and remember declaration of size_t
			Validate::SizeType = eliminator.pass.typedefNames["size_t"].first->base->clone();
		} else {
			// xxx - missing global typedef for size_t - default to long unsigned int, even though that may be wrong
			// eventually should have a warning for this case.
			Validate::SizeType = new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt );
		}
	}

	void ReplaceTypedef::premutate( QualifiedType * ) {
		visit_children = false;
	}

	Type * ReplaceTypedef::postmutate( QualifiedType * qualType ) {
		// replacing typedefs only makes sense for the 'oldest ancestor' of the qualified type
		qualType->parent = qualType->parent->acceptMutator( *visitor );
		return qualType;
	}

	Type * ReplaceTypedef::postmutate( TypeInstType * typeInst ) {
		// instances of typedef types will come here. If it is an instance
		// of a typdef type, link the instance to its actual type.
		TypedefMap::const_iterator def = typedefNames.find( typeInst->name );
		if ( def != typedefNames.end() ) {
			Type *ret = def->second.first->base->clone();
			ret->location = typeInst->location;
			ret->get_qualifiers() |= typeInst->get_qualifiers();
			// attributes are not carried over from typedef to function parameters/return values
			if ( ! inFunctionType ) {
				ret->attributes.splice( ret->attributes.end(), typeInst->attributes );
			} else {
				deleteAll( ret->attributes );
				ret->attributes.clear();
			}
			// place instance parameters on the typedef'd type
			if ( ! typeInst->parameters.empty() ) {
				ReferenceToType *rtt = dynamic_cast<ReferenceToType*>(ret);
				if ( ! rtt ) {
					SemanticError( typeInst->location, "Cannot apply type parameters to base type of " + typeInst->name );
				}
				rtt->parameters.clear();
				cloneAll( typeInst->parameters, rtt->parameters );
				mutateAll( rtt->parameters, *visitor );  // recursively fix typedefs on parameters
			} // if
			delete typeInst;
			return ret;
		} else {
			TypeDeclMap::const_iterator base = typedeclNames.find( typeInst->name );
			if ( base == typedeclNames.end() ) {
				SemanticError( typeInst->location, toString("Use of undefined type ", typeInst->name) );
			}
			typeInst->set_baseType( base->second );
			return typeInst;
		} // if
		assert( false );
	}

	struct VarLenChecker : WithShortCircuiting {
		void previsit( FunctionType * ) { visit_children = false; }
		void previsit( ArrayType * at ) {
			isVarLen |= at->isVarLen;
		}
		bool isVarLen = false;
	};

	bool isVariableLength( Type * t ) {
		PassVisitor<VarLenChecker> varLenChecker;
		maybeAccept( t, varLenChecker );
		return varLenChecker.pass.isVarLen;
	}

	Declaration * ReplaceTypedef::postmutate( TypedefDecl * tyDecl ) {
		if ( typedefNames.count( tyDecl->name ) == 1 && typedefNames[ tyDecl->name ].second == scopeLevel ) {
			// typedef to the same name from the same scope
			// must be from the same type

			Type * t1 = tyDecl->base;
			Type * t2 = typedefNames[ tyDecl->name ].first->base;
			if ( ! ResolvExpr::typesCompatible( t1, t2, Indexer() ) ) {
				SemanticError( tyDecl->location, "Cannot redefine typedef: " + tyDecl->name );
			}
			// Cannot redefine VLA typedefs. Note: this is slightly incorrect, because our notion of VLAs
			// at this point in the translator is imprecise. In particular, this will disallow redefining typedefs
			// with arrays whose dimension is an enumerator or a cast of a constant/enumerator. The effort required
			// to fix this corner case likely outweighs the utility of allowing it.
			if ( isVariableLength( t1 ) || isVariableLength( t2 ) ) {
				SemanticError( tyDecl->location, "Cannot redefine typedef: " + tyDecl->name );
			}
		} else {
			typedefNames[ tyDecl->name ] = std::make_pair( TypedefDeclPtr( tyDecl ), scopeLevel );
		} // if

		// When a typedef is a forward declaration:
		//    typedef struct screen SCREEN;
		// the declaration portion must be retained:
		//    struct screen;
		// because the expansion of the typedef is:
		//    void rtn( SCREEN *p ) => void rtn( struct screen *p )
		// hence the type-name "screen" must be defined.
		// Note, qualifiers on the typedef are superfluous for the forward declaration.

		Type *designatorType = tyDecl->base->stripDeclarator();
		if ( StructInstType *aggDecl = dynamic_cast< StructInstType * >( designatorType ) ) {
			declsToAddBefore.push_back( new StructDecl( aggDecl->name, DeclarationNode::Struct, noAttributes, tyDecl->linkage ) );
		} else if ( UnionInstType *aggDecl = dynamic_cast< UnionInstType * >( designatorType ) ) {
			declsToAddBefore.push_back( new UnionDecl( aggDecl->name, noAttributes, tyDecl->linkage ) );
		} else if ( EnumInstType *enumDecl = dynamic_cast< EnumInstType * >( designatorType ) ) {
			declsToAddBefore.push_back( new EnumDecl( enumDecl->name, noAttributes, tyDecl->linkage ) );
		} // if
		return tyDecl->clone();
	}

	void ReplaceTypedef::premutate( TypeDecl * typeDecl ) {
		TypedefMap::iterator i = typedefNames.find( typeDecl->name );
		if ( i != typedefNames.end() ) {
			typedefNames.erase( i ) ;
		} // if

		typedeclNames.insert( typeDecl->name, typeDecl );
	}

	void ReplaceTypedef::premutate( FunctionDecl * ) {
		GuardScope( typedefNames );
		GuardScope( typedeclNames );
	}

	void ReplaceTypedef::premutate( ObjectDecl * ) {
		GuardScope( typedefNames );
		GuardScope( typedeclNames );
	}

	DeclarationWithType * ReplaceTypedef::postmutate( ObjectDecl * objDecl ) {
		if ( FunctionType *funtype = dynamic_cast<FunctionType *>( objDecl->type ) ) { // function type?
			// replace the current object declaration with a function declaration
			FunctionDecl * newDecl = new FunctionDecl( objDecl->name, objDecl->get_storageClasses(), objDecl->linkage, funtype, 0, objDecl->attributes, objDecl->get_funcSpec() );
			objDecl->attributes.clear();
			objDecl->set_type( nullptr );
			delete objDecl;
			return newDecl;
		} // if
		return objDecl;
	}

	void ReplaceTypedef::premutate( CastExpr * ) {
		GuardScope( typedefNames );
		GuardScope( typedeclNames );
	}

	void ReplaceTypedef::premutate( CompoundStmt * ) {
		GuardScope( typedefNames );
		GuardScope( typedeclNames );
		scopeLevel += 1;
		GuardAction( [this](){ scopeLevel -= 1; } );
	}

	template<typename AggDecl>
	void ReplaceTypedef::addImplicitTypedef( AggDecl * aggDecl ) {
		if ( typedefNames.count( aggDecl->get_name() ) == 0 ) {
			Type *type = nullptr;
			if ( StructDecl * newDeclStructDecl = dynamic_cast< StructDecl * >( aggDecl ) ) {
				type = new StructInstType( Type::Qualifiers(), newDeclStructDecl->get_name() );
			} else if ( UnionDecl * newDeclUnionDecl = dynamic_cast< UnionDecl * >( aggDecl ) ) {
				type = new UnionInstType( Type::Qualifiers(), newDeclUnionDecl->get_name() );
			} else if ( EnumDecl * newDeclEnumDecl = dynamic_cast< EnumDecl * >( aggDecl )  ) {
				type = new EnumInstType( Type::Qualifiers(), newDeclEnumDecl->get_name() );
			} // if
			TypedefDeclPtr tyDecl( new TypedefDecl( aggDecl->get_name(), aggDecl->location, Type::StorageClasses(), type, aggDecl->get_linkage() ) );
			typedefNames[ aggDecl->get_name() ] = std::make_pair( std::move( tyDecl ), scopeLevel );
			// add the implicit typedef to the AST
			declsToAddBefore.push_back( new TypedefDecl( aggDecl->get_name(), aggDecl->location, Type::StorageClasses(), type->clone(), aggDecl->get_linkage() ) );
		} // if
	}

	template< typename AggDecl >
	void ReplaceTypedef::handleAggregate( AggDecl * aggr ) {
		SemanticErrorException errors;

		ValueGuard< std::list<Declaration * > > oldBeforeDecls( declsToAddBefore );
		ValueGuard< std::list<Declaration * > > oldAfterDecls ( declsToAddAfter  );
		declsToAddBefore.clear();
		declsToAddAfter.clear();

		GuardScope( typedefNames );
		GuardScope( typedeclNames );
		mutateAll( aggr->parameters, *visitor );

		// unroll mutateAll for aggr->members so that implicit typedefs for nested types are added to the aggregate body.
		for ( std::list< Declaration * >::iterator i = aggr->members.begin(); i != aggr->members.end(); ++i ) {
			if ( !declsToAddAfter.empty() ) { aggr->members.splice( i, declsToAddAfter ); }

			try {
				*i = maybeMutate( *i, *visitor );
			} catch ( SemanticErrorException &e ) {
				errors.append( e );
			}

			if ( !declsToAddBefore.empty() ) { aggr->members.splice( i, declsToAddBefore ); }
		}

		if ( !declsToAddAfter.empty() ) { aggr->members.splice( aggr->members.end(), declsToAddAfter ); }
		if ( !errors.isEmpty() ) { throw errors; }
	}

	void ReplaceTypedef::premutate( StructDecl * structDecl ) {
		visit_children = false;
		addImplicitTypedef( structDecl );
		handleAggregate( structDecl );
	}

	void ReplaceTypedef::premutate( UnionDecl * unionDecl ) {
		visit_children = false;
		addImplicitTypedef( unionDecl );
		handleAggregate( unionDecl );
	}

	void ReplaceTypedef::premutate( EnumDecl * enumDecl ) {
		addImplicitTypedef( enumDecl );
	}

	void ReplaceTypedef::premutate( FunctionType * ) {
		GuardValue( inFunctionType );
		inFunctionType = true;
	}

	void ReplaceTypedef::premutate( TraitDecl * ) {
		GuardScope( typedefNames );
		GuardScope( typedeclNames);
	}

	void VerifyCtorDtorAssign::verify( std::list< Declaration * > & translationUnit ) {
		PassVisitor<VerifyCtorDtorAssign> verifier;
		acceptAll( translationUnit, verifier );
	}

	void VerifyCtorDtorAssign::previsit( FunctionDecl * funcDecl ) {
		FunctionType * funcType = funcDecl->get_functionType();
		std::list< DeclarationWithType * > &returnVals = funcType->get_returnVals();
		std::list< DeclarationWithType * > &params = funcType->get_parameters();

		if ( CodeGen::isCtorDtorAssign( funcDecl->get_name() ) ) { // TODO: also check /=, etc.
			if ( params.size() == 0 ) {
				SemanticError( funcDecl, "Constructors, destructors, and assignment functions require at least one parameter " );
			}
			ReferenceType * refType = dynamic_cast< ReferenceType * >( params.front()->get_type() );
			if ( ! refType ) {
				SemanticError( funcDecl, "First parameter of a constructor, destructor, or assignment function must be a reference " );
			}
			if ( CodeGen::isCtorDtor( funcDecl->get_name() ) && returnVals.size() != 0 ) {
				SemanticError( funcDecl, "Constructors and destructors cannot have explicit return values " );
			}
		}
	}

	template< typename Aggr >
	void validateGeneric( Aggr * inst ) {
		std::list< TypeDecl * > * params = inst->get_baseParameters();
		if ( params ) {
			std::list< Expression * > & args = inst->get_parameters();

			// insert defaults arguments when a type argument is missing (currently only supports missing arguments at the end of the list).
			// A substitution is used to ensure that defaults are replaced correctly, e.g.,
			//   forall(otype T, otype alloc = heap_allocator(T)) struct vector;
			//   vector(int) v;
			// after insertion of default values becomes
			//   vector(int, heap_allocator(T))
			// and the substitution is built with T=int so that after substitution, the result is
			//   vector(int, heap_allocator(int))
			TypeSubstitution sub;
			auto paramIter = params->begin();
			for ( size_t i = 0; paramIter != params->end(); ++paramIter, ++i ) {
				if ( i < args.size() ) {
					TypeExpr * expr = strict_dynamic_cast< TypeExpr * >( *std::next( args.begin(), i ) );
					sub.add( (*paramIter)->get_name(), expr->get_type()->clone() );
				} else if ( i == args.size() ) {
					Type * defaultType = (*paramIter)->get_init();
					if ( defaultType ) {
						args.push_back( new TypeExpr( defaultType->clone() ) );
						sub.add( (*paramIter)->get_name(), defaultType->clone() );
					}
				}
			}

			sub.apply( inst );
			if ( args.size() < params->size() ) SemanticError( inst, "Too few type arguments in generic type " );
			if ( args.size() > params->size() ) SemanticError( inst, "Too many type arguments in generic type " );
		}
	}

	void ValidateGenericParameters::previsit( StructInstType * inst ) {
		validateGeneric( inst );
	}

	void ValidateGenericParameters::previsit( UnionInstType * inst ) {
		validateGeneric( inst );
	}

	void CompoundLiteral::premutate( ObjectDecl *objectDecl ) {
		storageClasses = objectDecl->get_storageClasses();
	}

	Expression *CompoundLiteral::postmutate( CompoundLiteralExpr *compLitExpr ) {
		// transform [storage_class] ... (struct S){ 3, ... };
		// into [storage_class] struct S temp =  { 3, ... };
		static UniqueName indexName( "_compLit" );

		ObjectDecl *tempvar = new ObjectDecl( indexName.newName(), storageClasses, LinkageSpec::C, nullptr, compLitExpr->get_result(), compLitExpr->get_initializer() );
		compLitExpr->set_result( nullptr );
		compLitExpr->set_initializer( nullptr );
		delete compLitExpr;
		declsToAddBefore.push_back( tempvar );					// add modified temporary to current block
		return new VariableExpr( tempvar );
	}

	void ReturnTypeFixer::fix( std::list< Declaration * > &translationUnit ) {
		PassVisitor<ReturnTypeFixer> fixer;
		acceptAll( translationUnit, fixer );
	}

	void ReturnTypeFixer::postvisit( FunctionDecl * functionDecl ) {
		FunctionType * ftype = functionDecl->get_functionType();
		std::list< DeclarationWithType * > & retVals = ftype->get_returnVals();
		assertf( retVals.size() == 0 || retVals.size() == 1, "Function %s has too many return values: %zu", functionDecl->get_name().c_str(), retVals.size() );
		if ( retVals.size() == 1 ) {
			// ensure all function return values have a name - use the name of the function to disambiguate (this also provides a nice bit of help for debugging).
			// ensure other return values have a name.
			DeclarationWithType * ret = retVals.front();
			if ( ret->get_name() == "" ) {
				ret->set_name( toString( "_retval_", CodeGen::genName( functionDecl ) ) );
			}
			ret->get_attributes().push_back( new Attribute( "unused" ) );
		}
	}

	void ReturnTypeFixer::postvisit( FunctionType * ftype ) {
		// xxx - need to handle named return values - this information needs to be saved somehow
		// so that resolution has access to the names.
		// Note that this pass needs to happen early so that other passes which look for tuple types
		// find them in all of the right places, including function return types.
		std::list< DeclarationWithType * > & retVals = ftype->get_returnVals();
		if ( retVals.size() > 1 ) {
			// generate a single return parameter which is the tuple of all of the return values
			TupleType * tupleType = strict_dynamic_cast< TupleType * >( ResolvExpr::extractResultType( ftype ) );
			// ensure return value is not destructed by explicitly creating an empty ListInit node wherein maybeConstruct is false.
			ObjectDecl * newRet = new ObjectDecl( "", Type::StorageClasses(), LinkageSpec::Cforall, 0, tupleType, new ListInit( std::list<Initializer*>(), noDesignators, false ) );
			deleteAll( retVals );
			retVals.clear();
			retVals.push_back( newRet );
		}
	}

	void FixObjectType::fix( std::list< Declaration * > & translationUnit ) {
		PassVisitor<FixObjectType> fixer;
		acceptAll( translationUnit, fixer );
	}

	void FixObjectType::previsit( ObjectDecl * objDecl ) {
		Type *new_type = ResolvExpr::resolveTypeof( objDecl->get_type(), indexer );
		new_type->get_qualifiers() -= Type::Lvalue; // even if typeof is lvalue, variable can never have lvalue-qualified type
		objDecl->set_type( new_type );
	}

	void FixObjectType::previsit( FunctionDecl * funcDecl ) {
		Type *new_type = ResolvExpr::resolveTypeof( funcDecl->type, indexer );
		new_type->get_qualifiers() -= Type::Lvalue; // even if typeof is lvalue, variable can never have lvalue-qualified type
		funcDecl->set_type( new_type );
	}

	void FixObjectType::previsit( TypeDecl *typeDecl ) {
		if ( typeDecl->get_base() ) {
			Type *new_type = ResolvExpr::resolveTypeof( typeDecl->get_base(), indexer );
			new_type->get_qualifiers() -= Type::Lvalue; // even if typeof is lvalue, variable can never have lvalue-qualified type
			typeDecl->set_base( new_type );
		} // if
	}

	void ArrayLength::computeLength( std::list< Declaration * > & translationUnit ) {
		PassVisitor<ArrayLength> len;
		acceptAll( translationUnit, len );
	}

	void ArrayLength::previsit( ObjectDecl * objDecl ) {
		if ( ArrayType * at = dynamic_cast< ArrayType * >( objDecl->type ) ) {
			if ( at->dimension ) return;
			if ( ListInit * init = dynamic_cast< ListInit * >( objDecl->init ) ) {
				at->dimension = new ConstantExpr( Constant::from_ulong( init->initializers.size() ) );
			}
		}
	}

	void ArrayLength::previsit( ArrayType * type ) {
		if ( type->dimension ) {
			// need to resolve array dimensions early so that constructor code can correctly determine
			// if a type is a VLA (and hence whether its elements need to be constructed)
			ResolvExpr::findSingleExpression( type->dimension, Validate::SizeType->clone(), indexer );

			// must re-evaluate whether a type is a VLA, now that more information is available
			// (e.g. the dimension may have been an enumerator, which was unknown prior to this step)
			type->isVarLen = ! InitTweak::isConstExpr( type->dimension );
		}
	}

	struct LabelFinder {
		std::set< Label > & labels;
		LabelFinder( std::set< Label > & labels ) : labels( labels ) {}
		void previsit( Statement * stmt ) {
			for ( Label & l : stmt->labels ) {
				labels.insert( l );
			}
		}
	};

	void LabelAddressFixer::premutate( FunctionDecl * funcDecl ) {
		GuardValue( labels );
		PassVisitor<LabelFinder> finder( labels );
		funcDecl->accept( finder );
	}

	Expression * LabelAddressFixer::postmutate( AddressExpr * addrExpr ) {
		// convert &&label into label address
		if ( AddressExpr * inner = dynamic_cast< AddressExpr * >( addrExpr->arg ) ) {
			if ( NameExpr * nameExpr = dynamic_cast< NameExpr * >( inner->arg ) ) {
				if ( labels.count( nameExpr->name ) ) {
					Label name = nameExpr->name;
					delete addrExpr;
					return new LabelAddressExpr( name );
				}
			}
		}
		return addrExpr;
	}
} // namespace SymTab

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
