//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Mutator.h --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Andrew Beach
// Last Modified On : Mon Jul 24 16:31:00 2017
// Update Count     : 16
//
#pragma once

#include <cassert>                 // for assert

#include "Common/SemanticError.h"  // for SemanticError
#include "SynTree/SynTree.h"       // for AST nodes

class Mutator {
  protected:
	Mutator();
	virtual ~Mutator();
  public:
	virtual DeclarationWithType * mutate( ObjectDecl * objectDecl );
	virtual DeclarationWithType * mutate( FunctionDecl * functionDecl );
	virtual Declaration * mutate( StructDecl * aggregateDecl );
	virtual Declaration * mutate( UnionDecl * aggregateDecl );
	virtual Declaration * mutate( EnumDecl * aggregateDecl );
	virtual Declaration * mutate( TraitDecl * aggregateDecl );
	virtual Declaration * mutate( TypeDecl * typeDecl );
	virtual Declaration * mutate( TypedefDecl * typeDecl );
	virtual AsmDecl * mutate( AsmDecl * asmDecl );

	virtual CompoundStmt * mutate( CompoundStmt * compoundStmt );
	virtual Statement * mutate( ExprStmt * exprStmt );
	virtual Statement * mutate( AsmStmt * asmStmt );
	virtual Statement * mutate( IfStmt * ifStmt );
	virtual Statement * mutate( WhileStmt * whileStmt );
	virtual Statement * mutate( ForStmt * forStmt );
	virtual Statement * mutate( SwitchStmt * switchStmt );
	virtual Statement * mutate( CaseStmt * caseStmt );
	virtual Statement * mutate( BranchStmt * branchStmt );
	virtual Statement * mutate( ReturnStmt * returnStmt );
	virtual Statement * mutate( ThrowStmt * throwStmt );
	virtual Statement * mutate( TryStmt * tryStmt );
	virtual Statement * mutate( CatchStmt * catchStmt );
	virtual Statement * mutate( FinallyStmt * catchStmt );
	virtual Statement * mutate( WaitForStmt * waitforStmt );
	virtual Statement * mutate( WithStmt * withStmt );
	virtual NullStmt * mutate( NullStmt * nullStmt );
	virtual Statement * mutate( DeclStmt * declStmt );
	virtual Statement * mutate( ImplicitCtorDtorStmt * impCtorDtorStmt );

	virtual Expression* mutate( ApplicationExpr * applicationExpr );
	virtual Expression* mutate( UntypedExpr * untypedExpr );
	virtual Expression* mutate( NameExpr * nameExpr );
	virtual Expression* mutate( AddressExpr * castExpr );
	virtual Expression* mutate( LabelAddressExpr * labAddressExpr );
	virtual Expression* mutate( CastExpr * castExpr );
	virtual Expression* mutate( VirtualCastExpr * castExpr );
	virtual Expression* mutate( UntypedMemberExpr * memberExpr );
	virtual Expression* mutate( MemberExpr * memberExpr );
	virtual Expression* mutate( VariableExpr * variableExpr );
	virtual Expression* mutate( ConstantExpr * constantExpr );
	virtual Expression* mutate( SizeofExpr * sizeofExpr );
	virtual Expression* mutate( AlignofExpr * alignofExpr );
	virtual Expression* mutate( UntypedOffsetofExpr * offsetofExpr );
	virtual Expression* mutate( OffsetofExpr * offsetofExpr );
	virtual Expression* mutate( OffsetPackExpr * offsetPackExpr );
	virtual Expression* mutate( AttrExpr * attrExpr );
	virtual Expression* mutate( LogicalExpr * logicalExpr );
	virtual Expression* mutate( ConditionalExpr * conditionalExpr );
	virtual Expression* mutate( CommaExpr * commaExpr );
	virtual Expression* mutate( TypeExpr * typeExpr );
	virtual Expression* mutate( AsmExpr * asmExpr );
	virtual Expression* mutate( ImplicitCopyCtorExpr * impCpCtorExpr );
	virtual Expression* mutate( ConstructorExpr * ctorExpr );
	virtual Expression* mutate( CompoundLiteralExpr * compLitExpr );
	virtual Expression* mutate( RangeExpr * rangeExpr );
	virtual Expression* mutate( UntypedTupleExpr * tupleExpr );
	virtual Expression* mutate( TupleExpr * tupleExpr );
	virtual Expression* mutate( TupleIndexExpr * tupleExpr );
	virtual Expression* mutate( TupleAssignExpr * assignExpr );
	virtual Expression* mutate( StmtExpr  * stmtExpr );
	virtual Expression* mutate( UniqueExpr  * uniqueExpr );
	virtual Expression* mutate( UntypedInitExpr  * initExpr );
	virtual Expression* mutate( InitExpr  * initExpr );

	virtual Type * mutate( VoidType * basicType );
	virtual Type * mutate( BasicType * basicType );
	virtual Type * mutate( PointerType * pointerType );
	virtual Type * mutate( ArrayType * arrayType );
	virtual Type * mutate( ReferenceType * refType );
	virtual Type * mutate( FunctionType * functionType );
	virtual Type * mutate( StructInstType * aggregateUseType );
	virtual Type * mutate( UnionInstType * aggregateUseType );
	virtual Type * mutate( EnumInstType * aggregateUseType );
	virtual Type * mutate( TraitInstType * aggregateUseType );
	virtual Type * mutate( TypeInstType * aggregateUseType );
	virtual Type * mutate( TupleType * tupleType );
	virtual Type * mutate( TypeofType * typeofType );
	virtual Type * mutate( AttrType * attrType );
	virtual Type * mutate( VarArgsType * varArgsType );
	virtual Type * mutate( ZeroType * zeroType );
	virtual Type * mutate( OneType * oneType );

	virtual Designation * mutate( Designation * designation );
	virtual Initializer * mutate( SingleInit * singleInit );
	virtual Initializer * mutate( ListInit * listInit );
	virtual Initializer * mutate( ConstructorInit * ctorInit );

	virtual Subrange * mutate( Subrange * subrange );

	virtual Constant * mutate( Constant * constant );

	virtual Attribute * mutate( Attribute * attribute );

	virtual TypeSubstitution * mutate( TypeSubstitution * sub );
  private:
	virtual Declaration * handleAggregateDecl(AggregateDecl * aggregateDecl );
	virtual Declaration * handleNamedTypeDecl(NamedTypeDecl * typeDecl );
	virtual Type * handleReferenceToType(ReferenceToType * aggregateUseType );
};

template< typename TreeType, typename MutatorType >
inline TreeType *maybeMutate( TreeType *tree, MutatorType &mutator ) {
	if ( tree ) {
		TreeType *newnode = dynamic_cast< TreeType * >( tree->acceptMutator( mutator ) );
		assert( newnode );
		return newnode;
	} else {
		return 0;
	} // if
}

template< typename Container, typename MutatorType >
inline void mutateAll( Container &container, MutatorType &mutator ) {
	SemanticError errors;
	for ( typename Container::iterator i = container.begin(); i != container.end(); ++i ) {
		try {
			if ( *i ) {
///		    *i = (*i)->acceptMutator( mutator );
				*i = dynamic_cast< typename Container::value_type >( (*i)->acceptMutator( mutator ) );
				assert( *i );
			} // if
		} catch( SemanticError &e ) {
			errors.append( e );
		} // try
	} // for
	if ( ! errors.isEmpty() ) {
		throw errors;
	} // if
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
