//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// AddressExpr.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 23:54:44 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Tue Apr 26 12:35:13 2016
// Update Count     : 6
//

#include <ostream>           // for ostream, operator<<, basic_ostream, endl
#include <string>            // for operator<<, string

#include "Common/utility.h"  // for maybeClone
#include "Expression.h"      // for AddressExpr, Expression
#include "Type.h"            // for PointerType, Type, Type::Qualifiers

// Address expressions are typed based on the following inference rules:
//    E : lvalue T  &..& (n references)
//   &E :        T *&..& (n references)
//
//    E : T  &..&        (m references)
//   &E : T *&..&        (m-1 references)
//
// That is, lvalues becomes

namespace {
	Type * addrType( Type * type ) {
		if ( ReferenceType * refType = dynamic_cast< ReferenceType * >( type ) ) {
			return new ReferenceType( refType->get_qualifiers(), addrType( refType->get_base() ) );
		} else {
			return new PointerType( Type::Qualifiers(), type->clone() );
		}
	}
}

AddressExpr::AddressExpr( Expression *arg, Expression *_aname ) : Expression( _aname ), arg( arg ) {
	if ( arg->has_result() ) {
		if ( arg->get_result()->get_lvalue() ) {
			// lvalue, retains all layers of reference and gains a pointer inside the references
			set_result( addrType( arg->get_result() ) );
		} else {
			// taking address of non-lvalue -- must be a reference, loses one layer of reference
			ReferenceType * refType = safe_dynamic_cast< ReferenceType * >( arg->get_result() );
			set_result( addrType( refType->get_base() ) );
		}
		// result of & is never an lvalue
		get_result()->set_lvalue( false );
	}
}

AddressExpr::AddressExpr( const AddressExpr &other ) : Expression( other ), arg( maybeClone( other.arg ) ) {
}

AddressExpr::~AddressExpr() {
	delete arg;
}

void AddressExpr::print( std::ostream &os, int indent ) const {
	os << "Address of:" << std::endl;
	if ( arg ) {
		os << std::string( indent+2, ' ' );
		arg->print( os, indent+2 );
	} // if
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
