#include "collector.h"

#ifdef __cforall
extern "C" {
#endif
#include <string.h>
#ifdef __cforall
}
#endif

#include <fstream>

#include "state.h"
#include "gcpointers.h"
#include "memory_pool.h"

void* gc_finish_alloc_block(void* block, size_t actual_size, size_t target_size);
void gc_assign_reference(void** ref, gc_object_header* ptr);

gcpointer_t** gc_find_previous_ref(gcpointer_t* target)
{
	if(!(target)) return NULL;

	bool managed = gc_is_managed(target);
	gc_object_header* obj = gc_get_object_ptr((void*)target->ptr);

	check(is_valid(obj));

	gcpointer_t** prev_next_ptr = managed ? &obj->type_chain : &obj->root_chain;
	while((*prev_next_ptr) && (*prev_next_ptr) != target)
	{
		prev_next_ptr = &(*prev_next_ptr)->next;
	}

	return prev_next_ptr;
}

void* gc_allocate(size_t target_size)
{
	// sout | "Allocating " | target_size | " bytes" | endl;

	size_t size = gc_compute_size(target_size + sizeof(gc_object_header));

	// sout | "Object header size: " | sizeof(gc_object_header) | " bytes" | endl;
	// sout | "Actual allocation size: " | size | " bytes" | endl;

	check(size < POOL_SIZE_BYTES);

	void* block = NULL;
	gc_state* gc = gc_get_state();

	if((intptr_t)(block = gc_try_allocate(gc, size))) return gc_finish_alloc_block(block, size, target_size);

	gc_collect(gc);

	if((intptr_t)(block = gc_try_allocate(gc, size))) return gc_finish_alloc_block(block, size, target_size);

	gc_allocate_pool(gc);

	if((intptr_t)(block = gc_try_allocate(gc, size))) return gc_finish_alloc_block(block, size, target_size);

	checkf( (int) 0, "ERROR: allocation in new pool failed");

	return NULL;
}

void* gc_finish_alloc_block(void* block, size_t actual_size, size_t target_size)
{
	intptr_t data = ((intptr_t)block) + sizeof(gc_object_header);
	void* header = block;

	check( data > ((intptr_t)block));
	check( data >= ((intptr_t)header));
	check( gc_is_aligned( (void*)data ) );
	check( data + target_size <= ((intptr_t)block) + actual_size );

	gc_object_header* obj = placement_ctor(header, actual_size);

	(void)obj; //remove unsused warning since this is for debug
	check(obj == gc_get_object_ptr( (void*)data ));

	gc_register_allocation(gc_get_state(), actual_size);

	return (void*)data;
}

void gc_process_reference(void** ref, worklist_t* worklist)
{
	check(!gc_is_in_heap(gc_get_state(), ref));

	gc_object_header* ptr = gc_get_object_ptr(*ref);
	if(ptr)
	{
		if(!ptr->is_forwarded)
		{
			gc_copy_object(ptr);

			gc_scan_object(ptr->forward, worklist);

			gc_assign_reference(ref, ptr->forward);
		}
		else
		{
			//duplication to help debug
			gc_assign_reference(ref, ptr->forward);
		}
	}
}

void gc_assign_reference(void** ref, gc_object_header* ptr)
{
	void* address = (void*)(((intptr_t)ptr) + sizeof(gc_object_header));

	gc_write_aligned_ptr(ref, address);
}

gc_object_header* gc_copy_object(gc_object_header* ptr)
{
	check(!ptr->forward);
	check(!ptr->is_forwarded);
	check(gc_pool_is_from_space(gc_pool_of(ptr)));

	gc_memory_pool* pool = gc_pool_of(ptr)->mirror;

	void* new_block = gc_pool_allocate(pool, ptr->size, true);

	memcpy(new_block, ptr, ptr->size);

	gc_object_header* fwd_ptr = placement_copy_ctor(new_block, ptr);

	ptr->forward = fwd_ptr;
	ptr->is_forwarded = true;

	return fwd_ptr;
}

void gc_scan_object(gc_object_header* object, worklist_t* worklist)
{
	gcpointer_t* field = object->type_chain;
	while(field)
	{
		check(((intptr_t)field) > ((intptr_t)object));
		check(((intptr_t)field) < ((intptr_t)((intptr_t)object) + object->size));

		check(gc_is_in_to_space(gc_get_state(), &field->ptr));

		intptr_t* ref = &field->ptr;
		push_back(worklist, ref);

		field = field->next;
	}
}
