//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Autogen.h --
//
// Author           : Rob Schluntz
// Created On       : Sun May 17 21:53:34 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Mar 14 23:13:28 2017
// Update Count     : 7
//

#ifndef AUTOGEN_H
#define AUTOGEN_H

#include <string>
#include "SynTree/Statement.h"
#include "SynTree/Expression.h"
#include "SynTree/Declaration.h"
#include "SynTree/Initializer.h"
#include "InitTweak/InitTweak.h"

namespace SymTab {
	/// Generates assignment operators, constructors, and destructor for aggregate types as required
	void autogenerateRoutines( std::list< Declaration * > &translationUnit );

	/// returns true if obj's name is the empty string and it has a bitfield width
	bool isUnnamedBitfield( ObjectDecl * obj );

	/// size_t type - set when size_t typedef is seen. Useful in a few places,
	/// such as in determining array dimension type
	extern Type * SizeType;

	/// inserts into out a generated call expression to function fname with arguments dstParam and srcParam. Intended to be used with generated ?=?, ?{}, and ^?{} calls.
	template< typename OutputIterator >
	Statement * genCall( InitTweak::InitExpander & srcParam, Expression * dstParam, const std::string & fname, OutputIterator out, Type * type, bool addCast = false, bool forward = true );

	/// inserts into out a generated call expression to function fname with arguments dstParam and srcParam. Should only be called with non-array types.
	/// optionally returns a statement which must be inserted prior to the containing loop, if there is one
	template< typename OutputIterator >
	Statement * genScalarCall( InitTweak::InitExpander & srcParam, Expression *dstParam, const std::string & fname, OutputIterator out, Type * type, bool addCast = false ) {
		// want to be able to generate assignment, ctor, and dtor generically,
		// so fname is either ?=?, ?{}, or ^?{}
		UntypedExpr *fExpr = new UntypedExpr( new NameExpr( fname ) );

		// do something special for unnamed members
		dstParam = new AddressExpr( dstParam );
		if ( addCast ) {
			// cast to T* with qualifiers removed, so that qualified objects can be constructed
			// and destructed with the same functions as non-qualified objects.
			// unfortunately, lvalue is considered a qualifier. For AddressExpr to resolve, its argument
			// must have an lvalue qualified type, so remove all qualifiers except lvalue. If we ever
			// remove lvalue as a qualifier, this can change to
			//   type->get_qualifiers() = Type::Qualifiers();
			assert( type );
			Type * castType = type->clone();
//			castType->get_qualifiers() -= Type::Qualifiers(true, true, true, false, true, false);
			castType->get_qualifiers() -= Type::Qualifiers( Type::Const | Type::Volatile | Type::Restrict | Type::Atomic );
			castType->set_isLvalue( true ); // xxx - might not need this
			dstParam = new CastExpr( dstParam, new PointerType( Type::Qualifiers(), castType ) );
		}
		fExpr->get_args().push_back( dstParam );

		Statement * listInit = srcParam.buildListInit( fExpr );

		std::list< Expression * > args = *++srcParam;
		fExpr->get_args().splice( fExpr->get_args().end(), args );

		*out++ = new ExprStmt( noLabels, fExpr );

		srcParam.clearArrayIndices();

		return listInit;
	}

	/// Store in out a loop which calls fname on each element of the array with srcParam and dstParam as arguments.
	/// If forward is true, loop goes from 0 to N-1, else N-1 to 0
	template< typename OutputIterator >
	void genArrayCall( InitTweak::InitExpander & srcParam, Expression *dstParam, const std::string & fname, OutputIterator out, ArrayType *array, bool addCast = false, bool forward = true ) {
		static UniqueName indexName( "_index" );

		// for a flexible array member nothing is done -- user must define own assignment
		if ( ! array->get_dimension() ) return ;

		Expression * begin, * end, * update, * cmp;
		if ( forward ) {
			// generate: for ( int i = 0; i < 0; ++i )
			begin = new ConstantExpr( Constant( new ZeroType( emptyQualifiers ), "0" ) );
			end = array->get_dimension()->clone();
			cmp = new NameExpr( "?<?" );
			update = new NameExpr( "++?" );
		} else {
			// generate: for ( int i = N-1; i >= 0; --i )
			begin = new UntypedExpr( new NameExpr( "?-?" ) );
			((UntypedExpr*)begin)->get_args().push_back( array->get_dimension()->clone() );
			((UntypedExpr*)begin)->get_args().push_back( new ConstantExpr( Constant( new OneType( emptyQualifiers ), "1" ) ) );
			end = new ConstantExpr( Constant( new ZeroType( emptyQualifiers ), "0" ) );
			cmp = new NameExpr( "?>=?" );
			update = new NameExpr( "--?" );
		}

		ObjectDecl *index = new ObjectDecl( indexName.newName(), DeclarationNode::StorageClasses(), LinkageSpec::C, 0, new BasicType( Type::Qualifiers(), BasicType::SignedInt ), new SingleInit( begin, std::list<Expression*>() ) );

		UntypedExpr *cond = new UntypedExpr( cmp );
		cond->get_args().push_back( new VariableExpr( index ) );
		cond->get_args().push_back( end );

		UntypedExpr *inc = new UntypedExpr( update );
		inc->get_args().push_back( new AddressExpr( new VariableExpr( index ) ) );

		UntypedExpr *dstIndex = new UntypedExpr( new NameExpr( "?[?]" ) );
		dstIndex->get_args().push_back( dstParam );
		dstIndex->get_args().push_back( new VariableExpr( index ) );
		dstParam = dstIndex;

		// srcParam must keep track of the array indices to build the
		// source parameter and/or array list initializer
		srcParam.addArrayIndex( new VariableExpr( index ), array->get_dimension()->clone() );

		// for stmt's body, eventually containing call
		CompoundStmt * body = new CompoundStmt( noLabels );
		Statement * listInit = genCall( srcParam, dstParam, fname, back_inserter( body->get_kids() ), array->get_base(), addCast, forward );

		// block containing for stmt and index variable
		std::list<Statement *> initList;
		CompoundStmt * block = new CompoundStmt( noLabels );
		block->get_kids().push_back( new DeclStmt( noLabels, index ) );
		if ( listInit ) block->get_kids().push_back( listInit );
		block->get_kids().push_back( new ForStmt( noLabels, initList, cond, inc, body ) );

		*out++ = block;
	}

	template< typename OutputIterator >
	Statement * genCall( InitTweak::InitExpander &  srcParam, Expression * dstParam, const std::string & fname, OutputIterator out, Type * type, bool addCast, bool forward ) {
		if ( ArrayType * at = dynamic_cast< ArrayType * >( type ) ) {
			genArrayCall( srcParam, dstParam, fname, out, at, addCast, forward );
			return 0;
		} else {
			return genScalarCall( srcParam, dstParam, fname, out, type, addCast );
		}
	}

	/// inserts into out a generated call expression to function fname with arguments dstParam
	/// and srcParam. Intended to be used with generated ?=?, ?{}, and ^?{} calls. decl is the
	/// object being constructed. The function wraps constructor and destructor calls in an
	/// ImplicitCtorDtorStmt node.
	template< typename OutputIterator >
	void genImplicitCall( InitTweak::InitExpander &  srcParam, Expression * dstParam, const std::string & fname, OutputIterator out, DeclarationWithType * decl, bool forward = true ) {
		ObjectDecl *obj = dynamic_cast<ObjectDecl *>( decl );
		assert( obj );
		// unnamed bit fields are not copied as they cannot be accessed
		if ( isUnnamedBitfield( obj ) ) return;

		bool addCast = (fname == "?{}" || fname == "^?{}") && ( !obj || ( obj && obj->get_bitfieldWidth() == NULL ) );
		std::list< Statement * > stmts;
		genCall( srcParam, dstParam, fname, back_inserter( stmts ), obj->get_type(), addCast, forward );

		// currently genCall should produce at most one element, but if that changes then the next line needs to be updated to grab the statement which contains the call
		assert( stmts.size() <= 1 );
		if ( stmts.size() == 1 ) {
			Statement * callStmt = stmts.front();
			if ( addCast ) {
				// implicitly generated ctor/dtor calls should be wrapped
				// so that later passes are aware they were generated.
				// xxx - don't mark as an implicit ctor/dtor if obj is a bitfield,
				// because this causes the address to be taken at codegen, which is illegal in C.
				callStmt = new ImplicitCtorDtorStmt( callStmt );
			}
			*out++ = callStmt;
		}
	}
} // namespace SymTab
#endif // AUTOGEN_H
