//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// FixGlobalInit.cc --
//
// Author           : Rob Schluntz
// Created On       : Mon May 04 15:14:56 2016
// Last Modified By : Rob Schluntz
// Last Modified On : Fri May 13 11:37:30 2016
// Update Count     : 2
//

#include "FixGlobalInit.h"
#include "InitTweak.h"
#include "SynTree/Declaration.h"
#include "SynTree/Type.h"
#include "SynTree/Expression.h"
#include "SynTree/Statement.h"
#include "SynTree/Initializer.h"
#include "SynTree/Visitor.h"
#include <algorithm>

namespace InitTweak {
	namespace {
		const std::list<Label> noLabels;
	}

	class GlobalFixer : public Visitor {
	  public:
		GlobalFixer( const std::string & name, bool inLibrary );

		virtual void visit( ObjectDecl *objDecl );
		virtual void visit( FunctionDecl *functionDecl );
		virtual void visit( StructDecl *aggregateDecl );
		virtual void visit( UnionDecl *aggregateDecl );
		virtual void visit( EnumDecl *aggregateDecl );
		virtual void visit( TraitDecl *aggregateDecl );
		virtual void visit( TypeDecl *typeDecl );

		UniqueName tempNamer;
		FunctionDecl * initFunction;
		FunctionDecl * destroyFunction;
	};

	class ConstExprChecker : public Visitor {
	public:
		ConstExprChecker() : isConstExpr( true ) {}

		virtual void visit( ApplicationExpr *applicationExpr ) { isConstExpr = false; }
		virtual void visit( UntypedExpr *untypedExpr ) { isConstExpr = false; }
		virtual void visit( NameExpr *nameExpr ) { isConstExpr = false; }
		virtual void visit( CastExpr *castExpr ) { isConstExpr = false; }
		virtual void visit( LabelAddressExpr *labAddressExpr ) { isConstExpr = false; }
		virtual void visit( UntypedMemberExpr *memberExpr ) { isConstExpr = false; }
		virtual void visit( MemberExpr *memberExpr ) { isConstExpr = false; }
		virtual void visit( VariableExpr *variableExpr ) { isConstExpr = false; }
		virtual void visit( ConstantExpr *constantExpr ) { /* bottom out */ }
		// these might be okay?
		// virtual void visit( SizeofExpr *sizeofExpr );
		// virtual void visit( AlignofExpr *alignofExpr );
		// virtual void visit( UntypedOffsetofExpr *offsetofExpr );
		// virtual void visit( OffsetofExpr *offsetofExpr );
		// virtual void visit( OffsetPackExpr *offsetPackExpr );
		// virtual void visit( AttrExpr *attrExpr );
		// virtual void visit( CommaExpr *commaExpr );
		// virtual void visit( LogicalExpr *logicalExpr );
		// virtual void visit( ConditionalExpr *conditionalExpr );
		virtual void visit( TupleExpr *tupleExpr ) { isConstExpr = false; }
		virtual void visit( SolvedTupleExpr *tupleExpr ) { isConstExpr = false; }
		virtual void visit( TypeExpr *typeExpr ) { isConstExpr = false; }
		virtual void visit( AsmExpr *asmExpr ) { isConstExpr = false; }
		virtual void visit( UntypedValofExpr *valofExpr ) { isConstExpr = false; }
		virtual void visit( CompoundLiteralExpr *compLitExpr ) { isConstExpr = false; }

		bool isConstExpr;
	};

	bool isConstExpr( Initializer * init ) {
		if ( init ) {
			ConstExprChecker checker;
			init->accept( checker );
			return checker.isConstExpr;
		}
		// for all intents and purposes, no initializer means const expr
		return true;
	}

	void fixGlobalInit( std::list< Declaration * > & translationUnit, const std::string & name, bool inLibrary ) {
		GlobalFixer fixer( name, inLibrary );
		acceptAll( translationUnit, fixer );
		// don't need to include function if it's empty
		if ( fixer.initFunction->get_statements()->get_kids().empty() ) {
			delete fixer.initFunction;
		} else {
			translationUnit.push_back( fixer.initFunction );
		}
		if ( fixer.destroyFunction->get_statements()->get_kids().empty() ) {
			delete fixer.destroyFunction;
		} else {
			translationUnit.push_back( fixer.destroyFunction );
		}
	}

  std::string globalFunctionName( const std::string & name ) {
  	// get basename
  	std::string ret = name.substr( 0, name.find( '.' ) );
  	// replace invalid characters with _
		static std::string invalid = "/-";
  	replace_if( ret.begin(), ret.end(), []( char c ) { return invalid.find(c) != std::string::npos; }, '_' );
  	return ret;
  }

	GlobalFixer::GlobalFixer( const std::string & name, bool inLibrary ) : tempNamer( "_global_init" ) {
		std::string fixedName = globalFunctionName( name );
		initFunction = new FunctionDecl( "_init_" + fixedName, DeclarationNode::Static, LinkageSpec::C, new FunctionType( Type::Qualifiers(), false ), new CompoundStmt( noLabels ), false, false, FunctionDecl::Attribute( FunctionDecl::Attribute::Constructor, inLibrary ? FunctionDecl::Attribute::High : FunctionDecl::Attribute::Default ) );

		destroyFunction = new FunctionDecl( "_destroy_" + fixedName, DeclarationNode::Static, LinkageSpec::C, new FunctionType( Type::Qualifiers(), false ), new CompoundStmt( noLabels ), false, false, FunctionDecl::Attribute( FunctionDecl::Attribute::Destructor, inLibrary ? FunctionDecl::Attribute::High : FunctionDecl::Attribute::Default ) );
	}

	void GlobalFixer::visit( ObjectDecl *objDecl ) {
		std::list< Statement * > & initStatements = initFunction->get_statements()->get_kids();
		std::list< Statement * > & destroyStatements = destroyFunction->get_statements()->get_kids();

		if ( ! tryConstruct( objDecl ) ) return; // don't construct @= or designated objects
		if ( objDecl->get_storageClass() == DeclarationNode::Extern ) return;
		// C allows you to initialize objects with constant expressions
		// xxx - this is an optimization. Need to first resolve constructors before we decide
		// to keep C-style initializer.
		// if ( isConstExpr( objDecl->get_init() ) ) return;

		if ( ArrayType * at = dynamic_cast< ArrayType * > ( objDecl->get_type() ) ) {
			// xxx - initialize each element of the array
		} else {
			// steal initializer from object and attach it to a new temporary
			ObjectDecl *newObj = new ObjectDecl( tempNamer.newName(), DeclarationNode::NoStorageClass, LinkageSpec::C, 0, objDecl->get_type()->clone(), objDecl->get_init() );
			objDecl->set_init( NULL );
			initStatements.push_back( new DeclStmt( noLabels, newObj ) );

			// copy construct objDecl using temporary
			UntypedExpr * init = new UntypedExpr( new NameExpr( "?{}" ) );
			init->get_args().push_back( new AddressExpr( new VariableExpr( objDecl ) ) );
			init->get_args().push_back( new VariableExpr( newObj ) );
			initStatements.push_back( new ImplicitCtorDtorStmt( new ExprStmt( noLabels, init ) ) );

			// add destructor calls to global destroy function
			UntypedExpr * destroy = new UntypedExpr( new NameExpr( "^?{}" ) );
			destroy->get_args().push_back( new AddressExpr( new VariableExpr( objDecl ) ) );
			destroyStatements.push_front( new ImplicitCtorDtorStmt( new ExprStmt( noLabels, destroy ) ) );
		}
	}

	// only modify global variables
	void GlobalFixer::visit( FunctionDecl *functionDecl ) {}
	void GlobalFixer::visit( StructDecl *aggregateDecl ) {}
	void GlobalFixer::visit( UnionDecl *aggregateDecl ) {}
	void GlobalFixer::visit( EnumDecl *aggregateDecl ) {}
	void GlobalFixer::visit( TraitDecl *aggregateDecl ) {}
	void GlobalFixer::visit( TypeDecl *typeDecl ) {}

} // namespace InitTweak

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //

