#pragma once
// IWYU pragma: private, include "PassVisitor.h"

#define VISIT_START( node )                     \
	__attribute__((unused))                   \
	guard_value_impl guard( at_cleanup_impl(pass, 0) );       \
	bool visit_children = true;               \
	set_visit_children( visit_children );	\
	call_previsit( node );                    \
	if( visit_children ) {                    \

#define VISIT_END( node )                       \
	}                                         \
	call_postvisit( node );                   \

#define MUTATE_START( node )                    \
	__attribute__((unused))                   \
	guard_value_impl guard( at_cleanup_impl(pass, 0) );       \
	bool visit_children = true;               \
	set_visit_children( visit_children );	\
	call_premutate( node );                   \
	if( visit_children ) {                    \

#define MUTATE_END( type, node )                \
	}                                         \
	return call_postmutate< type * >( node ); \


#define VISIT_BODY( node )        \
	VISIT_START( node );        \
	Visitor::visit( node );     \
	VISIT_END( node );          \


#define MUTATE_BODY( type, node ) \
	MUTATE_START( node );       \
	Mutator::mutate( node );    \
	MUTATE_END( type, node );   \



template<typename T>
static inline bool empty( T * ptr ) {
	return !ptr || ptr->empty();
}

typedef std::list< Statement   * > StmtList_t;
typedef std::list< Declaration * > DeclList_t;

template<typename iterator_t>
static inline void splice( iterator_t it, DeclList_t * decls ) {
	std::transform(
		decls->begin(),
		decls->end(),
		it,
		[](Declaration * decl) -> auto {
			return new DeclStmt( noLabels, decl );
		}
	);
	decls->clear();
}

template< typename pass_type >
static inline void acceptAll( std::list< Declaration* > &decls, PassVisitor< pass_type >& visitor ) {

	DeclList_t* beforeDecls = visitor.get_beforeDecls();
	DeclList_t* afterDecls  = visitor.get_afterDecls();

	for ( std::list< Declaration* >::iterator i = decls.begin(); ; ++i ) {
		// splice in new declarations after previous decl
		if ( !empty( afterDecls ) ) { decls.splice( i, *afterDecls ); }

		if ( i == decls.end() ) break;

		// run mutator on declaration
		maybeAccept( *i, visitor );

		// splice in new declarations before current decl
		if ( !empty( beforeDecls ) ) { decls.splice( i, *beforeDecls ); }
	}
}

template< typename pass_type >
static inline void mutateAll( std::list< Declaration* > &decls, PassVisitor< pass_type >& mutator ) {

	DeclList_t* beforeDecls = mutator.get_beforeDecls();
	DeclList_t* afterDecls  = mutator.get_afterDecls();

	for ( std::list< Declaration* >::iterator i = decls.begin(); ; ++i ) {
		// splice in new declarations after previous decl
		if ( !empty( afterDecls ) ) { decls.splice( i, *afterDecls ); }

		if ( i == decls.end() ) break;

		// run mutator on declaration
		*i = maybeMutate( *i, mutator );

		// splice in new declarations before current decl
		if ( !empty( beforeDecls ) ) { decls.splice( i, *beforeDecls ); }
	}
}

template< typename Container, typename VisitorType >
inline void maybeAccept( Container &container, VisitorType &visitor ) {
	SemanticError errors;
	for ( typename Container::iterator i = container.begin(); i != container.end(); ++i ) {
		try {
			if ( *i ) {
				(*i)->accept( visitor );
			}
		} catch( SemanticError &e ) {
			e.set_location( (*i)->location );
			errors.append( e );
		}
	}
	if ( ! errors.isEmpty() ) {
		throw errors;
	}
}

template< typename Container, typename MutatorType >
inline void maybeMutateRef( Container &container, MutatorType &mutator ) {
	SemanticError errors;
	for ( typename Container::iterator i = container.begin(); i != container.end(); ++i ) {
		try {
			if ( *i ) {
///		    *i = (*i)->acceptMutator( mutator );
				*i = dynamic_cast< typename Container::value_type >( (*i)->acceptMutator( mutator ) );
				assert( *i );
			} // if
		} catch( SemanticError &e ) {
			e.set_location( (*i)->location );
			errors.append( e );
		} // try
	} // for
	if ( ! errors.isEmpty() ) {
		throw errors;
	} // if
}

template< typename pass_type >
template< typename func_t >
void PassVisitor< pass_type >::handleStatementList( std::list< Statement * > & statements, func_t func ) {
	SemanticError errors;

	// don't want statements from outer CompoundStmts to be added to this CompoundStmt
	ValueGuardPtr< StmtList_t > oldBeforeStmts( get_beforeStmts() );
	ValueGuardPtr< StmtList_t > oldAfterStmts ( get_afterStmts () );
	ValueGuardPtr< DeclList_t > oldBeforeDecls( get_beforeDecls() );
	ValueGuardPtr< DeclList_t > oldAfterDecls ( get_afterDecls () );

	StmtList_t* beforeStmts = get_beforeStmts();
	StmtList_t* afterStmts  = get_afterStmts();
	DeclList_t* beforeDecls = get_beforeDecls();
	DeclList_t* afterDecls  = get_afterDecls();

	for ( std::list< Statement* >::iterator i = statements.begin(); i != statements.end(); ++i ) {

		if ( !empty( afterDecls ) ) { splice( std::inserter( statements, i ), afterDecls ); }
		if ( !empty( afterStmts ) ) { statements.splice( i, *afterStmts ); }

		try {
			func( *i );
			assert(( empty( beforeStmts ) && empty( afterStmts ))
			    || ( empty( beforeDecls ) && empty( afterDecls )) );

		} catch ( SemanticError &e ) {
			errors.append( e );
		}

		if ( !empty( beforeDecls ) ) { splice( std::inserter( statements, i ), beforeDecls ); }
		if ( !empty( beforeStmts ) ) { statements.splice( i, *beforeStmts ); }
	}

	if ( !empty( afterDecls ) ) { splice( std::back_inserter( statements ), afterDecls); }
	if ( !empty( afterStmts ) ) { statements.splice( statements.end(), *afterStmts ); }
	if ( !errors.isEmpty() ) { throw errors; }
}

template< typename pass_type >
void PassVisitor< pass_type >::visitStatementList( std::list< Statement * > & statements ) {
	handleStatementList( statements, [this]( Statement * stmt) {
		stmt->accept( *this );
	});
}

template< typename pass_type >
void PassVisitor< pass_type >::mutateStatementList( std::list< Statement * > & statements ) {
	handleStatementList( statements, [this]( Statement *& stmt) {
		stmt = stmt->acceptMutator( *this );
	});
}


template< typename pass_type >
template< typename func_t >
Statement * PassVisitor< pass_type >::handleStatement( Statement * stmt, func_t func ) {
	// don't want statements from outer CompoundStmts to be added to this CompoundStmt
	ValueGuardPtr< TypeSubstitution * >  oldEnv        ( get_env_ptr    () );
	ValueGuardPtr< DeclList_t >          oldBeforeDecls( get_beforeDecls() );
	ValueGuardPtr< DeclList_t >          oldAfterDecls ( get_afterDecls () );
	ValueGuardPtr< StmtList_t >          oldBeforeStmts( get_beforeStmts() );
	ValueGuardPtr< StmtList_t >          oldAfterStmts ( get_afterStmts () );

	Statement *newStmt = func( stmt );

	StmtList_t* beforeStmts = get_beforeStmts();
	StmtList_t* afterStmts  = get_afterStmts();
	DeclList_t* beforeDecls = get_beforeDecls();
	DeclList_t* afterDecls  = get_afterDecls();

	if( empty(beforeStmts) && empty(afterStmts) && empty(beforeDecls) && empty(afterDecls) ) { return newStmt; }
	assert(( empty( beforeStmts ) && empty( afterStmts ))
	    || ( empty( beforeDecls ) && empty( afterDecls )) );

	CompoundStmt *compound = new CompoundStmt( noLabels );
	if( !empty(beforeDecls) ) { splice( std::back_inserter( compound->get_kids() ), beforeDecls ); }
	if( !empty(beforeStmts) ) { compound->get_kids().splice( compound->get_kids().end(), *beforeStmts ); }
	compound->get_kids().push_back( newStmt );
	if( !empty(afterDecls) ) { splice( std::back_inserter( compound->get_kids() ), afterDecls ); }
	if( !empty(afterStmts) ) { compound->get_kids().splice( compound->get_kids().end(), *afterStmts ); }
	return compound;
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::visitStatement( Statement * stmt ) {
	return handleStatement( stmt, [this]( Statement * stmt ) {
		maybeAccept( stmt, *this );
		return stmt;
	});
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutateStatement( Statement * stmt ) {
	return handleStatement( stmt, [this]( Statement * stmt ) {
	 	return maybeMutate( stmt, *this );
	});
}

template< typename pass_type >
template< typename func_t >
Expression * PassVisitor< pass_type >::handleExpression( Expression * expr, func_t func ) {
	if( !expr ) return nullptr;

	auto env_ptr = get_env_ptr();
	if ( env_ptr && expr->get_env() ) {
		*env_ptr = expr->get_env();
	}

	// should env be cloned (or moved) onto the result of the mutate?
	return func( expr );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::visitExpression( Expression * expr ) {
	return handleExpression(expr, [this]( Expression * expr ) {
		expr->accept( *this );
		return expr;
	});
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutateExpression( Expression * expr ) {
	return handleExpression(expr, [this]( Expression * expr ) {
		return expr->acceptMutator( *this );
	});
}

//------------------------------------------------------------------------------------------------------------------------------------------------------------------------
//========================================================================================================================================================================
//========================================================================================================================================================================
//========================================================================================================================================================================
//========================================================================================================================================================================
//========================================================================================================================================================================
//------------------------------------------------------------------------------------------------------------------------------------------------------------------------


//--------------------------------------------------------------------------
// ObjectDecl
template< typename pass_type >
void PassVisitor< pass_type >::visit( ObjectDecl * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->type         , *this );
	maybeAccept        ( node->init         , *this );
	maybeAccept        ( node->bitfieldWidth, *this );

	if ( node->name != "" ) {
		indexerAddId( node );
	}

	VISIT_END( node );
}

template< typename pass_type >
DeclarationWithType * PassVisitor< pass_type >::mutate( ObjectDecl * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->type         , *this );
	maybeMutateRef     ( node->init         , *this );
	maybeMutateRef     ( node->bitfieldWidth, *this );

	if ( node->name != "" ) {
		indexerAddId( node );
	}

	MUTATE_END( DeclarationWithType, node );
}

//--------------------------------------------------------------------------
// FunctionDecl
template< typename pass_type >
void PassVisitor< pass_type >::visit( FunctionDecl * node ) {
	VISIT_START( node );

	if ( node->name != "" ) {
		indexerAddId( node );
	}

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeAccept( node->type, *this );
		maybeAccept( node->statements, *this );
	}

	VISIT_END( node );
}

template< typename pass_type >
DeclarationWithType * PassVisitor< pass_type >::mutate( FunctionDecl * node ) {
	MUTATE_START( node );

	if ( node->name != "" ) {
		indexerAddId( node );
	}

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeMutateRef( node->type, *this );
		maybeMutateRef( node->statements, *this );
	}

	MUTATE_END( DeclarationWithType, node );
}

//--------------------------------------------------------------------------
// StructDecl
template< typename pass_type >
void PassVisitor< pass_type >::visit( StructDecl * node ) {
	VISIT_START( node );

	// make up a forward declaration and add it before processing the members
	// needs to be on the heap because addStruct saves the pointer
	indexerAddStructFwd( node );

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeAccept( node->parameters, *this );
		maybeAccept( node->members   , *this );
	}

	// this addition replaces the forward declaration
	indexerAddStruct( node );

	VISIT_END( node );
}

template< typename pass_type >
Declaration * PassVisitor< pass_type >::mutate( StructDecl * node ) {
	MUTATE_START( node );

	// make up a forward declaration and add it before processing the members
	// needs to be on the heap because addStruct saves the pointer
	indexerAddStructFwd( node );

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeMutateRef( node->parameters, *this );
		maybeMutateRef( node->members   , *this );
	}

	// this addition replaces the forward declaration
	indexerAddStruct( node );

	MUTATE_END( Declaration, node );
}

//--------------------------------------------------------------------------
// UnionDecl
template< typename pass_type >
void PassVisitor< pass_type >::visit( UnionDecl * node ) {
	VISIT_START( node );

	// make up a forward declaration and add it before processing the members
	indexerAddUnionFwd( node );

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeAccept( node->parameters, *this );
		maybeAccept( node->members   , *this );
	}

	indexerAddUnion( node );

	VISIT_END( node );
}

template< typename pass_type >
Declaration * PassVisitor< pass_type >::mutate( UnionDecl * node ) {
	MUTATE_START( node );

	// make up a forward declaration and add it before processing the members
	indexerAddUnionFwd( node );

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeMutateRef( node->parameters, *this );
		maybeMutateRef( node->members   , *this );
	}

	indexerAddUnion( node );

	MUTATE_END( Declaration, node );
}

//--------------------------------------------------------------------------
// EnumDecl
template< typename pass_type >
void PassVisitor< pass_type >::visit( EnumDecl * node ) {
	VISIT_START( node );

	indexerAddEnum( node );

	// unlike structs, contexts, and unions, enums inject their members into the global scope
	maybeAccept( node->parameters, *this );
	maybeAccept( node->members   , *this );

	VISIT_END( node );
}

template< typename pass_type >
Declaration * PassVisitor< pass_type >::mutate( EnumDecl * node ) {
	MUTATE_START( node );

	indexerAddEnum( node );

	// unlike structs, contexts, and unions, enums inject their members into the global scope
	maybeMutateRef( node->parameters, *this );
	maybeMutateRef( node->members   , *this );

	MUTATE_END( Declaration, node );
}

//--------------------------------------------------------------------------
// TraitDecl
template< typename pass_type >
void PassVisitor< pass_type >::visit( TraitDecl * node ) {
	VISIT_START( node );

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeAccept( node->parameters, *this );
		maybeAccept( node->members   , *this );
	}

	indexerAddTrait( node );

	VISIT_END( node );
}

template< typename pass_type >
Declaration * PassVisitor< pass_type >::mutate( TraitDecl * node ) {
	MUTATE_START( node );

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeMutateRef( node->parameters, *this );
		maybeMutateRef( node->members   , *this );
	}

	indexerAddTrait( node );

	MUTATE_END( Declaration, node );
}

//--------------------------------------------------------------------------
// TypeDecl
template< typename pass_type >
void PassVisitor< pass_type >::visit( TypeDecl * node ) {
	VISIT_START( node );

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeAccept( node->parameters, *this );
		maybeAccept( node->base      , *this );
	}

	indexerAddType( node );

	maybeAccept( node->assertions, *this );

	indexerScopedAccept( node->init, *this );

	VISIT_END( node );
}

template< typename pass_type >
Declaration * PassVisitor< pass_type >::mutate( TypeDecl * node ) {
	MUTATE_START( node );

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeMutateRef( node->parameters, *this );
		maybeMutateRef( node->base      , *this );
	}

	indexerAddType( node );

	maybeMutateRef( node->assertions, *this );

	indexerScopedMutate( node->init, *this );

	MUTATE_END( Declaration, node );
}

//--------------------------------------------------------------------------
// TypedefDecl
template< typename pass_type >
void PassVisitor< pass_type >::visit( TypedefDecl * node ) {
	VISIT_START( node );

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeAccept( node->parameters, *this );
		maybeAccept( node->base      , *this );
	}

	indexerAddType( node );

	maybeAccept( node->assertions, *this );

	VISIT_END( node );
}

template< typename pass_type >
Declaration * PassVisitor< pass_type >::mutate( TypedefDecl * node ) {
	MUTATE_START( node );

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeMutateRef     ( node->parameters, *this );
		maybeMutateRef( node->base      , *this );
	}

	indexerAddType( node );

	maybeMutateRef( node->assertions, *this );

	MUTATE_END( Declaration, node );
}

//--------------------------------------------------------------------------
// AsmDecl
template< typename pass_type >
void PassVisitor< pass_type >::visit( AsmDecl * node ) {
	VISIT_START( node );

	maybeAccept( node->stmt, *this );

	VISIT_END( node );
}

template< typename pass_type >
AsmDecl * PassVisitor< pass_type >::mutate( AsmDecl * node ) {
	MUTATE_START( node );

	maybeMutateRef( node->stmt, *this );

	MUTATE_END( AsmDecl, node );
}

//--------------------------------------------------------------------------
// CompoundStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( CompoundStmt * node ) {
	VISIT_START( node );
	{
		auto guard1 = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		auto guard2 = makeFuncGuard( [this]() { call_beginScope();   }, [this]() { call_endScope();     } );
		visitStatementList( node->kids );
	}
	VISIT_END( node );
}

template< typename pass_type >
CompoundStmt * PassVisitor< pass_type >::mutate( CompoundStmt * node ) {
	MUTATE_START( node );
	{
		auto guard1 = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		auto guard2 = makeFuncGuard( [this]() { call_beginScope();   }, [this]() { call_endScope();     } );
		mutateStatementList( node->kids );
	}
	MUTATE_END( CompoundStmt, node );
}

//--------------------------------------------------------------------------
// ExprStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( ExprStmt * node ) {
	VISIT_START( node );

	visitExpression( node->expr );

	VISIT_END( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( ExprStmt * node ) {
	MUTATE_START( node );

	node->expr = mutateExpression( node->expr );

	MUTATE_END( Statement, node );
}

//--------------------------------------------------------------------------
// AsmStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( AsmStmt * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( AsmStmt * node ) {
	MUTATE_BODY( Statement, node );
}

//--------------------------------------------------------------------------
// IfStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( IfStmt * node ) {
	VISIT_START( node );

	visitExpression( node->condition );
	node->thenPart = visitStatement( node->thenPart );
	node->elsePart = visitStatement( node->elsePart );

	VISIT_END( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( IfStmt * node ) {
	MUTATE_START( node );
	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		node->condition = mutateExpression( node->condition );
		node->thenPart  = mutateStatement ( node->thenPart  );
		node->elsePart  = mutateStatement ( node->elsePart  );
	}
	MUTATE_END( Statement, node );
}

//--------------------------------------------------------------------------
// WhileStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( WhileStmt * node ) {
	VISIT_START( node );

	visitExpression( node->condition );
	node->body = visitStatement( node->body );

	VISIT_END( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( WhileStmt * node ) {
	MUTATE_START( node );

	node->condition = mutateExpression( node->condition );
	node->body      = mutateStatement ( node->body      );

	MUTATE_END( Statement, node );
}

//--------------------------------------------------------------------------
// ForStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( ForStmt * node ) {
	VISIT_START( node );
	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeAccept( node->initialization, *this );
		visitExpression( node->condition );
		visitExpression( node->increment );
		node->body = visitStatement( node->body );
	}
	VISIT_END( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( ForStmt * node ) {
	MUTATE_START( node );
	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeMutateRef( node->initialization, *this );
		node->condition = mutateExpression( node->condition );
		node->increment = mutateExpression( node->increment );
		node->body      = mutateStatement ( node->body      );
	}
	MUTATE_END( Statement, node );
}

//--------------------------------------------------------------------------
// SwitchStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( SwitchStmt * node ) {
	VISIT_START( node );

	visitExpression   ( node->condition  );
	visitStatementList( node->statements );

	VISIT_END( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( SwitchStmt * node ) {
	MUTATE_START( node );

	node->condition = mutateExpression( node->condition );
	mutateStatementList( node->statements );

	MUTATE_END( Statement, node );
}

//--------------------------------------------------------------------------
// CaseStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( CaseStmt * node ) {
	VISIT_START( node );

	visitExpression   ( node->condition );
	visitStatementList( node->stmts     );

	VISIT_END( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( CaseStmt * node ) {
	MUTATE_START( node );

	node->condition = mutateExpression( node->condition );
	mutateStatementList( node->stmts );

	MUTATE_END( Statement, node );
}

//--------------------------------------------------------------------------
// BranchStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( BranchStmt * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( BranchStmt * node ) {
	MUTATE_BODY( Statement, node );
}

//--------------------------------------------------------------------------
// ReturnStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( ReturnStmt * node ) {
	VISIT_START( node );

	visitExpression( node->expr );

	VISIT_END( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( ReturnStmt * node ) {
	MUTATE_START( node );

	node->expr = mutateExpression( node->expr );

	MUTATE_END( Statement, node );
}

//--------------------------------------------------------------------------
// ThrowStmt

template< typename pass_type >
void PassVisitor< pass_type >::visit( ThrowStmt * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( ThrowStmt * node ) {
	MUTATE_BODY( Statement, node );
}

//--------------------------------------------------------------------------
// TryStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( TryStmt * node ) {
	VISIT_START( node );

	maybeAccept( node->block       , *this );
	maybeAccept( node->handlers    , *this );
	maybeAccept( node->finallyBlock, *this );

	VISIT_END( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( TryStmt * node ) {
	MUTATE_START( node );

	maybeMutateRef( node->block       , *this );
	maybeMutateRef( node->handlers    , *this );
	maybeMutateRef( node->finallyBlock, *this );

	MUTATE_END( Statement, node );
}

//--------------------------------------------------------------------------
// CatchStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( CatchStmt * node ) {
	VISIT_START( node );
	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeAccept( node->decl, *this );
		node->cond = visitExpression( node->cond );
		node->body = visitStatement ( node->body );
	}
	VISIT_END( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( CatchStmt * node ) {
	MUTATE_START( node );
	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeMutateRef( node->decl, *this );
		node->cond = mutateExpression( node->cond );
		node->body = mutateStatement ( node->body );
	}
	MUTATE_END( Statement, node );
}

//--------------------------------------------------------------------------
// FinallyStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( FinallyStmt * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( FinallyStmt * node ) {
	MUTATE_BODY( Statement, node );
}

//--------------------------------------------------------------------------
// WaitForStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( WaitForStmt * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( WaitForStmt * node ) {
	MUTATE_BODY( Statement, node );
}

//--------------------------------------------------------------------------
// NullStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( NullStmt * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
NullStmt * PassVisitor< pass_type >::mutate( NullStmt * node ) {
	MUTATE_BODY( NullStmt, node );
}

//--------------------------------------------------------------------------
// DeclStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( DeclStmt * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( DeclStmt * node ) {
	MUTATE_BODY( Statement, node );
}

//--------------------------------------------------------------------------
// ImplicitCtorDtorStmt
template< typename pass_type >
void PassVisitor< pass_type >::visit( ImplicitCtorDtorStmt * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
Statement * PassVisitor< pass_type >::mutate( ImplicitCtorDtorStmt * node ) {
	MUTATE_BODY( Statement, node );
}

//--------------------------------------------------------------------------
// ApplicationExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( ApplicationExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result  , *this );
	maybeAccept        ( node->function, *this );
	maybeAccept        ( node->args    , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( ApplicationExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env     , *this );
	indexerScopedMutate( node->result  , *this );
	maybeMutateRef     ( node->function, *this );
	maybeMutateRef     ( node->args    , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// UntypedExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( UntypedExpr * node ) {
	VISIT_START( node );

	// maybeAccept( node->get_env(), *this );
	indexerScopedAccept( node->result, *this );

	for ( auto expr : node->args ) {
		visitExpression( expr );
	}

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( UntypedExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );

	for ( auto& expr : node->args ) {
		expr = mutateExpression( expr );
	}

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// NameExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( NameExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( NameExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// CastExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( CastExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept        ( node->arg   , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( CastExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->arg   , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// VirtualCastExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( VirtualCastExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept( node->arg, *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( VirtualCastExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->arg   , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// AddressExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( AddressExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept        ( node->arg   , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( AddressExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->arg   , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// LabelAddressExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( LabelAddressExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( LabelAddressExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// UntypedMemberExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( UntypedMemberExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result   , *this );
	maybeAccept        ( node->aggregate, *this );
	maybeAccept        ( node->member   , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( UntypedMemberExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env      , *this );
	indexerScopedMutate( node->result   , *this );
	maybeMutateRef     ( node->aggregate, *this );
	maybeMutateRef     ( node->member   , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// MemberExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( MemberExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result   , *this );
	maybeAccept        ( node->aggregate, *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( MemberExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env      , *this );
	indexerScopedMutate( node->result   , *this );
	maybeMutateRef     ( node->aggregate, *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// VariableExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( VariableExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( VariableExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// ConstantExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( ConstantExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result   , *this );
	maybeAccept        ( &node->constant, *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( ConstantExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	node->constant = *maybeMutate( &node->constant, *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// SizeofExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( SizeofExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	if ( node->get_isType() ) {
		maybeAccept( node->type, *this );
	} else {
		maybeAccept( node->expr, *this );
	}

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( SizeofExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	if ( node->get_isType() ) {
		maybeMutateRef( node->type, *this );
	} else {
		maybeMutateRef( node->expr, *this );
	}

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// AlignofExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( AlignofExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	if ( node->get_isType() ) {
		maybeAccept( node->type, *this );
	} else {
		maybeAccept( node->expr, *this );
	}

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( AlignofExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	if ( node->get_isType() ) {
		maybeMutateRef( node->type, *this );
	} else {
		maybeMutateRef( node->expr, *this );
	}

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// UntypedOffsetofExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( UntypedOffsetofExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept        ( node->type  , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( UntypedOffsetofExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->type  , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// OffsetofExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( OffsetofExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept        ( node->type  , *this );
	maybeAccept        ( node->member, *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( OffsetofExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->type  , *this );
	maybeMutateRef     ( node->member, *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// OffsetPackExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( OffsetPackExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept        ( node->type  , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( OffsetPackExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->type  , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// AttrExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( AttrExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	if ( node->get_isType() ) {
		maybeAccept( node->type, *this );
	} else {
		maybeAccept( node->expr, *this );
	}

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( AttrExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	if ( node->get_isType() ) {
		maybeMutateRef( node->type, *this );
	} else {
		maybeMutateRef( node->expr, *this );
	}

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// LogicalExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( LogicalExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept        ( node->arg1  , *this );
	maybeAccept        ( node->arg2  , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( LogicalExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->arg1  , *this );
	maybeMutateRef     ( node->arg2  , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// ConditionalExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( ConditionalExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept        ( node->arg1  , *this );
	maybeAccept        ( node->arg2  , *this );
	maybeAccept        ( node->arg3  , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( ConditionalExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->arg1  , *this );
	maybeMutateRef     ( node->arg2  , *this );
	maybeMutateRef     ( node->arg3  , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// CommaExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( CommaExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept        ( node->arg1  , *this );
	maybeAccept        ( node->arg2  , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( CommaExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->arg1  , *this );
	maybeMutateRef     ( node->arg2  , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// TypeExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( TypeExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept        ( node->type, *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( TypeExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->type  , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// AsmExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( AsmExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result    , *this );
	maybeAccept        ( node->inout     , *this );
	maybeAccept        ( node->constraint, *this );
	maybeAccept        ( node->operand   , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( AsmExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env       , *this );
	indexerScopedMutate( node->result    , *this );
	maybeMutateRef     ( node->inout     , *this );
	maybeMutateRef     ( node->constraint, *this );
	maybeMutateRef     ( node->operand   , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// ImplicitCopyCtorExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( ImplicitCopyCtorExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result     , *this );
	maybeAccept        ( node->callExpr   , *this );
	maybeAccept        ( node->tempDecls  , *this );
	maybeAccept        ( node->returnDecls, *this );
	maybeAccept        ( node->dtors      , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( ImplicitCopyCtorExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env        , *this );
	indexerScopedMutate( node->result     , *this );
	maybeMutateRef     ( node->callExpr   , *this );
	maybeMutateRef     ( node->tempDecls  , *this );
	maybeMutateRef     ( node->returnDecls, *this );
	maybeMutateRef     ( node->dtors      , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// ConstructorExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( ConstructorExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result  , *this );
	maybeAccept        ( node->callExpr, *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( ConstructorExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env     , *this );
	indexerScopedMutate( node->result  , *this );
	maybeMutateRef     ( node->callExpr, *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// CompoundLiteralExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( CompoundLiteralExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result     , *this );
	maybeAccept        ( node->initializer, *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( CompoundLiteralExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env        , *this );
	indexerScopedMutate( node->result     , *this );
	maybeMutateRef     ( node->initializer, *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// RangeExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( RangeExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept        ( node->low   , *this );
	maybeAccept        ( node->high  , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( RangeExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->low   , *this );
	maybeMutateRef     ( node->high  , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// UntypedTupleExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( UntypedTupleExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept        ( node->exprs , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( UntypedTupleExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->exprs , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// TupleExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( TupleExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept          ( node->exprs , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( TupleExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->exprs , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// TupleIndexExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( TupleIndexExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept        ( node->tuple , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( TupleIndexExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->tuple , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// TupleAssignExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( TupleAssignExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result  , *this );
	maybeAccept        ( node->stmtExpr, *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( TupleAssignExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env     , *this );
	indexerScopedMutate( node->result  , *this );
	maybeMutateRef     ( node->stmtExpr, *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// StmtExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( StmtExpr * node ) {
	VISIT_START( node );

	// don't want statements from outer CompoundStmts to be added to this StmtExpr
	ValueGuardPtr< TypeSubstitution * >      oldEnv        ( get_env_ptr() );
	ValueGuardPtr< std::list< Statement* > > oldBeforeStmts( get_beforeStmts() );
	ValueGuardPtr< std::list< Statement* > > oldAfterStmts ( get_afterStmts () );

	indexerScopedAccept( node->result     , *this );
	maybeAccept        ( node->statements , *this );
	maybeAccept        ( node->returnDecls, *this );
	maybeAccept        ( node->dtors      , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( StmtExpr * node ) {
	MUTATE_START( node );

	// don't want statements from outer CompoundStmts to be added to this StmtExpr
	ValueGuardPtr< TypeSubstitution * >      oldEnv        ( get_env_ptr() );
	ValueGuardPtr< std::list< Statement* > > oldBeforeStmts( get_beforeStmts() );
	ValueGuardPtr< std::list< Statement* > > oldAfterStmts ( get_afterStmts () );

	indexerScopedMutate( node->result     , *this );
	maybeMutateRef     ( node->statements , *this );
	maybeMutateRef     ( node->returnDecls, *this );
	maybeMutateRef     ( node->dtors      , *this );

	MUTATE_END( Expression, node );
}

//--------------------------------------------------------------------------
// UniqueExpr
template< typename pass_type >
void PassVisitor< pass_type >::visit( UniqueExpr * node ) {
	VISIT_START( node );

	indexerScopedAccept( node->result, *this );
	maybeAccept        ( node->expr  , *this );

	VISIT_END( node );
}

template< typename pass_type >
Expression * PassVisitor< pass_type >::mutate( UniqueExpr * node ) {
	MUTATE_START( node );

	indexerScopedMutate( node->env   , *this );
	indexerScopedMutate( node->result, *this );
	maybeMutateRef     ( node->expr  , *this );

	MUTATE_END( Expression, node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( VoidType * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( BasicType * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( PointerType * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( ArrayType * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( ReferenceType * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( FunctionType * node ) {
	VISIT_BODY( node );
}

//--------------------------------------------------------------------------
// StructInstType
template< typename pass_type >
void PassVisitor< pass_type >::visit( StructInstType * node ) {
	VISIT_START( node );

	indexerAddStruct( node->name );

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeAccept( node->forall    , *this );
		maybeAccept( node->parameters, *this );
	}

	VISIT_END( node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( StructInstType * node ) {
	MUTATE_START( node );

	indexerAddStruct( node->name );

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeMutateRef( node->forall    , *this );
		maybeMutateRef( node->parameters, *this );
	}

	MUTATE_END( Type, node );
}

//--------------------------------------------------------------------------
// UnionInstType
template< typename pass_type >
void PassVisitor< pass_type >::visit( UnionInstType * node ) {
	VISIT_START( node );

	indexerAddStruct( node->name );

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeAccept( node->forall    , *this );
		maybeAccept( node->parameters, *this );
	}

	VISIT_END( node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( UnionInstType * node ) {
	MUTATE_START( node );

	indexerAddStruct( node->name );

	{
		auto guard = makeFuncGuard( [this]() { indexerScopeEnter(); }, [this]() { indexerScopeLeave(); } );
		maybeMutateRef( node->forall    , *this );
		maybeMutateRef( node->parameters, *this );
	}

	MUTATE_END( Type, node );
}

//--------------------------------------------------------------------------
// EnumInstType
template< typename pass_type >
void PassVisitor< pass_type >::visit( EnumInstType * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( EnumInstType * node ) {
	MUTATE_BODY( Type, node );
}

//--------------------------------------------------------------------------
// TraitInstType
template< typename pass_type >
void PassVisitor< pass_type >::visit( TraitInstType * node ) {
	VISIT_START( node );

	maybeAccept( node->forall    , *this );
	maybeAccept( node->parameters, *this );

	VISIT_END( node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( TraitInstType * node ) {
	MUTATE_START( node );

	maybeMutateRef( node->forall    , *this );
	maybeMutateRef( node->parameters, *this );

	MUTATE_END( Type, node );
}

//--------------------------------------------------------------------------
// TypeInstType
template< typename pass_type >
void PassVisitor< pass_type >::visit( TypeInstType * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( TupleType * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( TypeofType * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( AttrType * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( VarArgsType * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( ZeroType * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( OneType * node ) {
	VISIT_BODY( node );
}

//--------------------------------------------------------------------------
// SingleInit
template< typename pass_type >
void PassVisitor< pass_type >::visit( SingleInit * node ) {
	VISIT_START( node );

	visitExpression( node->get_value() );

	VISIT_END( node );
}

template< typename pass_type >
Initializer * PassVisitor< pass_type >::mutate( SingleInit * node ) {
	MUTATE_START( node );

	node->set_value( mutateExpression( node->get_value() ) );

	MUTATE_END( Initializer, node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( ListInit * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( ConstructorInit * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( Subrange * node ) {
	VISIT_BODY( node );
}

template< typename pass_type >
void PassVisitor< pass_type >::visit( Constant * node ) {
	VISIT_BODY( node );
}

//---------------------------------------------------------------------------------------------------------------
template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( VoidType * node ) {
	MUTATE_BODY( Type, node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( BasicType * node ) {
	MUTATE_BODY( Type, node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( PointerType * node ) {
	MUTATE_BODY( Type, node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( ArrayType * node ) {
	MUTATE_BODY( Type, node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( ReferenceType * node ) {
	MUTATE_BODY( Type, node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( FunctionType * node ) {
	MUTATE_BODY( Type, node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( TypeInstType * node ) {
	MUTATE_BODY( Type, node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( TupleType * node ) {
	MUTATE_BODY( Type, node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( TypeofType * node ) {
	MUTATE_BODY( Type, node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( AttrType * node ) {
	MUTATE_BODY( Type, node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( VarArgsType * node ) {
	MUTATE_BODY( Type, node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( ZeroType * node ) {
	MUTATE_BODY( Type, node );
}

template< typename pass_type >
Type * PassVisitor< pass_type >::mutate( OneType * node ) {
	MUTATE_BODY( Type, node );
}

template< typename pass_type >
Initializer * PassVisitor< pass_type >::mutate( ListInit * node ) {
	MUTATE_BODY( Initializer, node );
}

template< typename pass_type >
Initializer * PassVisitor< pass_type >::mutate( ConstructorInit * node ) {
	MUTATE_BODY( Initializer, node );
}

template< typename pass_type >
Subrange * PassVisitor< pass_type >::mutate( Subrange * node  )  {
	MUTATE_BODY( Subrange, node );
}

template< typename pass_type >
Constant * PassVisitor< pass_type >::mutate( Constant * node  )  {
	MUTATE_BODY( Constant, node );
}
