//                              -*- Mode: CFA -*-
//
// Cforall Version 1.0.0 Copyright (C) 2016 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// signal.c --
//
// Author           : Thierry Delisle
// Created On       : Mon Jun 5 14:20:42 2017
// Last Modified By : Thierry Delisle
// Last Modified On : --
// Update Count     : 0
//

#include "libhdr.h"
#include "preemption.h"

extern "C" {
#include <errno.h>
#include <execinfo.h>
#define __USE_GNU
#include <signal.h>
#undef __USE_GNU
#include <stdio.h>
#include <string.h>
#include <unistd.h>
}


#ifdef __USE_STREAM__
#include "fstream"
#endif

#define __CFA_DEFAULT_PREEMPTION__ 10000

__attribute__((weak)) unsigned int default_preemption() {
	return __CFA_DEFAULT_PREEMPTION__;
}

#define __CFA_SIGCXT__ ucontext_t *
#define __CFA_SIGPARMS__ __attribute__((unused)) int sig, __attribute__((unused)) siginfo_t *sfp, __attribute__((unused)) __CFA_SIGCXT__ cxt

static void preempt( processor   * this );
static void timeout( thread_desc * this );

void sigHandler_ctxSwitch( __CFA_SIGPARMS__ );
void sigHandler_alarm    ( __CFA_SIGPARMS__ );
void sigHandler_segv     ( __CFA_SIGPARMS__ );
void sigHandler_abort    ( __CFA_SIGPARMS__ );

static void __kernel_sigaction( int sig, void (*handler)(__CFA_SIGPARMS__), int flags );

//=============================================================================================
// Kernel Preemption logic
//=============================================================================================

void kernel_start_preemption() {
	LIB_DEBUG_PRINT_SAFE("Kernel : Starting preemption\n");
	__kernel_sigaction( SIGUSR1, sigHandler_ctxSwitch, SA_SIGINFO );
	__kernel_sigaction( SIGALRM, sigHandler_alarm    , SA_SIGINFO );
	__kernel_sigaction( SIGSEGV, sigHandler_segv     , SA_SIGINFO );
	__kernel_sigaction( SIGBUS , sigHandler_segv     , SA_SIGINFO );
	// __kernel_sigaction( SIGABRT, sigHandler_abort    , SA_SIGINFO );
}

void kernel_stop_preemption() {
	//Block all signals, we are no longer in a position to handle them
	sigset_t mask;
	sigfillset( &mask );
	sigprocmask( SIG_BLOCK, &mask, NULL );
	LIB_DEBUG_PRINT_SAFE("Kernel : Preemption stopped\n");

	// assert( !systemProcessor->alarms.head );
	// assert( systemProcessor->alarms.tail == &systemProcessor->alarms.head );
}

LIB_DEBUG_DO( bool validate( alarm_list_t * this ); )

void tick_preemption() {
	// LIB_DEBUG_PRINT_BUFFER_DECL( STDERR_FILENO, "Ticking preemption\n" );

	alarm_list_t * alarms = &systemProcessor->alarms;
	__cfa_time_t currtime = __kernel_get_time();
	while( alarms->head && alarms->head->alarm < currtime ) {
		alarm_node_t * node = pop(alarms);
		// LIB_DEBUG_PRINT_BUFFER_LOCAL( STDERR_FILENO, "Ticking %p\n", node );

		if( node->kernel_alarm ) {
			preempt( node->proc );
		}
		else {
			timeout( node->thrd );
		}

		verify( validate( alarms ) );

		if( node->period > 0 ) {
			node->alarm = currtime + node->period;
			insert( alarms, node );
		}
		else {
			node->set = false;
		}
	}

	if( alarms->head ) {
		__kernel_set_timer( alarms->head->alarm - currtime );
	}

	verify( validate( alarms ) );
	// LIB_DEBUG_PRINT_BUFFER_LOCAL( STDERR_FILENO, "Ticking preemption done\n" );
}

void update_preemption( processor * this, __cfa_time_t duration ) {
	LIB_DEBUG_PRINT_BUFFER_DECL( STDERR_FILENO, "Processor : %p updating preemption to %lu\n", this, duration );

	alarm_node_t * alarm = this->preemption_alarm;
	duration *= 1000;

	// Alarms need to be enabled
	if ( duration > 0 && !alarm->set ) {
		alarm->alarm = __kernel_get_time() + duration;
		alarm->period = duration;
		register_self( alarm );
	}
	// Zero duraction but alarm is set
	else if ( duration == 0 && alarm->set ) {
		unregister_self( alarm );
		alarm->alarm = 0;
		alarm->period = 0;
	}
	// If alarm is different from previous, change it
	else if ( duration > 0 && alarm->period != duration ) {
		unregister_self( alarm );
		alarm->alarm = __kernel_get_time() + duration;
		alarm->period = duration;
		register_self( alarm );
	}
}

void ?{}( preemption_scope * this, processor * proc ) {
	(&this->alarm){ proc };
	this->proc = proc;
	this->proc->preemption_alarm = &this->alarm;
	update_preemption( this->proc, this->proc->preemption );
}

void ^?{}( preemption_scope * this ) {
	disable_interrupts();

	update_preemption( this->proc, 0 );
}

//=============================================================================================
// Kernel Signal logic
//=============================================================================================

LIB_DEBUG_DO( static thread_local void * last_interrupt = 0; )

extern "C" {
	void disable_interrupts() {
		__attribute__((unused)) unsigned short new_val = __atomic_add_fetch_2( &disable_preempt_count, 1, __ATOMIC_SEQ_CST );
		verify( new_val < (unsigned short)65_000 );
		verify( new_val != (unsigned short) 0 );
	}

	void enable_interrupts_noRF() {
		__attribute__((unused)) unsigned short prev = __atomic_fetch_add_2( &disable_preempt_count, -1, __ATOMIC_SEQ_CST );
		verify( prev != (unsigned short) 0 );
	}

	void enable_interrupts( DEBUG_CTX_PARAM ) {
		processor * proc   = this_processor;
		thread_desc * thrd = this_thread;
		unsigned short prev = __atomic_fetch_add_2( &disable_preempt_count, -1, __ATOMIC_SEQ_CST );
		verify( prev != (unsigned short) 0 );
		if( prev == 1 && proc->pending_preemption ) {
			proc->pending_preemption = false;
			LIB_DEBUG_PRINT_BUFFER_DECL( STDERR_FILENO, "Executing deferred CtxSwitch on %p\n", this_processor );
			BlockInternal( thrd );
			LIB_DEBUG_PRINT_BUFFER_LOCAL( STDERR_FILENO, "Executing deferred back\n" );
		}

		LIB_DEBUG_DO( proc->last_enable = caller; )
	}
}

static inline void signal_unblock( int sig ) {
	// LIB_DEBUG_PRINT_BUFFER_DECL( STDERR_FILENO, "Processor : %p unblocking sig %i\n", this_processor, sig );

	// LIB_DEBUG_DO(
	// 	sigset_t waiting;
	// 	sigemptyset(&waiting);
	// 	sigpending(&waiting);
	// 	verify( !sigismember(&waiting, sig) );
	// )

	sigset_t mask;
	sigemptyset( &mask );
	sigaddset( &mask, sig );

	if ( sigprocmask( SIG_UNBLOCK, &mask, NULL ) == -1 ) {
	    abortf( "internal error, sigprocmask" );
	} // if
}

static inline bool preemption_ready() {
	return disable_preempt_count == 0;
}

static inline void defer_ctxSwitch() {
	this_processor->pending_preemption = true;
}

static inline void defer_alarm() {
	systemProcessor->pending_alarm = true;
}

extern "C" {
	__attribute__((noinline)) void __debug_break() {
		pthread_kill( pthread_self(), SIGTRAP );
	}
}

void sigHandler_ctxSwitch( __CFA_SIGPARMS__ ) {
	LIB_DEBUG_PRINT_BUFFER_DECL( STDERR_FILENO, "CtxSw IRH %10p running %10p @ %10p\n", this_processor, this_thread, (void *)(cxt->uc_mcontext.gregs[REG_RIP]) );
	LIB_DEBUG_DO( last_interrupt = (void *)(cxt->uc_mcontext.gregs[REG_RIP]); )

	if( preemption_ready() ) {
		// LIB_DEBUG_PRINT_BUFFER_LOCAL( STDERR_FILENO, "Ctx Switch IRH : Blocking thread %p on %p\n", this_thread, this_processor );
		signal_unblock( SIGUSR1 );
		BlockInternal( (thread_desc*)this_thread );
		// LIB_DEBUG_PRINT_BUFFER_LOCAL( STDERR_FILENO, "Ctx Switch IRH : Back\n\n");
	}
	else {
		// LIB_DEBUG_PRINT_BUFFER_LOCAL( STDERR_FILENO, "Ctx Switch IRH : Defering\n" );
		defer_ctxSwitch();
		signal_unblock( SIGUSR1 );
	}
}

void sigHandler_alarm( __CFA_SIGPARMS__ ) {
	LIB_DEBUG_PRINT_BUFFER_DECL( STDERR_FILENO, "\nAlarm IRH %10p running %10p @ %10p\n", this_processor, this_thread, (void *)(cxt->uc_mcontext.gregs[REG_RIP]) );
	LIB_DEBUG_DO( last_interrupt = (void *)(cxt->uc_mcontext.gregs[REG_RIP]); )

	if( try_lock( &systemProcessor->alarm_lock DEBUG_CTX2 ) ) {
		tick_preemption();
		unlock( &systemProcessor->alarm_lock );
	}
	else {
		defer_alarm();
	}

	signal_unblock( SIGALRM );

	if( preemption_ready() && this_processor->pending_preemption ) {
		// LIB_DEBUG_PRINT_BUFFER_LOCAL( STDERR_FILENO, "Alarm IRH : Blocking thread %p on %p\n", this_thread, this_processor );
		this_processor->pending_preemption = false;
		BlockInternal( (thread_desc*)this_thread );
		// LIB_DEBUG_PRINT_BUFFER_LOCAL( STDERR_FILENO, "Alarm Switch IRH : Back\n\n");
	}
}

static void preempt( processor * this ) {
	// LIB_DEBUG_PRINT_BUFFER_DECL( STDERR_FILENO, "Processor : signalling %p\n", this );

	if( this != systemProcessor ) {
		pthread_kill( this->kernel_thread, SIGUSR1 );
	}
	else {
		defer_ctxSwitch();
	}
}

static void timeout( thread_desc * this ) {
	//TODO : implement waking threads
}

static void __kernel_sigaction( int sig, void (*handler)(__CFA_SIGPARMS__), int flags ) {
	struct sigaction act;

	act.sa_sigaction = (void (*)(int, siginfo_t *, void *))handler;
	act.sa_flags = flags;

	// disabled during signal handler
	sigemptyset( &act.sa_mask );
	sigaddset( &act.sa_mask, sig );

	if ( sigaction( sig, &act, NULL ) == -1 ) {
		LIB_DEBUG_PRINT_BUFFER_DECL( STDERR_FILENO,
			" __kernel_sigaction( sig:%d, handler:%p, flags:%d ), problem installing signal handler, error(%d) %s.\n",
			sig, handler, flags, errno, strerror( errno )
		);
		_exit( EXIT_FAILURE );
	}
}

typedef void (*sa_handler_t)(int);

static void __kernel_sigdefault( int sig ) {
	struct sigaction act;

	// act.sa_handler = SIG_DFL;
	act.sa_flags = 0;
	sigemptyset( &act.sa_mask );

	if ( sigaction( sig, &act, NULL ) == -1 ) {
		LIB_DEBUG_PRINT_BUFFER_DECL( STDERR_FILENO,
			" __kernel_sigdefault( sig:%d ), problem reseting signal handler, error(%d) %s.\n",
			sig, errno, strerror( errno )
		);
		_exit( EXIT_FAILURE );
	}
}

//=============================================================================================
// Terminating Signals logic
//=============================================================================================

LIB_DEBUG_DO(
	static void __kernel_backtrace( int start ) {
		// skip first N stack frames

		enum { Frames = 50 };
		void * array[Frames];
		int size = backtrace( array, Frames );
		char ** messages = backtrace_symbols( array, size );

		// find executable name
		*index( messages[0], '(' ) = '\0';
		#ifdef __USE_STREAM__
		serr | "Stack back trace for:" | messages[0] | endl;
		#else
		fprintf( stderr, "Stack back trace for: %s\n", messages[0]);
		#endif

		// skip last 2 stack frames after main
		for ( int i = start; i < size && messages != NULL; i += 1 ) {
			char * name = NULL;
			char * offset_begin = NULL;
			char * offset_end = NULL;

			for ( char *p = messages[i]; *p; ++p ) {
				// find parantheses and +offset
				if ( *p == '(' ) {
					name = p;
				}
				else if ( *p == '+' ) {
					offset_begin = p;
				}
				else if ( *p == ')' ) {
					offset_end = p;
					break;
				}
			}

			// if line contains symbol print it
			int frameNo = i - start;
			if ( name && offset_begin && offset_end && name < offset_begin ) {
				// delimit strings
				*name++ = '\0';
				*offset_begin++ = '\0';
				*offset_end++ = '\0';

				#ifdef __USE_STREAM__
				serr 	| "("  | frameNo | ")" | messages[i] | ":"
					| name | "+" | offset_begin | offset_end | endl;
				#else
				fprintf( stderr, "(%i) %s : %s + %s %s\n", frameNo, messages[i], name, offset_begin, offset_end);
				#endif
			}
			// otherwise, print the whole line
			else {
				#ifdef __USE_STREAM__
				serr | "(" | frameNo | ")" | messages[i] | endl;
				#else
				fprintf( stderr, "(%i) %s\n", frameNo, messages[i] );
				#endif
			}
		}

		free( messages );
	}
)

void sigHandler_segv( __CFA_SIGPARMS__ ) {
	LIB_DEBUG_DO(
		#ifdef __USE_STREAM__
		serr 	| "*CFA runtime error* program cfa-cpp terminated with"
			| (sig == SIGSEGV ? "segment fault." : "bus error.")
			| endl;
		#else
		fprintf( stderr, "*CFA runtime error* program cfa-cpp terminated with %s\n", sig == SIGSEGV ? "segment fault." : "bus error." );
		#endif

		// skip first 2 stack frames
		__kernel_backtrace( 1 );
	)
	exit( EXIT_FAILURE );
}

// void sigHandler_abort( __CFA_SIGPARMS__ ) {
// 	// skip first 6 stack frames
// 	LIB_DEBUG_DO( __kernel_backtrace( 6 ); )

// 	// reset default signal handler
// 	__kernel_sigdefault( SIGABRT );

// 	raise( SIGABRT );
// }