//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// LabelFixer.cc -- 
//
// Author           : Rodolfo G. Esteves
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Tue Jul 28 13:32:43 2015
// Update Count     : 156
//

#include <list>
#include <cassert>

#include "LabelFixer.h"
#include "MLEMutator.h"
#include "SynTree/Expression.h"
#include "SynTree/Statement.h"
#include "SynTree/Declaration.h"
#include "utility.h"

#include <iostream>

namespace ControlStruct {
	bool LabelFixer::Entry::insideLoop() {
		return ( dynamic_cast< ForStmt * > ( definition ) ||
			dynamic_cast< WhileStmt * > ( definition )  );
	}

	LabelFixer::LabelFixer( LabelGenerator *gen ) : generator ( gen ) {
		if ( generator == 0 )
			generator = LabelGenerator::getGenerator();
	}

	void LabelFixer::visit( FunctionDecl *functionDecl ) {
		// need to go into a nested function in a fresh state
		std::map < Label, Entry *> oldLabelTable = labelTable;
		labelTable.clear();

		maybeAccept( functionDecl->get_statements(), *this );

		MLEMutator mlemut( resolveJumps(), generator );
		functionDecl->acceptMutator( mlemut );

		// and remember the outer function's labels when
		// returning to it
		labelTable = oldLabelTable;
	}

	// prune to at most one label definition for each statement
	void LabelFixer::visit( Statement *stmt ) {
		std::list< Label > &labels = stmt->get_labels();

		if ( ! labels.empty() ) {
			// only remember one label for each statement
			Label current = setLabelsDef( labels, stmt );
		} // if
	}

	void LabelFixer::visit( BranchStmt *branchStmt ) {
		visit ( ( Statement * )branchStmt );

		// for labeled branches, add an entry to the label table
		Label target = branchStmt->get_target();
		if ( target != "" ) {
			setLabelsUsg( target, branchStmt );
		}
	}

	void LabelFixer::visit( UntypedExpr *untyped ) {
		if ( NameExpr * func = dynamic_cast< NameExpr * >( untyped->get_function() ) ) {
			if ( func->get_name() == "&&" ) {
				NameExpr * arg = dynamic_cast< NameExpr * >( untyped->get_args().front() );
				Label target = arg->get_name();
				assert( target != "" );
				setLabelsUsg( target, untyped );
			} else {
				Visitor::visit( untyped );
			}
		}
	}


	// sets the definition of the labelTable entry to be the provided 
	// statement for every label in the list parameter. Happens for every kind of statement
	Label LabelFixer::setLabelsDef( std::list< Label > &llabel, Statement *definition ) {
		assert( definition != 0 );
		assert( llabel.size() > 0 );

		Entry * e = new Entry( definition );

		for ( std::list< Label >::iterator i = llabel.begin(); i != llabel.end(); i++ ) {
			if ( labelTable.find( *i ) == labelTable.end() ) {
				// all labels on this statement need to use the same entry, so this should only be created once
				// undefined and unused until now, add an entry
				labelTable[ *i ] =  e;
			} else if ( labelTable[ *i ]->defined() ) {
				// defined twice, error
				throw SemanticError( "Duplicate definition of label: " + *i );
			}	else {
				// used previously, but undefined until now -> link with this entry
				delete labelTable[ *i ];
				labelTable[ *i ] = e;
			} // if
		} // for

		// produce one of the labels attached to this statement to be 
		// temporarily used as the canonical label
		return labelTable[ llabel.front() ]->get_label();
	}

	// A label was used, add it ot the table if it isn't already there
	template< typename UsageNode >
	void LabelFixer::setLabelsUsg( Label orgValue, UsageNode *use ) {
		assert( use != 0 );

		// add label with an unknown origin
		if ( labelTable.find( orgValue ) == labelTable.end() ) {
			labelTable[ orgValue ] = new Entry( 0 );
		}
	}

	// Builds a table that maps a label to its defining statement.
	std::map<Label, Statement * > *LabelFixer::resolveJumps() throw ( SemanticError ) {
		std::map< Label, Statement * > *ret = new std::map< Label, Statement * >();
		for ( std::map< Label, Entry * >::iterator i = labelTable.begin(); i != labelTable.end(); ++i ) {
			if ( ! i->second->defined() ) {
				throw SemanticError( "Use of undefined label: " + i->first );
			}
			(*ret)[ i->first ] = i->second->get_definition();
		}

		return ret;
	}
}  // namespace ControlStruct

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
