//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// SemanticError.h --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Aug 29 22:03:36 2017
// Update Count     : 17
//

#pragma once

#include "ErrorObjects.h"

//-----------------------------------------------------------------------------
// Errors

__attribute__((noreturn)) void SemanticError( CodeLocation location, std::string error );

template< typename T >
__attribute__((noreturn)) static inline void SemanticError( const T * obj, const std::string & error ) {
	SemanticError( obj->location, toString( error, obj ) );
}

template< typename T >
__attribute__((noreturn)) static inline void SemanticError( CodeLocation location, const T * obj, const std::string & error ) {
	SemanticError( location, toString( error, obj ) );
}

//-----------------------------------------------------------------------------
// Warnings

constexpr const char * const WarningFormats[] = {
	"self assignment of expression: %s",
	"rvalue to reference conversion of rvalue: %s",
};

enum class Warning {
	SelfAssignment,
	RvalueToReferenceConversion,
	NUMBER_OF_WARNINGS, //This MUST be the last warning
};

static_assert(
	(sizeof(WarningFormats) / sizeof(WarningFormats[0])) == ((unsigned long)Warning::NUMBER_OF_WARNINGS),
	"Each warning format should have a corresponding warning enum value"
);

// ## used here to allow empty __VA_ARGS__
#define SemanticWarning(loc, id, ...) SemanticWarningImpl(loc, id, WarningFormats[(int)id], ## __VA_ARGS__)

void SemanticWarningImpl (CodeLocation loc, Warning warn, const char * const fmt, ...) __attribute__((format(printf, 3, 4)));


//-----------------------------------------------------------------------------
// Helpers
namespace ErrorHelpers {
	const std::string & error_str();
	const std::string & warning_str();
	const std::string & bold_ttycode();
	const std::string & reset_font_ttycode();

	std::string make_bold( const std::string & str );

	struct bold {};
	std::ostream & operator<<(std::ostream & os, bold);

	struct reset_font {};
	std::ostream & operator<<(std::ostream & os, reset_font);
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
