//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// RenameVars.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 12:05:18 2015
// Last Modified By : Andrew Beach
// Last Modified On : Thr Jun 20 17:39:00 2019
// Update Count     : 8
//

#include <ext/alloc_traits.h>      // for __alloc_traits<>::value_type
#include <memory>                  // for allocator_traits<>::value_type
#include <sstream>                 // for operator<<, basic_ostream, ostring...
#include <utility>                 // for pair

#include "AST/Pass.hpp"
#include "AST/Type.hpp"
#include "Common/PassVisitor.h"
#include "Common/ScopedMap.h"
#include "Common/SemanticError.h"  // for SemanticError
#include "RenameVars.h"
#include "SynTree/Declaration.h"   // for DeclarationWithType, TypeDecl, Dec...
#include "SynTree/Expression.h"    // for Expression
#include "SynTree/Type.h"          // for Type, TypeInstType, TraitInstType
#include "SynTree/Visitor.h"       // for acceptAll, maybeAccept

namespace ResolvExpr {

namespace {
	class RenamingData {
		int level = 0;
		int resetCount = 0;
		ScopedMap< std::string, std::string > nameMap;

	public:
		void reset() {
			level = 0;
			++resetCount;
		}

		using mapConstIterator = ScopedMap< std::string, std::string >::const_iterator;

		void rename( TypeInstType * type ) {
			mapConstIterator it = nameMap.find( type->name );
			if ( it != nameMap.end() ) {
				type->name = it->second;
			}
		}

		void openLevel( Type * type ) {
			if ( ! type->forall.empty() ) {
				nameMap.beginScope();
				// renames all "forall" type names to `_${level}_${name}'
				for ( auto td : type->forall ) {
					std::ostringstream output;
					output << "_" << resetCount << "_" << level << "_" << td->name;
					std::string newname( output.str() );
					nameMap[ td->get_name() ] = newname;
					td->name = newname;
					// ditto for assertion names, the next level in
					level++;
					// acceptAll( td->assertions, *this );
				} // for
			} // if
		}

		void closeLevel( Type * type ) {
			if ( !type->forall.empty() ) {
				nameMap.endScope();
			}
		}

		const ast::TypeInstType * rename( const ast::TypeInstType * type ) {
			mapConstIterator it = nameMap.find( type->name );
			if ( it != nameMap.end() ) {
				ast::TypeInstType * mutType = ast::mutate( type );
				mutType->name = it->second;
	            type = mutType;
			}
			return type;
		}

		template<typename NodeT>
		const NodeT * openLevel( const NodeT * type ) {
			if ( !type->forall.empty() ) {
				nameMap.beginScope();
				// Load new names from this forall clause and perform renaming.
				NodeT * mutType = ast::mutate( type );
				for ( ast::ptr< ast::TypeDecl > & td : mutType->forall ) {
					std::ostringstream output;
					output << "_" << resetCount << "_" << level << "_" << td->name;
					std::string newname( output.str() );
					nameMap[ td->name ] = newname;
					++level;

					ast::TypeDecl * decl = ast::mutate( td.get() );
					decl->name = newname;
					td = decl;
				}
			}
			return type;
		}

		template<typename NodeT>
		const NodeT * closeLevel( const NodeT * type ) {
			if ( !type->forall.empty() ) {
				nameMap.endScope();
			}
			return type;
		}
	};

	// Global State:
	RenamingData renaming;

	struct RenameVars {
		void previsit( TypeInstType * instType ) {
			renaming.openLevel( (Type*)instType );
			renaming.rename( instType );
		}
		void previsit( Type * type ) {
			renaming.openLevel( type );
		}
		void postvisit( Type * type ) {
			renaming.closeLevel( type );
		}

		const ast::FunctionType * previsit( const ast::FunctionType * type ) {
			return renaming.openLevel( type );
		}
		const ast::StructInstType * previsit( const ast::StructInstType * type ) {
			return renaming.openLevel( type );
		}
		const ast::UnionInstType * previsit( const ast::UnionInstType * type ) {
			return renaming.openLevel( type );
		}
		const ast::TraitInstType * previsit( const ast::TraitInstType * type ) {
			return renaming.openLevel( type );
		}
		const ast::TypeInstType * previsit( const ast::TypeInstType * type ) {
			return renaming.rename( renaming.openLevel( type ) );
		}
		const ast::ParameterizedType * postvisit( const ast::ParameterizedType * type ) {
			return renaming.closeLevel( type );
		}
	};

} // namespace

void renameTyVars( Type * t ) {
	PassVisitor<RenameVars> renamer;
	t->accept( renamer );
}

const ast::Type * renameTyVars( const ast::Type * t ) {
	ast::Pass<RenameVars> renamer;
	return t->accept( renamer );
}

void resetTyVarRenaming() {
	renaming.reset();
}

} // namespace ResolvExpr

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
