//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// AlternativeFinder.h --
//
// Author           : Richard C. Bilson
// Created On       : Sat May 16 23:56:12 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Tue Apr 19 11:44:53 2016
// Update Count     : 2
//

#ifndef ALTERNATIVEFINDER_H
#define ALTERNATIVEFINDER_H

#include <set>

#include "Alternative.h"
#include "Unify.h"
#include "SynTree/SynTree.h"
#include "SymTab/Indexer.h"
#include "SynTree/TypeSubstitution.h"

namespace ResolvExpr {
	class AlternativeFinder : public Visitor {
	  public:
		AlternativeFinder( const SymTab::Indexer &indexer, const TypeEnvironment &env );
		void find( Expression *expr, bool adjust = false, bool prune = true );
		/// Calls find with the adjust flag set; adjustment turns array and function types into equivalent pointer types
		void findWithAdjustment( Expression *expr, bool prune = true );
		AltList &get_alternatives() { return alternatives; }

		// make this look like an STL container so that we can apply generic algorithms
		typedef Alternative value_type;
		typedef AltList::iterator iterator;
		typedef AltList::const_iterator const_iterator;
		AltList::iterator begin() { return alternatives.begin(); }
		AltList::iterator end() { return alternatives.end(); }
		AltList::const_iterator begin() const { return alternatives.begin(); }
		AltList::const_iterator end() const { return alternatives.end(); }

		const SymTab::Indexer &get_indexer() const { return indexer; }
		const TypeEnvironment &get_environ() const { return env; }
	  private:
		virtual void visit( ApplicationExpr *applicationExpr );
		virtual void visit( UntypedExpr *untypedExpr );
		virtual void visit( AddressExpr *addressExpr );
		virtual void visit( CastExpr *castExpr );
		virtual void visit( UntypedMemberExpr *memberExpr );
		virtual void visit( MemberExpr *memberExpr );
		virtual void visit( NameExpr *variableExpr );
		virtual void visit( VariableExpr *variableExpr );
		virtual void visit( ConstantExpr *constantExpr );
		virtual void visit( SizeofExpr *sizeofExpr );
		virtual void visit( AlignofExpr *alignofExpr );
		virtual void visit( UntypedOffsetofExpr *offsetofExpr );
		virtual void visit( OffsetofExpr *offsetofExpr );
		virtual void visit( OffsetPackExpr *offsetPackExpr );
		virtual void visit( AttrExpr *attrExpr );
		virtual void visit( LogicalExpr *logicalExpr );
		virtual void visit( ConditionalExpr *conditionalExpr );
		virtual void visit( CommaExpr *commaExpr );
		virtual void visit( TupleExpr *tupleExpr );
		virtual void visit( ImplicitCopyCtorExpr * impCpCtorExpr );
		virtual void visit( ConstructorExpr * ctorExpr );
		virtual void visit( TupleIndexExpr *tupleExpr );
		virtual void visit( TupleAssignExpr *tupleExpr );
		/// Runs a new alternative finder on each element in [begin, end)
		/// and writes each alternative finder to out.
		template< typename InputIterator, typename OutputIterator >
		void findSubExprs( InputIterator begin, InputIterator end, OutputIterator out );

		/// Adds alternatives for member expressions, given the aggregate, conversion cost for that aggregate, and name of the member
		template< typename StructOrUnionType > void addAggMembers( StructOrUnionType *aggInst, Expression *expr, const Cost &newCost, const TypeEnvironment & env, Expression * member );
		/// Adds alternatives for offsetof expressions, given the base type and name of the member
		template< typename StructOrUnionType > void addOffsetof( StructOrUnionType *aggInst, const std::string &name );
		bool instantiateFunction( std::list< DeclarationWithType* >& formals, const AltList &actuals, bool isVarArgs, OpenVarSet& openVars, TypeEnvironment &resultEnv, AssertionSet &resultNeed, AssertionSet &resultHave, AltList & out );
		template< typename OutputIterator >
		void makeFunctionAlternatives( const Alternative &func, FunctionType *funcType, const AltList &actualAlt, OutputIterator out );
		template< typename OutputIterator >
		void inferParameters( const AssertionSet &need, AssertionSet &have, const Alternative &newAlt, OpenVarSet &openVars, OutputIterator out );
		void resolveAttr( DeclarationWithType *funcDecl, FunctionType *function, Type *argType, const TypeEnvironment &env );

		const SymTab::Indexer &indexer;
		AltList alternatives;
		const TypeEnvironment &env;
	}; // AlternativeFinder

	Expression *resolveInVoidContext( Expression *expr, const SymTab::Indexer &indexer, TypeEnvironment &env );

	template< typename InputIterator, typename OutputIterator >
	void findMinCost( InputIterator begin, InputIterator end, OutputIterator out ) {
		AltList alternatives;

		// select the alternatives that have the minimum parameter cost
		Cost minCost = Cost::infinity;
		for ( InputIterator i = begin; i != end; ++i ) {
			if ( i->cost < minCost ) {
				minCost = i->cost;
				i->cost = i->cvtCost;
				alternatives.clear();
				alternatives.push_back( *i );
			} else if ( i->cost == minCost ) {
				i->cost = i->cvtCost;
				alternatives.push_back( *i );
			}
		}
		std::copy( alternatives.begin(), alternatives.end(), out );
	}

	Cost sumCost( const AltList &in );
} // namespace ResolvExpr

#endif // ALTERNATIVEFINDER_H

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
