//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// RemoveInit.cc -- 
//
// Author           : Rob Schluntz
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Dec 15 15:37:26 2015
// Update Count     : 15
//

#include "RemoveInit.h"
#include "SynTree/Declaration.h"
#include "SynTree/Type.h"
#include "SynTree/Expression.h"
#include "SynTree/Statement.h"
#include "SynTree/Initializer.h"
#include "SynTree/Mutator.h"

namespace InitTweak {
	namespace {
		const std::list<Label> noLabels;
	}
	
	class RemoveInit : public Mutator {
	  public:
		RemoveInit();
		virtual ObjectDecl * mutate(ObjectDecl *objDecl);
		virtual DeclarationWithType * mutate( FunctionDecl *functionDecl );

		virtual Statement * mutate( ReturnStmt * returnStmt );
		
		virtual CompoundStmt * mutate(CompoundStmt * compoundStmt);
		
	  protected:
		std::list< Statement* > stmtsToAddBefore;
		std::list< Statement* > stmtsToAddAfter;
		void mutateStatementList( std::list< Statement* > &statements );

		std::list<DeclarationWithType*> returnVals;
		UniqueName tempNamer;
		std::string funcName;
	};

	void tweak( std::list< Declaration * > translationUnit ) {
		RemoveInit remover;
		mutateAll( translationUnit, remover );
	}

	RemoveInit::RemoveInit() : tempNamer( "_retVal" ) {}
	
	void RemoveInit::mutateStatementList( std::list< Statement* > &statements ) {
		for ( std::list< Statement* >::iterator i = statements.begin(); i != statements.end(); ++i ) {
			if ( ! stmtsToAddAfter.empty() ) {
				statements.splice( i, stmtsToAddAfter );
			} // if
			*i = (*i)->acceptMutator( *this );
			if ( ! stmtsToAddBefore.empty() ) {
				statements.splice( i, stmtsToAddBefore );
			} // if
		} // for
		if ( ! stmtsToAddAfter.empty() ) {
			statements.splice( statements.end(), stmtsToAddAfter );
		} // if
	}

	CompoundStmt *RemoveInit::mutate(CompoundStmt *compoundStmt) {
		mutateStatementList( compoundStmt->get_kids() );
		return compoundStmt;
	}

	// in the case where an object has an initializer and a polymorphic type, insert an assignment immediately after the
	// declaration. This will (seemingly) cause the later phases to do the right thing with the assignment
	ObjectDecl *RemoveInit::mutate( ObjectDecl *objDecl ) {
		if (objDecl->get_init() && dynamic_cast<TypeInstType*>(objDecl->get_type())) {
			if (SingleInit * single = dynamic_cast<SingleInit*>(objDecl->get_init())) {
				UntypedExpr *assign = new UntypedExpr( new NameExpr( "?=?" ) );
				assign->get_args().push_back( new AddressExpr (new NameExpr( objDecl->get_name() ) ) );
				assign->get_args().push_back( single->get_value()->clone() );
				stmtsToAddAfter.push_back(new ExprStmt(noLabels, assign));
			} // if
		} // if
		return objDecl;
	}

	Statement *RemoveInit::mutate( ReturnStmt *returnStmt ) {
		// update for multiple return values
		assert( returnVals.size() == 0 || returnVals.size() == 1 );
		// hands off if the function returns an lvalue - we don't want to allocate a temporary if a variable's address
		// is being returned
		if ( returnStmt->get_expr() && returnVals.size() == 1 && funcName != "?=?" && ! returnVals.front()->get_type()->get_isLvalue()  ) {
			ObjectDecl *newObj = new ObjectDecl( tempNamer.newName(), DeclarationNode::NoStorageClass, LinkageSpec::C, 0, returnVals.front()->get_type()->clone(), 0 );
			stmtsToAddBefore.push_back( new DeclStmt( noLabels, newObj ) );
			
			UntypedExpr *assign = new UntypedExpr( new NameExpr( "?=?" ) );
			assign->get_args().push_back( new AddressExpr (new NameExpr( newObj->get_name() ) ) );
			assign->get_args().push_back( returnStmt->get_expr() );
			stmtsToAddBefore.push_back(new ExprStmt(noLabels, assign));

			returnStmt->set_expr( new VariableExpr( newObj ) );
		} // if
		return returnStmt;
	}

	DeclarationWithType* RemoveInit::mutate( FunctionDecl *functionDecl ) {
		std::list<DeclarationWithType*> oldReturnVals = returnVals;
		std::string oldFuncName = funcName;
		
		FunctionType * type = functionDecl->get_functionType();
		returnVals = type->get_returnVals();
		funcName = functionDecl->get_name();
		DeclarationWithType * decl = Mutator::mutate( functionDecl );
		returnVals = oldReturnVals;
		funcName = oldFuncName;
		return decl;
	}
} // namespace InitTweak

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
