//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// PtrsCastable.cc --
//
// Author           : Richard C. Bilson
// Created On       : Sun May 17 11:48:00 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Mar  2 17:36:18 2016
// Update Count     : 8
//

#include "Common/PassVisitor.h"
#include "ResolvExpr/TypeEnvironment.h"  // for EqvClass, TypeEnvironment
#include "SymTab/Indexer.h"              // for Indexer
#include "SynTree/Declaration.h"         // for TypeDecl, TypeDecl::Kind::Ftype
#include "SynTree/Type.h"                // for TypeInstType, Type, BasicType
#include "SynTree/Visitor.h"             // for Visitor
#include "typeops.h"                     // for ptrsAssignable

namespace ResolvExpr {
	struct PtrsCastable : public WithShortCircuiting  {
	  public:
		PtrsCastable( Type *dest, const TypeEnvironment &env, const SymTab::Indexer &indexer );

		int get_result() const { return result; }

		void previsit( Type * ) { visit_children = false; }

		void postvisit( VoidType * voidType );
		void postvisit( BasicType * basicType );
		void postvisit( PointerType * pointerType );
		void postvisit( ArrayType * arrayType );
		void postvisit( FunctionType * functionType );
		void postvisit( StructInstType * inst );
		void postvisit( UnionInstType * inst );
		void postvisit( EnumInstType * inst );
		void postvisit( TraitInstType * inst );
		void postvisit( TypeInstType * inst );
		void postvisit( TupleType * tupleType );
		void postvisit( VarArgsType * varArgsType );
		void postvisit( ZeroType * zeroType );
		void postvisit( OneType * oneType );
	  private:
		Type *dest;
		int result;
		const TypeEnvironment &env;
		const SymTab::Indexer &indexer;
	};

	namespace {
		int objectCast( Type *src, const TypeEnvironment &env, const SymTab::Indexer &indexer ) {
			if ( dynamic_cast< FunctionType* >( src ) ) {
				return -1;
			} else if ( TypeInstType *typeInst = dynamic_cast< TypeInstType* >( src ) ) {
				if ( NamedTypeDecl *ntDecl = indexer.lookupType( typeInst->get_name() ) ) {
					if ( TypeDecl *tyDecl = dynamic_cast< TypeDecl* >( ntDecl ) ) {
						if ( tyDecl->get_kind() == TypeDecl::Ftype ) {
							return -1;
						} // if
					} //if
				} else if ( const EqvClass *eqvClass = env.lookup( typeInst->get_name() ) ) {
					if ( eqvClass->data.kind == TypeDecl::Ftype ) {
						return -1;
					} // if
				} // if
			} //if
			return 1;
		}
		int functionCast( Type *src, const TypeEnvironment &env, const SymTab::Indexer &indexer ) {
			return -1 * objectCast( src, env, indexer );  // reverse the sense of objectCast
		}
	}

	int ptrsCastable( Type *src, Type *dest, const TypeEnvironment &env, const SymTab::Indexer &indexer ) {
		if ( TypeInstType *destAsTypeInst = dynamic_cast< TypeInstType* >( dest ) ) {
			if ( const EqvClass *eqvClass = env.lookup( destAsTypeInst->get_name() ) ) {
				// xxx - should this be ptrsCastable?
				return ptrsAssignable( src, eqvClass->type, env );
			} // if
		} // if
		if ( dynamic_cast< VoidType* >( dest ) ) {
			return objectCast( src, env, indexer );
		} else {
			PassVisitor<PtrsCastable> ptrs( dest, env, indexer );
			src->accept( ptrs );
			return ptrs.pass.get_result();
		} // if
	}

	PtrsCastable::PtrsCastable( Type *dest, const TypeEnvironment &env, const SymTab::Indexer &indexer )
		: dest( dest ), result( 0 ), env( env ), indexer( indexer )	{
	}

	void PtrsCastable::postvisit( VoidType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable::postvisit( BasicType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable::postvisit( PointerType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable::postvisit( ArrayType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable::postvisit( FunctionType * ) {
		// result = -1;
		result = functionCast( dest, env, indexer );
	}

	void PtrsCastable::postvisit( StructInstType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable::postvisit( UnionInstType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable::postvisit( EnumInstType * ) {
		if ( dynamic_cast< EnumInstType* >( dest ) ) {
			result = 1;
		} else if ( BasicType *bt = dynamic_cast< BasicType* >( dest ) ) {
			if ( bt->get_kind() == BasicType::SignedInt ) {
				result = 0;
			} else {
				result = 1;
			}
		} else {
			result = objectCast( dest, env, indexer );
		}
	}

	void PtrsCastable::postvisit( TraitInstType * ) {}

	void PtrsCastable::postvisit(TypeInstType *inst) {
		//result = objectCast( inst, env, indexer ) > 0 && objectCast( dest, env, indexer ) > 0 ? 1 : -1;
		result = objectCast( inst, env, indexer ) == objectCast( dest, env, indexer ) ? 1 : -1;
	}

	void PtrsCastable::postvisit( TupleType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable::postvisit( VarArgsType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable::postvisit( ZeroType * ) {
		result = objectCast( dest, env, indexer );
	}

	void PtrsCastable::postvisit( OneType * ) {
		result = objectCast( dest, env, indexer );
	}
} // namespace ResolvExpr

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
