//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// CodeGenerator.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Sat Jul 30 11:27:07 2016
// Update Count     : 340
//

#include <algorithm>
#include <iostream>
#include <cassert>
#include <list>

#include "Parser/ParseNode.h"

#include "SynTree/Declaration.h"
#include "SynTree/Expression.h"
#include "SynTree/Initializer.h"
#include "SynTree/Statement.h"
#include "SynTree/Type.h"
#include "SynTree/Attribute.h"

#include "Common/utility.h"
#include "Common/UnimplementedError.h"

#include "CodeGenerator.h"
#include "OperatorTable.h"
#include "GenType.h"

#include "InitTweak/InitTweak.h"

using namespace std;

namespace CodeGen {
	int CodeGenerator::tabsize = 4;

	// the kinds of statements that would ideally be followed by whitespace
	bool wantSpacing( Statement * stmt) {
		return dynamic_cast< IfStmt * >( stmt ) || dynamic_cast< CompoundStmt * >( stmt ) ||
			dynamic_cast< WhileStmt * >( stmt ) || dynamic_cast< ForStmt * > ( stmt ) || dynamic_cast< SwitchStmt *>( stmt );
	}

	void CodeGenerator::extension( Expression *expr ) {
		if ( expr->get_extension() ) {
			output << "__extension__ ";
		} // if
	} // extension

	void CodeGenerator::extension( Declaration *decl ) {
		if ( decl->get_extension() ) {
			output << "__extension__ ";
		} // if
	} // extension

	ostream & CodeGenerator::Indenter::operator()( ostream & output ) const {
	  return output << string( cg.cur_indent, ' ' );
	}

	ostream & operator<<( ostream & output, const CodeGenerator::Indenter &indent ) {
		return indent( output );
	}

	CodeGenerator::LabelPrinter & CodeGenerator::LabelPrinter::operator()( std::list< Label > & l ) {
		labels = &l;
		return *this;
	}

	ostream & operator<<( ostream & output, CodeGenerator::LabelPrinter &printLabels ) {
		std::list< Label > & labs = *printLabels.labels;
		// l.unique(); // assumes a sorted list. Why not use set? Does order matter?
		for ( Label & l : labs ) {
			output << l.get_name() + ": ";
			printLabels.cg.genAttributes( l.get_attributes() );
		}
		return output;
	}

	CodeGenerator::CodeGenerator( std::ostream &os ) : indent( *this), cur_indent( 0 ), insideFunction( false ), output( os ), printLabels( *this ) { }

	CodeGenerator::CodeGenerator( std::ostream &os, std::string init, int indentation, bool infunp )
			: indent( *this), cur_indent( indentation ), insideFunction( infunp ), output( os ), printLabels( *this ) {
		//output << std::string( init );
	}

	CodeGenerator::CodeGenerator( std::ostream &os, char *init, int indentation, bool infunp )
			: indent( *this ), cur_indent( indentation ), insideFunction( infunp ), output( os ), printLabels( *this ) {
		//output << std::string( init );
	}

	string mangleName( DeclarationWithType *decl ) {
		if ( decl->get_mangleName() != "" ) {
			// need to incorporate scope level in order to differentiate names for destructors
			return decl->get_scopedMangleName();
		} else {
			return decl->get_name();
		} // if
	}

	void CodeGenerator::genAttributes( std::list< Attribute * > & attributes ) {
		if ( ! attributes.empty() ) {
			output << "__attribute__ ((";
			for ( Attribute *& attr : attributes ) {
				if ( ! attr->empty() ) {
					output << attr->get_name() << "(";
					genCommaList( attr->get_parameters().begin(), attr->get_parameters().end() );
					output << ")";
				}
				output << ",";
			}
			output << ")) ";
		}
	}


	//*** Declarations
	void CodeGenerator::visit( FunctionDecl *functionDecl ) {
		extension( functionDecl );
		genAttributes( functionDecl->get_attributes() );

		handleStorageClass( functionDecl );
		if ( functionDecl->get_isInline() ) {
			output << "inline ";
		} // if
		if ( functionDecl->get_isNoreturn() ) {
			output << "_Noreturn ";
		} // if
		output << genType( functionDecl->get_functionType(), mangleName( functionDecl ) );

		// how to get this to the Functype?
		std::list< Declaration * > olds = functionDecl->get_oldDecls();
		if ( ! olds.empty() ) {
			output << " /* function has old declaration */";
		} // if

		// acceptAll( functionDecl->get_oldDecls(), *this );
		if ( functionDecl->get_statements() ) {
			functionDecl->get_statements()->accept( *this );
		} // if
	}

	void CodeGenerator::visit( ObjectDecl *objectDecl ) {
		extension( objectDecl );
		handleStorageClass( objectDecl );
		output << genType( objectDecl->get_type(), mangleName( objectDecl ) );

		if ( objectDecl->get_init() ) {
			output << " = ";
			objectDecl->get_init()->accept( *this );
		} // if
		if ( objectDecl->get_bitfieldWidth() ) {
			output << ":";
			objectDecl->get_bitfieldWidth()->accept( *this );
		} // if
	}

	void CodeGenerator::handleAggregate( AggregateDecl *aggDecl ) {
		if ( aggDecl->get_name() != "" )
			output << aggDecl->get_name();

		std::list< Declaration * > &memb = aggDecl->get_members();
		if ( ! memb.empty() ) {
//		if ( aggDecl->has_body() ) {
//			std::list< Declaration * > &memb = aggDecl->get_members();
			output << " {" << endl;

			cur_indent += CodeGenerator::tabsize;
			for ( std::list< Declaration* >::iterator i = memb.begin(); i != memb.end();  i++) {
				output << indent;
				(*i)->accept( *this );
				output << ";" << endl;
			}

			cur_indent -= CodeGenerator::tabsize;

			output << indent << "}";
		} // if
	}

	void CodeGenerator::visit( StructDecl *structDecl ) {
		extension( structDecl );
		output << "struct ";
		handleAggregate( structDecl );
	}

	void CodeGenerator::visit( UnionDecl *unionDecl ) {
		extension( unionDecl );
		output << "union ";
		handleAggregate( unionDecl );
	}

	void CodeGenerator::visit( EnumDecl *enumDecl ) {
		extension( enumDecl );
		output << "enum ";

		if ( enumDecl->get_name() != "" )
			output << enumDecl->get_name();

		std::list< Declaration* > &memb = enumDecl->get_members();

		if ( ! memb.empty() ) {
			output << " {" << endl;

			cur_indent += CodeGenerator::tabsize;
			for ( std::list< Declaration* >::iterator i = memb.begin(); i != memb.end();  i++) {
				ObjectDecl *obj = dynamic_cast< ObjectDecl* >( *i );
				assert( obj );
				output << indent << mangleName( obj );
				if ( obj->get_init() ) {
					output << " = ";
					obj->get_init()->accept( *this );
				} // if
				output << "," << endl;
			} // for

			cur_indent -= CodeGenerator::tabsize;

			output << indent << "}";
		} // if
	}

	void CodeGenerator::visit( TraitDecl *traitDecl ) {}

	void CodeGenerator::visit( TypedefDecl *typeDecl ) {
		assert( false && "Typedefs are removed and substituted in earlier passes." );
		//output << "typedef ";
		//output << genType( typeDecl->get_base(), typeDecl->get_name() );
	}

	void CodeGenerator::visit( TypeDecl *typeDecl ) {
		// really, we should mutate this into something that isn't a TypeDecl but that requires large-scale changes,
		// still to be done
		extension( typeDecl );
		output << "extern unsigned long " << typeDecl->get_name();
		if ( typeDecl->get_base() ) {
			output << " = sizeof( " << genType( typeDecl->get_base(), "" ) << " )";
		} // if
	}

	void CodeGenerator::printDesignators( std::list< Expression * > & designators ) {
		typedef std::list< Expression * > DesignatorList;
		if ( designators.size() == 0 ) return;
		for ( DesignatorList::iterator iter = designators.begin(); iter != designators.end(); ++iter ) {
			if ( dynamic_cast< NameExpr * >( *iter ) ) {
				// if expression is a name, then initializing aggregate member
				output << ".";
				(*iter)->accept( *this );
			} else {
				// if not a simple name, it has to be a constant expression, i.e. an array designator
				output << "[";
				(*iter)->accept( *this );
				output << "]";
			}
		}
		output << " = ";
	}

	void CodeGenerator::visit( SingleInit *init ) {
		printDesignators( init->get_designators() );
		init->get_value()->accept( *this );
	}

	void CodeGenerator::visit( ListInit *init ) {
		printDesignators( init->get_designators() );
		output << "{ ";
		if ( init->begin_initializers() == init->end_initializers() ) {
			// illegal to leave initializer list empty for scalar initializers, but always legal to have 0
			output << "0";
		} else {
			genCommaList( init->begin_initializers(), init->end_initializers() );
		}
		output << " }";
	}

	void CodeGenerator::visit( Constant *constant ) {
		output << constant->get_value() ;
	}

	//*** Expressions
	void CodeGenerator::visit( ApplicationExpr *applicationExpr ) {
		extension( applicationExpr );
		if ( VariableExpr *varExpr = dynamic_cast< VariableExpr* >( applicationExpr->get_function() ) ) {
			OperatorInfo opInfo;
			if ( varExpr->get_var()->get_linkage() == LinkageSpec::Intrinsic && operatorLookup( varExpr->get_var()->get_name(), opInfo ) ) {
				std::list< Expression* >::iterator arg = applicationExpr->get_args().begin();
				switch ( opInfo.type ) {
				  case OT_PREFIXASSIGN:
				  case OT_POSTFIXASSIGN:
				  case OT_INFIXASSIGN:
				  case OT_CTOR:
				  case OT_DTOR:
					{
						assert( arg != applicationExpr->get_args().end() );
						if ( AddressExpr *addrExpr = dynamic_cast< AddressExpr * >( *arg ) ) {
							// remove & from first assignment/ctor argument
							*arg = addrExpr->get_arg();
						} else {
							// no address-of operator, so must be a pointer - add dereference
							UntypedExpr *newExpr = new UntypedExpr( new NameExpr( "*?" ) );
							newExpr->get_args().push_back( *arg );
							assert( (*arg)->get_results().size() == 1 );
							Type * type = InitTweak::getPointerBase( (*arg)->get_results().front() );
							assert( type );
							newExpr->get_results().push_back( type );
							*arg = newExpr;
						} // if
						break;
					}

				  default:
					// do nothing
					;
				}

				switch ( opInfo.type ) {
				  case OT_INDEX:
					assert( applicationExpr->get_args().size() == 2 );
					(*arg++)->accept( *this );
					output << "[";
					(*arg)->accept( *this );
					output << "]";
					break;

				  case OT_CALL:
					// there are no intrinsic definitions of the function call operator
					assert( false );
					break;

				  case OT_CTOR:
				  case OT_DTOR:
					if ( applicationExpr->get_args().size() == 1 ) {
						// the expression fed into a single parameter constructor or destructor may contain side
						// effects, so must still output this expression
						output << "(";
						(*arg++)->accept( *this );
						output << ") /* " << opInfo.inputName << " */";
					} else if ( applicationExpr->get_args().size() == 2 ) {
						// intrinsic two parameter constructors are essentially bitwise assignment
						output << "(";
						(*arg++)->accept( *this );
						output << opInfo.symbol;
						(*arg)->accept( *this );
						output << ") /* " << opInfo.inputName << " */";
					} else {
						// no constructors with 0 or more than 2 parameters
						assert( false );
					}
					break;

				  case OT_PREFIX:
				  case OT_PREFIXASSIGN:
					assert( applicationExpr->get_args().size() == 1 );
					output << "(";
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;

				  case OT_POSTFIX:
				  case OT_POSTFIXASSIGN:
					assert( applicationExpr->get_args().size() == 1 );
					(*arg)->accept( *this );
					output << opInfo.symbol;
					break;


				  case OT_INFIX:
				  case OT_INFIXASSIGN:
					assert( applicationExpr->get_args().size() == 2 );
					output << "(";
					(*arg++)->accept( *this );
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;

				  case OT_CONSTANT:
				  case OT_LABELADDRESS:
					// there are no intrinsic definitions of 0/1 or label addresses as functions
					assert( false );
				}
			} else {
				varExpr->accept( *this );
				output << "(";
				genCommaList( applicationExpr->get_args().begin(), applicationExpr->get_args().end() );
				output << ")";
			} // if
		} else {
			applicationExpr->get_function()->accept( *this );
			output << "(";
			genCommaList( applicationExpr->get_args().begin(), applicationExpr->get_args().end() );
			output << ")";
		} // if
	}

	void CodeGenerator::visit( UntypedExpr *untypedExpr ) {
		extension( untypedExpr );
		if ( NameExpr *nameExpr = dynamic_cast< NameExpr* >( untypedExpr->get_function() ) ) {
			OperatorInfo opInfo;
			if ( operatorLookup( nameExpr->get_name(), opInfo ) ) {
				std::list< Expression* >::iterator arg = untypedExpr->get_args().begin();
				switch ( opInfo.type ) {
				  case OT_INDEX:
					assert( untypedExpr->get_args().size() == 2 );
					(*arg++)->accept( *this );
					output << "[";
					(*arg)->accept( *this );
					output << "]";
					break;

				  case OT_CALL:
					assert( false );


				  case OT_CTOR:
				  case OT_DTOR:
					if ( untypedExpr->get_args().size() == 1 ) {
						// the expression fed into a single parameter constructor or destructor may contain side
						// effects, so must still output this expression
						output << "(";
						(*arg++)->accept( *this );
						output << ") /* " << opInfo.inputName << " */";
					} else if ( untypedExpr->get_args().size() == 2 ) {
						// intrinsic two parameter constructors are essentially bitwise assignment
						output << "(";
						(*arg++)->accept( *this );
						output << opInfo.symbol;
						(*arg)->accept( *this );
						output << ") /* " << opInfo.inputName << " */";
					} else {
						// no constructors with 0 or more than 2 parameters
						assert( false );
					}
					break;

				  case OT_PREFIX:
				  case OT_PREFIXASSIGN:
				  case OT_LABELADDRESS:
					assert( untypedExpr->get_args().size() == 1 );
					output << "(";
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;

				  case OT_POSTFIX:
				  case OT_POSTFIXASSIGN:
					assert( untypedExpr->get_args().size() == 1 );
					(*arg)->accept( *this );
					output << opInfo.symbol;
					break;

				  case OT_INFIX:
				  case OT_INFIXASSIGN:
					assert( untypedExpr->get_args().size() == 2 );
					output << "(";
					(*arg++)->accept( *this );
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;

				  case OT_CONSTANT:
					// there are no intrinsic definitions of 0 or 1 as functions
					assert( false );
				}
			} else {
				if ( nameExpr->get_name() != "Range" ) {
					// builtin routines
					nameExpr->accept( *this );
					output << "(";
					genCommaList( untypedExpr->get_args().begin(), untypedExpr->get_args().end() );
					output << ")";
				} else {
					// case V1 ... V2 or case V1~V2
					assert( untypedExpr->get_args().size() == 2 );
					(*untypedExpr->get_args().begin())->accept( *this );
					output << " ... ";
					(*--untypedExpr->get_args().end())->accept( *this );
				} // if
			} // if
		} else {
			untypedExpr->get_function()->accept( *this );
			output << "(";
			genCommaList( untypedExpr->get_args().begin(), untypedExpr->get_args().end() );
			output << ")";
		} // if
	}

	void CodeGenerator::visit( NameExpr *nameExpr ) {
		extension( nameExpr );
		OperatorInfo opInfo;
		if ( operatorLookup( nameExpr->get_name(), opInfo ) ) {
			assert( opInfo.type == OT_CONSTANT );
			output << opInfo.symbol;
		} else {
			output << nameExpr->get_name();
		} // if
	}

	void CodeGenerator::visit( AddressExpr *addressExpr ) {
		extension( addressExpr );
		output << "(&";
		// this hack makes sure that we don't convert "constant_zero" to "0" if we're taking its address
		if ( VariableExpr *variableExpr = dynamic_cast< VariableExpr* >( addressExpr->get_arg() ) ) {
			output << mangleName( variableExpr->get_var() );
		} else {
			addressExpr->get_arg()->accept( *this );
		} // if
		output << ")";
	}

	void CodeGenerator::visit( CastExpr *castExpr ) {
		extension( castExpr );
		output << "(";
		if ( castExpr->get_results().empty() ) {
			output << "(void)" ;
		} else if ( ! castExpr->get_results().front()->get_isLvalue() ) {
			// at least one result type of cast, but not an lvalue
			output << "(";
			output << genType( castExpr->get_results().front(), "" );
			output << ")";
		} else {
			// otherwise, the cast is to an lvalue type, so the cast should be dropped, since the result of a cast is
			// never an lvalue in C
		}
		castExpr->get_arg()->accept( *this );
		output << ")";
	}

	void CodeGenerator::visit( UntypedMemberExpr *memberExpr ) {
		assert( false );
	}

	void CodeGenerator::visit( MemberExpr *memberExpr ) {
		extension( memberExpr );
		memberExpr->get_aggregate()->accept( *this );
		output << "." << mangleName( memberExpr->get_member() );
	}

	void CodeGenerator::visit( VariableExpr *variableExpr ) {
		extension( variableExpr );
		OperatorInfo opInfo;
		if ( variableExpr->get_var()->get_linkage() == LinkageSpec::Intrinsic && operatorLookup( variableExpr->get_var()->get_name(), opInfo ) && opInfo.type == OT_CONSTANT ) {
			output << opInfo.symbol;
		} else {
			output << mangleName( variableExpr->get_var() );
		} // if
	}

	void CodeGenerator::visit( ConstantExpr *constantExpr ) {
		assert( constantExpr->get_constant() );
		extension( constantExpr );
		constantExpr->get_constant()->accept( *this );
	}

	void CodeGenerator::visit( SizeofExpr *sizeofExpr ) {
		extension( sizeofExpr );
		output << "sizeof(";
		if ( sizeofExpr->get_isType() ) {
			output << genType( sizeofExpr->get_type(), "" );
		} else {
			sizeofExpr->get_expr()->accept( *this );
		} // if
		output << ")";
	}

	void CodeGenerator::visit( AlignofExpr *alignofExpr ) {
		// use GCC extension to avoid bumping std to C11
		extension( alignofExpr );
		output << "__alignof__(";
		if ( alignofExpr->get_isType() ) {
			output << genType( alignofExpr->get_type(), "" );
		} else {
			alignofExpr->get_expr()->accept( *this );
		} // if
		output << ")";
	}

	void CodeGenerator::visit( UntypedOffsetofExpr *offsetofExpr ) {
		assert( false && "UntypedOffsetofExpr should not reach code generation." );
	}

	void CodeGenerator::visit( OffsetofExpr *offsetofExpr ) {
		// use GCC builtin
		output << "__builtin_offsetof(";
		output << genType( offsetofExpr->get_type(), "" );
		output << ", " << mangleName( offsetofExpr->get_member() );
		output << ")";
	}

	void CodeGenerator::visit( OffsetPackExpr *offsetPackExpr ) {
		assert( false && "OffsetPackExpr should not reach code generation." );
	}

	void CodeGenerator::visit( LogicalExpr *logicalExpr ) {
		extension( logicalExpr );
		output << "(";
		logicalExpr->get_arg1()->accept( *this );
		if ( logicalExpr->get_isAnd() ) {
			output << " && ";
		} else {
			output << " || ";
		} // if
		logicalExpr->get_arg2()->accept( *this );
		output << ")";
	}

	void CodeGenerator::visit( ConditionalExpr *conditionalExpr ) {
		extension( conditionalExpr );
		output << "(";
		conditionalExpr->get_arg1()->accept( *this );
		output << " ? ";
		conditionalExpr->get_arg2()->accept( *this );
		output << " : ";
		conditionalExpr->get_arg3()->accept( *this );
		output << ")";
	}

	void CodeGenerator::visit( CommaExpr *commaExpr ) {
		extension( commaExpr );
		output << "(";
		commaExpr->get_arg1()->accept( *this );
		output << " , ";
		commaExpr->get_arg2()->accept( *this );
		output << ")";
	}

	void CodeGenerator::visit( TupleExpr *tupleExpr ) {}

	void CodeGenerator::visit( TypeExpr *typeExpr ) {}

	void CodeGenerator::visit( AsmExpr *asmExpr ) {
		if ( asmExpr->get_inout() ) {
			output << "[ ";
			asmExpr->get_inout()->accept( *this );
			output << " ] ";
		} // if
		asmExpr->get_constraint()->accept( *this );
		output << " ( ";
		asmExpr->get_operand()->accept( *this );
		output << " )";
	}

	//*** Statements
	void CodeGenerator::visit( CompoundStmt *compoundStmt ) {
		std::list<Statement*> ks = compoundStmt->get_kids();
		output << "{" << endl;

		cur_indent += CodeGenerator::tabsize;

		for ( std::list<Statement *>::iterator i = ks.begin(); i != ks.end();  i++ ) {
			output << indent << printLabels( (*i)->get_labels() );
			(*i)->accept( *this );

			output << endl;
			if ( wantSpacing( *i ) ) {
				output << endl;
			}
		}
		cur_indent -= CodeGenerator::tabsize;

		output << indent << "}";
	}

	void CodeGenerator::visit( ExprStmt *exprStmt ) {
		assert( exprStmt );
		// cast the top-level expression to void to reduce gcc warnings.
		Expression * expr = new CastExpr( exprStmt->get_expr() );
		expr->accept( *this );
		output << ";";
	}

	void CodeGenerator::visit( AsmStmt *asmStmt ) {
		output << "asm ";
		if ( asmStmt->get_voltile() ) output << "volatile ";
		if ( ! asmStmt->get_gotolabels().empty()  ) output << "goto ";
		output << "( ";
		if ( asmStmt->get_instruction() ) asmStmt->get_instruction()->accept( *this );
		output << " : ";
		genCommaList( asmStmt->get_output().begin(), asmStmt->get_output().end() );
		output << " : ";
		genCommaList( asmStmt->get_input().begin(), asmStmt->get_input().end() );
		output << " : ";
		genCommaList( asmStmt->get_clobber().begin(), asmStmt->get_clobber().end() );
		if ( ! asmStmt->get_gotolabels().empty() ) {
			output << " : ";
			for ( std::list<Label>::iterator begin = asmStmt->get_gotolabels().begin();; ) {
				output << *begin++;
				if ( begin == asmStmt->get_gotolabels().end() ) break;
				output << ", ";
			} // for
		} // if
		output << " );" ;
	}

	void CodeGenerator::visit( IfStmt *ifStmt ) {
		output << "if ( ";
		ifStmt->get_condition()->accept( *this );
		output << " ) ";

		ifStmt->get_thenPart()->accept( *this );

		if ( ifStmt->get_elsePart() != 0) {
			output << " else ";
			ifStmt->get_elsePart()->accept( *this );
		} // if
	}

	void CodeGenerator::visit( SwitchStmt *switchStmt ) {
		output << "switch ( " ;
		switchStmt->get_condition()->accept( *this );
		output << " ) ";

		output << "{" << std::endl;
		cur_indent += CodeGenerator::tabsize;

		acceptAll( switchStmt->get_branches(), *this );

		cur_indent -= CodeGenerator::tabsize;

		output << indent << "}";
	}

	void CodeGenerator::visit( CaseStmt *caseStmt ) {
		output << indent;
		if ( caseStmt->isDefault()) {
			output << "default";
		} else {
			output << "case ";
			caseStmt->get_condition()->accept( *this );
		} // if
		output << ":\n";

		std::list<Statement *> sts = caseStmt->get_statements();

		cur_indent += CodeGenerator::tabsize;
		for ( std::list<Statement *>::iterator i = sts.begin(); i != sts.end();  i++) {
			output << indent << printLabels( (*i)->get_labels() )  ;
			(*i)->accept( *this );
			output << endl;
		}
		cur_indent -= CodeGenerator::tabsize;
	}

	void CodeGenerator::visit( BranchStmt *branchStmt ) {
		switch ( branchStmt->get_type()) {
		  case BranchStmt::Goto:
			if ( ! branchStmt->get_target().empty() )
				output << "goto " << branchStmt->get_target();
			else {
				if ( branchStmt->get_computedTarget() != 0 ) {
					output << "goto *";
					branchStmt->get_computedTarget()->accept( *this );
				} // if
			} // if
			break;
		  case BranchStmt::Break:
			output << "break";
			break;
		  case BranchStmt::Continue:
			output << "continue";
			break;
		}
		output << ";";
	}


	void CodeGenerator::visit( ReturnStmt *returnStmt ) {
		output << "return ";
		maybeAccept( returnStmt->get_expr(), *this );
		output << ";";
	}

	void CodeGenerator::visit( WhileStmt *whileStmt ) {
		if ( whileStmt->get_isDoWhile() ) {
			output << "do" ;
		} else {
			output << "while (" ;
			whileStmt->get_condition()->accept( *this );
			output << ")";
		} // if
		output << " ";

		output << CodeGenerator::printLabels( whileStmt->get_body()->get_labels() );
		whileStmt->get_body()->accept( *this );

		output << indent;

		if ( whileStmt->get_isDoWhile() ) {
			output << " while (" ;
			whileStmt->get_condition()->accept( *this );
			output << ");";
		} // if
	}

	void CodeGenerator::visit( ForStmt *forStmt ) {
		// initialization is always hoisted, so don't bother doing anything with that
		output << "for (;";

		if ( forStmt->get_condition() != 0 ) {
			forStmt->get_condition()->accept( *this );
		}
		output << ";";

		if ( forStmt->get_increment() != 0 ) {
			// cast the top-level expression to void to reduce gcc warnings.
			Expression * expr = new CastExpr( forStmt->get_increment() );
			expr->accept( *this );
		}
		output << ") ";

		if ( forStmt->get_body() != 0 ) {
			output << CodeGenerator::printLabels( forStmt->get_body()->get_labels() );
			forStmt->get_body()->accept( *this );
		} // if
	}

	void CodeGenerator::visit( NullStmt *nullStmt ) {
		//output << indent << CodeGenerator::printLabels( nullStmt->get_labels() );
		output << "/* null statement */ ;";
	}

	void CodeGenerator::visit( DeclStmt *declStmt ) {
		declStmt->get_decl()->accept( *this );

		if ( doSemicolon( declStmt->get_decl() ) ) {
			output << ";";
		} // if
	}

	void CodeGenerator::handleStorageClass( Declaration *decl ) {
		switch ( decl->get_storageClass() ) {
		  case DeclarationNode::Extern:
			output << "extern ";
			break;
		  case DeclarationNode::Static:
			output << "static ";
			break;
		  case DeclarationNode::Auto:
			// silently drop storage class
			break;
		  case DeclarationNode::Register:
			output << "register ";
			break;
		  case DeclarationNode::Inline:
			output << "inline ";
			break;
		  case DeclarationNode::Fortran:
			output << "fortran ";
			break;
		  case DeclarationNode::Noreturn:
			output << "_Noreturn ";
			break;
		  case DeclarationNode::Threadlocal:
			output << "_Thread_local ";
			break;
		  case DeclarationNode::NoStorageClass:
			break;
		} // switch
	}
} // namespace CodeGen

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
