//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// GenPoly.cc -- 
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Dec 15 16:11:18 2015
// Update Count     : 13
//

#include "GenPoly.h"

#include "SymTab/Mangler.h"
#include "SynTree/Expression.h"
#include "SynTree/Type.h"

#include <iostream>
using namespace std;

namespace GenPoly {
	bool needsAdapter( FunctionType *adaptee, const TyVarMap &tyVars ) {
		if ( ! adaptee->get_returnVals().empty() && isPolyType( adaptee->get_returnVals().front()->get_type(), tyVars ) ) {
			return true;
		} // if
		for ( std::list< DeclarationWithType* >::const_iterator innerArg = adaptee->get_parameters().begin(); innerArg != adaptee->get_parameters().end(); ++innerArg ) {
			if ( isPolyType( (*innerArg)->get_type(), tyVars ) ) {
				return true;
			} // if
		} // for
		return false;
	}

	ReferenceToType *isPolyRet( FunctionType *function ) {
		if ( ! function->get_returnVals().empty() ) {
			TyVarMap forallTypes;
			makeTyVarMap( function, forallTypes );
			return (ReferenceToType*)isPolyType( function->get_returnVals().front()->get_type(), forallTypes );
		} // if
		return 0;
	}

	namespace {
		/// Checks a parameter list for polymorphic parameters; will substitute according to env if present
		bool hasPolyParams( std::list< Expression* >& params, const TypeSubstitution *env ) {
			for ( std::list< Expression* >::iterator param = params.begin(); param != params.end(); ++param ) {
				TypeExpr *paramType = dynamic_cast< TypeExpr* >( *param );
				assert(paramType && "Aggregate parameters should be type expressions");
				if ( isPolyType( paramType->get_type(), env ) ) return true;
			}
			return false;
		}

		/// Checks a parameter list for polymorphic parameters from tyVars; will substitute according to env if present
		bool hasPolyParams( std::list< Expression* >& params, const TyVarMap &tyVars, const TypeSubstitution *env ) {
			for ( std::list< Expression* >::iterator param = params.begin(); param != params.end(); ++param ) {
				TypeExpr *paramType = dynamic_cast< TypeExpr* >( *param );
				assert(paramType && "Aggregate parameters should be type expressions");
				if ( isPolyType( paramType->get_type(), tyVars, env ) ) return true;
			}
			return false;
		}
	}

	Type *isPolyType( Type *type, const TypeSubstitution *env ) {
		if ( TypeInstType *typeInst = dynamic_cast< TypeInstType * >( type ) ) {
			if ( env ) {
				if ( Type *newType = env->lookup( typeInst->get_name() ) ) {
					return isPolyType( newType, env );
				} // if
			} // if
			return type;
		} else if ( StructInstType *structType = dynamic_cast< StructInstType* >( type ) ) {
			if ( hasPolyParams( structType->get_parameters(), env ) ) return type;
		} else if ( UnionInstType *unionType = dynamic_cast< UnionInstType* >( type ) ) {
			if ( hasPolyParams( unionType->get_parameters(), env ) ) return type;
		}
		return 0;
	}
	
	Type *isPolyType( Type *type, const TyVarMap &tyVars, const TypeSubstitution *env ) {
		if ( TypeInstType *typeInst = dynamic_cast< TypeInstType * >( type ) ) {
			if ( env ) {
				if ( Type *newType = env->lookup( typeInst->get_name() ) ) {
					return isPolyType( newType, tyVars, env );
				} // if
			} // if
			if ( tyVars.find( typeInst->get_name() ) != tyVars.end() ) {
				return type;
			}
		} else if ( StructInstType *structType = dynamic_cast< StructInstType* >( type ) ) {
			if ( hasPolyParams( structType->get_parameters(), tyVars, env ) ) return type;
		} else if ( UnionInstType *unionType = dynamic_cast< UnionInstType* >( type ) ) {
			if ( hasPolyParams( unionType->get_parameters(), tyVars, env ) ) return type;
		}
		return 0;
	}

	Type *isPolyPtr( Type *type, const TypeSubstitution *env ) {
		if ( PointerType *ptr = dynamic_cast< PointerType *>( type ) ) {
			return isPolyType( ptr->get_base(), env );
		} else if ( env ) {
			if ( TypeInstType *typeInst = dynamic_cast< TypeInstType *>( type ) ) {
				if ( Type *newType = env->lookup( typeInst->get_name() ) ) {
					return isPolyPtr( newType, env );
				} // if
			} // if
		} // if
		return 0;
	}
	
	Type *isPolyPtr( Type *type, const TyVarMap &tyVars, const TypeSubstitution *env ) {
		if ( PointerType *ptr = dynamic_cast< PointerType *>( type ) ) {
			return isPolyType( ptr->get_base(), tyVars, env );
		} else if ( env ) {
			if ( TypeInstType *typeInst = dynamic_cast< TypeInstType *>( type ) ) {
				if ( Type *newType = env->lookup( typeInst->get_name() ) ) {
					return isPolyPtr( newType, tyVars, env );
				} // if
			} // if
		} // if
		return 0;
	}

	FunctionType * getFunctionType( Type *ty ) {
		PointerType *ptrType;
		if ( ( ptrType = dynamic_cast< PointerType* >( ty ) ) ) {
			return dynamic_cast< FunctionType* >( ptrType->get_base() ); // pointer if FunctionType, NULL otherwise
		} else {
			return dynamic_cast< FunctionType* >( ty ); // pointer if FunctionType, NULL otherwise
		}
	}

	void makeTyVarMap( Type *type, TyVarMap &tyVarMap ) {
		for ( std::list< TypeDecl* >::const_iterator tyVar = type->get_forall().begin(); tyVar != type->get_forall().end(); ++tyVar ) {
			assert( *tyVar );
			tyVarMap[ (*tyVar)->get_name() ] = (*tyVar)->get_kind();
		}
		if ( PointerType *pointer = dynamic_cast< PointerType* >( type ) ) {
			makeTyVarMap( pointer->get_base(), tyVarMap );
		}
	}
	
	void printTyVarMap( std::ostream &os, const TyVarMap &tyVarMap ) {
		for ( TyVarMap::const_iterator i = tyVarMap.begin(); i != tyVarMap.end(); ++i ) {
			os << i->first << " (" << i->second << ") ";
		} // for
		os << std::endl;
	}

	std::string sizeofName( Type *ty ) {
		return std::string( "_sizeof_" ) + SymTab::Mangler::mangleType( ty );
	}

	std::string alignofName( Type *ty ) {
		return std::string( "_alignof_" ) + SymTab::Mangler::mangleType( ty );
	}
} // namespace GenPoly

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
