//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// TupleAssignment.cc --
//
// Author           : Rodolfo G. Esteves
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon May 18 15:02:53 2015
// Update Count     : 2
//

#include "ResolvExpr/AlternativeFinder.h"
#include "ResolvExpr/Alternative.h"
#include "ResolvExpr/typeops.h"
#include "SynTree/Expression.h"
#include "SynTree/Initializer.h"
#include "Tuples.h"
#include "Common/SemanticError.h"

#include <functional>
#include <algorithm>
#include <iterator>
#include <iostream>
#include <cassert>
#include <set>
#include <unordered_set>

namespace Tuples {
	class TupleAssignSpotter {
	  public:
		// dispatcher for Tuple (multiple and mass) assignment operations
		TupleAssignSpotter( ResolvExpr::AlternativeFinder & );
		void spot( UntypedExpr * expr, std::list<ResolvExpr::AltList> &possibilities );

	  private:
		void match();
		// records for assignment generation
		struct Options {
			void print( std::ostream & );
			int size() const { return options.size(); }
			bool empty() const { return options.empty(); }
			typedef std::list< ResolvExpr::AltList >::iterator iterator;
			iterator begin() { return options.begin(); }
			iterator end() { return options.end(); }

			std::list< ResolvExpr::AltList > options;
		};

		struct Matcher {
		  public:
			Matcher( TupleAssignSpotter &spotter, Expression *_lhs, Expression *_rhs );
			virtual ~Matcher() {}
			virtual void match( std::list< Expression * > &out ) = 0;
			std::list< Expression * > lhs, rhs;
			TupleAssignSpotter &spotter;
			std::list< ObjectDecl * > tmpDecls;
		};

		struct MassAssignMatcher : public Matcher {
		  public:
			MassAssignMatcher( TupleAssignSpotter &spotter, Expression *lhs, Expression *rhs ) : Matcher( spotter, lhs, rhs ) {
				this->rhs.push_back( rhs );
			}
			virtual void match( std::list< Expression * > &out );
		};

		struct MultipleAssignMatcher : public Matcher {
		  public:
			MultipleAssignMatcher( TupleAssignSpotter &spot, Expression *lhs, Expression *rhs );
			virtual void match( std::list< Expression * > &out );
		};

		ResolvExpr::AlternativeFinder &currentFinder;
		// Expression *rhs, *lhs;
		Matcher *matcher = nullptr;
		Options options;
	};

	bool isTupleVar( DeclarationWithType *decl ) {
		return dynamic_cast< TupleType * >( decl->get_type() );
	}

	/// true if expr is an expression of tuple type, i.e. a tuple expression, tuple variable, or MRV (multiple-return-value) function
	bool isTuple( Expression *expr ) {
		if ( ! expr ) return false;
		assert( expr->has_result() );
		// xxx - used to include cast to varExpr and call to isTupleVar, but this doesn't seem like it should be necessary
		return dynamic_cast<TupleExpr *>(expr) || expr->get_result()->size() > 1;
	}

	bool pointsToTuple( Expression *expr ) {
		// also check for function returning tuple of reference types
		if ( AddressExpr *addr = dynamic_cast< AddressExpr * >( expr) ) {
			return isTuple( addr->get_arg() );
		}
		return false;
	}

	bool isTupleExpr( Expression *expr ) {
		assert( expr->has_result() );
		return expr->get_result()->size() > 1;
	}

	void handleTupleAssignment( ResolvExpr::AlternativeFinder & currentFinder, UntypedExpr * expr, std::list<ResolvExpr::AltList> &possibilities ) {
		TupleAssignSpotter spotter( currentFinder );
		spotter.spot( expr, possibilities );
	}

	TupleAssignSpotter::TupleAssignSpotter( ResolvExpr::AlternativeFinder &f )
		: currentFinder(f) {}

	void TupleAssignSpotter::spot( UntypedExpr * expr, std::list<ResolvExpr::AltList> &possibilities ) {
		if (  NameExpr *assgnop = dynamic_cast< NameExpr * >(expr->get_function()) ) {
			if ( assgnop->get_name() == std::string("?=?") ) {
				for ( std::list<ResolvExpr::AltList>::iterator ali = possibilities.begin(); ali != possibilities.end(); ++ali ) {
					assert( ali->size() == 2 );
					ResolvExpr::Alternative op1 = ali->front(), op2 = ali->back();

					MultipleAssignMatcher multiMatcher( *this, op1.expr, op2.expr );
					MassAssignMatcher massMatcher( *this, op1.expr, op2.expr );
					if ( pointsToTuple(op1.expr) ) { // also handles tuple vars
						if ( isTuple( op2.expr ) ) {
							matcher = &multiMatcher;
						} else {
							// mass assignment
							matcher = &massMatcher;
						}
						match();
					} else if ( isTuple( op2.expr ) ) {
						throw SemanticError("Cannot assign a tuple value into a non-tuple lvalue.", expr);
					}
				}
			}
		}
	}

	void TupleAssignSpotter::match() {
		assert ( matcher != 0 );

		std::list< Expression * > new_assigns;
		matcher->match( new_assigns );

		if ( new_assigns.empty() ) return;
		ResolvExpr::AltList current;
		// now resolve new assignments
		for ( std::list< Expression * >::iterator i = new_assigns.begin(); i != new_assigns.end(); ++i ) {
			ResolvExpr::AlternativeFinder finder( currentFinder.get_indexer(), currentFinder.get_environ() );
			finder.findWithAdjustment(*i);
			// prune expressions that don't coincide with
			ResolvExpr::AltList alts = finder.get_alternatives();
			assert( alts.size() == 1 );
			assert( alts.front().expr != 0 );
			current.push_back( alts.front() );
		}

		// extract expressions from the assignment alternatives to produce a list of assignments that
		// together form a single alternative
		std::list< Expression *> solved_assigns;
		for ( ResolvExpr::Alternative & alt : current ) {
			solved_assigns.push_back( alt.expr->clone() );
		}
		// xxx - need to do this??
		// TypeEnvironment compositeEnv;
		// simpleCombineEnvironments( i->begin(), i->end(), compositeEnv );
		currentFinder.get_alternatives().push_front( ResolvExpr::Alternative(new TupleAssignExpr(solved_assigns, matcher->tmpDecls), currentFinder.get_environ(), ResolvExpr::sumCost( current ) ) );
	}

	TupleAssignSpotter::Matcher::Matcher( TupleAssignSpotter &spotter, Expression *lhs, Expression *rhs ) : spotter(spotter) {
		// xxx - shouldn't need to be &<tuple-expr>, just &<lvalue-tuple-type>
		if (AddressExpr *addr = dynamic_cast<AddressExpr *>(lhs) )
			if ( TupleExpr *tuple = dynamic_cast<TupleExpr *>(addr->get_arg()) )
				std::copy( tuple->get_exprs().begin(), tuple->get_exprs().end(), back_inserter(this->lhs) );
	}

	TupleAssignSpotter::MultipleAssignMatcher::MultipleAssignMatcher( TupleAssignSpotter &spotter, Expression *lhs, Expression *rhs ) : Matcher( spotter, lhs, rhs ) {

		if ( TupleExpr *tuple = dynamic_cast<TupleExpr *>(rhs) )
			std::copy( tuple->get_exprs().begin(), tuple->get_exprs().end(), back_inserter(this->rhs) );
	}

	UntypedExpr * createAssgn( ObjectDecl *left, ObjectDecl *right ) {
		assert( left && right );
		std::list< Expression * > args;
		args.push_back( new AddressExpr( new UntypedExpr( new NameExpr("*?"), std::list< Expression * >{ new VariableExpr( left ) } ) ) );
		args.push_back( new VariableExpr( right ) );
		return new UntypedExpr( new NameExpr( "?=?" ), args );
	}

	ObjectDecl * newObject( UniqueName & namer, Expression * expr ) {
		assert( expr->has_result() && ! expr->get_result()->isVoid() );
		return new ObjectDecl( namer.newName(), DeclarationNode::NoStorageClass, LinkageSpec::Cforall, nullptr, expr->get_result()->clone(), new SingleInit( expr->clone() ) );
	}

	void TupleAssignSpotter::MassAssignMatcher::match( std::list< Expression * > &out ) {
		static UniqueName lhsNamer( "__massassign_L" );
		static UniqueName rhsNamer( "__massassign_R" );
		assert ( ! lhs.empty() && rhs.size() == 1);

		ObjectDecl * rtmp = newObject( rhsNamer, rhs.front() );
		for ( Expression * l : lhs ) {
			ObjectDecl * ltmp = newObject( lhsNamer, new AddressExpr( l ) );
			out.push_back( createAssgn( ltmp, rtmp ) );
			tmpDecls.push_back( ltmp );
		}
		tmpDecls.push_back( rtmp );
	}

	void TupleAssignSpotter::MultipleAssignMatcher::match( std::list< Expression * > &out ) {
		static UniqueName lhsNamer( "__multassign_L" );
		static UniqueName rhsNamer( "__multassign_R" );
		// xxx - need more complicated matching?
		if ( lhs.size() == rhs.size() ) {
			std::list< ObjectDecl * > ltmp;
			std::list< ObjectDecl * > rtmp;
			std::transform( lhs.begin(), lhs.end(), back_inserter( ltmp ), []( Expression * expr ){
				return newObject( lhsNamer, new AddressExpr( expr ) );
			});
			std::transform( rhs.begin(), rhs.end(), back_inserter( rtmp ), []( Expression * expr ){
				return newObject( rhsNamer, expr );
			});
			zipWith( ltmp.begin(), ltmp.end(), rtmp.begin(), rtmp.end(), back_inserter(out), createAssgn );
			tmpDecls.splice( tmpDecls.end(), ltmp );
			tmpDecls.splice( tmpDecls.end(), rtmp );
		}
	}

	void TupleAssignSpotter::Options::print( std::ostream &ostr ) {
		for ( ResolvExpr::AltList & l : options ) {
			for ( ResolvExpr::Alternative & alt : l ) {
				alt.print( ostr );
				ostr << " ";
			}
			ostr << std::endl;
		} // for
	}
} // namespace Tuples

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
