package main

import (
	"context"
	"flag"
	"fmt"
	"math/rand"
	"os"
	"syscall"
	"sync/atomic"
	"time"
	"unsafe"
	"golang.org/x/sync/semaphore"
	"golang.org/x/text/language"
	"golang.org/x/text/message"
)

// ==================================================
type MyData struct {
	ttid int
	id int
	data [] uint64
}

func NewData(id int, size uint64) (*MyData) {
	var data [] uint64
	data = make([]uint64, size)
	for i := uint64(0); i < size; i++ {
		data[i] = 0
	}
	return &MyData{syscall.Gettid(), id, data}
}

func (this * MyData) moved( ttid int ) (uint64) {
	if this.ttid == ttid {
		return 0
	}
	this.ttid = ttid
	return 1
}

func (this * MyData) access( idx uint64 ) {
	this.data[idx % uint64(len(this.data))] += 1
}

// ==================================================
type MyCtx struct {
	s * semaphore.Weighted
	d unsafe.Pointer
	c context.Context
	ttid int
	id int
}

func NewCtx( sem * semaphore.Weighted, data * MyData, id int ) (MyCtx) {
	return MyCtx{sem, unsafe.Pointer(data), context.Background(), syscall.Gettid(), id}
}

func (this * MyCtx) moved( ttid int ) (uint64) {
	if this.ttid == ttid {
		return 0
	}
	this.ttid = ttid
	return 1
}

// ==================================================
type Spot struct {
	ptr uintptr
	id int
}

// Main handshake of the code
// Single seat, first thread arriving waits
// Next threads unblocks current one and blocks in its place
// if share == true, exchange data in the process
func (this * Spot) put( ctx * MyCtx, data * MyData, share bool) (* MyData, bool) {
	new := uintptr(unsafe.Pointer(ctx))
	// old_d := ctx.d
	var raw uintptr
	for true {
		raw = this.ptr
		if raw == uintptr(1) {
			return nil, true
		}
		if atomic.CompareAndSwapUintptr(&this.ptr, raw, new) {
			break
		}
	}

	if raw != uintptr(0) {
		var val *MyCtx
		val = (*MyCtx)(unsafe.Pointer(raw))
		if share {
			// fmt.Printf("[%d] - %d update %d: %p -> %p\n", this.id, ctx.id, val.id, val.d, data)
			atomic.StorePointer(&val.d, unsafe.Pointer(data))
		}

		// fmt.Printf("[%d] - %d release %d\n", this.id, ctx.id, val.id)
		val.s.Release(1)
	}

	// fmt.Printf("[%d] - %d enter\n", this.id, ctx.id)
	ctx.s.Acquire(ctx.c, 1)
	ret := (* MyData)(atomic.LoadPointer(&ctx.d))
	// fmt.Printf("[%d] - %d leave: %p -> %p\n", this.id, ctx.id, ret, old_d)

	return ret, false
}

func (this * Spot) release() {
	val := (*MyCtx)(unsafe.Pointer(atomic.SwapUintptr(&this.ptr, uintptr(1))))
	if val == nil {
		return
	}

	val.s.Release(1)
}

// ==================================================
type Result struct {
	count uint64
	gmigs uint64
	dmigs uint64
}

func NewResult() (Result) {
	return Result{0, 0, 0}
}

// ==================================================
func __xorshift64( state * uint64 ) (uint64) {
	x := *state
	x ^= x << 13
	x ^= x >> 7
	x ^= x << 17
	*state = x
	return x
}

func work(data * MyData, cnt uint64, state * uint64) {
	for i := uint64(0); i < cnt; i++ {
		data.access(__xorshift64(state))
	}
}

func local(result chan Result, start chan struct{}, size uint64, cnt uint64, channels [] Spot, share bool, id int) {
    	state := rand.Uint64()

	data := NewData(id, size)
	sem := semaphore.NewWeighted(1)
	sem.Acquire(context.Background(), 1)
	ctx := NewCtx(sem, data, id)

	r := NewResult()
	<- start
	for true {
		work(data, cnt, &state)

		i := __xorshift64(&state) % uint64(len(channels))
		var closed bool
		data, closed = channels[i].put(&ctx, data, share)

		if closed { break }
		if  clock_mode && atomic.LoadInt32(&stop) == 1 { break }
		if !clock_mode && r.count >= stop_count { break }
		if uint64(len(data.data)) != size {
			panic("Data has weird size")
		}

		ttid := syscall.Gettid()
		r.count += 1
		r.gmigs += ctx .moved(ttid)
		r.dmigs += data.moved(ttid)
	}

	atomic.AddInt64(&threads_left, -1);
	result <- r
}

func main() {
	work_sizeOpt := flag.Uint64("w", 2    , "Number of words (uint64) per threads")
	countOpt     := flag.Uint64("c", 2    , "Number of words (uint64) to touch")
	shareOpt     := flag.Bool  ("s", false, "Pass the work data to the next thread when blocking")

	defer bench_init()()

	size  := *work_sizeOpt
	cnt   := *countOpt
	share := *shareOpt

	if ! (nthreads > nprocs) {
		fmt.Fprintf(os.Stderr, "Must have more threads than procs\n")
		os.Exit(1)
	}

	barrierStart := make(chan struct{})
	threads_left = int64(nprocs)
	result  := make(chan Result)
	channels := make([]Spot, nthreads - nprocs)
	for i := range channels {
		channels[i] = Spot{uintptr(0), i}
	}

	for i := 0; i < nthreads; i++ {
		go local(result, barrierStart, size, cnt, channels, share, i)
	}
	fmt.Printf("Starting\n");

	atomic.StoreInt32(&stop, 0)
	start := time.Now()
	close(barrierStart)

	wait(start, true);

	atomic.StoreInt32(&stop, 1)
	end := time.Now()
	delta := end.Sub(start)

	fmt.Printf("\nDone\n")

	for i := range channels {
		channels[i].release()
	}

	global_result := NewResult()
	for i := 0; i < nthreads; i++ {
		r := <- result
		global_result.count += r.count
		global_result.gmigs += r.gmigs
		global_result.dmigs += r.dmigs
	}

	p := message.NewPrinter(language.English)
	p.Printf("Duration (ms)          : %f\n", delta.Seconds());
	p.Printf("Number of processors   : %d\n", nprocs);
	p.Printf("Number of threads      : %d\n", nthreads);
	p.Printf("Work size (64bit words): %d\n", size);
	p.Printf("Total Operations(ops)  : %15d\n", global_result.count)
	p.Printf("Total G Migrations     : %15d\n", global_result.gmigs)
	p.Printf("Total D Migrations     : %15d\n", global_result.dmigs)
	p.Printf("Ops per second         : %18.2f\n", float64(global_result.count) / delta.Seconds())
	p.Printf("ns per ops             : %18.2f\n", float64(delta.Nanoseconds()) / float64(global_result.count))
	p.Printf("Ops per threads        : %15d\n", global_result.count / uint64(nthreads))
	p.Printf("Ops per procs          : %15d\n", global_result.count / uint64(nprocs))
	p.Printf("Ops/sec/procs          : %18.2f\n", (float64(global_result.count) / float64(nprocs)) / delta.Seconds())
	p.Printf("ns per ops/procs       : %18.2f\n", float64(delta.Nanoseconds()) / (float64(global_result.count) / float64(nprocs)))
}