/*                               -*- Mode: C -*- 
 * 
 * Pretty Printer Lexer, Copyright (C) Rodolfo Gabriel Esteves and Peter A. Buhr 2001
 *      Permission is granted to copy this grammar and to use it within software systems.
 *      THIS GRAMMAR IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
 * 
 * lex.l -- 
 * 
 * Author           : Rodolfo Gabriel Esteves
 * Created On       : Sat Dec 15 11:45:59 2001
 * Last Modified By : Peter A. Buhr
 * Last Modified On : Mon Apr  8 15:48:24 2002
 * Update Count     : 245
 */

%option stack
%option yylineno

%{
#include <list>
#include <string>
#include <iostream>

#include "parse.h" 
#include "yacc.tab.h" 

#define RETURN_TOKEN( kind ) yylval.tokenp = new Token( yytext, ws_list, kind ); return kind;

list<string> ws_list;
string comment_str;
string code_str;
%}

integer [0-9]+
identifier [a-zA-Z_$][0-9a-zA-Z_$]*

simple_escape ['"?\\]
escape_sequence [\\]{simple_escape}
c_char [^'\\\n]|{escape_sequence}
s_char [^"\\\n]|{escape_sequence}

%x C_COMMENT STR C_CODE

/* ---------------------------- Token Section ---------------------------- */
%%
<INITIAL,C_CODE>"/*"	{				/* C style comments */
			#if defined(DEBUG_ALL) | defined(DEBUG_COMMENT)
    			    cerr << "\"/*\" : " << yytext << endl;
			#endif
			    if ( YYSTATE == C_CODE ) code_str += yytext;
			    else comment_str += yytext;
			    yy_push_state(C_COMMENT);
			}
<C_COMMENT>(.|"\n")	{				/* C style comments */
			#if defined(DEBUG_ALL) | defined(DEBUG_COMMENT)
    			    cerr << "<C_COMMENT>(.|\\n) : " << yytext << endl;
			#endif
			    if ( yy_top_state() == C_CODE ) code_str += yytext;
			    else comment_str += yytext;
			}
<C_COMMENT>"*/"		{				/* C style comments */
			#if defined(DEBUG_ALL) | defined(DEBUG_COMMENT)
    			    cerr << "<C_COMMENT>\"*/\" : " << yytext << endl;
			#endif
			    if ( yy_top_state() == C_CODE ) code_str += yytext;
			    else {
				comment_str += yytext;
				//cerr << "C COMMENT : " << endl << comment_str << endl;
				ws_list.push_back( comment_str );
				comment_str = "";
			    }
			    yy_pop_state();
			}
<INITIAL,C_CODE>"//"[^\n]*"\n" {			/* C++ style comments */
			#if defined(DEBUG_ALL) | defined(DEBUG_COMMENT)
    			    cerr << "\"//\"[^\\n]*\"\n\" : " << yytext << endl;
			#endif
			    if ( YYSTATE == C_CODE ) code_str += yytext;
			    else {
				comment_str += yytext;
				//cerr << "C++ COMMENT : " << endl << comment_str << endl;
				ws_list.push_back( comment_str );
				comment_str = "";
			    }
			}

";"			{ RETURN_TOKEN( ';' ) }
":"			{ RETURN_TOKEN( ':' ) }
"|"			{ RETURN_TOKEN( '|' ) }
","			{ RETURN_TOKEN( ',' ) }
"<"			{ RETURN_TOKEN( '<' ) }
">"			{ RETURN_TOKEN( '>' ) }

[[:space:]]+		{				/* [ \t\n]+ */
			    ws_list.push_back( yytext );
			    //cerr << "WS : " << "\"" << yytext << "\"" << endl;
			}

<INITIAL>"{"		{ RETURN_TOKEN( '{' ) }
<INITIAL>"}"		{ RETURN_TOKEN( '}' ) }
<C_CODE>"{"		{
			#if defined(DEBUG_ALL) | defined(DEBUG_C)
			    cerr << "<C_CODE>. : " << yytext << endl;
			#endif
			    code_str += yytext;
			    RETURN_TOKEN( '{' )
			}
<C_CODE>"}"		{
			#if defined(DEBUG_ALL) | defined(DEBUG_C)
			    cerr << "<C_CODE>. : " << yytext << endl;
			#endif
			    code_str += yytext;
			    RETURN_TOKEN( '}' )
			}

"%%"			{ RETURN_TOKEN( MARK ) }
"%{"			{ RETURN_TOKEN( LCURL ) }
<C_CODE>"%}"		{ RETURN_TOKEN( RCURL ) }

^"%union"       	{ RETURN_TOKEN( UNION ) }
^"%start"       	{ RETURN_TOKEN( START ) }
^"%token"       	{ RETURN_TOKEN( TOKEN ) }
^"%type"	       	{ RETURN_TOKEN( TYPE ) }
^"%left"	       	{ RETURN_TOKEN( LEFT ) }
^"%right"	       	{ RETURN_TOKEN( RIGHT ) }
^"%nonassoc"    	{ RETURN_TOKEN( NONASSOC ) }
^"%pure_parser"    	{ RETURN_TOKEN( PURE_PARSER ) }
^"%semantic_parser"    	{ RETURN_TOKEN( SEMANTIC_PARSER ) }
^"%expect"  	  	{ RETURN_TOKEN( EXPECT ) }
^"%thong" 	   	{ RETURN_TOKEN( THONG ) }

"%prec" 	   	{ RETURN_TOKEN( PREC ) }

{integer}	    	{ RETURN_TOKEN( INTEGER ); }
[']{c_char}[']    	{ RETURN_TOKEN( CHARACTER ); }
{identifier}    	{ RETURN_TOKEN( IDENTIFIER ); }

<C_CODE>["]{s_char}*["]	{				/* hide braces "{}" in strings */
			#if defined(DEBUG_ALL) | defined(DEBUG_C)
			    cerr << "<C_CODE>. : " << yytext << endl;
			#endif
			    code_str += yytext;
			}

<C_CODE>(.|\n)		{				/* must be last rule of C_CODE */
			#if defined(DEBUG_ALL) | defined(DEBUG_C)
			    cerr << "<C_CODE>. : " << yytext << endl;
			#endif
			    code_str += yytext;
			}

.			{ printf("UNKNOWN CHARACTER:%s\n", yytext); } /* unknown characters */
%%
void lexC(void) {
    BEGIN(C_CODE);
}

string lexYacc(void) {
    BEGIN(INITIAL);
    //cerr << "CODE: " << endl << code_str << endl;
    string temp( code_str );
    code_str = "";
    return temp;
}

/* Local Variables: */
/* fill-column: 100 */
/* compile-command: "gmake" */
/* End: */
