#pragma once
#include <cstddef>
#include <exception>
#include <iomanip>
#include <memory>
#include <ostream>
#include <string>
#include <typeinfo>
#include <typeindex>

class bad_cast : public std::exception {
	std::string why;
public:
	bad_cast( const std::type_index& f, const std::type_index& t ) : std::exception() {
		why = std::string{"bad cast of "} + f.name() + " to " + t.name();
	}
	~bad_cast() override = default;
	
	const char* what() const noexcept override { return why.c_str(); }
};

template<typename T> std::type_index class_of() { return { typeid(T) }; }

template<typename T> using ptr = std::unique_ptr<T>;

struct object {
	std::type_index get_class() const { return { this ? typeid(*this) : typeid(std::nullptr_t) }; }

	template<typename T> T& as() {
		T* p = dynamic_cast<T*>(this);
		if ( !p ) throw bad_cast{ get_class(), class_of<T>() };
		return *p;
	}

	template<typename T> const T& as() const {
		const T* p = dynamic_cast<const T*>(this);
		if ( !p ) throw bad_cast{ get_class(), class_of<T>() };
		return *p;
	}

	virtual ptr<object> new_inst() const = 0;
	virtual ptr<object> new_copy() const = 0;
	virtual object& operator= (const object&) = 0;
	virtual ~object() = default;
};

template<typename T, typename... Args> static inline ptr<T> make(Args&&... args) {
	return std::make_unique<T>(std::forward<Args>(args)...);
}

template<typename To, typename From> 
ptr<To> as_ptr( ptr<From>&& p ) { return ptr<To>{ &p.release()->template as<To>() }; }

struct ordered : public virtual object {
	virtual int cmp(const ordered&) const = 0;

	bool operator< (const ordered& that) const { return cmp(that) < 0; }
	bool operator<= ( const ordered& that ) const { return cmp(that) <= 0; }
	bool operator== ( const ordered& that ) const { return cmp(that) == 0; }
	bool operator!= ( const ordered& that ) const { return cmp(that) != 0; }
	bool operator> ( const ordered& that ) const { return cmp(that) > 0; }
	bool operator>= ( const ordered& that ) const { return cmp(that) >= 0; }
};

struct printable : public virtual object {
	virtual void print(std::ostream&) const = 0;
};

class boolean : public ordered, public printable {
	bool x;
public:
	boolean() = default;
	boolean(bool x) : x(x) {}
	ptr<object> new_inst() const override { return make<boolean>(); }
	ptr<object> new_copy() const override { return make<boolean>(*this); }
	boolean& operator= (const boolean& that) {
		x = that.x;
		return *this;	
	}
	object& operator= (const object& that) override { return *this = that.as<boolean>(); } /***/
	~boolean() override = default;

	int cmp(const boolean& that) const { return x == that.x ? 0 : x == false ? -1 : 1; }
	int cmp(const ordered& that) const override { return cmp( that.as<boolean>() ); } /***/

	void print(std::ostream& out) const override { out << (x ? "true" : "false"); }
};

class character : public ordered, public printable {
	char x;
public:
	character() = default;
	character(char x) : x(x) {}
	ptr<object> new_inst() const override { return make<character>(); }
	ptr<object> new_copy() const override { return make<character>(*this); }
	character& operator= (const character& that) {
		x = that.x;
		return *this;	
	}
	object& operator= (const object& that) override { return *this = that.as<character>(); } /***/
	~character() override = default;

	int cmp(const character& that) const { return x == that.x ? 0 : x < that.x ? -1 : 1; }
	int cmp(const ordered& that) const override { return cmp( that.as<character>() ); } /***/

	void print(std::ostream& out) const override {
		if ( 0x20 <= x && x <= 0x7E ) { out << "'" << x << "'"; }
		else { out << "'\\" << std::hex << (unsigned int)x << std::setbase(0) << "'"; }
	}
};

class integer : public ordered, public printable {
	int x;
public:
	integer() = default;
	integer(int x) : x(x) {}
	ptr<object> new_inst() const override { return make<integer>(); }
	ptr<object> new_copy() const override { return make<integer>(*this); }
	integer& operator= (const integer& that) {
		x = that.x;
		return *this;	
	}
	object& operator= (const object& that) override { return *this = that.as<integer>(); } /***/
	~integer() override = default;

	int cmp(const integer& that) const { return x == that.x ? 0 : x < that.x ? -1 : 1; }
	int cmp(const ordered& that) const override { return cmp( that.as<integer>() ); } /***/

	void print(std::ostream& out) const override { out << x; }
};

class c_string : public printable {
	static constexpr const char* empty = "";
	const char* s;
public:
	c_string() : s(empty) {}
	c_string(const char* s) : s(s) {}
	ptr<object> new_inst() const override { return make<c_string>(); }
	ptr<object> new_copy() const override { return make<c_string>(s); }
	c_string& operator= (const c_string& that) {
		s = that.s;
		return *this;
	}
	object& operator= (const object& that) override { return *this = that.as<c_string>(); } /***/
	~c_string() override = default;

	void print(std::ostream& out) const override { out << s; }
};

class pair : public ordered, public printable {
	ptr<object> x;
	ptr<object> y;
public:
	pair() = default;
	pair(ptr<object>&& x, ptr<object>&& y) : x(std::move(x)), y(std::move(y)) {}
	ptr<object> new_inst() const override { return make<pair>(); }
	ptr<object> new_copy() const override { return make<pair>(x->new_copy(), y->new_copy()); }
	pair& operator= (const pair& that) {
		x = that.x->new_copy();
		y = that.y->new_copy();
		return *this;
	}
	object& operator= (const object& that) override { return *this = that.as<pair>(); } /***/
	~pair() override = default;

	int cmp(const pair& that) const {
		int c = x->as<ordered>().cmp( that.x->as<ordered>() ); /***/
		if ( c != 0 ) return c;
		return y->as<ordered>().cmp( that.y->as<ordered>() ); /***/
	}
	int cmp(const ordered& that) const override { return cmp( that.as<pair>() ); }

	void print(std::ostream& out) const override {
		out << "[";
		x->as<printable>().print(out); /***/
		out << ", ";
		y->as<printable>().print(out); /***/
		out << "]";
	}
};
