//
// Cforall Version 1.0.0 Copyright (C) 2016 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// InstantiateGeneric.cc --
//
// Author           : Aaron B. Moss
// Created On       : Thu Aug 04 18:33:00 2016
// Last Modified By : Aaron B. Moss
// Last Modified On : Thu Aug 04 18:33:00 2016
// Update Count     : 1
//

#include <cassert>
#include <list>
#include <utility>
#include <vector>
#include <unordered_map>

#include "InstantiateGeneric.h"

#include "DeclMutator.h"
#include "GenPoly.h"
#include "ScopedSet.h"
#include "PolyMutator.h"

#include "ResolvExpr/typeops.h"

#include "SynTree/Declaration.h"
#include "SynTree/Expression.h"
#include "SynTree/Type.h"

#include "Common/ScopedMap.h"
#include "Common/UniqueName.h"
#include "Common/utility.h"

namespace GenPoly {

	/// Abstracts type equality for a list of parameter types
	struct TypeList {
		TypeList() : params() {}
		TypeList( const std::list< Type* > &_params ) : params() { cloneAll(_params, params); }
		TypeList( std::list< Type* > &&_params ) : params( _params ) {}

		TypeList( const TypeList &that ) : params() { cloneAll(that.params, params); }
		TypeList( TypeList &&that ) : params( std::move( that.params ) ) {}

		/// Extracts types from a list of TypeExpr*
		TypeList( const std::list< TypeExpr* >& _params ) : params() {
			for ( std::list< TypeExpr* >::const_iterator param = _params.begin(); param != _params.end(); ++param ) {
				params.push_back( (*param)->get_type()->clone() );
			}
		}

		TypeList& operator= ( const TypeList &that ) {
			deleteAll( params );

			params.clear();
			cloneAll( that.params, params );

			return *this;
		}

		TypeList& operator= ( TypeList &&that ) {
			deleteAll( params );

			params = std::move( that.params );

			return *this;
		}

		~TypeList() { deleteAll( params ); }

		bool operator== ( const TypeList& that ) const {
			if ( params.size() != that.params.size() ) return false;

			SymTab::Indexer dummy;
			for ( std::list< Type* >::const_iterator it = params.begin(), jt = that.params.begin(); it != params.end(); ++it, ++jt ) {
				if ( ! ResolvExpr::typesCompatible( *it, *jt, dummy ) ) return false;
			}
			return true;
		}

		std::list< Type* > params;  ///< Instantiation parameters
	};

	/// Maps a key and a TypeList to the some value, accounting for scope
	template< typename Key, typename Value >
	class InstantiationMap {
		/// Wraps value for a specific (Key, TypeList) combination
		typedef std::pair< TypeList, Value* > Instantiation;
		/// List of TypeLists paired with their appropriate values
		typedef std::vector< Instantiation > ValueList;
		/// Underlying map type; maps keys to a linear list of corresponding TypeLists and values
		typedef ScopedMap< Key*, ValueList > InnerMap;

		InnerMap instantiations;  ///< instantiations

	public:
		/// Starts a new scope
		void beginScope() { instantiations.beginScope(); }

		/// Ends a scope
		void endScope() { instantiations.endScope(); }

		/// Gets the value for the (key, typeList) pair, returns NULL on none such.
		Value *lookup( Key *key, const std::list< TypeExpr* >& params ) const {
			TypeList typeList( params );

			// scan scopes for matches to the key
			for ( typename InnerMap::const_iterator insts = instantiations.find( key ); insts != instantiations.end(); insts = instantiations.findNext( insts, key ) ) {
				for ( typename ValueList::const_reverse_iterator inst = insts->second.rbegin(); inst != insts->second.rend(); ++inst ) {
					if ( inst->first == typeList ) return inst->second;
				}
			}
			// no matching instantiations found
			return 0;
		}

		/// Adds a value for a (key, typeList) pair to the current scope
		void insert( Key *key, const std::list< TypeExpr* > &params, Value *value ) {
			instantiations[ key ].push_back( Instantiation( TypeList( params ), value ) );
		}
	};

	/// Possible options for a given specialization of a generic type
	enum class genericType {
		dtypeStatic,  ///< Concrete instantiation based solely on {d,f}type-to-void conversions
		concrete,     ///< Concrete instantiation requiring at least one parameter type
		dynamic       ///< No concrete instantiation
	};

	genericType& operator |= ( genericType& gt, const genericType& ht ) {
		switch ( gt ) {
		case genericType::dtypeStatic:
			gt = ht;
			break;
		case genericType::concrete:
			if ( ht == genericType::dynamic ) { gt = genericType::dynamic; }
			break;
		case genericType::dynamic:
			// nothing possible
			break;
		}
		return gt;
	}

	// collect the environments of each TypeInstType so that type variables can be replaced
	// xxx - possibly temporary solution. Access to type environments is required in GenericInstantiator, but it needs to be a DeclMutator which does not provide easy access to the type environments.
	class EnvFinder final : public GenPoly::PolyMutator {
	public:
		using GenPoly::PolyMutator::mutate;
		virtual Type * mutate( TypeInstType * inst ) override {
			if ( env ) envMap[inst] = env;
			return inst;
		}

		// don't want to associate an environment with TypeInstTypes that occur in function types - this may actually only apply to function types belonging to DeclarationWithTypes (or even just FunctionDecl)?
		virtual Type * mutate( FunctionType * ftype ) override {
			return ftype;
		}
		std::unordered_map< ReferenceToType *, TypeSubstitution * > envMap;
	};

	/// Mutator pass that replaces concrete instantiations of generic types with actual struct declarations, scoped appropriately
	class GenericInstantiator final : public DeclMutator {
		/// Map of (generic type, parameter list) pairs to concrete type instantiations
		InstantiationMap< AggregateDecl, AggregateDecl > instantiations;
		/// Set of types which are dtype-only generic (and therefore have static layout)
		ScopedSet< AggregateDecl* > dtypeStatics;
		/// Namer for concrete types
		UniqueName typeNamer;
		/// Reference to mapping of environments
		const std::unordered_map< ReferenceToType *, TypeSubstitution * > & envMap;
	public:
		GenericInstantiator( const std::unordered_map< ReferenceToType *, TypeSubstitution * > & envMap ) : DeclMutator(), instantiations(), dtypeStatics(), typeNamer("_conc_"), envMap( envMap ) {}

		using DeclMutator::mutate;
		virtual Type* mutate( StructInstType *inst ) override;
		virtual Type* mutate( UnionInstType *inst ) override;

		virtual void doBeginScope() override;
		virtual void doEndScope() override;
	private:
		/// Wrap instantiation lookup for structs
		StructDecl* lookup( StructInstType *inst, const std::list< TypeExpr* > &typeSubs ) { return (StructDecl*)instantiations.lookup( inst->get_baseStruct(), typeSubs ); }
		/// Wrap instantiation lookup for unions
		UnionDecl* lookup( UnionInstType *inst, const std::list< TypeExpr* > &typeSubs ) { return (UnionDecl*)instantiations.lookup( inst->get_baseUnion(), typeSubs ); }
		/// Wrap instantiation insertion for structs
		void insert( StructInstType *inst, const std::list< TypeExpr* > &typeSubs, StructDecl *decl ) { instantiations.insert( inst->get_baseStruct(), typeSubs, decl ); }
		/// Wrap instantiation insertion for unions
		void insert( UnionInstType *inst, const std::list< TypeExpr* > &typeSubs, UnionDecl *decl ) { instantiations.insert( inst->get_baseUnion(), typeSubs, decl ); }

		void replaceParametersWithConcrete( std::list< Expression* >& params );
		Type *replaceWithConcrete( Type *type, bool doClone );

		/// Strips a dtype-static aggregate decl of its type parameters, marks it as stripped
		void stripDtypeParams( AggregateDecl *base, std::list< TypeDecl* >& baseParams, const std::list< TypeExpr* >& typeSubs );
	};

	void instantiateGeneric( std::list< Declaration* > &translationUnit ) {
		EnvFinder finder;
		mutateAll( translationUnit, finder );
		GenericInstantiator instantiator( finder.envMap );
		instantiator.mutateDeclarationList( translationUnit );
	}

	/// Makes substitutions of params into baseParams; returns dtypeStatic if there is a concrete instantiation based only on {d,f}type-to-void conversions,
	/// concrete if there is a concrete instantiation requiring at least one parameter type, and dynamic if there is no concrete instantiation
	genericType makeSubstitutions( const std::list< TypeDecl* >& baseParams, const std::list< Expression* >& params, std::list< TypeExpr* >& out ) {
		genericType gt = genericType::dtypeStatic;

		// substitute concrete types for given parameters, and incomplete types for placeholders
		std::list< TypeDecl* >::const_iterator baseParam = baseParams.begin();
		std::list< Expression* >::const_iterator param = params.begin();
		for ( ; baseParam != baseParams.end() && param != params.end(); ++baseParam, ++param ) {
			TypeExpr *paramType = dynamic_cast< TypeExpr* >( *param );
			assert(paramType && "Aggregate parameters should be type expressions");

			if ( (*baseParam)->isComplete() ) {
				// substitute parameter for complete (otype or sized dtype) type; makes the struct concrete or dynamic depending on the parameter
				out.push_back( paramType->clone() );
				gt |= isPolyType( paramType->get_type() ) ? genericType::dynamic : genericType::concrete;
			} else switch ( (*baseParam)->get_kind() ) {
				case TypeDecl::Dtype:
					// can pretend that any incomplete dtype is `void`
					out.push_back( new TypeExpr( new VoidType( Type::Qualifiers() ) ) );
					break;
				case TypeDecl::Ftype:
					// can pretend that any ftype is `void (*)(void)`
					out.push_back( new TypeExpr( new FunctionType( Type::Qualifiers(), false ) ) );
					break;
				case TypeDecl::Ttype:
					assertf( false, "Ttype parameters are not currently allowed as parameters to generic types." );
					break;
				case TypeDecl::Any:
					assertf( false, "otype parameters handled by baseParam->isComplete()." );
					break;
			}
		}

		assert( baseParam == baseParams.end() && param == params.end() && "Type parameters should match type variables" );
		return gt;
	}

	/// Substitutes types of members of in according to baseParams => typeSubs, appending the result to out
	void substituteMembers( const std::list< Declaration* >& in, const std::list< TypeDecl* >& baseParams, const std::list< TypeExpr* >& typeSubs,
							std::list< Declaration* >& out ) {
		// substitute types into new members
		TypeSubstitution subs( baseParams.begin(), baseParams.end(), typeSubs.begin() );
		for ( std::list< Declaration* >::const_iterator member = in.begin(); member != in.end(); ++member ) {
			Declaration *newMember = (*member)->clone();
			subs.apply(newMember);
			out.push_back( newMember );
		}
	}

	/// Substitutes types of members according to baseParams => typeSubs, working in-place
	void substituteMembers( std::list< Declaration* >& members, const std::list< TypeDecl* >& baseParams, const std::list< TypeExpr* >& typeSubs ) {
		// substitute types into new members
		TypeSubstitution subs( baseParams.begin(), baseParams.end(), typeSubs.begin() );
		for ( std::list< Declaration* >::iterator member = members.begin(); member != members.end(); ++member ) {
			subs.apply(*member);
		}
	}

	/// Strips the instances's type parameters
	void stripInstParams( ReferenceToType *inst ) {
		deleteAll( inst->get_parameters() );
		inst->get_parameters().clear();
	}

	void GenericInstantiator::stripDtypeParams( AggregateDecl *base, std::list< TypeDecl* >& baseParams, const std::list< TypeExpr* >& typeSubs ) {
		substituteMembers( base->get_members(), baseParams, typeSubs );

		deleteAll( baseParams );
		baseParams.clear();

		dtypeStatics.insert( base );
	}

	/// xxx - more or less copied from box -- these should be merged with those somehow...
	void GenericInstantiator::replaceParametersWithConcrete( std::list< Expression* >& params ) {
		for ( std::list< Expression* >::iterator param = params.begin(); param != params.end(); ++param ) {
			TypeExpr *paramType = dynamic_cast< TypeExpr* >( *param );
			assertf(paramType, "Aggregate parameters should be type expressions");
			paramType->set_type( replaceWithConcrete( paramType->get_type(), false ) );
		}
	}

	Type *GenericInstantiator::replaceWithConcrete( Type *type, bool doClone ) {
		if ( TypeInstType *typeInst = dynamic_cast< TypeInstType * >( type ) ) {
			if ( envMap.count( typeInst ) ) {
				TypeSubstitution * env = envMap.at( typeInst );
				Type *concrete = env->lookup( typeInst->get_name() );
				if ( concrete ) {
					return concrete->clone();
				}
				else return typeInst->clone();
			}
		} else if ( StructInstType *structType = dynamic_cast< StructInstType* >( type ) ) {
			if ( doClone ) {
				structType = structType->clone();
			}
			replaceParametersWithConcrete( structType->get_parameters() );
			return structType;
		} else if ( UnionInstType *unionType = dynamic_cast< UnionInstType* >( type ) ) {
			if ( doClone ) {
				unionType = unionType->clone();
			}
			replaceParametersWithConcrete( unionType->get_parameters() );
			return unionType;
		}
		return type;
	}


	Type* GenericInstantiator::mutate( StructInstType *inst ) {
		// mutate subtypes
		Type *mutated = Mutator::mutate( inst );
		inst = dynamic_cast< StructInstType* >( mutated );
		if ( ! inst ) return mutated;

		// exit early if no need for further mutation
		if ( inst->get_parameters().empty() ) return inst;

		// need to replace type variables to ensure that generic types are instantiated for the return values of polymorphic functions (in particular, for thunks, because they are not [currently] copy constructed).
		replaceWithConcrete( inst, false );

		// check for an already-instantiatiated dtype-static type
		if ( dtypeStatics.find( inst->get_baseStruct() ) != dtypeStatics.end() ) {
			stripInstParams( inst );
			return inst;
		}

		// check if type can be concretely instantiated; put substitutions into typeSubs
		assertf( inst->get_baseParameters(), "Base struct has parameters" );
		std::list< TypeExpr* > typeSubs;
		genericType gt = makeSubstitutions( *inst->get_baseParameters(), inst->get_parameters(), typeSubs );
		switch ( gt ) {
		case genericType::dtypeStatic:
			stripDtypeParams( inst->get_baseStruct(), *inst->get_baseParameters(), typeSubs );
			stripInstParams( inst );
			break;

		case genericType::concrete: {
			// make concrete instantiation of generic type
			StructDecl *concDecl = lookup( inst, typeSubs );
			if ( ! concDecl ) {
				// set concDecl to new type, insert type declaration into statements to add
				concDecl = new StructDecl( typeNamer.newName( inst->get_name() ) );
				concDecl->set_body( inst->get_baseStruct()->has_body() );
				substituteMembers( inst->get_baseStruct()->get_members(), *inst->get_baseParameters(), typeSubs, 	concDecl->get_members() );
				DeclMutator::addDeclaration( concDecl );
				insert( inst, typeSubs, concDecl );
			}
			StructInstType *newInst = new StructInstType( inst->get_qualifiers(), concDecl->get_name() );
			newInst->set_baseStruct( concDecl );

			delete inst;
			inst = newInst;
			break;
		}

		case genericType::dynamic:
			// do nothing
			break;
		}

		deleteAll( typeSubs );
		return inst;
	}

	Type* GenericInstantiator::mutate( UnionInstType *inst ) {
		// mutate subtypes
		Type *mutated = Mutator::mutate( inst );
		inst = dynamic_cast< UnionInstType* >( mutated );
		if ( ! inst ) return mutated;

		// exit early if no need for further mutation
		if ( inst->get_parameters().empty() ) return inst;

		// check for an already-instantiatiated dtype-static type
		if ( dtypeStatics.find( inst->get_baseUnion() ) != dtypeStatics.end() ) {
			stripInstParams( inst );
			return inst;
		}

		// check if type can be concretely instantiated; put substitutions into typeSubs
		assert( inst->get_baseParameters() && "Base union has parameters" );
		std::list< TypeExpr* > typeSubs;
		genericType gt = makeSubstitutions( *inst->get_baseParameters(), inst->get_parameters(), typeSubs );
		switch ( gt ) {
		case genericType::dtypeStatic:
			stripDtypeParams( inst->get_baseUnion(), *inst->get_baseParameters(), typeSubs );
			stripInstParams( inst );
			break;

		case genericType::concrete:
		{
			// make concrete instantiation of generic type
			UnionDecl *concDecl = lookup( inst, typeSubs );
			if ( ! concDecl ) {
				// set concDecl to new type, insert type declaration into statements to add
				concDecl = new UnionDecl( typeNamer.newName( inst->get_name() ) );
				concDecl->set_body( inst->get_baseUnion()->has_body() );
				substituteMembers( inst->get_baseUnion()->get_members(), *inst->get_baseParameters(), typeSubs, concDecl->get_members() );
				DeclMutator::addDeclaration( concDecl );
				insert( inst, typeSubs, concDecl );
			}
			UnionInstType *newInst = new UnionInstType( inst->get_qualifiers(), concDecl->get_name() );
			newInst->set_baseUnion( concDecl );

			delete inst;
			inst = newInst;
			break;
		}
		case genericType::dynamic:
			// do nothing
			break;
		}

		deleteAll( typeSubs );
		return inst;
	}

	void GenericInstantiator::doBeginScope() {
		DeclMutator::doBeginScope();
		instantiations.beginScope();
		dtypeStatics.beginScope();
	}

	void GenericInstantiator::doEndScope() {
		DeclMutator::doEndScope();
		instantiations.endScope();
		dtypeStatics.endScope();
	}

} // namespace GenPoly

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
