//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Box.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Fri Feb  5 16:45:07 2016
// Update Count     : 286
//

#include <set>
#include <stack>
#include <string>
#include <iterator>
#include <algorithm>
#include <cassert>

#include "Box.h"
#include "InstantiateGeneric.h"
#include "PolyMutator.h"
#include "FindFunction.h"
#include "ScopedMap.h"
#include "ScrubTyVars.h"

#include "Parser/ParseNode.h"

#include "SynTree/Constant.h"
#include "SynTree/Type.h"
#include "SynTree/Expression.h"
#include "SynTree/Initializer.h"
#include "SynTree/Statement.h"
#include "SynTree/Mutator.h"

#include "ResolvExpr/TypeEnvironment.h"

#include "SymTab/Mangler.h"

#include "Common/SemanticError.h"
#include "Common/UniqueName.h"
#include "Common/utility.h"

#include <ext/functional> // temporary

namespace GenPoly {
	namespace {
		const std::list<Label> noLabels;

		FunctionType *makeAdapterType( FunctionType *adaptee, const TyVarMap &tyVars );

		/// Replaces polymorphic return types with out-parameters, replaces calls to polymorphic functions with adapter calls as needed, and adds appropriate type variables to the function call
		class Pass1 : public PolyMutator {
		  public:
			Pass1();
			virtual Expression *mutate( ApplicationExpr *appExpr );
			virtual Expression *mutate( AddressExpr *addrExpr );
			virtual Expression *mutate( UntypedExpr *expr );
			virtual DeclarationWithType* mutate( FunctionDecl *functionDecl );
			virtual TypeDecl *mutate( TypeDecl *typeDecl );
			virtual Expression *mutate( CommaExpr *commaExpr );
			virtual Expression *mutate( ConditionalExpr *condExpr );
			virtual Statement * mutate( ReturnStmt *returnStmt );
			virtual Type *mutate( PointerType *pointerType );
			virtual Type * mutate( FunctionType *functionType );

			virtual void doBeginScope();
			virtual void doEndScope();
		  private:
			/// Makes a new temporary array holding the offsets of the fields of `type`, and returns a new variable expression referencing it
			Expression *makeOffsetArray( StructInstType *type );
			/// Pass the extra type parameters from polymorphic generic arguments or return types into a function application
			void passArgTypeVars( ApplicationExpr *appExpr, Type *parmType, Type *argBaseType, std::list< Expression *>::iterator &arg, const TyVarMap &exprTyVars, std::set< std::string > &seenTypes );
			/// passes extra type parameters into a polymorphic function application
			void passTypeVars( ApplicationExpr *appExpr, ReferenceToType *polyRetType, std::list< Expression *>::iterator &arg, const TyVarMap &exprTyVars );
			/// wraps a function application with a new temporary for the out-parameter return value
			Expression *addRetParam( ApplicationExpr *appExpr, FunctionType *function, Type *retType, std::list< Expression *>::iterator &arg );
			/// Replaces all the type parameters of a generic type with their concrete equivalents under the current environment
			void replaceParametersWithConcrete( ApplicationExpr *appExpr, std::list< Expression* >& params );
			/// Replaces a polymorphic type with its concrete equivalant under the current environment (returns itself if concrete).
			/// If `doClone` is set to false, will not clone interior types
			Type *replaceWithConcrete( ApplicationExpr *appExpr, Type *type, bool doClone = true );
			/// wraps a function application returning a polymorphic type with a new temporary for the out-parameter return value
			Expression *addPolyRetParam( ApplicationExpr *appExpr, FunctionType *function, ReferenceToType *polyType, std::list< Expression *>::iterator &arg );
			Expression *applyAdapter( ApplicationExpr *appExpr, FunctionType *function, std::list< Expression *>::iterator &arg, const TyVarMap &exprTyVars );
			void boxParam( Type *formal, Expression *&arg, const TyVarMap &exprTyVars );
			void boxParams( ApplicationExpr *appExpr, FunctionType *function, std::list< Expression *>::iterator &arg, const TyVarMap &exprTyVars );
			void addInferredParams( ApplicationExpr *appExpr, FunctionType *functionType, std::list< Expression *>::iterator &arg, const TyVarMap &tyVars );
			/// Stores assignment operators from assertion list in local map of assignment operations
			void findAssignOps( const std::list< TypeDecl *> &forall );
			void passAdapters( ApplicationExpr *appExpr, FunctionType *functionType, const TyVarMap &exprTyVars );
			FunctionDecl *makeAdapter( FunctionType *adaptee, FunctionType *realType, const std::string &mangleName, const TyVarMap &tyVars );
			/// Replaces intrinsic operator functions with their arithmetic desugaring
			Expression *handleIntrinsics( ApplicationExpr *appExpr );
			/// Inserts a new temporary variable into the current scope with an auto-generated name
			ObjectDecl *makeTemporary( Type *type );

			typedef std::map< std::string, DeclarationWithType *> AdapterMap;
			std::map< std::string, DeclarationWithType *> assignOps;
			ScopedMap< std::string, DeclarationWithType *> scopedAssignOps;
			std::stack< AdapterMap > adapters;
			DeclarationWithType *retval;
			bool useRetval;
			UniqueName tempNamer;
		};

		/// Moves polymorphic returns in function types to pointer-type parameters, adds type size and assertion parameters to parameter lists as well
		class Pass2 : public PolyMutator {
		  public:
			template< typename DeclClass >
			DeclClass *handleDecl( DeclClass *decl, Type *type );
			virtual DeclarationWithType *mutate( FunctionDecl *functionDecl );
			virtual ObjectDecl *mutate( ObjectDecl *objectDecl );
			virtual TypeDecl *mutate( TypeDecl *typeDecl );
			virtual TypedefDecl *mutate( TypedefDecl *typedefDecl );
			virtual Type *mutate( PointerType *pointerType );
			virtual Type *mutate( FunctionType *funcType );
		  private:
			void addAdapters( FunctionType *functionType );

			std::map< UniqueId, std::string > adapterName;
		};

		/// Replaces member expressions for polymorphic types with calculated add-field-offset-and-dereference;
		/// also fixes offsetof expressions.
		class MemberExprFixer : public PolyMutator {
		  public:
			template< typename DeclClass >
			DeclClass *handleDecl( DeclClass *decl, Type *type );
			virtual DeclarationWithType *mutate( FunctionDecl *functionDecl );
			virtual ObjectDecl *mutate( ObjectDecl *objectDecl );
			virtual TypedefDecl *mutate( TypedefDecl *objectDecl );
			virtual TypeDecl *mutate( TypeDecl *objectDecl );
			virtual Statement *mutate( DeclStmt *declStmt );
			virtual Type *mutate( PointerType *pointerType );
			virtual Type *mutate( FunctionType *funcType );
			virtual Expression *mutate( MemberExpr *memberExpr );
			virtual Expression *mutate( OffsetofExpr *offsetofExpr );
		};

		/// Replaces initialization of polymorphic values with alloca, declaration of dtype/ftype with appropriate void expression, and sizeof expressions of polymorphic types with the proper variable
		class Pass3 : public PolyMutator {
		  public:
			template< typename DeclClass >
			DeclClass *handleDecl( DeclClass *decl, Type *type );
			virtual DeclarationWithType *mutate( FunctionDecl *functionDecl );
			virtual ObjectDecl *mutate( ObjectDecl *objectDecl );
			virtual TypedefDecl *mutate( TypedefDecl *objectDecl );
			virtual TypeDecl *mutate( TypeDecl *objectDecl );
			virtual Type *mutate( PointerType *pointerType );
			virtual Type *mutate( FunctionType *funcType );
		  private:
		};

	} // anonymous namespace

	void printAllNotBuiltin( const std::list< Declaration *>& translationUnit, std::ostream &os ) {
		for ( std::list< Declaration *>::const_iterator i = translationUnit.begin(); i != translationUnit.end(); ++i ) {
			if ( ! LinkageSpec::isBuiltin( (*i)->get_linkage() ) ) {
				(*i)->print( os );
				os << std::endl;
			} // if
		} // for
	}

	/// version of mutateAll with special handling for translation unit so you can check the end of the prelude when debugging
	template< typename MutatorType >
	inline void mutateTranslationUnit( std::list< Declaration* > &translationUnit, MutatorType &mutator ) {
		bool seenIntrinsic = false;
		SemanticError errors;
		for ( typename std::list< Declaration* >::iterator i = translationUnit.begin(); i != translationUnit.end(); ++i ) {
			try {
				if ( *i ) {
					if ( (*i)->get_linkage() == LinkageSpec::Intrinsic ) {
						seenIntrinsic = true;
					} else if ( seenIntrinsic ) {
						seenIntrinsic = false; // break on this line when debugging for end of prelude
					}

					*i = dynamic_cast< Declaration* >( (*i)->acceptMutator( mutator ) );
					assert( *i );
				} // if
			} catch( SemanticError &e ) {
				errors.append( e );
			} // try
		} // for
		if ( ! errors.isEmpty() ) {
			throw errors;
		} // if
	}

	void box( std::list< Declaration *>& translationUnit ) {
		Pass1 pass1;
		Pass2 pass2;
		MemberExprFixer memberFixer;
		Pass3 pass3;
		mutateTranslationUnit/*All*/( translationUnit, pass1 );
		mutateTranslationUnit/*All*/( translationUnit, pass2 );
		instantiateGeneric( translationUnit );
		mutateTranslationUnit/*All*/( translationUnit, memberFixer );
		mutateTranslationUnit/*All*/( translationUnit, pass3 );
	}

	////////////////////////////////////////// Pass1 ////////////////////////////////////////////////////

	namespace {
		std::string makePolyMonoSuffix( FunctionType * function, const TyVarMap &tyVars ) {
			std::stringstream name;

			// NOTE: this function previously used isPolyObj, which failed to produce
			// the correct thing in some situations. It's not clear to me why this wasn't working.

			// if the return type or a parameter type involved polymorphic types, then the adapter will need
			// to take those polymorphic types as pointers. Therefore, there can be two different functions
			// with the same mangled name, so we need to further mangle the names.
			for ( std::list< DeclarationWithType *>::iterator retval = function->get_returnVals().begin(); retval != function->get_returnVals().end(); ++retval ) {
				if ( isPolyType( (*retval)->get_type(), tyVars ) ) {
					name << "P";
				} else {
					name << "M";
				}
			}
			name << "_";
			std::list< DeclarationWithType *> &paramList = function->get_parameters();
			for ( std::list< DeclarationWithType *>::iterator arg = paramList.begin(); arg != paramList.end(); ++arg ) {
				if ( isPolyType( (*arg)->get_type(), tyVars ) ) {
					name << "P";
				} else {
					name << "M";
				}
			} // for
			return name.str();
		}

		std::string mangleAdapterName( FunctionType * function, const TyVarMap &tyVars ) {
			return SymTab::Mangler::mangle( function ) + makePolyMonoSuffix( function, tyVars );
		}

		std::string makeAdapterName( const std::string &mangleName ) {
			return "_adapter" + mangleName;
		}

		Pass1::Pass1() : useRetval( false ), tempNamer( "_temp" ) {
			adapters.push(AdapterMap());
		}

		/// returns T if the given declaration is: (*?=?)(T *, T) for some T (return not checked, but maybe should be), NULL otherwise
		ReferenceToType *isAssignment( DeclarationWithType *decl ) {
			if ( decl->get_name() == "?=?" ) {
				if ( FunctionType *funType = getFunctionType( decl->get_type() ) ) {
					if ( funType->get_parameters().size() == 2 ) {
						if ( PointerType *pointer = dynamic_cast< PointerType *>( funType->get_parameters().front()->get_type() ) ) {
							if ( ReferenceToType *refType = dynamic_cast< ReferenceToType *>( pointer->get_base() ) ) {
								if ( ReferenceToType *refType2 = dynamic_cast< ReferenceToType *>( funType->get_parameters().back()->get_type() ) ) {
									if ( refType->get_name() == refType2->get_name() ) {
										return refType;
									} // if
								} // if
							} // if
						} // if
					} // if
				} // if
			} // if
			return 0;
		}

		void Pass1::findAssignOps( const std::list< TypeDecl *> &forall ) {
			// what if a nested function uses an assignment operator?
			// assignOps.clear();
			for ( std::list< TypeDecl *>::const_iterator i = forall.begin(); i != forall.end(); ++i ) {
				for ( std::list< DeclarationWithType *>::const_iterator assert = (*i)->get_assertions().begin(); assert != (*i)->get_assertions().end(); ++assert ) {
					std::string typeName;
					if ( TypeInstType *typeInst = dynamic_cast< TypeInstType* >( isAssignment( *assert ) ) ) {
						assignOps[ typeInst->get_name() ] = *assert;
					} // if
				} // for
			} // for
		}

		DeclarationWithType *Pass1::mutate( FunctionDecl *functionDecl ) {
			// if this is a polymorphic assignment function, put it in the map for this scope
			if ( ReferenceToType *refType = isAssignment( functionDecl ) ) {
				if ( ! dynamic_cast< TypeInstType* >( refType ) ) {
					scopedAssignOps.insert( refType->get_name(), functionDecl );
				}
			}

			if ( functionDecl->get_statements() ) {		// empty routine body ?
				doBeginScope();
				TyVarMap oldtyVars = scopeTyVars;
				std::map< std::string, DeclarationWithType *> oldassignOps = assignOps;
				DeclarationWithType *oldRetval = retval;
				bool oldUseRetval = useRetval;

				// process polymorphic return value
				retval = 0;
				if ( isPolyRet( functionDecl->get_functionType() ) && functionDecl->get_linkage() == LinkageSpec::Cforall ) {
					retval = functionDecl->get_functionType()->get_returnVals().front();

					// give names to unnamed return values
					if ( retval->get_name() == "" ) {
						retval->set_name( "_retparm" );
						retval->set_linkage( LinkageSpec::C );
					} // if
				} // if

				FunctionType *functionType = functionDecl->get_functionType();
				makeTyVarMap( functionDecl->get_functionType(), scopeTyVars );
				findAssignOps( functionDecl->get_functionType()->get_forall() );

				std::list< DeclarationWithType *> &paramList = functionType->get_parameters();
				std::list< FunctionType *> functions;
				for ( std::list< TypeDecl *>::iterator tyVar = functionType->get_forall().begin(); tyVar != functionType->get_forall().end(); ++tyVar ) {
					for ( std::list< DeclarationWithType *>::iterator assert = (*tyVar)->get_assertions().begin(); assert != (*tyVar)->get_assertions().end(); ++assert ) {
						findFunction( (*assert)->get_type(), functions, scopeTyVars, needsAdapter );
					} // for
				} // for
				for ( std::list< DeclarationWithType *>::iterator arg = paramList.begin(); arg != paramList.end(); ++arg ) {
					findFunction( (*arg)->get_type(), functions, scopeTyVars, needsAdapter );
				} // for

				AdapterMap & adapters = Pass1::adapters.top();
				for ( std::list< FunctionType *>::iterator funType = functions.begin(); funType != functions.end(); ++funType ) {
					std::string mangleName = mangleAdapterName( *funType, scopeTyVars );
					if ( adapters.find( mangleName ) == adapters.end() ) {
						std::string adapterName = makeAdapterName( mangleName );
						adapters.insert( std::pair< std::string, DeclarationWithType *>( mangleName, new ObjectDecl( adapterName, DeclarationNode::NoStorageClass, LinkageSpec::C, 0, new PointerType( Type::Qualifiers(), makeAdapterType( *funType, scopeTyVars ) ), 0 ) ) );
					} // if
				} // for

				functionDecl->set_statements( functionDecl->get_statements()->acceptMutator( *this ) );

				scopeTyVars = oldtyVars;
				assignOps = oldassignOps;
				// std::cerr << "end FunctionDecl: ";
				// for ( TyVarMap::iterator i = scopeTyVars.begin(); i != scopeTyVars.end(); ++i ) {
				// 	std::cerr << i->first << " ";
				// }
				// std::cerr << "\n";
				retval = oldRetval;
				useRetval = oldUseRetval;
				doEndScope();
			} // if
			return functionDecl;
		}

		TypeDecl *Pass1::mutate( TypeDecl *typeDecl ) {
			scopeTyVars[ typeDecl->get_name() ] = typeDecl->get_kind();
			return Mutator::mutate( typeDecl );
		}

		Expression *Pass1::mutate( CommaExpr *commaExpr ) {
			bool oldUseRetval = useRetval;
			useRetval = false;
			commaExpr->set_arg1( maybeMutate( commaExpr->get_arg1(), *this ) );
			useRetval = oldUseRetval;
			commaExpr->set_arg2( maybeMutate( commaExpr->get_arg2(), *this ) );
			return commaExpr;
		}

		Expression *Pass1::mutate( ConditionalExpr *condExpr ) {
			bool oldUseRetval = useRetval;
			useRetval = false;
			condExpr->set_arg1( maybeMutate( condExpr->get_arg1(), *this ) );
			useRetval = oldUseRetval;
			condExpr->set_arg2( maybeMutate( condExpr->get_arg2(), *this ) );
			condExpr->set_arg3( maybeMutate( condExpr->get_arg3(), *this ) );
			return condExpr;

		}

		Expression *Pass1::makeOffsetArray( StructInstType *ty ) {
			std::list< Declaration* > &baseMembers = ty->get_baseStruct()->get_members();

			// make a new temporary array
			Type *offsetType = new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt );
			std::stringstream lenGen;
			lenGen << baseMembers.size();
			ConstantExpr *lenExpr = new ConstantExpr( Constant( offsetType->clone(), lenGen.str() ) );
			ObjectDecl *arrayTemp = makeTemporary( new ArrayType( Type::Qualifiers(), offsetType, lenExpr, false, false ) );

			// build initializer list for temporary
			std::list< Initializer* > inits;
			for ( std::list< Declaration* >::const_iterator member = baseMembers.begin(); member != baseMembers.end(); ++member ) {
				DeclarationWithType *memberDecl;
				if ( DeclarationWithType *origMember = dynamic_cast< DeclarationWithType* >( *member ) ) {
					memberDecl = origMember->clone();
				} else {
					memberDecl = new ObjectDecl( (*member)->get_name(), DeclarationNode::NoStorageClass, LinkageSpec::Cforall, 0, offsetType->clone(), 0 );
				}
				inits.push_back( new SingleInit( new OffsetofExpr( ty->clone(), memberDecl ) ) );
			}
			arrayTemp->set_init( new ListInit( inits ) );

			// return variable pointing to temporary
			return new VariableExpr( arrayTemp );
		}

		void Pass1::passArgTypeVars( ApplicationExpr *appExpr, Type *parmType, Type *argBaseType, std::list< Expression *>::iterator &arg, const TyVarMap &exprTyVars, std::set< std::string > &seenTypes ) {
			Type *polyBase = hasPolyBase( parmType, exprTyVars );
			if ( polyBase && ! dynamic_cast< TypeInstType* >( polyBase ) ) {
				std::string sizeName = sizeofName( polyBase );
				if ( seenTypes.count( sizeName ) ) return;

				arg = appExpr->get_args().insert( arg, new SizeofExpr( argBaseType->clone() ) );
				arg++;
				arg = appExpr->get_args().insert( arg, new AlignofExpr( argBaseType->clone() ) );
				arg++;
				if ( dynamic_cast< StructInstType* >( polyBase ) ) {
					if ( StructInstType *argBaseStructType = dynamic_cast< StructInstType* >( argBaseType ) ) {
						arg = appExpr->get_args().insert( arg, makeOffsetArray( argBaseStructType ) );
						arg++;
					} else {
						throw SemanticError( "Cannot pass non-struct type for generic struct" );
					}
				}

				seenTypes.insert( sizeName );
			}
		}

		void Pass1::passTypeVars( ApplicationExpr *appExpr, ReferenceToType *polyRetType, std::list< Expression *>::iterator &arg, const TyVarMap &exprTyVars ) {
			// pass size/align for type variables
			for ( TyVarMap::const_iterator tyParm = exprTyVars.begin(); tyParm != exprTyVars.end(); ++tyParm ) {
				ResolvExpr::EqvClass eqvClass;
				assert( env );
				if ( tyParm->second == TypeDecl::Any ) {
					Type *concrete = env->lookup( tyParm->first );
					if ( concrete ) {
						arg = appExpr->get_args().insert( arg, new SizeofExpr( concrete->clone() ) );
						arg++;
						arg = appExpr->get_args().insert( arg, new AlignofExpr( concrete->clone() ) );
						arg++;
					} else {
						throw SemanticError( "unbound type variable in application ", appExpr );
					} // if
				} // if
			} // for

			// add size/align for generic types to parameter list
			if ( appExpr->get_function()->get_results().empty() ) return;
			FunctionType *funcType = getFunctionType( appExpr->get_function()->get_results().front() );
			assert( funcType );

			std::list< DeclarationWithType* >::const_iterator fnParm = funcType->get_parameters().begin();
			std::list< Expression* >::const_iterator fnArg = arg;
			std::set< std::string > seenTypes; //< names for generic types we've seen

			// a polymorphic return type may need to be added to the argument list
			if ( polyRetType ) {
				Type *concRetType = replaceWithConcrete( appExpr, polyRetType );
				passArgTypeVars( appExpr, polyRetType, concRetType, arg, exprTyVars, seenTypes );
			}
			
			// add type information args for presently unseen types in parameter list
			for ( ; fnParm != funcType->get_parameters().end() && fnArg != appExpr->get_args().end(); ++fnParm, ++fnArg ) {
				VariableExpr *fnArgBase = getBaseVar( *fnArg );
				if ( ! fnArgBase || fnArgBase->get_results().empty() ) continue;
				passArgTypeVars( appExpr, (*fnParm)->get_type(), fnArgBase->get_results().front(), arg, exprTyVars, seenTypes );
			}
		}

		ObjectDecl *Pass1::makeTemporary( Type *type ) {
			ObjectDecl *newObj = new ObjectDecl( tempNamer.newName(), DeclarationNode::NoStorageClass, LinkageSpec::C, 0, type, 0 );
			stmtsToAdd.push_back( new DeclStmt( noLabels, newObj ) );
			return newObj;
		}

		Expression *Pass1::addRetParam( ApplicationExpr *appExpr, FunctionType *function, Type *retType, std::list< Expression *>::iterator &arg ) {
			// ***** Code Removal ***** After introducing a temporary variable for all return expressions, the following code appears superfluous.
			// if ( useRetval ) {
			// 	assert( retval );
			// 	arg = appExpr->get_args().insert( arg, new VariableExpr( retval ) );
			// 	arg++;
			// } else {

			// Create temporary to hold return value of polymorphic function and produce that temporary as a result
			// using a comma expression.  Possibly change comma expression into statement expression "{}" for multiple
			// return values.
			ObjectDecl *newObj = makeTemporary( retType->clone() );
			Expression *paramExpr = new VariableExpr( newObj );

			// If the type of the temporary is not polymorphic, box temporary by taking its address;
			// otherwise the temporary is already boxed and can be used directly.
			if ( ! isPolyType( newObj->get_type(), scopeTyVars, env ) ) {
				paramExpr = new AddressExpr( paramExpr );
			} // if
			arg = appExpr->get_args().insert( arg, paramExpr ); // add argument to function call
			arg++;
			// Build a comma expression to call the function and emulate a normal return.
			CommaExpr *commaExpr = new CommaExpr( appExpr, new VariableExpr( newObj ) );
			commaExpr->set_env( appExpr->get_env() );
			appExpr->set_env( 0 );
			return commaExpr;
			// } // if
			// return appExpr;
		}

		void Pass1::replaceParametersWithConcrete( ApplicationExpr *appExpr, std::list< Expression* >& params ) {
			for ( std::list< Expression* >::iterator param = params.begin(); param != params.end(); ++param ) {
				TypeExpr *paramType = dynamic_cast< TypeExpr* >( *param );
				assert(paramType && "Aggregate parameters should be type expressions");
				paramType->set_type( replaceWithConcrete( appExpr, paramType->get_type(), false ) );
			}
		}

		Type *Pass1::replaceWithConcrete( ApplicationExpr *appExpr, Type *type, bool doClone ) {
			if ( TypeInstType *typeInst = dynamic_cast< TypeInstType * >( type ) ) {
				Type *concrete = env->lookup( typeInst->get_name() );
				if ( concrete == 0 ) {
					throw SemanticError( "Unbound type variable " + typeInst->get_name() + " in ", appExpr );
				} // if
				return concrete;
			} else if ( StructInstType *structType = dynamic_cast< StructInstType* >( type ) ) {
				if ( doClone ) {
					structType = structType->clone();
				}
				replaceParametersWithConcrete( appExpr, structType->get_parameters() );
				return structType;
			} else if ( UnionInstType *unionType = dynamic_cast< UnionInstType* >( type ) ) {
				if ( doClone ) {
					unionType = unionType->clone();
				}
				replaceParametersWithConcrete( appExpr, unionType->get_parameters() );
				return unionType;
			}
			return type;
		}

		Expression *Pass1::addPolyRetParam( ApplicationExpr *appExpr, FunctionType *function, ReferenceToType *polyType, std::list< Expression *>::iterator &arg ) {
			assert( env );
			Type *concrete = replaceWithConcrete( appExpr, polyType );
			// add out-parameter for return value	
			return addRetParam( appExpr, function, concrete, arg );
		}

		Expression *Pass1::applyAdapter( ApplicationExpr *appExpr, FunctionType *function, std::list< Expression *>::iterator &arg, const TyVarMap &tyVars ) {
			Expression *ret = appExpr;
			if ( ! function->get_returnVals().empty() && isPolyType( function->get_returnVals().front()->get_type(), tyVars ) ) {
				ret = addRetParam( appExpr, function, function->get_returnVals().front()->get_type(), arg );
			} // if
			std::string mangleName = mangleAdapterName( function, tyVars );
			std::string adapterName = makeAdapterName( mangleName );

			// cast adaptee to void (*)(), since it may have any type inside a polymorphic function
			Type * adapteeType = new PointerType( Type::Qualifiers(), new FunctionType( Type::Qualifiers(), true ) );
			appExpr->get_args().push_front( new CastExpr( appExpr->get_function(), adapteeType ) );
			appExpr->set_function( new NameExpr( adapterName ) );

			return ret;
		}

		void Pass1::boxParam( Type *param, Expression *&arg, const TyVarMap &exprTyVars ) {
			assert( ! arg->get_results().empty() );
			if ( isPolyType( param, exprTyVars ) ) {
				if ( isPolyType( arg->get_results().front() ) ) {
					// if the argument's type is polymorphic, we don't need to box again!
					return;
				} else if ( arg->get_results().front()->get_isLvalue() ) {
					// VariableExpr and MemberExpr are lvalues
					arg = new AddressExpr( arg );
				} else {
					// use type computed in unification to declare boxed variables
					Type * newType = param->clone();
					if ( env ) env->apply( newType );
					ObjectDecl *newObj = new ObjectDecl( tempNamer.newName(), DeclarationNode::NoStorageClass, LinkageSpec::C, 0, newType, 0 );
					newObj->get_type()->get_qualifiers() = Type::Qualifiers(); // TODO: is this right???
					stmtsToAdd.push_back( new DeclStmt( noLabels, newObj ) );
					UntypedExpr *assign = new UntypedExpr( new NameExpr( "?=?" ) );
					assign->get_args().push_back( new VariableExpr( newObj ) );
					assign->get_args().push_back( arg );
					stmtsToAdd.push_back( new ExprStmt( noLabels, assign ) );
					arg = new AddressExpr( new VariableExpr( newObj ) );
				} // if
			} // if
		}

		/// cast parameters to polymorphic functions so that types are replaced with
		/// void * if they are type parameters in the formal type.
		/// this gets rid of warnings from gcc.
		void addCast( Expression *&actual, Type *formal, const TyVarMap &tyVars ) {
			Type * newType = formal->clone();
			if ( getFunctionType( newType ) ) {
				newType = ScrubTyVars::scrub( newType, tyVars );
				actual = new CastExpr( actual, newType );
			} // if
		}

		void Pass1::boxParams( ApplicationExpr *appExpr, FunctionType *function, std::list< Expression *>::iterator &arg, const TyVarMap &exprTyVars ) {
			for ( std::list< DeclarationWithType *>::const_iterator param = function->get_parameters().begin(); param != function->get_parameters().end(); ++param, ++arg ) {
				assert( arg != appExpr->get_args().end() );
				addCast( *arg, (*param)->get_type(), exprTyVars );
				boxParam( (*param)->get_type(), *arg, exprTyVars );
			} // for
		}

		void Pass1::addInferredParams( ApplicationExpr *appExpr, FunctionType *functionType, std::list< Expression *>::iterator &arg, const TyVarMap &tyVars ) {
			std::list< Expression *>::iterator cur = arg;
			for ( std::list< TypeDecl *>::iterator tyVar = functionType->get_forall().begin(); tyVar != functionType->get_forall().end(); ++tyVar ) {
				for ( std::list< DeclarationWithType *>::iterator assert = (*tyVar)->get_assertions().begin(); assert != (*tyVar)->get_assertions().end(); ++assert ) {
					InferredParams::const_iterator inferParam = appExpr->get_inferParams().find( (*assert)->get_uniqueId() );
					assert( inferParam != appExpr->get_inferParams().end() && "NOTE: Explicit casts of polymorphic functions to compatible monomorphic functions are currently unsupported" );
					Expression *newExpr = inferParam->second.expr->clone();
					addCast( newExpr, (*assert)->get_type(), tyVars );
					boxParam( (*assert)->get_type(), newExpr, tyVars );
					appExpr->get_args().insert( cur, newExpr );
				} // for
			} // for
		}

		void makeRetParm( FunctionType *funcType ) {
			DeclarationWithType *retParm = funcType->get_returnVals().front();

			// make a new parameter that is a pointer to the type of the old return value
			retParm->set_type( new PointerType( Type::Qualifiers(), retParm->get_type() ) );
			funcType->get_parameters().push_front( retParm );

			// we don't need the return value any more
			funcType->get_returnVals().clear();
		}

		FunctionType *makeAdapterType( FunctionType *adaptee, const TyVarMap &tyVars ) {
			// actually make the adapter type
			FunctionType *adapter = adaptee->clone();
			if ( ! adapter->get_returnVals().empty() && isPolyType( adapter->get_returnVals().front()->get_type(), tyVars ) ) {
				makeRetParm( adapter );
			} // if
			adapter->get_parameters().push_front( new ObjectDecl( "", DeclarationNode::NoStorageClass, LinkageSpec::C, 0, new PointerType( Type::Qualifiers(), new FunctionType( Type::Qualifiers(), true ) ), 0 ) );
			return adapter;
		}

		Expression *makeAdapterArg( DeclarationWithType *param, DeclarationWithType *arg, DeclarationWithType *realParam, const TyVarMap &tyVars ) {
			assert( param );
			assert( arg );
			if ( isPolyType( realParam->get_type(), tyVars ) ) {
				if ( dynamic_cast<TypeInstType *>(arg->get_type()) == NULL ) {
					UntypedExpr *deref = new UntypedExpr( new NameExpr( "*?" ) );
					deref->get_args().push_back( new CastExpr( new VariableExpr( param ), new PointerType( Type::Qualifiers(), arg->get_type()->clone() ) ) );
					deref->get_results().push_back( arg->get_type()->clone() );
					return deref;
				} // if
			} // if
			return new VariableExpr( param );
		}

		void addAdapterParams( ApplicationExpr *adapteeApp, std::list< DeclarationWithType *>::iterator arg, std::list< DeclarationWithType *>::iterator param, std::list< DeclarationWithType *>::iterator paramEnd, std::list< DeclarationWithType *>::iterator realParam, const TyVarMap &tyVars ) {
			UniqueName paramNamer( "_p" );
			for ( ; param != paramEnd; ++param, ++arg, ++realParam ) {
				if ( (*param)->get_name() == "" ) {
					(*param)->set_name( paramNamer.newName() );
					(*param)->set_linkage( LinkageSpec::C );
				} // if
				adapteeApp->get_args().push_back( makeAdapterArg( *param, *arg, *realParam, tyVars ) );
			} // for
		}



		FunctionDecl *Pass1::makeAdapter( FunctionType *adaptee, FunctionType *realType, const std::string &mangleName, const TyVarMap &tyVars ) {
			FunctionType *adapterType = makeAdapterType( adaptee, tyVars );
			adapterType = ScrubTyVars::scrub( adapterType, tyVars );
			DeclarationWithType *adapteeDecl = adapterType->get_parameters().front();
			adapteeDecl->set_name( "_adaptee" );
			ApplicationExpr *adapteeApp = new ApplicationExpr( new CastExpr( new VariableExpr( adapteeDecl ), new PointerType( Type::Qualifiers(), realType ) ) );
			Statement *bodyStmt;

			std::list< TypeDecl *>::iterator tyArg = realType->get_forall().begin();
			std::list< TypeDecl *>::iterator tyParam = adapterType->get_forall().begin();
			std::list< TypeDecl *>::iterator realTyParam = adaptee->get_forall().begin();
			for ( ; tyParam != adapterType->get_forall().end(); ++tyArg, ++tyParam, ++realTyParam ) {
				assert( tyArg != realType->get_forall().end() );
				std::list< DeclarationWithType *>::iterator assertArg = (*tyArg)->get_assertions().begin();
				std::list< DeclarationWithType *>::iterator assertParam = (*tyParam)->get_assertions().begin();
				std::list< DeclarationWithType *>::iterator realAssertParam = (*realTyParam)->get_assertions().begin();
				for ( ; assertParam != (*tyParam)->get_assertions().end(); ++assertArg, ++assertParam, ++realAssertParam ) {
					assert( assertArg != (*tyArg)->get_assertions().end() );
					adapteeApp->get_args().push_back( makeAdapterArg( *assertParam, *assertArg, *realAssertParam, tyVars ) );
				} // for
			} // for

			std::list< DeclarationWithType *>::iterator arg = realType->get_parameters().begin();
			std::list< DeclarationWithType *>::iterator param = adapterType->get_parameters().begin();
			std::list< DeclarationWithType *>::iterator realParam = adaptee->get_parameters().begin();
			param++;		// skip adaptee parameter
			if ( realType->get_returnVals().empty() ) {
				addAdapterParams( adapteeApp, arg, param, adapterType->get_parameters().end(), realParam, tyVars );
				bodyStmt = new ExprStmt( noLabels, adapteeApp );
			} else if ( isPolyType( adaptee->get_returnVals().front()->get_type(), tyVars ) ) {
				if ( (*param)->get_name() == "" ) {
					(*param)->set_name( "_ret" );
					(*param)->set_linkage( LinkageSpec::C );
				} // if
				UntypedExpr *assign = new UntypedExpr( new NameExpr( "?=?" ) );
				UntypedExpr *deref = new UntypedExpr( new NameExpr( "*?" ) );
				deref->get_args().push_back( new CastExpr( new VariableExpr( *param++ ), new PointerType( Type::Qualifiers(), realType->get_returnVals().front()->get_type()->clone() ) ) );
				assign->get_args().push_back( deref );
				addAdapterParams( adapteeApp, arg, param, adapterType->get_parameters().end(), realParam, tyVars );
				assign->get_args().push_back( adapteeApp );
				bodyStmt = new ExprStmt( noLabels, assign );
			} else {
				// adapter for a function that returns a monomorphic value
				addAdapterParams( adapteeApp, arg, param, adapterType->get_parameters().end(), realParam, tyVars );
				bodyStmt = new ReturnStmt( noLabels, adapteeApp );
			} // if
			CompoundStmt *adapterBody = new CompoundStmt( noLabels );
			adapterBody->get_kids().push_back( bodyStmt );
			std::string adapterName = makeAdapterName( mangleName );
			return new FunctionDecl( adapterName, DeclarationNode::NoStorageClass, LinkageSpec::C, adapterType, adapterBody, false, false );
		}

		void Pass1::passAdapters( ApplicationExpr * appExpr, FunctionType * functionType, const TyVarMap & exprTyVars ) {
			// collect a list of function types passed as parameters or implicit parameters (assertions)
			std::list< DeclarationWithType *> &paramList = functionType->get_parameters();
			std::list< FunctionType *> functions;
			for ( std::list< TypeDecl *>::iterator tyVar = functionType->get_forall().begin(); tyVar != functionType->get_forall().end(); ++tyVar ) {
				for ( std::list< DeclarationWithType *>::iterator assert = (*tyVar)->get_assertions().begin(); assert != (*tyVar)->get_assertions().end(); ++assert ) {
					findFunction( (*assert)->get_type(), functions, exprTyVars, needsAdapter );
				} // for
			} // for
			for ( std::list< DeclarationWithType *>::iterator arg = paramList.begin(); arg != paramList.end(); ++arg ) {
				findFunction( (*arg)->get_type(), functions, exprTyVars, needsAdapter );
			} // for

			// parameter function types for which an appropriate adapter has been generated.  we cannot use the types
			// after applying substitutions, since two different parameter types may be unified to the same type
			std::set< std::string > adaptersDone;

			for ( std::list< FunctionType *>::iterator funType = functions.begin(); funType != functions.end(); ++funType ) {
				FunctionType *originalFunction = (*funType)->clone();
				FunctionType *realFunction = (*funType)->clone();
				std::string mangleName = SymTab::Mangler::mangle( realFunction );

				// only attempt to create an adapter or pass one as a parameter if we haven't already done so for this
				// pre-substitution parameter function type.
				if ( adaptersDone.find( mangleName ) == adaptersDone.end() ) {
					adaptersDone.insert( adaptersDone.begin(), mangleName );

					// apply substitution to type variables to figure out what the adapter's type should look like
					assert( env );
					env->apply( realFunction );
					mangleName = SymTab::Mangler::mangle( realFunction );
					mangleName += makePolyMonoSuffix( originalFunction, exprTyVars );

					AdapterMap & adapters = Pass1::adapters.top();
					AdapterMap::iterator adapter = adapters.find( mangleName );
					if ( adapter == adapters.end() ) {
						// adapter has not been created yet in the current scope, so define it
						FunctionDecl *newAdapter = makeAdapter( *funType, realFunction, mangleName, exprTyVars );
						adapter = adapters.insert( adapters.begin(), std::pair< std::string, DeclarationWithType *>( mangleName, newAdapter ) );
						stmtsToAdd.push_back( new DeclStmt( noLabels, newAdapter ) );
					} // if
					assert( adapter != adapters.end() );

					// add the appropriate adapter as a parameter
					appExpr->get_args().push_front( new VariableExpr( adapter->second ) );
				} // if
			} // for
		} // passAdapters

		Expression *makeIncrDecrExpr( ApplicationExpr *appExpr, Type *polyType, bool isIncr ) {
			NameExpr *opExpr;
			if ( isIncr ) {
				opExpr = new NameExpr( "?+=?" );
			} else {
				opExpr = new NameExpr( "?-=?" );
			} // if
			UntypedExpr *addAssign = new UntypedExpr( opExpr );
			if ( AddressExpr *address = dynamic_cast< AddressExpr *>( appExpr->get_args().front() ) ) {
				addAssign->get_args().push_back( address->get_arg() );
			} else {
				addAssign->get_args().push_back( appExpr->get_args().front() );
			} // if
			addAssign->get_args().push_back( new NameExpr( sizeofName( polyType ) ) );
			addAssign->get_results().front() = appExpr->get_results().front()->clone();
			if ( appExpr->get_env() ) {
				addAssign->set_env( appExpr->get_env() );
				appExpr->set_env( 0 );
			} // if
			appExpr->get_args().clear();
			delete appExpr;
			return addAssign;
		}

		Expression *Pass1::handleIntrinsics( ApplicationExpr *appExpr ) {
			if ( VariableExpr *varExpr = dynamic_cast< VariableExpr *>( appExpr->get_function() ) ) {
				if ( varExpr->get_var()->get_linkage() == LinkageSpec::Intrinsic ) {
					if ( varExpr->get_var()->get_name() == "?[?]" ) {
						assert( ! appExpr->get_results().empty() );
						assert( appExpr->get_args().size() == 2 );
						Type *baseType1 = isPolyPtr( appExpr->get_args().front()->get_results().front(), scopeTyVars, env );
						Type *baseType2 = isPolyPtr( appExpr->get_args().back()->get_results().front(), scopeTyVars, env );
						assert( ! baseType1 || ! baseType2 ); // the arguments cannot both be polymorphic pointers
						UntypedExpr *ret = 0;
						if ( baseType1 || baseType2 ) { // one of the arguments is a polymorphic pointer
							ret = new UntypedExpr( new NameExpr( "?+?" ) );
						} // if
						if ( baseType1 ) {
							UntypedExpr *multiply = new UntypedExpr( new NameExpr( "?*?" ) );
							multiply->get_args().push_back( appExpr->get_args().back() );
							multiply->get_args().push_back( new NameExpr( sizeofName( baseType1 ) ) );
							ret->get_args().push_back( appExpr->get_args().front() );
							ret->get_args().push_back( multiply );
						} else if ( baseType2 ) {
							UntypedExpr *multiply = new UntypedExpr( new NameExpr( "?*?" ) );
							multiply->get_args().push_back( appExpr->get_args().front() );
							multiply->get_args().push_back( new NameExpr( sizeofName( baseType2 ) ) );
							ret->get_args().push_back( multiply );
							ret->get_args().push_back( appExpr->get_args().back() );
						} // if
						if ( baseType1 || baseType2 ) {
							ret->get_results().push_front( appExpr->get_results().front()->clone() );
							if ( appExpr->get_env() ) {
								ret->set_env( appExpr->get_env() );
								appExpr->set_env( 0 );
							} // if
							appExpr->get_args().clear();
							delete appExpr;
							return ret;
						} // if
					} else if ( varExpr->get_var()->get_name() == "*?" ) {
						assert( ! appExpr->get_results().empty() );
						assert( ! appExpr->get_args().empty() );
						if ( isPolyType( appExpr->get_results().front(), scopeTyVars, env ) ) {
							Expression *ret = appExpr->get_args().front();
							delete ret->get_results().front();
							ret->get_results().front() = appExpr->get_results().front()->clone();
							if ( appExpr->get_env() ) {
								ret->set_env( appExpr->get_env() );
								appExpr->set_env( 0 );
							} // if
							appExpr->get_args().clear();
							delete appExpr;
							return ret;
						} // if
					} else if ( varExpr->get_var()->get_name() == "?++" || varExpr->get_var()->get_name() == "?--" ) {
						assert( ! appExpr->get_results().empty() );
						assert( appExpr->get_args().size() == 1 );
						if ( Type *baseType = isPolyPtr( appExpr->get_results().front(), scopeTyVars, env ) ) {
							Type *tempType = appExpr->get_results().front()->clone();
							if ( env ) {
								env->apply( tempType );
							} // if
							ObjectDecl *newObj = makeTemporary( tempType );
							VariableExpr *tempExpr = new VariableExpr( newObj );
							UntypedExpr *assignExpr = new UntypedExpr( new NameExpr( "?=?" ) );
							assignExpr->get_args().push_back( tempExpr->clone() );
							if ( AddressExpr *address = dynamic_cast< AddressExpr *>( appExpr->get_args().front() ) ) {
								assignExpr->get_args().push_back( address->get_arg()->clone() );
							} else {
								assignExpr->get_args().push_back( appExpr->get_args().front()->clone() );
							} // if
							CommaExpr *firstComma = new CommaExpr( assignExpr, makeIncrDecrExpr( appExpr, baseType, varExpr->get_var()->get_name() == "?++" ) );
							return new CommaExpr( firstComma, tempExpr );
						} // if
					} else if ( varExpr->get_var()->get_name() == "++?" || varExpr->get_var()->get_name() == "--?" ) {
						assert( ! appExpr->get_results().empty() );
						assert( appExpr->get_args().size() == 1 );
						if ( Type *baseType = isPolyPtr( appExpr->get_results().front(), scopeTyVars, env ) ) {
							return makeIncrDecrExpr( appExpr, baseType, varExpr->get_var()->get_name() == "++?" );
						} // if
					} else if ( varExpr->get_var()->get_name() == "?+?" || varExpr->get_var()->get_name() == "?-?" ) {
						assert( ! appExpr->get_results().empty() );
						assert( appExpr->get_args().size() == 2 );
						Type *baseType1 = isPolyPtr( appExpr->get_args().front()->get_results().front(), scopeTyVars, env );
						Type *baseType2 = isPolyPtr( appExpr->get_args().back()->get_results().front(), scopeTyVars, env );
						if ( baseType1 && baseType2 ) {
							UntypedExpr *divide = new UntypedExpr( new NameExpr( "?/?" ) );
							divide->get_args().push_back( appExpr );
							divide->get_args().push_back( new NameExpr( sizeofName( baseType1 ) ) );
							divide->get_results().push_front( appExpr->get_results().front()->clone() );
							if ( appExpr->get_env() ) {
								divide->set_env( appExpr->get_env() );
								appExpr->set_env( 0 );
							} // if
							return divide;
						} else if ( baseType1 ) {
							UntypedExpr *multiply = new UntypedExpr( new NameExpr( "?*?" ) );
							multiply->get_args().push_back( appExpr->get_args().back() );
							multiply->get_args().push_back( new NameExpr( sizeofName( baseType1 ) ) );
							appExpr->get_args().back() = multiply;
						} else if ( baseType2 ) {
							UntypedExpr *multiply = new UntypedExpr( new NameExpr( "?*?" ) );
							multiply->get_args().push_back( appExpr->get_args().front() );
							multiply->get_args().push_back( new NameExpr( sizeofName( baseType2 ) ) );
							appExpr->get_args().front() = multiply;
						} // if
					} else if ( varExpr->get_var()->get_name() == "?+=?" || varExpr->get_var()->get_name() == "?-=?" ) {
						assert( ! appExpr->get_results().empty() );
						assert( appExpr->get_args().size() == 2 );
						Type *baseType = isPolyPtr( appExpr->get_results().front(), scopeTyVars, env );
						if ( baseType ) {
							UntypedExpr *multiply = new UntypedExpr( new NameExpr( "?*?" ) );
							multiply->get_args().push_back( appExpr->get_args().back() );
							multiply->get_args().push_back( new NameExpr( sizeofName( baseType ) ) );
							appExpr->get_args().back() = multiply;
						} // if
					} // if
					return appExpr;
				} // if
			} // if
			return 0;
		}

		Expression *Pass1::mutate( ApplicationExpr *appExpr ) {
			// std::cerr << "mutate appExpr: ";
			// for ( TyVarMap::iterator i = scopeTyVars.begin(); i != scopeTyVars.end(); ++i ) {
			// 	std::cerr << i->first << " ";
			// }
			// std::cerr << "\n";
			bool oldUseRetval = useRetval;
			useRetval = false;
			appExpr->get_function()->acceptMutator( *this );
			mutateAll( appExpr->get_args(), *this );
			useRetval = oldUseRetval;

			assert( ! appExpr->get_function()->get_results().empty() );
			PointerType *pointer = dynamic_cast< PointerType *>( appExpr->get_function()->get_results().front() );
			assert( pointer );
			FunctionType *function = dynamic_cast< FunctionType *>( pointer->get_base() );
			assert( function );

			if ( Expression *newExpr = handleIntrinsics( appExpr ) ) {
				return newExpr;
			} // if

			Expression *ret = appExpr;

			std::list< Expression *>::iterator arg = appExpr->get_args().begin();
			std::list< Expression *>::iterator paramBegin = appExpr->get_args().begin();

			TyVarMap exprTyVars;
			makeTyVarMap( function, exprTyVars );
			ReferenceToType *polyRetType = 0;

			if ( polyRetType = isPolyRet( function ) ) {
				ret = addPolyRetParam( appExpr, function, polyRetType, arg );
			} else if ( needsAdapter( function, scopeTyVars ) ) {
				// std::cerr << "needs adapter: ";
				// for ( TyVarMap::iterator i = scopeTyVars.begin(); i != scopeTyVars.end(); ++i ) {
				// 	std::cerr << i->first << " ";
				// }
				// std::cerr << "\n";
				// change the application so it calls the adapter rather than the passed function
				ret = applyAdapter( appExpr, function, arg, scopeTyVars );
			} // if
			arg = appExpr->get_args().begin();

			passTypeVars( appExpr, polyRetType, arg, exprTyVars );
			addInferredParams( appExpr, function, arg, exprTyVars );

			arg = paramBegin;

			boxParams( appExpr, function, arg, exprTyVars );

			passAdapters( appExpr, function, exprTyVars );

			return ret;
		}

		Expression *Pass1::mutate( UntypedExpr *expr ) {
			if ( ! expr->get_results().empty() && isPolyType( expr->get_results().front(), scopeTyVars, env ) ) {
				if ( NameExpr *name = dynamic_cast< NameExpr *>( expr->get_function() ) ) {
					if ( name->get_name() == "*?" ) {
						Expression *ret = expr->get_args().front();
						expr->get_args().clear();
						delete expr;
						return ret->acceptMutator( *this );
					} // if
				} // if
			} // if
			return PolyMutator::mutate( expr );
		}

		Expression *Pass1::mutate( AddressExpr *addrExpr ) {
			assert( ! addrExpr->get_arg()->get_results().empty() );

			bool needs = false;
			if ( UntypedExpr *expr = dynamic_cast< UntypedExpr *>( addrExpr->get_arg() ) ) {
				if ( ! expr->get_results().empty() && isPolyType( expr->get_results().front(), scopeTyVars, env ) ) {
					if ( NameExpr *name = dynamic_cast< NameExpr *>( expr->get_function() ) ) {
						if ( name->get_name() == "*?" ) {
							if ( ApplicationExpr * appExpr = dynamic_cast< ApplicationExpr * >( expr->get_args().front() ) ) {
								assert( ! appExpr->get_function()->get_results().empty() );
								PointerType *pointer = dynamic_cast< PointerType *>( appExpr->get_function()->get_results().front() );
								assert( pointer );
								FunctionType *function = dynamic_cast< FunctionType *>( pointer->get_base() );
								assert( function );
								needs = needsAdapter( function, scopeTyVars );
							} // if
						} // if
					} // if
				} // if
			} // if
			addrExpr->set_arg( mutateExpression( addrExpr->get_arg() ) );
			if ( isPolyType( addrExpr->get_arg()->get_results().front(), scopeTyVars, env ) || needs ) {
				Expression *ret = addrExpr->get_arg();
				delete ret->get_results().front();
				ret->get_results().front() = addrExpr->get_results().front()->clone();
				addrExpr->set_arg( 0 );
				delete addrExpr;
				return ret;
			} else {
				return addrExpr;
			} // if
		}

		/// Wraps a function declaration in a new pointer-to-function variable expression
		VariableExpr *wrapFunctionDecl( DeclarationWithType *functionDecl ) {
			// line below cloned from FixFunction.cc
			ObjectDecl *functionObj = new ObjectDecl( functionDecl->get_name(), functionDecl->get_storageClass(), functionDecl->get_linkage(), 0,
			                                          new PointerType( Type::Qualifiers(), functionDecl->get_type()->clone() ), 0 );
			functionObj->set_mangleName( functionDecl->get_mangleName() );
			return new VariableExpr( functionObj );
		}
		
		Statement * Pass1::mutate( ReturnStmt *returnStmt ) {
			if ( retval && returnStmt->get_expr() ) {
				assert( ! returnStmt->get_expr()->get_results().empty() );
				// ***** Code Removal ***** After introducing a temporary variable for all return expressions, the following code appears superfluous.
				// if ( returnStmt->get_expr()->get_results().front()->get_isLvalue() ) {
				// by this point, a cast expr on a polymorphic return value is redundant
				while ( CastExpr *castExpr = dynamic_cast< CastExpr *>( returnStmt->get_expr() ) ) {
					returnStmt->set_expr( castExpr->get_arg() );
					returnStmt->get_expr()->set_env( castExpr->get_env() );
					castExpr->set_env( 0 );
					castExpr->set_arg( 0 );
					delete castExpr;
				} //while

				// find assignment operator for (polymorphic) return type
				ApplicationExpr *assignExpr = 0;
				if ( TypeInstType *typeInst = dynamic_cast< TypeInstType *>( retval->get_type() ) ) {
					// find assignment operator for type variable
					std::map< std::string, DeclarationWithType *>::const_iterator assignIter = assignOps.find( typeInst->get_name() );
					if ( assignIter == assignOps.end() ) {
						throw SemanticError( "Attempt to return dtype or ftype object in ", returnStmt->get_expr() );
					} // if
					assignExpr = new ApplicationExpr( new VariableExpr( assignIter->second ) );
				} else if ( ReferenceToType *refType = dynamic_cast< ReferenceToType *>( retval->get_type() ) ) {
					// find assignment operator for generic type
					ScopedMap< std::string, DeclarationWithType *>::const_iterator assignIter = scopedAssignOps.find( refType->get_name() );
					if ( assignIter == scopedAssignOps.end() ) {
						throw SemanticError( "Attempt to return dtype or ftype generic object in ", returnStmt->get_expr() );
					}

					// wrap it up in an application expression
					DeclarationWithType *functionDecl = assignIter->second;
					assignExpr = new ApplicationExpr( wrapFunctionDecl( functionDecl ) );
					assignExpr->set_env( env->clone() );

					// find each of its needed secondary assignment operators
					std::list< Expression* > &tyParams = refType->get_parameters();
					std::list< TypeDecl* > &forallParams = functionDecl->get_type()->get_forall();
					std::list< Expression* >::const_iterator tyIt = tyParams.begin();
					std::list< TypeDecl* >::const_iterator forallIt = forallParams.begin();
					for ( ; tyIt != tyParams.end() && forallIt != forallParams.end(); ++tyIt, ++forallIt ) {
						if ( (*forallIt)->get_kind() != TypeDecl::Any ) continue; // skip types with no assign op (ftype/dtype)

						std::list< DeclarationWithType* > &asserts = (*forallIt)->get_assertions();
						assert( ! asserts.empty() && "Type param needs assignment operator assertion" );
						DeclarationWithType *actualDecl = asserts.front();
						ReferenceToType *actualType = isAssignment( actualDecl );
						assert( actualType && "First assertion of type with assertions should be assignment operator" );
						TypeExpr *formalTypeExpr = dynamic_cast< TypeExpr* >( *tyIt );
						assert( formalTypeExpr && "type parameters must be type expressions" );
						Type *formalType = formalTypeExpr->get_type();
						assignExpr->get_env()->add( actualType->get_name(), formalType );
						
						DeclarationWithType *assertAssign = 0;
						if ( TypeInstType *formalTypeInstType = dynamic_cast< TypeInstType* >( formalType ) ) {
							std::map< std::string, DeclarationWithType *>::const_iterator assertAssignIt = assignOps.find( formalTypeInstType->get_name() );
							if ( assertAssignIt == assignOps.end() ) {
								throw SemanticError( "No assignment operation found for ", formalTypeInstType );
							}
							assertAssign = assertAssignIt->second;
							//assignExpr->get_env()->add( formalTypeInstType->get_name(), actualType );
						} else if ( ReferenceToType *formalReferenceType = dynamic_cast< ReferenceToType* >( formalType ) )  {
							ScopedMap< std::string, DeclarationWithType *>::const_iterator assertAssignIt = scopedAssignOps.find( formalReferenceType->get_name() );
							if ( assertAssignIt == scopedAssignOps.end() ) {
								throw SemanticError( "No assignment operation found for ", formalReferenceType );
							}
							assertAssign = assertAssignIt->second;
						} else assert( false && "returning polymorphic types with non struct/polymorphic parameters not yet supported" );
						

						assignExpr->get_inferParams()[ actualDecl->get_uniqueId() ]
							= ParamEntry( assertAssign->get_uniqueId(), assertAssign->get_type()->clone(), actualDecl->get_type()->clone(), wrapFunctionDecl( assertAssign ) );
					}
				}
				assert( assignExpr );

				// replace return statement with appropriate assignment to out parameter
				Expression *retParm = new NameExpr( retval->get_name() );
				retParm->get_results().push_back( new PointerType( Type::Qualifiers(), retval->get_type()->clone() ) );
				assignExpr->get_args().push_back( retParm );
				assignExpr->get_args().push_back( returnStmt->get_expr() );
				stmtsToAdd.push_back( new ExprStmt( noLabels, mutateExpression( assignExpr ) ) );
				// } else {
				// 	useRetval = true;
				// 	stmtsToAdd.push_back( new ExprStmt( noLabels, mutateExpression( returnStmt->get_expr() ) ) );
				// 	useRetval = false;
				// } // if
				returnStmt->set_expr( 0 );
			} else {
				returnStmt->set_expr( mutateExpression( returnStmt->get_expr() ) );
			} // if
			return returnStmt;
		}

		Type * Pass1::mutate( PointerType *pointerType ) {
			TyVarMap oldtyVars = scopeTyVars;
			makeTyVarMap( pointerType, scopeTyVars );

			Type *ret = Mutator::mutate( pointerType );

			scopeTyVars = oldtyVars;
			return ret;
		}

		Type * Pass1::mutate( FunctionType *functionType ) {
			TyVarMap oldtyVars = scopeTyVars;
			makeTyVarMap( functionType, scopeTyVars );

			Type *ret = Mutator::mutate( functionType );

			scopeTyVars = oldtyVars;
			return ret;
		}

		void Pass1::doBeginScope() {
			// push a copy of the current map
			adapters.push(adapters.top());
			scopedAssignOps.beginScope();
		}

		void Pass1::doEndScope() {
			adapters.pop();
			scopedAssignOps.endScope();
		}

////////////////////////////////////////// Pass2 ////////////////////////////////////////////////////

		void Pass2::addAdapters( FunctionType *functionType ) {
			std::list< DeclarationWithType *> &paramList = functionType->get_parameters();
			std::list< FunctionType *> functions;
			for ( std::list< DeclarationWithType *>::iterator arg = paramList.begin(); arg != paramList.end(); ++arg ) {
				Type *orig = (*arg)->get_type();
				findAndReplaceFunction( orig, functions, scopeTyVars, needsAdapter );
				(*arg)->set_type( orig );
			}
			std::set< std::string > adaptersDone;
			for ( std::list< FunctionType *>::iterator funType = functions.begin(); funType != functions.end(); ++funType ) {
				std::string mangleName = mangleAdapterName( *funType, scopeTyVars );
				if ( adaptersDone.find( mangleName ) == adaptersDone.end() ) {
					std::string adapterName = makeAdapterName( mangleName );
					paramList.push_front( new ObjectDecl( adapterName, DeclarationNode::NoStorageClass, LinkageSpec::C, 0, new PointerType( Type::Qualifiers(), makeAdapterType( *funType, scopeTyVars ) ), 0 ) );
					adaptersDone.insert( adaptersDone.begin(), mangleName );
				}
			}
//  deleteAll( functions );
		}

		template< typename DeclClass >
		DeclClass * Pass2::handleDecl( DeclClass *decl, Type *type ) {
			DeclClass *ret = static_cast< DeclClass *>( Mutator::mutate( decl ) );

			return ret;
		}

		DeclarationWithType * Pass2::mutate( FunctionDecl *functionDecl ) {
			return handleDecl( functionDecl, functionDecl->get_functionType() );
		}

		ObjectDecl * Pass2::mutate( ObjectDecl *objectDecl ) {
			return handleDecl( objectDecl, objectDecl->get_type() );
		}

		TypeDecl * Pass2::mutate( TypeDecl *typeDecl ) {
			scopeTyVars[ typeDecl->get_name() ] = typeDecl->get_kind();
			if ( typeDecl->get_base() ) {
				return handleDecl( typeDecl, typeDecl->get_base() );
			} else {
				return Mutator::mutate( typeDecl );
			}
		}

		TypedefDecl * Pass2::mutate( TypedefDecl *typedefDecl ) {
			return handleDecl( typedefDecl, typedefDecl->get_base() );
		}

		Type * Pass2::mutate( PointerType *pointerType ) {
			TyVarMap oldtyVars = scopeTyVars;
			makeTyVarMap( pointerType, scopeTyVars );

			Type *ret = Mutator::mutate( pointerType );

			scopeTyVars = oldtyVars;
			return ret;
		}

		Type *Pass2::mutate( FunctionType *funcType ) {
			TyVarMap oldtyVars = scopeTyVars;
			makeTyVarMap( funcType, scopeTyVars );

			// move polymorphic return type to parameter list
			if ( isPolyRet( funcType ) ) {
				DeclarationWithType *ret = funcType->get_returnVals().front();
				ret->set_type( new PointerType( Type::Qualifiers(), ret->get_type() ) );
				funcType->get_parameters().push_front( ret );
				funcType->get_returnVals().pop_front();
			}

			// add size/align and assertions for type parameters to parameter list
			std::list< DeclarationWithType *>::iterator last = funcType->get_parameters().begin();
			std::list< DeclarationWithType *> inferredParams;
			ObjectDecl newObj( "", DeclarationNode::NoStorageClass, LinkageSpec::C, 0, new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ), 0 );
			ObjectDecl newPtr( "", DeclarationNode::NoStorageClass, LinkageSpec::C, 0,
			                   new PointerType( Type::Qualifiers(), new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ) ), 0 );
//   ObjectDecl *newFunPtr = new ObjectDecl( "", DeclarationNode::NoStorageClass, LinkageSpec::Cforall, 0, new PointerType( Type::Qualifiers(), new FunctionType( Type::Qualifiers(), true ) ), 0 );
			for ( std::list< TypeDecl *>::const_iterator tyParm = funcType->get_forall().begin(); tyParm != funcType->get_forall().end(); ++tyParm ) {
				ObjectDecl *sizeParm, *alignParm;
				// add all size and alignment parameters to parameter list
				if ( (*tyParm)->get_kind() == TypeDecl::Any ) {
					TypeInstType parmType( Type::Qualifiers(), (*tyParm)->get_name(), *tyParm );

					sizeParm = newObj.clone();
					sizeParm->set_name( sizeofName( &parmType ) );
					last = funcType->get_parameters().insert( last, sizeParm );
					++last;

					alignParm = newObj.clone();
					alignParm->set_name( alignofName( &parmType ) );
					last = funcType->get_parameters().insert( last, alignParm );
					++last;
				}
				// move all assertions into parameter list
				for ( std::list< DeclarationWithType *>::iterator assert = (*tyParm)->get_assertions().begin(); assert != (*tyParm)->get_assertions().end(); ++assert ) {
//      *assert = (*assert)->acceptMutator( *this );
					inferredParams.push_back( *assert );
				}
				(*tyParm)->get_assertions().clear();
			}

			// add size/align for generic parameter types to parameter list
			std::set< std::string > seenTypes; // sizeofName for generic types we've seen
			for ( std::list< DeclarationWithType* >::const_iterator fnParm = last; fnParm != funcType->get_parameters().end(); ++fnParm ) {
				Type *polyBase = hasPolyBase( (*fnParm)->get_type(), scopeTyVars );
				if ( polyBase && ! dynamic_cast< TypeInstType* >( polyBase ) ) {
					std::string sizeName = sizeofName( polyBase );
					if ( seenTypes.count( sizeName ) ) continue;

					ObjectDecl *sizeParm, *alignParm, *offsetParm;
					sizeParm = newObj.clone();
					sizeParm->set_name( sizeName );
					last = funcType->get_parameters().insert( last, sizeParm );
					++last;

					alignParm = newObj.clone();
					alignParm->set_name( alignofName( polyBase ) );
					last = funcType->get_parameters().insert( last, alignParm );
					++last;

					if ( dynamic_cast< StructInstType* >( polyBase ) ) {
						offsetParm = newPtr.clone();
						offsetParm->set_name( offsetofName( polyBase ) );
						last = funcType->get_parameters().insert( last, offsetParm );
						++last;
					}

					seenTypes.insert( sizeName );
				}
			}

			// splice assertion parameters into parameter list
			funcType->get_parameters().splice( last, inferredParams );
			addAdapters( funcType );
			mutateAll( funcType->get_returnVals(), *this );
			mutateAll( funcType->get_parameters(), *this );

			scopeTyVars = oldtyVars;
			return funcType;
		}

////////////////////////////////////////// MemberExprFixer ////////////////////////////////////////////////////

		template< typename DeclClass >
		DeclClass * MemberExprFixer::handleDecl( DeclClass *decl, Type *type ) {
			TyVarMap oldtyVars = scopeTyVars;
			makeTyVarMap( type, scopeTyVars );

			DeclClass *ret = static_cast< DeclClass *>( Mutator::mutate( decl ) );

			scopeTyVars = oldtyVars;
			return ret;
		}

		ObjectDecl * MemberExprFixer::mutate( ObjectDecl *objectDecl ) {
			return handleDecl( objectDecl, objectDecl->get_type() );
		}

		DeclarationWithType * MemberExprFixer::mutate( FunctionDecl *functionDecl ) {
			return handleDecl( functionDecl, functionDecl->get_functionType() );
		}

		TypedefDecl * MemberExprFixer::mutate( TypedefDecl *typedefDecl ) {
			return handleDecl( typedefDecl, typedefDecl->get_base() );
		}

		TypeDecl * MemberExprFixer::mutate( TypeDecl *typeDecl ) {
			scopeTyVars[ typeDecl->get_name() ] = typeDecl->get_kind();
			return Mutator::mutate( typeDecl );
		}

		Type * MemberExprFixer::mutate( PointerType *pointerType ) {
			TyVarMap oldtyVars = scopeTyVars;
			makeTyVarMap( pointerType, scopeTyVars );

			Type *ret = Mutator::mutate( pointerType );

			scopeTyVars = oldtyVars;
			return ret;
		}

		Type * MemberExprFixer::mutate( FunctionType *functionType ) {
			TyVarMap oldtyVars = scopeTyVars;
			makeTyVarMap( functionType, scopeTyVars );

			Type *ret = Mutator::mutate( functionType );

			scopeTyVars = oldtyVars;
			return ret;
		}

		Statement *MemberExprFixer::mutate( DeclStmt *declStmt ) {
			if ( ObjectDecl *objectDecl = dynamic_cast< ObjectDecl *>( declStmt->get_decl() ) ) {
				if ( isPolyType( objectDecl->get_type(), scopeTyVars ) ) {
					// change initialization of a polymorphic value object
					// to allocate storage with alloca
					Type *declType = objectDecl->get_type();
					UntypedExpr *alloc = new UntypedExpr( new NameExpr( "__builtin_alloca" ) );
					alloc->get_args().push_back( new NameExpr( sizeofName( declType ) ) );

					delete objectDecl->get_init();

					std::list<Expression*> designators;
					objectDecl->set_init( new SingleInit( alloc, designators ) );
				}
			}
			return Mutator::mutate( declStmt );
		}

		/// Finds the member in the base list that matches the given declaration; returns its index, or -1 if not present
		long findMember( DeclarationWithType *memberDecl, std::list< Declaration* > &baseDecls ) {
			long i = 0;
			for(std::list< Declaration* >::const_iterator decl = baseDecls.begin(); decl != baseDecls.end(); ++decl, ++i ) {
				if ( memberDecl->get_name() != (*decl)->get_name() ) continue;

				if ( DeclarationWithType *declWithType = dynamic_cast< DeclarationWithType* >( *decl ) ) {
					if ( memberDecl->get_mangleName().empty() || declWithType->get_mangleName().empty()
					     || memberDecl->get_mangleName() == declWithType->get_mangleName() ) return i;
					else continue;
				} else return i;
			}
			return -1;
		}

		/// Returns an index expression into the offset array for a type
		Expression *makeOffsetIndex( Type *objectType, long i ) {
			std::stringstream offset_namer;
			offset_namer << i;
			ConstantExpr *fieldIndex = new ConstantExpr( Constant( new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ), offset_namer.str() ) );
			UntypedExpr *fieldOffset = new UntypedExpr( new NameExpr( "?[?]" ) );
			fieldOffset->get_args().push_back( new NameExpr( offsetofName( objectType ) ) );
			fieldOffset->get_args().push_back( fieldIndex );
			return fieldOffset;
		}

		/// Returns an expression dereferenced n times
		Expression *makeDerefdVar( Expression *derefdVar, long n ) {
			for ( int i = 1; i < n; ++i ) {
				UntypedExpr *derefExpr = new UntypedExpr( new NameExpr( "*?" ) );
				derefExpr->get_args().push_back( derefdVar );
				derefdVar = derefExpr;
			}
			return derefdVar;
		}
		
		Expression *MemberExprFixer::mutate( MemberExpr *memberExpr ) {
			// mutate, exiting early if no longer MemberExpr
			Expression *expr = Mutator::mutate( memberExpr );
			memberExpr = dynamic_cast< MemberExpr* >( expr );
			if ( ! memberExpr ) return expr;

			// get declaration for base struct, exiting early if not found
			int varDepth;
			VariableExpr *varExpr = getBaseVar( memberExpr->get_aggregate(), &varDepth );
			if ( ! varExpr ) return memberExpr;
			ObjectDecl *objectDecl = dynamic_cast< ObjectDecl* >( varExpr->get_var() );
			if ( ! objectDecl ) return memberExpr;

			// only mutate member expressions for polymorphic types
			int tyDepth;
			Type *objectType = hasPolyBase( objectDecl->get_type(), scopeTyVars, &tyDepth );
			if ( ! objectType ) return memberExpr;

			if ( StructInstType *structType = dynamic_cast< StructInstType* >( objectType ) ) {
				// look up offset index
				long i = findMember( memberExpr->get_member(), structType->get_baseStruct()->get_members() );
				if ( i == -1 ) return memberExpr;

				// replace member expression with pointer to base plus offset
				UntypedExpr *fieldLoc = new UntypedExpr( new NameExpr( "?+?" ) );
				fieldLoc->get_args().push_back( makeDerefdVar( varExpr->clone(), varDepth ) );
				fieldLoc->get_args().push_back( makeOffsetIndex( objectType, i ) );

				delete memberExpr;
				return fieldLoc;
			} else if ( dynamic_cast< UnionInstType* >( objectType ) ) {
				// union members are all at offset zero, so build appropriately-dereferenced variable
				Expression *derefdVar = makeDerefdVar( varExpr->clone(), varDepth );
				delete memberExpr;
				return derefdVar;
			} else return memberExpr;
		}

		Expression *MemberExprFixer::mutate( OffsetofExpr *offsetofExpr ) {
			// mutate, exiting early if no longer OffsetofExpr
			Expression *expr = Mutator::mutate( offsetofExpr );
			offsetofExpr = dynamic_cast< OffsetofExpr* >( expr );
			if ( ! offsetofExpr ) return expr;

			// only mutate expressions for polymorphic structs/unions
			Type *ty = isPolyType( offsetofExpr->get_type(), scopeTyVars );
			if ( ! ty ) return offsetofExpr;

			if ( StructInstType *structType = dynamic_cast< StructInstType* >( ty ) ) {
				// replace offsetof expression by index into offset array
				long i = findMember( offsetofExpr->get_member(), structType->get_baseStruct()->get_members() );
				if ( i == -1 ) return offsetofExpr;

				Expression *offsetInd = makeOffsetIndex( ty, i );
				delete offsetofExpr;
				return offsetInd;
			} else if ( dynamic_cast< UnionInstType* >( ty ) ) {
				// all union members are at offset zero
				delete offsetofExpr;
				return new ConstantExpr( Constant( new BasicType( Type::Qualifiers(), BasicType::LongUnsignedInt ), std::string("0") ) );
			} else return offsetofExpr;
		}

////////////////////////////////////////// Pass3 ////////////////////////////////////////////////////

		template< typename DeclClass >
		DeclClass * Pass3::handleDecl( DeclClass *decl, Type *type ) {
			TyVarMap oldtyVars = scopeTyVars;
			makeTyVarMap( type, scopeTyVars );

			DeclClass *ret = static_cast< DeclClass *>( Mutator::mutate( decl ) );
			ScrubTyVars::scrub( decl, scopeTyVars );

			scopeTyVars = oldtyVars;
			return ret;
		}

		ObjectDecl * Pass3::mutate( ObjectDecl *objectDecl ) {
			return handleDecl( objectDecl, objectDecl->get_type() );
		}

		DeclarationWithType * Pass3::mutate( FunctionDecl *functionDecl ) {
			return handleDecl( functionDecl, functionDecl->get_functionType() );
		}

		TypedefDecl * Pass3::mutate( TypedefDecl *typedefDecl ) {
			return handleDecl( typedefDecl, typedefDecl->get_base() );
		}

		TypeDecl * Pass3::mutate( TypeDecl *typeDecl ) {
//   Initializer *init = 0;
//   std::list< Expression *> designators;
//   scopeTyVars[ typeDecl->get_name() ] = typeDecl->get_kind();
//   if ( typeDecl->get_base() ) {
//     init = new SimpleInit( new SizeofExpr( handleDecl( typeDecl, typeDecl->get_base() ) ), designators );
//   }
//   return new ObjectDecl( typeDecl->get_name(), Declaration::Extern, LinkageSpec::C, 0, new BasicType( Type::Qualifiers(), BasicType::UnsignedInt ), init );

			scopeTyVars[ typeDecl->get_name() ] = typeDecl->get_kind();
			return Mutator::mutate( typeDecl );
		}

		Type * Pass3::mutate( PointerType *pointerType ) {
			TyVarMap oldtyVars = scopeTyVars;
			makeTyVarMap( pointerType, scopeTyVars );

			Type *ret = Mutator::mutate( pointerType );

			scopeTyVars = oldtyVars;
			return ret;
		}

		Type * Pass3::mutate( FunctionType *functionType ) {
			TyVarMap oldtyVars = scopeTyVars;
			makeTyVarMap( functionType, scopeTyVars );

			Type *ret = Mutator::mutate( functionType );

			scopeTyVars = oldtyVars;
			return ret;
		}
	} // anonymous namespace
} // namespace GenPoly

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
