//
// Cforall Version 1.0.0 Copyright (C) 2016 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// algorithm.c --
//
// Author           : Peter A. Buhr
// Created On       : Thu Jan 28 17:10:29 2016
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon Jan  1 19:03:16 2018
// Update Count     : 437
//

#include "stdlib"

//---------------------------------------

#define _XOPEN_SOURCE 600								// posix_memalign, *rand48
#include <string.h>										// memcpy, memset
#include <malloc.h>										// malloc_usable_size
#include <math.h>										// fabsf, fabs, fabsl
#include <complex.h>									// _Complex_I
#include <assert.h>

// resize, non-array types
forall( dtype T | sized(T) ) T * alloc( T ptr[], size_t dim, char fill ) {
	size_t olen = malloc_usable_size( ptr );			// current allocation
    char * nptr = (void *)realloc( (void *)ptr, dim * (size_t)sizeof(T) ); // C realloc
	size_t nlen = malloc_usable_size( nptr );			// new allocation
	if ( nlen > olen ) {								// larger ?
		memset( nptr + olen, (int)fill, nlen - olen );	// initialize added storage
	} //
    return (T *)nptr;
} // alloc

// allocation/deallocation and constructor/destructor, non-array types
forall( dtype T | sized(T), ttype Params | { void ?{}( T &, Params ); } )
T * new( Params p ) {
	return &(*malloc()){ p };								// run constructor
} // new

forall( dtype T | sized(T) | { void ^?{}( T & ); } )
void delete( T * ptr ) {
	if ( ptr ) {										// ignore null
		^(*ptr){};											// run destructor
		free( ptr );
	} // if
} // delete

forall( dtype T, ttype Params | sized(T) | { void ^?{}( T & ); void delete( Params ); } )
void delete( T * ptr, Params rest ) {
	if ( ptr ) {										// ignore null
		^(*ptr){};											// run destructor
		free( ptr );
	} // if
	delete( rest );
} // delete


// allocation/deallocation and constructor/destructor, array types
forall( dtype T | sized(T), ttype Params | { void ?{}( T &, Params ); } )
T * anew( size_t dim, Params p ) {
	T *arr = alloc( dim );
	for ( unsigned int i = 0; i < dim; i += 1 ) {
		(arr[i]){ p };									// run constructor
	} // for
	return arr;
} // anew

forall( dtype T | sized(T) | { void ^?{}( T & ); } )
void adelete( size_t dim, T arr[] ) {
	if ( arr ) {										// ignore null
		for ( int i = dim - 1; i >= 0; i -= 1 ) {		// reverse allocation order, must be unsigned
			^(arr[i]){};								// run destructor
		} // for
		free( arr );
	} // if
} // adelete

forall( dtype T | sized(T) | { void ^?{}( T & ); }, ttype Params | { void adelete( Params ); } )
void adelete( size_t dim, T arr[], Params rest ) {
	if ( arr ) {										// ignore null
		for ( int i = dim - 1; i >= 0; i -= 1 ) {		// reverse allocation order, must be unsigned
			^(arr[i]){};								// run destructor
		} // for
		free( arr );
	} // if
	adelete( rest );
} // adelete

//---------------------------------------

float _Complex strto( const char * sptr, char ** eptr ) {
	float re, im;
	char * eeptr;
	re = strtof( sptr, &eeptr );
	if ( sptr == *eeptr ) { if ( eptr != 0 ) *eptr = eeptr; return 0.0f + 0.0f * _Complex_I; }
	im = strtof( eeptr, &eeptr );
	if ( sptr == *eeptr ) { if ( eptr != 0 ) *eptr = eeptr; return 0.0f + 0.0f * _Complex_I; }
	if ( *eeptr != 'i' ) { if ( eptr != 0 ) *eptr = eeptr; return 0.0f + 0.0f * _Complex_I; }
	return re + im * _Complex_I;
} // strto

double _Complex strto( const char * sptr, char ** eptr ) {
	double re, im;
	char * eeptr;
	re = strtod( sptr, &eeptr );
	if ( sptr == *eeptr ) { if ( eptr != 0 ) *eptr = eeptr; return 0.0 + 0.0 * _Complex_I; }
	im = strtod( eeptr, &eeptr );
	if ( sptr == *eeptr ) { if ( eptr != 0 ) *eptr = eeptr; return 0.0 + 0.0 * _Complex_I; }
	if ( *eeptr != 'i' ) { if ( eptr != 0 ) *eptr = eeptr; return 0.0 + 0.0 * _Complex_I; }
	return re + im * _Complex_I;
} // strto

long double _Complex strto( const char * sptr, char ** eptr ) {
	long double re, im;
	char * eeptr;
	re = strtold( sptr, &eeptr );
	if ( sptr == *eeptr ) { if ( eptr != 0 ) *eptr = eeptr; return 0.0L + 0.0L * _Complex_I; }
	im = strtold( eeptr, &eeptr );
	if ( sptr == *eeptr ) { if ( eptr != 0 ) *eptr = eeptr; return 0.0L + 0.0L * _Complex_I; }
	if ( *eeptr != 'i' ) { if ( eptr != 0 ) *eptr = eeptr; return 0.0L + 0.0L * _Complex_I; }
	return re + im * _Complex_I;
} // strto

//---------------------------------------

forall( otype E | { int ?<?( E, E ); } )
E * bsearch( E key, const E * vals, size_t dim ) {
	int cmp( const void * t1, const void * t2 ) {
		return *(E *)t1 < *(E *)t2 ? -1 : *(E *)t2 < *(E *)t1 ? 1 : 0;
	} // cmp
	return (E *)bsearch( &key, vals, dim, sizeof(E), cmp );
} // bsearch

forall( otype E | { int ?<?( E, E ); } )
size_t bsearch( E key, const E * vals, size_t dim ) {
	E * result = bsearch( key, vals, dim );
	return result ? result - vals : dim;				// pointer subtraction includes sizeof(E)
} // bsearch

forall( otype K, otype E | { int ?<?( K, K ); K getKey( const E & ); } )
E * bsearch( K key, const E * vals, size_t dim ) {
	int cmp( const void * t1, const void * t2 ) {
		return *(K *)t1 < getKey( *(E *)t2 ) ? -1 : getKey( *(E *)t2 ) < *(K *)t1 ? 1 : 0;
	} // cmp
	return (E *)bsearch( &key, vals, dim, sizeof(E), cmp );
} // bsearch

forall( otype K, otype E | { int ?<?( K, K ); K getKey( const E & ); } )
size_t bsearch( K key, const E * vals, size_t dim ) {
	E * result = bsearch( key, vals, dim );
	return result ? result - vals : dim;				// pointer subtraction includes sizeof(E)
} // bsearch


forall( otype E | { int ?<?( E, E ); } )
size_t bsearchl( E key, const E * vals, size_t dim ) {
	size_t l = 0, m, h = dim;
	while ( l < h ) {
		m = (l + h) / 2;
		if ( (E &)(vals[m]) < key ) {					// cast away const
			l = m + 1;
		} else {
			h = m;
		} // if
	} // while
	return l;
} // bsearchl

forall( otype E | { int ?<?( E, E ); } )
E * bsearchl( E key, const E * vals, size_t dim ) {
	size_t posn = bsearchl( key, vals, dim );
	return (E *)(&vals[posn]);							// cast away const
} // bsearchl

forall( otype K, otype E | { int ?<?( K, K ); K getKey( const E & ); } )
size_t bsearchl( K key, const E * vals, size_t dim ) {
	size_t l = 0, m, h = dim;
	while ( l < h ) {
		m = (l + h) / 2;
		if ( getKey( vals[m] ) < key ) {
			l = m + 1;
		} else {
			h = m;
		} // if
	} // while
	return l;
} // bsearchl

forall( otype K, otype E | { int ?<?( K, K ); K getKey( const E & ); } )
E * bsearchl( K key, const E * vals, size_t dim ) {
	size_t posn = bsearchl( key, vals, dim );
	return (E *)(&vals[posn]);							// cast away const
} // bsearchl


forall( otype E | { int ?<?( E, E ); } )
size_t bsearchu( E key, const E * vals, size_t dim ) {
	size_t l = 0, m, h = dim;
	while ( l < h ) {
		m = (l + h) / 2;
		if ( ! ( key < (E &)(vals[m]) ) ) {				// cast away const
			l = m + 1;
		} else {
			h = m;
		} // if
	} // while
	return l;
} // bsearchu

forall( otype E | { int ?<?( E, E ); } )
E * bsearchu( E key, const E * vals, size_t dim ) {
	size_t posn = bsearchu( key, vals, dim );
	return (E *)(&vals[posn]);
} // bsearchu

forall( otype K, otype E | { int ?<?( K, K ); K getKey( const E & ); } )
size_t bsearchu( K key, const E * vals, size_t dim ) {
	size_t l = 0, m, h = dim;
	while ( l < h ) {
		m = (l + h) / 2;
		if ( ! ( key < getKey( vals[m] ) ) ) {
			l = m + 1;
		} else {
			h = m;
		} // if
	} // while
	return l;
} // bsearchu

forall( otype K, otype E | { int ?<?( K, K ); K getKey( const E & ); } )
E * bsearchu( K key, const E * vals, size_t dim ) {
	size_t posn = bsearchu( key, vals, dim );
	return (E *)(&vals[posn]);
} // bsearchu


forall( otype E | { int ?<?( E, E ); } )
void qsort( E * vals, size_t dim ) {
	int cmp( const void * t1, const void * t2 ) {
		return *(E *)t1 < *(E *)t2 ? -1 : *(E *)t2 < *(E *)t1 ? 1 : 0;
	} // cmp
	qsort( vals, dim, sizeof(E), cmp );
} // qsort

//---------------------------------------

[ int, int ] div( int num, int denom ) { div_t qr = div( num, denom ); return [ qr.quot, qr.rem ]; }
[ long int, long int ] div( long int num, long int denom ) { ldiv_t qr = ldiv( num, denom ); return [ qr.quot, qr.rem ]; }
[ long long int, long long int ] div( long long int num, long long int denom ) { lldiv_t qr = lldiv( num, denom ); return [ qr.quot, qr.rem ]; }
forall( otype T | { T ?/?( T, T ); T ?%?( T, T ); } )
[ T, T ] div( T num, T denom ) { return [ num / denom, num % denom ]; }

//---------------------------------------

void random_seed( long int s ) { srand48( s ); srandom( s ); } // call srandom to harmonize with C-lib random
char random( void ) { return (unsigned long int)random(); }
char random( char u ) { return random( (unsigned long int)u ); }
char random( char l, char u ) { return random( (unsigned long int)l, (unsigned long int)u ); }
int random( void ) { return (long int)random(); }
int random( int u ) { return random( (long int)u ); }
int random( int l, int u ) { return random( (long int)l, (long int)u ); }
unsigned int random( void ) { return (unsigned long int)random(); }
unsigned int random( unsigned int u ) { return random( (unsigned long int)u ); }
unsigned int random( unsigned int l, unsigned int u ) { return random( (unsigned long int)l, (unsigned long int)u ); }
//extern "C" { long int random() { return mrand48(); } }
long int random( long int u ) { if ( u < 0 ) return random( u, 0 ); else return random( 0, u ); }
long int random( long int l, long int u ) { assert( l < u ); return lrand48() % (u - l) + l; }
unsigned long int random( void ) { return lrand48(); }
unsigned long int random( unsigned long int u ) { return lrand48() % u; }
unsigned long int random( unsigned long int l, unsigned long int u ) { assert( l < u ); return lrand48() % (u - l) + l; }
float random( void ) { return (float)drand48(); }		// cast otherwise float uses lrand48
double random( void ) { return drand48(); }
float _Complex random( void ) { return (float)drand48() + (float _Complex)(drand48() * _Complex_I); }
double _Complex random( void ) { return drand48() + (double _Complex)(drand48() * _Complex_I); }
long double _Complex random( void ) { return (long double)drand48() + (long double _Complex)(drand48() * _Complex_I); }


// Local Variables: //
// tab-width: 4 //
// End: //
