//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Statement.cc --
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Peter A. Buhr
// Last Modified On : Sun Sep  3 20:46:44 2017
// Update Count     : 68
//

#include "SynTree/Statement.h"

#include <stddef.h>                // for NULL
#include <cassert>                 // for assert, assertf
#include <iostream>                // for operator<<, basic_ostream, endl
#include <list>                    // for list, list<>::const_iterator, _Lis...
#include <string>                  // for operator<<, string, char_traits

#include "Common/SemanticError.h"  // for SemanticError
#include "Common/utility.h"        // for maybeClone, cloneAll, deleteAll
#include "Declaration.h"           // for Declaration
#include "Expression.h"            // for Expression, ConstantExpr
#include "Statement.h"             // for Statement, ForStmt, AsmStmt, Catch...
#include "SynTree/Label.h"         // for Label, operator<<

using std::string;
using std::endl;

Statement::Statement( std::list<Label> labels ) : labels( labels ) {}

void Statement::print( __attribute__((unused)) std::ostream &, __attribute__((unused)) int indent ) const {}

Statement::~Statement() {}

ExprStmt::ExprStmt( std::list<Label> labels, Expression *expr ) : Statement( labels ), expr( expr ) {}

ExprStmt::ExprStmt( const ExprStmt &other ) : Statement( other ), expr( maybeClone( other.expr ) ) {}

ExprStmt::~ExprStmt() {
	delete expr;
}

void ExprStmt::print( std::ostream &os, int indent ) const {
	os << "Expression Statement:" << endl << std::string( indent + 2, ' ' );
	expr->print( os, indent + 2 );
}


AsmStmt::AsmStmt( std::list<Label> labels, bool voltile, Expression *instruction, std::list<Expression *> output, std::list<Expression *> input, std::list<ConstantExpr *> clobber, std::list<Label> gotolabels ) : Statement( labels ), voltile( voltile ), instruction( instruction ), output( output ), input( input ), clobber( clobber ), gotolabels( gotolabels ) {}

AsmStmt::AsmStmt( const AsmStmt & other ) : Statement( other ), voltile( other.voltile ), instruction( maybeClone( other.instruction ) ), gotolabels( other.gotolabels ) {
  cloneAll( other.output, output );
  cloneAll( other.input, input );
  cloneAll( other.clobber, clobber );
}

AsmStmt::~AsmStmt() {
	delete instruction;
	deleteAll( output );
	deleteAll( input );
	deleteAll( clobber );
}

void AsmStmt::print( std::ostream &os, int indent ) const {
	os << "Assembler Statement:" << endl;
	os << std::string( indent, ' ' ) << "instruction: " << endl << std::string( indent, ' ' );
	instruction->print( os, indent + 2 );
	if ( ! output.empty() ) {
		os << endl << std::string( indent, ' ' ) << "output: " << endl;
		printAll( output, os, indent + 2 );
	} // if
	if ( ! input.empty() ) {
		os << std::string( indent, ' ' ) << "input: " << endl << std::string( indent, ' ' );
		printAll( input, os, indent + 2 );
	} // if
	if ( ! clobber.empty() ) {
		os << std::string( indent, ' ' ) << "clobber: " << endl;
		printAll( clobber, os, indent + 2 );
	} // if
}


const char *BranchStmt::brType[] = { "Goto", "Break", "Continue" };

BranchStmt::BranchStmt( std::list<Label> labels, Label target, Type type ) throw ( SemanticError ) :
	Statement( labels ), originalTarget( target ), target( target ), computedTarget( nullptr ), type( type ) {
	//actually this is a syntactic error signaled by the parser
	if ( type == BranchStmt::Goto && target.empty() ) {
		throw SemanticError("goto without target");
	}
}

BranchStmt::BranchStmt( std::list<Label> labels, Expression *computedTarget, Type type ) throw ( SemanticError ) :
	Statement( labels ), computedTarget( computedTarget ), type( type ) {
	if ( type != BranchStmt::Goto || computedTarget == nullptr ) {
		throw SemanticError("Computed target not valid in branch statement");
	}
}

void BranchStmt::print( std::ostream &os, int indent ) const {
	os << string( indent, ' ' ) << "Branch (" << brType[type] << ")" << endl ;
	if ( target != "" ) os << string( indent+2, ' ' ) << "with target: " << target << endl;
	if ( originalTarget != "" ) os << string( indent+2, ' ' ) << "with original target: " << originalTarget << endl;
	if ( computedTarget != nullptr ) os << string( indent+2, ' ' ) << "with computed target: " << computedTarget << endl;
}

ReturnStmt::ReturnStmt( std::list<Label> labels, Expression *expr ) : Statement( labels ), expr( expr ) {}

ReturnStmt::ReturnStmt( const ReturnStmt & other ) : Statement( other ), expr( maybeClone( other.expr ) ) {}

ReturnStmt::~ReturnStmt() {
	delete expr;
}

void ReturnStmt::print( std::ostream &os, int indent ) const {
	os <<  "Return Statement, returning: ";
	if ( expr != 0 ) {
		os << endl << string( indent+2, ' ' );
		expr->print( os, indent + 2 );
	}
	os << endl;
}

IfStmt::IfStmt( std::list<Label> labels, Expression *condition, Statement *thenPart, Statement *elsePart, std::list<Statement *> initialization ):
	Statement( labels ), condition( condition ), thenPart( thenPart ), elsePart( elsePart ), initialization( initialization ) {}

IfStmt::IfStmt( const IfStmt & other ) :
	Statement( other ), condition( maybeClone( other.condition ) ), thenPart( maybeClone( other.thenPart ) ), elsePart( maybeClone( other.elsePart ) ) {
	cloneAll( other.initialization, initialization );
}

IfStmt::~IfStmt() {
	deleteAll( initialization );
	delete condition;
	delete thenPart;
	delete elsePart;
}

void IfStmt::print( std::ostream &os, int indent ) const {
	os << "If on condition: " << endl ;
	os << string( indent+4, ' ' );
	condition->print( os, indent + 4 );

	if ( !initialization.empty() ) {
		os << string( indent + 2, ' ' ) << "initialization: \n";
		for ( std::list<Statement *>::const_iterator it = initialization.begin(); it != initialization.end(); ++it ) {
			os << string( indent + 4, ' ' );
			(*it)->print( os, indent + 4 );
		}
		os << endl;
	}

	os << string( indent+2, ' ' ) << "... then: " << endl;

	os << string( indent+4, ' ' );
	thenPart->print( os, indent + 4 );

	if ( elsePart != 0 ) {
		os << string( indent+2, ' ' ) << "... else: " << endl;
		os << string( indent+4, ' ' );
		elsePart->print( os, indent + 4 );
	} // if
}

SwitchStmt::SwitchStmt( std::list<Label> labels, Expression * condition, const std::list<Statement *> &statements ):
	Statement( labels ), condition( condition ), statements( statements ) {
}

SwitchStmt::SwitchStmt( const SwitchStmt & other ):
	Statement( other ), condition( maybeClone( other.condition ) ) {
	cloneAll( other.statements, statements );
}

SwitchStmt::~SwitchStmt() {
	delete condition;
	// destroy statements
	deleteAll( statements );
}

void SwitchStmt::print( std::ostream &os, int indent ) const {
	os << "Switch on condition: ";
	condition->print( os );
	os << endl;

	// statements
	std::list<Statement *>::const_iterator i;
	for ( i = statements.begin(); i != statements.end(); i++)
		(*i)->print( os, indent + 4 );

	//for_each( statements.begin(), statements.end(), mem_fun( bind1st(&Statement::print ), os ));
}

CaseStmt::CaseStmt( std::list<Label> labels, Expression *condition, const std::list<Statement *> &statements, bool deflt ) throw ( SemanticError ) :
	Statement( labels ), condition( condition ), stmts( statements ), _isDefault( deflt ) {
	if ( isDefault() && condition != 0 )
		throw SemanticError("default with conditions");
}

CaseStmt::CaseStmt( const CaseStmt & other ) :
	Statement( other ), condition( maybeClone(other.condition ) ), _isDefault( other._isDefault ) {
	cloneAll( other.stmts, stmts );
}

CaseStmt::~CaseStmt() {
	delete condition;
	deleteAll( stmts );
}

CaseStmt * CaseStmt::makeDefault( std::list<Label> labels, std::list<Statement *> stmts ) {
	return new CaseStmt( labels, 0, stmts, true );
}

void CaseStmt::print( std::ostream &os, int indent ) const {
	os << string( indent, ' ' );

	if ( isDefault() )
		os << "Default ";
	else {
		os << "Case ";
		condition->print( os );
	} // if

	os << endl;

	std::list<Statement *>::const_iterator i;
	for ( i = stmts.begin(); i != stmts.end(); i++)
		(*i )->print( os, indent + 4 );
}

WhileStmt::WhileStmt( std::list<Label> labels, Expression *condition, Statement *body, bool isDoWhile ):
	Statement( labels ), condition( condition), body( body), isDoWhile( isDoWhile) {
}

WhileStmt::WhileStmt( const WhileStmt & other ):
	Statement( other ), condition( maybeClone( other.condition ) ), body( maybeClone( other.body ) ), isDoWhile( other.isDoWhile ) {
}

WhileStmt::~WhileStmt() {
	delete body;
	delete condition;
}

void WhileStmt::print( std::ostream &os, int indent ) const {
	os << "While on condition: " << endl ;
	condition->print( os, indent + 4 );

	os << string( indent, ' ' ) << ".... with body: " << endl;

	if ( body != 0 ) body->print( os, indent + 4 );
}

ForStmt::ForStmt( std::list<Label> labels, std::list<Statement *> initialization, Expression *condition, Expression *increment, Statement *body ):
	Statement( labels ), initialization( initialization ), condition( condition ), increment( increment ), body( body ) {
}

ForStmt::ForStmt( const ForStmt & other ):
	Statement( other ), condition( maybeClone( other.condition ) ), increment( maybeClone( other.increment ) ), body( maybeClone( other.body ) ) {
		cloneAll( other.initialization, initialization );

}

ForStmt::~ForStmt() {
	deleteAll( initialization );
	delete condition;
	delete increment;
	delete body;
}

void ForStmt::print( std::ostream &os, int indent ) const {
	os << "Labels: {";
	for ( std::list<Label>::const_iterator it = get_labels().begin(); it != get_labels().end(); ++it) {
		os << *it << ",";
	}
	os << "}" << endl;

	os << string( indent, ' ' ) << "For Statement" << endl ;

	os << string( indent + 2, ' ' ) << "initialization: \n";
	for ( std::list<Statement *>::const_iterator it = initialization.begin(); it != initialization.end(); ++it ) {
		os << string( indent + 4, ' ' );
		(*it)->print( os, indent + 4 );
	}

	os << "\n" << string( indent + 2, ' ' ) << "condition: \n";
	if ( condition != 0 ) {
		os << string( indent + 4, ' ' );
		condition->print( os, indent + 4 );
	}

	os << "\n" << string( indent + 2, ' ' ) << "increment: \n";
	if ( increment != 0 ) {
		os << string( indent + 4, ' ' );
		increment->print( os, indent + 4 );
	}

	os << "\n" << string( indent + 2, ' ' ) << "statement block: \n";
	if ( body != 0 ) {
		os << string( indent + 4, ' ' );
		body->print( os, indent + 4 );
	}

	os << endl;
}

ThrowStmt::ThrowStmt( std::list<Label> labels, Kind kind, Expression * expr, Expression * target ) :
		Statement( labels ), kind(kind), expr(expr), target(target)	{
	assertf(Resume == kind || nullptr == target, "Non-local termination throw is not accepted." );
}

ThrowStmt::ThrowStmt( const ThrowStmt &other ) :
	Statement ( other ), kind( other.kind ), expr( maybeClone( other.expr ) ), target( maybeClone( other.target ) ) {
}

ThrowStmt::~ThrowStmt() {
	delete expr;
	delete target;
}

void ThrowStmt::print( std::ostream &os, int indent) const {
	if ( target ) {
		os << "Non-Local ";
	}
	os << "Throw Statement, raising: ";
	expr->print(os, indent + 4);
	if ( target ) {
		os << "At: ";
		target->print(os, indent + 4);
	}
}

TryStmt::TryStmt( std::list<Label> labels, CompoundStmt *tryBlock, std::list<CatchStmt *> &handlers, FinallyStmt *finallyBlock ) :
	Statement( labels ), block( tryBlock ),  handlers( handlers ), finallyBlock( finallyBlock ) {
}

TryStmt::TryStmt( const TryStmt &other ) : Statement( other ), block( maybeClone( other.block ) ), finallyBlock( maybeClone( other.finallyBlock ) ) {
	cloneAll( other.handlers, handlers );
}

TryStmt::~TryStmt() {
	delete block;
	deleteAll( handlers );
	delete finallyBlock;
}

void TryStmt::print( std::ostream &os, int indent ) const {
	os << "Try Statement" << endl;
	os << string( indent + 2, ' ' ) << "with block:" << endl;
	os << string( indent + 4, ' ' );
	block->print( os, indent + 4 );

	// handlers
	os << string( indent + 2, ' ' ) << "and handlers:" << endl;
	for ( std::list<CatchStmt *>::const_iterator i = handlers.begin(); i != handlers.end(); i++) {
		os << string( indent + 4, ' ' );
		(*i )->print( os, indent + 4 );
	}

	// finally block
	if ( finallyBlock != 0 ) {
		os << string( indent + 2, ' ' ) << "and finally:" << endl;
		finallyBlock->print( os, indent + 4 );
	} // if
}

CatchStmt::CatchStmt( std::list<Label> labels, Kind kind, Declaration *decl, Expression *cond, Statement *body ) :
	Statement( labels ), kind ( kind ), decl ( decl ), cond ( cond ), body( body ) {
}

CatchStmt::CatchStmt( const CatchStmt & other ) :
	Statement( other ), kind ( other.kind ), decl ( maybeClone( other.decl ) ), cond ( maybeClone( other.cond ) ), body( maybeClone( other.body ) ) {
}

CatchStmt::~CatchStmt() {
	delete decl;
	delete body;
}

void CatchStmt::print( std::ostream &os, int indent ) const {
	os << "Catch " << ((Terminate == kind) ? "Terminate" : "Resume") << " Statement" << endl;

	os << string( indent + 2, ' ' ) << "... catching: ";
	if ( decl ) {
		decl->printShort( os, indent + 4 );
		os << endl;
	}
	else
		os << string( indent + 4 , ' ' ) << ">>> Error:  this catch clause must have a declaration <<<" << endl;

	if ( cond ) {
		os << string( indent + 2, ' ' ) << "with conditional:" << endl;
		os << string( indent + 4, ' ' );
		cond->print( os, indent + 4 );
	}
	else
		os << string( indent + 2, ' ' ) << "with no conditional" << endl;

	os << string( indent + 2, ' ' ) << "with block:" << endl;
	os << string( indent + 4, ' ' );
	body->print( os, indent + 4 );
}


FinallyStmt::FinallyStmt( std::list<Label> labels, CompoundStmt *block ) : Statement( labels ), block( block ) {
	assert( labels.empty() ); // finally statement cannot be labeled
}

FinallyStmt::FinallyStmt( const FinallyStmt & other ) : Statement( other ), block( maybeClone( other.block ) ) {
}

FinallyStmt::~FinallyStmt() {
	delete block;
}

void FinallyStmt::print( std::ostream &os, int indent ) const {
	os << "Finally Statement" << endl;
	os << string( indent + 2, ' ' ) << "with block:" << endl;
	os << string( indent + 4, ' ' );
	block->print( os, indent + 4 );
}

WaitForStmt::WaitForStmt( std::list<Label> labels ) : Statement( labels ) {
	timeout.time      = nullptr;
	timeout.statement = nullptr;
	timeout.condition = nullptr;
 	orelse .statement = nullptr;
	orelse .condition = nullptr;
}

WaitForStmt::WaitForStmt( const WaitForStmt & other ) : Statement( other ) {
	clauses.reserve( other.clauses.size() );
	for( auto & ocl : other.clauses ) {
		clauses.emplace_back();
		clauses.back().target.function = ocl.target.function->clone();
		cloneAll( ocl.target.arguments, clauses.back().target.arguments );
		clauses.back().statement = ocl.statement->clone();
		clauses.back().condition = ocl.condition->clone();
	}

	timeout.time      = other.timeout.time     ->clone();
	timeout.statement = other.timeout.statement->clone();
	timeout.condition = other.timeout.condition->clone();
	orelse .statement = other.orelse .statement->clone();
	orelse .condition = other.orelse .condition->clone();
}

WaitForStmt::~WaitForStmt() {
	for( auto & clause : clauses ) {
		delete clause.target.function;
		deleteAll( clause.target.arguments );
		delete clause.statement;
		delete clause.condition;
	}

	delete timeout.time;
	delete timeout.statement;
	delete timeout.condition;

	delete orelse.statement;
	delete orelse.condition;
}

void WaitForStmt::print( std::ostream &os, int indent ) const {
	os << "Waitfor Statement" << endl;
	os << string( indent + 2, ' ' ) << "with block:" << endl;
	os << string( indent + 4, ' ' );
	// block->print( os, indent + 4 );
}

NullStmt::NullStmt( std::list<Label> labels ) : Statement( labels ) {}
NullStmt::NullStmt() : Statement( std::list<Label>() ) {}

void NullStmt::print( std::ostream &os, __attribute__((unused)) int indent ) const {
	os << "Null Statement" << endl ;
}

ImplicitCtorDtorStmt::ImplicitCtorDtorStmt( Statement * callStmt ) : Statement( std::list<Label>() ), callStmt( callStmt ) {
	assert( callStmt );
}

ImplicitCtorDtorStmt::ImplicitCtorDtorStmt( const ImplicitCtorDtorStmt & other ) : Statement( other ), callStmt( maybeClone( other.callStmt ) ) {
}

ImplicitCtorDtorStmt::~ImplicitCtorDtorStmt() {
	delete callStmt;
}

void ImplicitCtorDtorStmt::print( std::ostream &os, int indent ) const {
	os << "Implicit Ctor Dtor Statement" << endl;
	os << string( indent + 2, ' ' ) << "with Ctor/Dtor: ";
	callStmt->print( os, indent + 2);
	os << endl;
}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
