//
// Cforall Version 1.0.0 Copyright (C) 2019 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Heap.h --
//
// Author           : Thierry Delisle
// Created On       : Thu Feb 28 12::05:10 2019
// Last Modified By :
// Last Modified On :
// Update Count     :
//

#pragma once

#include <cstdint>
#include <iostream>

namespace Stats {
	namespace Counters {
		void print();

		class BaseCounter {
		public:
			BaseCounter(const char * const name) : name(name) {
				top.append(this);
			}

			BaseCounter(const char * const name, BaseCounter * parent) : name(name) {
				parent->children.append(this);
			}
		protected:
			virtual ~BaseCounter() = default;

			struct list_t {
				BaseCounter * head = nullptr;
				BaseCounter * tail = nullptr;

				void append(BaseCounter * node) {
					if(!head) { head = node; }
					else      { tail->next = node;}
					tail = node;
				}
			};

		private:
			virtual void print(std::ostream &) = 0;
			template<typename T>
			friend void ForAllCounters(BaseCounter::list_t &, size_t, T );
			friend void print();

		private:
			const char * const name;

			BaseCounter * next = nullptr;
			list_t children;

			static list_t top;
		};

		class CounterGroup : public BaseCounter {
		public:
			CounterGroup(const char * const name ) : BaseCounter(name) {}
			CounterGroup(const char * const name, BaseCounter * parent) : BaseCounter(name, parent) {}

		protected:
			virtual ~CounterGroup() = default;

		private:
			virtual void print(std::ostream & os) {
				os << "";
			}
			template<typename T>
			friend void ForAllCounters(BaseCounter::list_t &, size_t, T );
			friend void print();
		};

		class SimpleCounter : public BaseCounter {
		public:
			SimpleCounter(const char * const name ) : BaseCounter(name) {}
			SimpleCounter(const char * const name, BaseCounter * parent) : BaseCounter(name, parent) {}

			inline void operator++(int)          { count++;        }
			inline void operator+=(size_t value) { count += value; }
		protected:
			virtual ~SimpleCounter() = default;

		private:
			virtual void print(std::ostream & os) {
				os << count;
			}
			template<typename T>
			friend void ForAllCounters(BaseCounter::list_t &, size_t, T );
			friend void print();

			size_t count = 0;

		};

		template<typename T>
		class AverageCounter : public BaseCounter {
		public:
			AverageCounter(const char * const name ) : BaseCounter(name), sum{} {}
			AverageCounter(const char * const name, BaseCounter * parent) : BaseCounter(name, parent), sum{} {}

			inline void push(T value) {
				sum += value;
				count++;
			}

		protected:
			virtual ~AverageCounter() = default;

		private:
			virtual void print(std::ostream & os) {
				os << sum / count;
			}
			template<typename F>
			friend void ForAllCounters(BaseCounter::list_t &, size_t, F );
			friend void print();

			T sum;
			size_t count = 1;
		};
	}
}
