//
// Cforall Version 1.0.0 Copyright (C) 2016 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// algorithm.c --
//
// Author           : Peter A. Buhr
// Created On       : Thu Jan 28 17:10:29 2016
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed May 24 18:13:15 2017
// Update Count     : 198
//

#include "stdlib"

//---------------------------------------

extern "C" {
#define _XOPEN_SOURCE 600								// posix_memalign, *rand48
#include <stdlib.h>										// malloc, free, calloc, realloc, memalign, posix_memalign, bsearch
#include <string.h>										// memset
#include <malloc.h>										// malloc_usable_size
#include <math.h>										// fabsf, fabs, fabsl
#include <complex.h>									// _Complex_I
} // extern "C"

forall( dtype T | sized(T) ) T * malloc( void ) {		// type-safe
    return (T *)(void *)malloc( (size_t)sizeof(T) );
} // malloc

forall( dtype T | sized(T) ) T * malloc( char fill ) {	// initial with fill value (like calloc)
	T * ptr = (T *)(void *)malloc( (size_t)sizeof(T) );
    return memset( ptr, (int)fill, sizeof(T) );
} // malloc

forall( dtype T | sized(T) ) T * malloc( T * ptr, size_t size ) { // alternative realloc
    return (T *)realloc( ptr, size );
} // malloc

forall( dtype T | sized(T) ) T * malloc( T * ptr, size_t size, unsigned char fill ) { // alternative realloc with fill value
    return (T *)realloc( ptr, size, fill );
} // malloc


forall( dtype T | sized(T) ) T * calloc( size_t nmemb ) { // type-safe array initialization with fill 0
    return (T *)calloc( nmemb, sizeof(T) );
} // calloc


forall( dtype T | sized(T) ) T * realloc( T * ptr, size_t size ) { // type-safe
    return (T *)(void *)realloc( (void *)ptr, size );
} // realloc

forall( dtype T | sized(T) ) T * realloc( T * ptr, size_t size, unsigned char fill ) { // alternative realloc with fill value
    char * nptr = (T *)(void *)realloc( (void *)ptr, size );
    size_t unused = malloc_usable_size( nptr );
    memset( nptr + size - unused, (int)fill, unused );	// initialize any new storage
    return nptr;
} // realloc


forall( dtype T | sized(T) ) T * aligned_alloc( size_t alignment ) { // aligned allocation
    return (T *)memalign( alignment, sizeof(T) );
} // aligned_alloc

forall( dtype T | sized(T) ) T * memalign( size_t alignment ) {
    return (T *)memalign( alignment, sizeof(T) );
} // memalign

forall( dtype T | sized(T) ) int posix_memalign( T ** ptr, size_t alignment ) {
    return posix_memalign( (void **)ptr, alignment, sizeof(T) );
} // posix_memalign


forall( dtype T, ttype Params | sized(T) | { void ?{}( T *, Params ); } ) //  new
T * new( Params p ) {
	return ((T *)malloc()){ p };
} // new

forall( dtype T | { void ^?{}(T *); } )					// delete
void delete( T * ptr ) {
	if ( ptr ) {
		^ptr{};
		free( ptr );
	}
} // delete

forall( dtype T, ttype Params | { void ^?{}(T *); void delete(Params); } )
void delete( T * ptr, Params rest ) {
	if ( ptr ) {
		^ptr{};
		free( ptr );
	}
	delete( rest );
} // delete

//---------------------------------------

int ato( const char * ptr ) {
	int i;
	if ( sscanf( ptr, "%d", &i ) == EOF ) {}
	return i;
} // ato

unsigned int ato( const char * ptr ) {
	unsigned int ui;
	if ( sscanf( ptr, "%u", &ui ) == EOF ) {}
	return ui;
} // ato

long int ato( const char * ptr ) {
	long int li;
	if ( sscanf( ptr, "%ld", &li ) == EOF ) {}
	return li;
} // ato

unsigned long int ato( const char * ptr ) {
	unsigned long int uli;
	if ( sscanf( ptr, "%lu", &uli ) == EOF ) {}
	return uli;
} // ato

long long int ato( const char * ptr ) {
	long long int lli;
	if ( sscanf( ptr, "%lld", &lli ) == EOF ) {}
	return lli;
} // ato

unsigned long long int ato( const char * ptr ) {
	unsigned long long int ulli;
	if ( sscanf( ptr, "%llu", &ulli ) == EOF ) {}
	return ulli;
} // ato


float ato( const char * ptr ) {
	float f;
	if ( sscanf( ptr, "%f", &f ) == EOF ) {}
	return f;
} // ato

double ato( const char * ptr ) {
	double d;
	if ( sscanf( ptr, "%lf", &d ) == EOF ) {}
	return d;
} // ato

long double ato( const char * ptr ) {
	long double ld;
	if ( sscanf( ptr, "%Lf", &ld ) == EOF ) {}
	return ld;
} // ato


float _Complex ato( const char * ptr ) {
	float re, im;
	if ( sscanf( ptr, "%g%gi", &re, &im ) == EOF ) {}
	return re + im * _Complex_I;
} // ato

double _Complex ato( const char * ptr ) {
	double re, im;
	if ( sscanf( ptr, "%lf%lfi", &re, &im ) == EOF ) {}
	return re + im * _Complex_I;
} // ato

long double _Complex ato( const char * ptr ) {
	long double re, im;
	if ( sscanf( ptr, "%Lf%Lfi", &re, &im ) == EOF ) {}
	return re + im * _Complex_I;
} // ato


int strto( const char * sptr, char ** eptr, int base ) {
	return (int)strtol( sptr, eptr, base );
} // strto

unsigned int strto( const char * sptr, char ** eptr, int base ) {
	return (unsigned int)strtoul( sptr, eptr, base );
} // strto

long int strto( const char * sptr, char ** eptr, int base ) {
	return strtol( sptr, eptr, base );
} // strto

unsigned long int strto( const char * sptr, char ** eptr, int base ) {
	return strtoul( sptr, eptr, base );
} // strto

long long int strto( const char * sptr, char ** eptr, int base ) {
	return strtoll( sptr, eptr, base );
} // strto

unsigned long long int strto( const char * sptr, char ** eptr, int base ) {
	return strtoull( sptr, eptr, base );
} // strto


float strto( const char * sptr, char ** eptr ) {
	return strtof( sptr, eptr );
} // strto

double strto( const char * sptr, char ** eptr ) {
	return strtod( sptr, eptr );
} // strto

long double strto( const char * sptr, char ** eptr ) {
	return strtold( sptr, eptr );
} // strto


float _Complex strto( const char * sptr, char ** eptr ) {
	float re, im;
	re = strtof( sptr, eptr );
	if ( sptr == *eptr ) return 0.0;
	im = strtof( sptr, eptr );
	if ( sptr == *eptr ) return 0.0;
	return re + im * _Complex_I;
} // strto

double _Complex strto( const char * sptr, char ** eptr ) {
	double re, im;
	re = strtod( sptr, eptr );
	if ( sptr == *eptr ) return 0.0;
	im = strtod( sptr, eptr );
	if ( sptr == *eptr ) return 0.0;
	return re + im * _Complex_I;
} // strto

long double _Complex strto( const char * sptr, char ** eptr ) {
	long double re, im;
	re = strtold( sptr, eptr );
	if ( sptr == *eptr ) return 0.0;
	im = strtold( sptr, eptr );
	if ( sptr == *eptr ) return 0.0;
	return re + im * _Complex_I;
} // strto

//---------------------------------------

forall( otype T | { int ?<?( T, T ); } )
T * bsearch( T key, const T * arr, size_t dimension ) {
	int comp( const void * t1, const void * t2 ) { return *(T *)t1 < *(T *)t2 ? -1 : *(T *)t2 < *(T *)t1 ? 1 : 0; }
	return (T *)bsearch( &key, arr, dimension, sizeof(T), comp );
} // bsearch

forall( otype T | { int ?<?( T, T ); } )
unsigned int bsearch( T key, const T * arr, size_t dimension ) {
	T *result = bsearch( key, arr, dimension );
	return result ? result - arr : dimension;			// pointer subtraction includes sizeof(T)
} // bsearch

forall( otype T | { int ?<?( T, T ); } )
void qsort( const T * arr, size_t dimension ) {
	int comp( const void * t1, const void * t2 ) { return *(T *)t1 < *(T *)t2 ? -1 : *(T *)t2 < *(T *)t1 ? 1 : 0; }
	qsort( arr, dimension, sizeof(T), comp );
} // qsort

//---------------------------------------

forall( otype T | { T ?/?( T, T ); T ?%?( T, T ); } )
[ T, T ] div( T t1, T t2 ) { return [ t1 / t2, t1 % t2 ]; }

//---------------------------------------

unsigned char abs( signed char v ) { return abs( (int)v ); }
unsigned long int abs( long int v ) { return labs( v ); }
unsigned long long int abs( long long int v ) { return llabs( v ); }
float abs( float x ) { return fabsf( x ); }
double abs( double x ) { return fabs( x ); }
long double abs( long double x ) { return fabsl( x ); }
float abs( float _Complex x ) { return cabsf( x ); }
double abs( double _Complex x ) { return cabs( x ); }
long double abs( long double _Complex x ) { return cabsl( x ); }

//---------------------------------------

void rand48seed( long int s ) { srand48( s ); }
char rand48( void ) { return mrand48(); }
int rand48( void ) { return mrand48(); }
unsigned int rand48( void ) { return lrand48(); }
long int rand48( void ) { return mrand48(); }
unsigned long int rand48( void ) { return lrand48(); }
float rand48( void ) { return (float)drand48(); }		// otherwise float uses lrand48
double rand48( void ) { return drand48(); }
float _Complex rand48( void ) { return (float)drand48() + (float _Complex)(drand48() * _Complex_I); }
double _Complex rand48( void ) { return drand48() + (double _Complex)(drand48() * _Complex_I); }
long double _Complex rand48( void) { return (long double)drand48() + (long double _Complex)(drand48() * _Complex_I); }

//---------------------------------------

forall( otype T | { int ?<?( T, T ); } )
T min( T t1, T t2 ) {
	return t1 < t2 ? t1 : t2;
} // min

forall( otype T | { int ?>?( T, T ); } )
T max( T t1, T t2 ) {
	return t1 > t2 ? t1 : t2;
} // max

forall( otype T | { T min( T, T ); T max( T, T ); } )
T clamp( T value, T min_val, T max_val ) {
	return max( min_val, min( value, max_val ) );
} // clamp

forall( otype T )
void swap( T * t1, T * t2 ) {
	T temp = *t1;
	*t1 = *t2;
	*t2 = temp;
} // swap

// Local Variables: //
// tab-width: 4 //
// End: //
