//
// Cforall Version 1.0.0 Copyright (C) 2019 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Convert.cpp -- Convert between the new and old syntax trees.
//
// Author           : Thierry Delisle
// Created On       : Thu May 09 15::37::05 2019
// Last Modified By : Andrew Beach
// Last Modified On : Man Jun 10 11:51:00 2019
// Update Count     : 11
//

#include "Convert.hpp"

#include <deque>
#include <unordered_map>

#include "AST/Attribute.hpp"
#include "AST/Decl.hpp"
#include "AST/Expr.hpp"
#include "AST/Init.hpp"
#include "AST/Stmt.hpp"
#include "AST/TypeSubstitution.hpp"

#include "SymTab/Autogen.h"
#include "SynTree/Attribute.h"
#include "SynTree/Declaration.h"
#include "SynTree/TypeSubstitution.h"

#include "Validate/FindSpecialDecls.h"

//================================================================================================
// Utilities
template<template <class...> class C>
struct to {
	template<typename T>
	static auto from( T && v ) -> C< typename T::value_type > {
		C< typename T::value_type > l;
		std::move(std::begin(v), std::end(v), std::back_inserter(l));
		return l;
	}
};

//================================================================================================
namespace {

// This is to preserve the FindSpecialDecls hack. It does not (and perhaps should not)
// allow us to use the same stratagy in the new ast.
ast::Type * sizeType = nullptr;
ast::FunctionDecl * dereferenceOperator = nullptr;
ast::StructDecl   * dtorStruct = nullptr;
ast::FunctionDecl * dtorStructDestroy = nullptr;

}

//================================================================================================
class ConverterNewToOld : public ast::Visitor {
	BaseSyntaxNode * node = nullptr;
	using Cache = std::unordered_map< const ast::Node *, BaseSyntaxNode * >;
	Cache cache;

	template<typename T>
	struct Getter {
		ConverterNewToOld & visitor;

		template<typename U, enum ast::Node::ref_type R>
		T * accept1( const ast::ptr_base<U, R> & ptr ) {
			if ( ! ptr ) return nullptr;
			ptr->accept( visitor );
			T * ret = strict_dynamic_cast< T * >( visitor.node );
			visitor.node = nullptr;
			return ret;
		}

		template<typename U>
		std::list< T * > acceptL( const U & container ) {
			std::list< T * > ret;
			for ( auto ptr : container ) {
				ret.emplace_back( accept1( ptr ) );
			}
			return ret;
		}
	};

    template<typename T>
    Getter<T> get() {
        return Getter<T>{ *this };
    }

	Label makeLabel(Statement * labelled, const ast::Label& label) {
		// This probably will leak memory, but only until we get rid of the old tree.
		if ( nullptr == labelled && label.location.isSet() ) {
			labelled = new NullStmt();
			labelled->location = label.location;
		}
		return Label(
			label.name,
			labelled,
			get<Attribute>().acceptL(label.attributes)
		);
	}

	template<template <class...> class C>
	std::list<Label> makeLabelL(Statement * labelled, const C<ast::Label>& labels) {
		std::list<Label> ret;
		for (auto label : labels) {
			ret.push_back( makeLabel(labelled, label) );
		}
		return ret;
	}

	/// get new qualifiers from old type
	Type::Qualifiers cv( const ast::Type * ty ) { return { ty->qualifiers.val }; }

	/// returns true and sets `node` if in cache
	bool inCache( const ast::Node * node ) {
		auto it = cache.find( node );
		if ( it == cache.end() ) return false;
		this->node = it->second;
		return true;
	}

public:
	Declaration * decl( const ast::Decl * declNode ) {
		return get<Declaration>().accept1( ast::ptr<ast::Decl>( declNode ) );
	}

private:
	void declPostamble( Declaration * decl, const ast::Decl * node ) {
		decl->location = node->location;
		// name comes from constructor
		// linkage comes from constructor
		decl->extension = node->extension;
		decl->uniqueId = node->uniqueId;
		// storageClasses comes from constructor
		this->node = decl;
	}

	const ast::DeclWithType * declWithTypePostamble (
			DeclarationWithType * decl, const ast::DeclWithType * node ) {
		cache.emplace( node, decl );
		decl->mangleName = node->mangleName;
		decl->scopeLevel = node->scopeLevel;
		decl->asmName = get<Expression>().accept1( node->asmName );
		// attributes comes from constructor
		decl->isDeleted = node->isDeleted;
		// fs comes from constructor
		declPostamble( decl, node );
		return nullptr;
	}

	const ast::DeclWithType * visit( const ast::ObjectDecl * node ) override final {
		auto&& bfwd = get<Expression>().accept1( node->bitfieldWidth );
		auto&& type = get<Type>().accept1( node->type );
		auto&& init = get<Initializer>().accept1( node->init );
		auto&& attr = get<Attribute>().acceptL( node->attributes );
		if ( inCache( node ) ) {
			return nullptr;
		}
		auto decl = new ObjectDecl(
			node->name,
			Type::StorageClasses( node->storage.val ),
			LinkageSpec::Spec( node->linkage.val ),
			bfwd,
			type,
			init,
			attr,
			Type::FuncSpecifiers( node->funcSpec.val )
		);
		return declWithTypePostamble( decl, node );
	}

	const ast::DeclWithType * visit( const ast::FunctionDecl * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto decl = new FunctionDecl(
			node->name,
			Type::StorageClasses( node->storage.val ),
			LinkageSpec::Spec( node->linkage.val ),
			get<FunctionType>().accept1( node->type ),
			{},
			get<Attribute>().acceptL( node->attributes ),
			Type::FuncSpecifiers( node->funcSpec.val )
		);
		cache.emplace( node, decl );
		decl->statements = get<CompoundStmt>().accept1( node->stmts );
		decl->withExprs = get<Expression>().acceptL( node->withExprs );
		if ( dereferenceOperator == node ) {
			Validate::dereferenceOperator = decl;
		}
		if ( dtorStructDestroy == node ) {
			Validate::dtorStructDestroy = decl;
		}
		return declWithTypePostamble( decl, node );
	}

	const ast::Decl * namedTypePostamble( NamedTypeDecl * decl, const ast::NamedTypeDecl * node ) {
		// base comes from constructor
		decl->parameters = get<TypeDecl>().acceptL( node->params );
		decl->assertions = get<DeclarationWithType>().acceptL( node->assertions );
		declPostamble( decl, node );
		return nullptr;
	}

	const ast::Decl * visit( const ast::TypeDecl * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto decl = new TypeDecl(
			node->name,
			Type::StorageClasses( node->storage.val ),
			get<Type>().accept1( node->base ),
			(TypeDecl::Kind)(unsigned)node->kind,
			node->sized,
			get<Type>().accept1( node->init )
		);
		cache.emplace( node, decl );
		return namedTypePostamble( decl, node );
	}

	const ast::Decl * visit( const ast::TypedefDecl * node ) override final {
		auto decl = new TypedefDecl(
			node->name,
			node->location,
			Type::StorageClasses( node->storage.val ),
            get<Type>().accept1( node->base ),
			LinkageSpec::Spec( node->linkage.val )
		);
		return namedTypePostamble( decl, node );
	}

	const ast::Decl * aggregatePostamble( AggregateDecl * decl, const ast::AggregateDecl * node ) {
		cache.emplace( node, decl );
		decl->members = get<Declaration>().acceptL( node->members );
		decl->parameters = get<TypeDecl>().acceptL( node->params );
		decl->body = node->body;
		// attributes come from constructor
		decl->parent = get<AggregateDecl>().accept1( node->parent );
		declPostamble( decl, node );
		return nullptr;
	}

	const ast::Decl * visit( const ast::StructDecl * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto decl = new StructDecl(
			node->name,
			node->kind,
			get<Attribute>().acceptL( node->attributes ),
			LinkageSpec::Spec( node->linkage.val )
		);

		if ( dtorStruct == node ) {
			Validate::dtorStruct = decl;
		}

		return aggregatePostamble( decl, node );
	}

	const ast::Decl * visit( const ast::UnionDecl * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto decl = new UnionDecl(
			node->name,
			get<Attribute>().acceptL( node->attributes ),
			LinkageSpec::Spec( node->linkage.val )
		);
		return aggregatePostamble( decl, node );
	}

	const ast::Decl * visit( const ast::EnumDecl * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto decl = new EnumDecl(
			node->name,
			get<Attribute>().acceptL( node->attributes ),
			LinkageSpec::Spec( node->linkage.val )
		);
		return aggregatePostamble( decl, node );
	}

	const ast::Decl * visit( const ast::TraitDecl * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto decl = new TraitDecl(
			node->name,
			{},
			LinkageSpec::Spec( node->linkage.val )
		);
		return aggregatePostamble( decl, node );
	}

	const ast::AsmDecl * visit( const ast::AsmDecl * node ) override final {
		auto decl = new AsmDecl( get<AsmStmt>().accept1( node->stmt ) );
		declPostamble( decl, node );
		return nullptr;
	}

	const ast::StaticAssertDecl * visit( const ast::StaticAssertDecl * node ) override final {
		auto decl = new StaticAssertDecl(
			get<Expression>().accept1( node->cond ),
			get<ConstantExpr>().accept1( node->msg )
		);
		declPostamble( decl, node );
		return nullptr;
	}

	const ast::Stmt * stmtPostamble( Statement * stmt, const ast::Stmt * node ) {
		cache.emplace( node, stmt );
		stmt->location = node->location;
		stmt->labels = makeLabelL( stmt, node->labels );
		this->node = stmt;
		return nullptr;
	}

	const ast::CompoundStmt * visit( const ast::CompoundStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new CompoundStmt( get<Statement>().acceptL( node->kids ) );
		stmtPostamble( stmt, node );
		return nullptr;
	}

	const ast::Stmt * visit( const ast::ExprStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new ExprStmt( nullptr );
		cache.emplace( node, stmt );
		stmt->expr = get<Expression>().accept1( node->expr );
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::AsmStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new AsmStmt(
			node->isVolatile,
			get<Expression>().accept1( node->instruction ),
			get<Expression>().acceptL( node->output ),
			get<Expression>().acceptL( node->input ),
			get<ConstantExpr>().acceptL( node->clobber ),
			makeLabelL( nullptr, node->gotoLabels ) // What are these labelling?
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::DirectiveStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new DirectiveStmt( node->directive );
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::IfStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new IfStmt(
			get<Expression>().accept1( node->cond ),
			get<Statement>().accept1( node->thenPart ),
			get<Statement>().accept1( node->elsePart ),
			get<Statement>().acceptL( node->inits )
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::SwitchStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new SwitchStmt(
			get<Expression>().accept1( node->cond ),
			get<Statement>().acceptL( node->stmts )
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::CaseStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new CaseStmt(
			get<Expression>().accept1( node->cond ),
			get<Statement>().acceptL( node->stmts ),
			node->isDefault()
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::WhileStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto inits = get<Statement>().acceptL( node->inits );
		auto stmt = new WhileStmt(
			get<Expression>().accept1( node->cond ),
			get<Statement>().accept1( node->body ),
			inits,
			node->isDoWhile
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::ForStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new ForStmt(
			get<Statement>().acceptL( node->inits ),
			get<Expression>().accept1( node->cond ),
			get<Expression>().accept1( node->inc ),
			get<Statement>().accept1( node->body )
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::BranchStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		BranchStmt * stmt;
		if (node->computedTarget) {
			stmt = new BranchStmt( get<Expression>().accept1( node->computedTarget ),
				BranchStmt::Goto );
		} else {
			BranchStmt::Type type;
			switch (node->kind) {
			#define CASE(n) \
			case ast::BranchStmt::n: \
				type = BranchStmt::n; \
				break
			CASE(Goto);
			CASE(Break);
			CASE(Continue);
			CASE(FallThrough);
			CASE(FallThroughDefault);
			#undef CASE
			default:
				assertf(false, "Invalid ast::BranchStmt::Kind: %d\n", node->kind);
			}

			// The labels here are also weird.
			stmt = new BranchStmt( makeLabel( nullptr, node->originalTarget ), type );
			stmt->target = makeLabel( stmt, node->target );
		}
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::ReturnStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new ReturnStmt( get<Expression>().accept1( node->expr ) );
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::ThrowStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		ThrowStmt::Kind kind;
		switch (node->kind) {
		case ast::ExceptionKind::Terminate:
			kind = ThrowStmt::Terminate;
			break;
		case ast::ExceptionKind::Resume:
			kind = ThrowStmt::Resume;
			break;
		default:
			assertf(false, "Invalid ast::ThrowStmt::Kind: %d\n", node->kind);
		}
		auto stmt = new ThrowStmt(
			kind,
			get<Expression>().accept1( node->expr ),
			get<Expression>().accept1( node->target )
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::TryStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto handlers = get<CatchStmt>().acceptL( node->handlers );
		auto stmt = new TryStmt(
			get<CompoundStmt>().accept1( node->body ),
			handlers,
			get<FinallyStmt>().accept1( node->finally )
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::CatchStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		CatchStmt::Kind kind;
		switch (node->kind) {
		case ast::ExceptionKind::Terminate:
			kind = CatchStmt::Terminate;
			break;
		case ast::ExceptionKind::Resume:
			kind = CatchStmt::Resume;
			break;
		default:
			assertf(false, "Invalid ast::CatchStmt::Kind: %d\n", node->kind);
		}
		auto stmt = new CatchStmt(
			kind,
			get<Declaration>().accept1( node->decl ),
			get<Expression>().accept1( node->cond ),
			get<Statement>().accept1( node->body )
		);
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::FinallyStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new FinallyStmt( get<CompoundStmt>().accept1( node->body ) );
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::WaitForStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new WaitForStmt;
		stmt->clauses.reserve( node->clauses.size() );
		for ( auto clause : node->clauses ) {
			stmt->clauses.push_back({{
					get<Expression>().accept1( clause.target.func ),
					get<Expression>().acceptL( clause.target.args ),
				},
				get<Statement>().accept1( clause.stmt ),
				get<Expression>().accept1( clause.cond ),
			});
		}
		stmt->timeout = {
			get<Expression>().accept1( node->timeout.time ),
			get<Statement>().accept1( node->timeout.stmt ),
			get<Expression>().accept1( node->timeout.cond ),
		};
		stmt->orelse = {
			get<Statement>().accept1( node->orElse.stmt ),
			get<Expression>().accept1( node->orElse.cond ),
		};
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::WithStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new WithStmt(
			get<Expression>().acceptL( node->exprs ),
			get<Statement>().accept1( node->stmt )
		);
		return stmtPostamble( stmt, node );
	}

	const ast::NullStmt * visit( const ast::NullStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new NullStmt();
		stmtPostamble( stmt, node );
		return nullptr;
	}

	const ast::Stmt * visit( const ast::DeclStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new DeclStmt( get<Declaration>().accept1( node->decl ) );
		return stmtPostamble( stmt, node );
	}

	const ast::Stmt * visit( const ast::ImplicitCtorDtorStmt * node ) override final {
		if ( inCache( node ) ) return nullptr;
		auto stmt = new ImplicitCtorDtorStmt{
			get<Statement>().accept1( node->callStmt )
		};
		return stmtPostamble( stmt, node );
	}

	TypeSubstitution * convertTypeSubstitution(const ast::TypeSubstitution * src) {

		if (!src) return nullptr;

		TypeSubstitution *rslt = new TypeSubstitution();

		for (decltype(src->begin()) src_i = src->begin(); src_i != src->end(); src_i++) {
			rslt->add( src_i->first,
			           get<Type>().accept1(src_i->second) );
		}

		for (decltype(src->beginVar()) src_i = src->beginVar(); src_i != src->endVar(); src_i++) {
			rslt->addVar( src_i->first,
			              get<Expression>().accept1(src_i->second) );
		}

		return rslt;
	}

	void convertInferUnion(std::map<UniqueId,ParamEntry> &tgtInferParams,
						   std::vector<UniqueId>         &tgtResnSlots,
						   const ast::Expr::InferUnion   &srcInferred ) {

		assert( tgtInferParams.empty() );
		assert( tgtResnSlots.empty() );

		if ( srcInferred.mode == ast::Expr::InferUnion::Params ) {
			const ast::InferredParams &srcParams = srcInferred.inferParams();
			for (auto & srcParam : srcParams) {
				auto res = tgtInferParams.emplace(srcParam.first, ParamEntry(
					srcParam.second.decl,
					get<Declaration>().accept1(srcParam.second.declptr),
					get<Type>().accept1(srcParam.second.actualType),
					get<Type>().accept1(srcParam.second.formalType),
					get<Expression>().accept1(srcParam.second.expr)
				));
				assert(res.second);
			}
		} else if ( srcInferred.mode == ast::Expr::InferUnion::Slots  ) {
			const ast::ResnSlots &srcSlots = srcInferred.resnSlots();
			for (auto srcSlot : srcSlots) {
				tgtResnSlots.push_back(srcSlot);
			}
		}
	}

	Expression * visitBaseExpr_skipResultType(const ast::Expr * src, Expression * tgt) {

		tgt->location  = src->location;
		tgt->env       = convertTypeSubstitution(src->env);
		tgt->extension = src->extension;

		convertInferUnion(tgt->inferParams, tgt->resnSlots, src->inferred);
		return tgt;
	}

	Expression * visitBaseExpr(const ast::Expr * src, Expression * tgt) {

		tgt->result = get<Type>().accept1(src->result);
		return visitBaseExpr_skipResultType(src, tgt);
	}

	const ast::Expr * visit( const ast::ApplicationExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new ApplicationExpr(
				get<Expression>().accept1(node->func),
				get<Expression>().acceptL(node->args)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new UntypedExpr(
				get<Expression>().accept1(node->func),
				get<Expression>().acceptL(node->args)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::NameExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new NameExpr(
				node->name
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::AddressExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new AddressExpr(
				get<Expression>().accept1(node->arg)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::LabelAddressExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new LabelAddressExpr(
				makeLabel(nullptr, node->arg)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::CastExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new CastExpr(
				get<Expression>().accept1(node->arg),
				(node->isGenerated == ast::GeneratedCast)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::KeywordCastExpr * node ) override final {
		KeywordCastExpr::Target castTarget = KeywordCastExpr::NUMBER_OF_TARGETS;
		switch (node->target) {
			case ast::KeywordCastExpr::Coroutine:
				castTarget = KeywordCastExpr::Coroutine;
				break;
			case ast::KeywordCastExpr::Thread:
				castTarget = KeywordCastExpr::Thread;
				break;
			case ast::KeywordCastExpr::Monitor:
				castTarget = KeywordCastExpr::Monitor;
				break;
			default:
				break;
		}
		assert ( castTarget < KeywordCastExpr::NUMBER_OF_TARGETS );
		auto expr = visitBaseExpr( node,
			new KeywordCastExpr(
				get<Expression>().accept1(node->arg),
				castTarget
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::VirtualCastExpr * node ) override final {
		auto expr = visitBaseExpr_skipResultType( node,
			new VirtualCastExpr(
				get<Expression>().accept1(node->arg),
				get<Type>().accept1(node->result)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedMemberExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new UntypedMemberExpr(
				get<Expression>().accept1(node->member),
				get<Expression>().accept1(node->aggregate)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::MemberExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new MemberExpr(
				get<DeclarationWithType>().accept1(node->member),
				get<Expression>().accept1(node->aggregate)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::VariableExpr * node ) override final {
		auto expr = new VariableExpr();
		visitBaseExpr( node, expr );
		expr->var = get<DeclarationWithType>().accept1(node->var);
		Type * type = expr->var->get_type()->clone();
		if(FunctionType * ft = dynamic_cast<FunctionType*>(type)) {
			if(node->result.as<ast::PointerType>()) {
				type = new PointerType({}, ft);
			}
		}

		type->set_lvalue( true );
		expr->result = type ;
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::ConstantExpr * node ) override final {
		ConstantExpr *rslt = nullptr;
		switch ( node->kind ) {
		case ast::ConstantExpr::Integer:
			rslt = new ConstantExpr{Constant{
				get<Type>().accept1( node->result ),
				node->rep,
				(unsigned long long) node->intValue()
			}};
			break;
		case ast::ConstantExpr::FloatingPoint:
			rslt = new ConstantExpr{Constant{
				get<Type>().accept1(node->result),
				node->rep,
				(double) node->floatValue()
			}};
			break;
		case ast::ConstantExpr::String:
			rslt = new ConstantExpr{Constant{
				get<Type>().accept1( node->result ),
				node->rep,
				(long long unsigned int)0
			}};
			break;
		}
		assert(rslt);
		auto expr = visitBaseExpr( node, rslt );
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::SizeofExpr * node ) override final {
		assert (node->expr || node->type);
		assert (! (node->expr && node->type));
		SizeofExpr *rslt;
		if (node->expr) {
			rslt = new SizeofExpr(
				get<Expression>().accept1(node->expr)
			);
			assert (!rslt->isType);
		}
		else {
			assert(node->type);
			rslt = new SizeofExpr(
				get<Type>().accept1(node->type)
			);
			assert (rslt->isType);
		}
		auto expr = visitBaseExpr( node, rslt );
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::AlignofExpr * node ) override final {
		assert (node->expr || node->type);
		assert (! (node->expr && node->type));
		AlignofExpr *rslt;
		if (node->expr) {
			rslt = new AlignofExpr(
				get<Expression>().accept1(node->expr)
			);
			assert (!rslt->isType);
		}
		else {
			assert(node->type);
			rslt = new AlignofExpr(
				get<Type>().accept1(node->type)
			);
			assert (rslt->isType);
		}
		auto expr = visitBaseExpr( node, rslt );
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedOffsetofExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new UntypedOffsetofExpr(
				get<Type>().accept1(node->type),
				node->member
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::OffsetofExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new OffsetofExpr(
				get<Type>().accept1(node->type),
				get<DeclarationWithType>().accept1(node->member)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::OffsetPackExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new OffsetPackExpr(
				get<StructInstType>().accept1(node->type)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::LogicalExpr * node ) override final {
		assert (node->isAnd == ast::LogicalFlag::AndExpr ||
				node->isAnd == ast::LogicalFlag::OrExpr	);
		auto expr = visitBaseExpr( node,
			new LogicalExpr(
				get<Expression>().accept1(node->arg1),
				get<Expression>().accept1(node->arg2),
				(node->isAnd == ast::LogicalFlag::AndExpr)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::ConditionalExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new ConditionalExpr(
				get<Expression>().accept1(node->arg1),
				get<Expression>().accept1(node->arg2),
				get<Expression>().accept1(node->arg3)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::CommaExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new CommaExpr(
				get<Expression>().accept1(node->arg1),
				get<Expression>().accept1(node->arg2)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::TypeExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new TypeExpr(
				get<Type>().accept1(node->type)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::AsmExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new AsmExpr(
				get<Expression>().accept1(node->inout),
				get<Expression>().accept1(node->constraint),
				get<Expression>().accept1(node->operand)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::ImplicitCopyCtorExpr * node ) override final {
		auto rslt = new ImplicitCopyCtorExpr(
			get<ApplicationExpr>().accept1(node->callExpr)
		);

		auto expr = visitBaseExpr( node, rslt );
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::ConstructorExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new ConstructorExpr(
				get<Expression>().accept1(node->callExpr)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::CompoundLiteralExpr * node ) override final {
		auto expr = visitBaseExpr_skipResultType( node,
			new CompoundLiteralExpr(
				get<Type>().accept1(node->result),
				get<Initializer>().accept1(node->init)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::RangeExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new RangeExpr(
				get<Expression>().accept1(node->low),
				get<Expression>().accept1(node->high)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedTupleExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new UntypedTupleExpr(
				get<Expression>().acceptL(node->exprs)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::TupleExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new UntypedTupleExpr(
				get<Expression>().acceptL(node->exprs)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::TupleIndexExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new TupleIndexExpr(
				get<Expression>().accept1(node->tuple),
				node->index
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::TupleAssignExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new TupleAssignExpr(
				get<StmtExpr>().accept1(node->stmtExpr)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::StmtExpr * node ) override final {
		auto rslt = new StmtExpr(
			get<CompoundStmt>().accept1(node->stmts)
		);

		rslt->returnDecls = get<ObjectDecl>().acceptL(node->returnDecls);
		rslt->dtors       = get<Expression>().acceptL(node->dtors);

		auto expr = visitBaseExpr( node, rslt );
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UniqueExpr * node ) override final {
		auto rslt = new UniqueExpr(
			get<Expression>().accept1(node->expr)
		);

		rslt->object = get<ObjectDecl>  ().accept1(node->object);
		rslt->var    = get<VariableExpr>().accept1(node->var);

		auto expr = visitBaseExpr( node, rslt );
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::UntypedInitExpr * node ) override final {
		std::list<InitAlternative> initAlts;
		for (auto ia : node->initAlts) {
			initAlts.push_back(InitAlternative(
				get<Type>       ().accept1(ia.type),
				get<Designation>().accept1(ia.designation)
			));
		}
		auto expr = visitBaseExpr( node,
			new UntypedInitExpr(
				get<Expression>().accept1(node->expr),
				initAlts
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::InitExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new InitExpr(
				get<Expression>().accept1(node->expr),
				get<Designation>().accept1(node->designation)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::DeletedExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new DeletedExpr(
				get<Expression>().accept1(node->expr),
				inCache(node->deleteStmt) ?
					this->node :
					get<BaseSyntaxNode>().accept1(node->deleteStmt)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::DefaultArgExpr * node ) override final {
		auto expr = visitBaseExpr( node,
			new DefaultArgExpr(
				get<Expression>().accept1(node->expr)
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Expr * visit( const ast::GenericExpr * node ) override final {
		std::list<GenericExpr::Association> associations;
		for (auto association : node->associations) {
			associations.push_back(GenericExpr::Association(
				get<Type>      ().accept1(association.type),
				get<Expression>().accept1(association.expr)
			));
		}
		auto expr = visitBaseExpr( node,
			new GenericExpr(
				get<Expression>().accept1(node->control),
				associations
			)
		);
		this->node = expr;
		return nullptr;
	}

	const ast::Type * visit( const ast::VoidType * node ) override final {
		this->node = new VoidType{ cv( node ) };
		return nullptr;
	}

	const ast::Type * visit( const ast::BasicType * node ) override final {
		auto type = new BasicType{ cv( node ), (BasicType::Kind)(unsigned)node->kind };
		// I believe this should always be a BasicType.
		if ( sizeType == node ) {
			Validate::SizeType = type;
		}
		this->node = type;
		return nullptr;
	}

	const ast::Type * visit( const ast::PointerType * node ) override final {
		this->node = new PointerType{
			cv( node ),
			get<Type>().accept1( node->base ),
			get<Expression>().accept1( node->dimension ),
			(bool)node->isVarLen,
			(bool)node->isStatic
		};
		return nullptr;
	}

	const ast::Type * visit( const ast::ArrayType * node ) override final {
		this->node = new ArrayType{
			cv( node ),
			get<Type>().accept1( node->base ),
			get<Expression>().accept1( node->dimension ),
			(bool)node->isVarLen,
			(bool)node->isStatic
		};
		return nullptr;
	}

	const ast::Type * visit( const ast::ReferenceType * node ) override final {
		this->node = new ReferenceType{
			cv( node ),
			get<Type>().accept1( node->base )
		};
		return nullptr;
	}

	const ast::Type * visit( const ast::QualifiedType * node ) override final {
		this->node = new QualifiedType{
			cv( node ),
			get<Type>().accept1( node->parent ),
			get<Type>().accept1( node->child )
		};
		return nullptr;
	}

	const ast::Type * visit( const ast::FunctionType * node ) override final {
		auto ty = new FunctionType {
			cv( node ),
			(bool)node->isVarArgs
		};
		ty->returnVals = get<DeclarationWithType>().acceptL( node->returns );
		ty->parameters = get<DeclarationWithType>().acceptL( node->params );
		ty->forall = get<TypeDecl>().acceptL( node->forall );
		this->node = ty;
		return nullptr;
	}

	void postvisit( const ast::ReferenceToType * old, ReferenceToType * ty ) {
		ty->forall = get<TypeDecl>().acceptL( old->forall );
		ty->parameters = get<Expression>().acceptL( old->params );
		ty->hoistType = old->hoistType;
	}

	const ast::Type * visit( const ast::StructInstType * node ) override final {
		StructInstType * ty;
		if ( node->base ) {
			ty = new StructInstType{
				cv( node ),
				get<StructDecl>().accept1( node->base ),
				get<Attribute>().acceptL( node->attributes )
			};
		} else {
			ty = new StructInstType{
				cv( node ),
				node->name,
				get<Attribute>().acceptL( node->attributes )
			};
		}
		postvisit( node, ty );
		this->node = ty;
		return nullptr;
	}

	const ast::Type * visit( const ast::UnionInstType * node ) override final {
		UnionInstType * ty;
		if ( node->base ) {
			ty = new UnionInstType{
				cv( node ),
				get<UnionDecl>().accept1( node->base ),
				get<Attribute>().acceptL( node->attributes )
			};
		} else {
			ty = new UnionInstType{
				cv( node ),
				node->name,
				get<Attribute>().acceptL( node->attributes )
			};
		}
		postvisit( node, ty );
		this->node = ty;
		return nullptr;
	}

	const ast::Type * visit( const ast::EnumInstType * node ) override final {
		EnumInstType * ty;
		if ( node->base ) {
			ty = new EnumInstType{
				cv( node ),
				get<EnumDecl>().accept1( node->base ),
				get<Attribute>().acceptL( node->attributes )
			};
		} else {
			ty = new EnumInstType{
				cv( node ),
				node->name,
				get<Attribute>().acceptL( node->attributes )
			};
		}
		postvisit( node, ty );
		this->node = ty;
		return nullptr;
	}

	const ast::Type * visit( const ast::TraitInstType * node ) override final {
		TraitInstType * ty;
		if ( node->base ) {
			ty = new TraitInstType{
				cv( node ),
				get<TraitDecl>().accept1( node->base ),
				get<Attribute>().acceptL( node->attributes )
			};
		} else {
			ty = new TraitInstType{
				cv( node ),
				node->name,
				get<Attribute>().acceptL( node->attributes )
			};
		}
		postvisit( node, ty );
		this->node = ty;
		return nullptr;
	}

	const ast::Type * visit( const ast::TypeInstType * node ) override final {
		TypeInstType * ty;
		if ( node->base ) {
			ty = new TypeInstType{
				cv( node ),
				node->name,
				get<TypeDecl>().accept1( node->base ),
				get<Attribute>().acceptL( node->attributes )
			};
		} else {
			ty = new TypeInstType{
				cv( node ),
				node->name,
				node->kind == ast::TypeVar::Ftype,
				get<Attribute>().acceptL( node->attributes )
			};
		}
		postvisit( node, ty );
		this->node = ty;
		return nullptr;
	}

	const ast::Type * visit( const ast::TupleType * node ) override final {
		this->node = new TupleType{
			cv( node ),
			get<Type>().acceptL( node->types )
			// members generated by TupleType c'tor
		};
		return nullptr;
	}

	const ast::Type * visit( const ast::TypeofType * node ) override final {
		this->node = new TypeofType{
			cv( node ),
			get<Expression>().accept1( node->expr ),
			(bool)node->kind
		};
		return nullptr;
	}

	const ast::Type * visit( const ast::VarArgsType * node ) override final {
		this->node = new VarArgsType{ cv( node ) };
		return nullptr;
	}

	const ast::Type * visit( const ast::ZeroType * node ) override final {
		this->node = new ZeroType{ cv( node ) };
		return nullptr;
	}

	const ast::Type * visit( const ast::OneType * node ) override final {
		this->node = new OneType{ cv( node ) };
		return nullptr;
	}

	const ast::Type * visit( const ast::GlobalScopeType * ) override final {
		this->node = new GlobalScopeType{};
		return nullptr;
	}

	const ast::Designation * visit( const ast::Designation * node ) override final {
		auto designation = new Designation( get<Expression>().acceptL( node->designators ) );
		designation->location = node->location;
		this->node = designation;
		return nullptr;
	}

	const ast::Init * visit( const ast::SingleInit * node ) override final {
		auto init = new SingleInit(
			get<Expression>().accept1( node->value ),
			ast::MaybeConstruct == node->maybeConstructed
		);
		init->location = node->location;
		this->node = init;
		return nullptr;
	}

	const ast::Init * visit( const ast::ListInit * node ) override final {
		auto init = new ListInit(
			get<Initializer>().acceptL( node->initializers ),
			get<Designation>().acceptL( node->designations ),
			ast::MaybeConstruct == node->maybeConstructed
		);
		init->location = node->location;
		this->node = init;
		return nullptr;
	}

	const ast::Init * visit( const ast::ConstructorInit * node ) override final {
		auto init = new ConstructorInit(
			get<Statement>().accept1( node->ctor ),
			get<Statement>().accept1( node->dtor ),
			get<Initializer>().accept1( node->init )
		);
		init->location = node->location;
		this->node = init;
		return nullptr;
	}

	const ast::Attribute * visit( const ast::Attribute * node ) override final {
		auto attr = new Attribute(
			node->name,
			get<Expression>().acceptL(node->params)
		);
		this->node = attr;
		return nullptr;
	}

	const ast::TypeSubstitution * visit( const ast::TypeSubstitution * node ) override final {
		// Handled by convertTypeSubstitution helper instead.
		// TypeSubstitution is not a node in the old model, so the conversion result wouldn't fit in this->node.
		assert( 0 );
		(void)node;
		return nullptr;
	}
};

std::list< Declaration * > convert( const std::list< ast::ptr< ast::Decl > > && translationUnit ) {
	ConverterNewToOld c;
	std::list< Declaration * > decls;
	for(auto d : translationUnit) {
		decls.emplace_back( c.decl( d ) );
	}
	return decls;
}

//================================================================================================

class ConverterOldToNew : public Visitor {
public:
	ast::Decl * decl() {
		return strict_dynamic_cast< ast::Decl * >( node );
	}

	ConverterOldToNew() = default;
	ConverterOldToNew(const ConverterOldToNew &) = delete;
	ConverterOldToNew(ConverterOldToNew &&) = delete;
private:
	/// conversion output
	ast::Node * node = nullptr;
	/// cache of nodes that might be referenced by readonly<> for de-duplication
	std::unordered_map< BaseSyntaxNode *, ast::Node * > cache = {};

	// Local Utilities:

	template<typename NewT, typename OldT>
	NewT * getAccept1( OldT old ) {
		if ( ! old ) return nullptr;
		old->accept(*this);
		ast::Node * ret = node;
		node = nullptr;
		return strict_dynamic_cast< NewT * >( ret );
	}

#	define GET_ACCEPT_1(child, type) \
		getAccept1< ast::type, decltype( old->child ) >( old->child )

	template<typename NewT, typename OldC>
	std::vector< ast::ptr<NewT> > getAcceptV( const OldC& old ) {
		std::vector< ast::ptr<NewT> > ret;
		ret.reserve( old.size() );
		for ( auto a : old ) {
			a->accept( *this );
			ret.emplace_back( strict_dynamic_cast< NewT * >(node) );
			node = nullptr;
		}
		return ret;
	}

#	define GET_ACCEPT_V(child, type) \
		getAcceptV< ast::type, decltype( old->child ) >( old->child )

	template<typename NewT, typename OldC>
	std::deque< ast::ptr<NewT> > getAcceptD( const OldC& old ) {
		std::deque< ast::ptr<NewT> > ret;
		for ( auto a : old ) {
			a->accept( *this );
			ret.emplace_back( strict_dynamic_cast< NewT * >(node) );
			node = nullptr;
		}
		return ret;
	}

#	define GET_ACCEPT_D(child, type) \
		getAcceptD< ast::type, decltype( old->child ) >( old->child )

	ast::Label make_label(const Label* old) {
		CodeLocation const & location =
		    ( old->labelled ) ? old->labelled->location : CodeLocation();
		return ast::Label(
			location,
			old->name,
			GET_ACCEPT_V(attributes, Attribute)
		);
	}

	template<template <class...> class C>
	C<ast::Label> make_labels(C<Label> olds) {
		C<ast::Label> ret;
		for (auto oldn : olds) {
			ret.push_back( make_label( &oldn ) );
		}
		return ret;
	}

#	define GET_LABELS_V(labels) \
		to<std::vector>::from( make_labels( std::move( labels ) ) )

	static ast::CV::Qualifiers cv( Type * ty ) { return { ty->get_qualifiers().val }; }

	/// returns true and sets `node` if in cache
	bool inCache( BaseSyntaxNode * old ) {
		auto it = cache.find( old );
		if ( it == cache.end() ) return false;
		node = it->second;
		return true;
	}

	// Now all the visit functions:

	virtual void visit( ObjectDecl * old ) override final {
		auto&& type = GET_ACCEPT_1(type, Type);
		auto&& init = GET_ACCEPT_1(init, Init);
		auto&& bfwd = GET_ACCEPT_1(bitfieldWidth, Expr);
		auto&& attr = GET_ACCEPT_V(attributes, Attribute);
		if ( inCache( old ) ) {
			return;
		}
		auto decl = new ast::ObjectDecl(
			old->location,
			old->name,
			type,
			init,
			{ old->get_storageClasses().val },
			{ old->linkage.val },
			bfwd,
			std::move(attr),
			{ old->get_funcSpec().val }
		);
		cache.emplace(old, decl);
		assert(cache.find( old ) != cache.end());
		decl->scopeLevel = old->scopeLevel;
		decl->mangleName = old->mangleName;
		decl->isDeleted  = old->isDeleted;
		decl->uniqueId   = old->uniqueId;
		decl->extension  = old->extension;

		this->node = decl;
	}

	virtual void visit( FunctionDecl * old ) override final {
		if ( inCache( old ) ) return;
		auto decl = new ast::FunctionDecl{
			old->location,
			old->name,
			GET_ACCEPT_1(type, FunctionType),
			{},
			{ old->storageClasses.val },
			{ old->linkage.val },
			GET_ACCEPT_V(attributes, Attribute),
			{ old->get_funcSpec().val }
		};
		cache.emplace( old, decl );
		decl->withExprs = GET_ACCEPT_V(withExprs, Expr);
		decl->stmts = GET_ACCEPT_1(statements, CompoundStmt);
		decl->scopeLevel = old->scopeLevel;
		decl->mangleName = old->mangleName;
		decl->isDeleted  = old->isDeleted;
		decl->uniqueId   = old->uniqueId;
		decl->extension  = old->extension;

		this->node = decl;

		if ( Validate::dereferenceOperator == old ) {
			dereferenceOperator = decl;
		}

		if ( Validate::dtorStructDestroy == old ) {
			dtorStructDestroy = decl;
		}
	}

	virtual void visit( StructDecl * old ) override final {
		if ( inCache( old ) ) return;
		auto decl = new ast::StructDecl(
			old->location,
			old->name,
			old->kind,
			GET_ACCEPT_V(attributes, Attribute),
			{ old->linkage.val }
		);
		cache.emplace( old, decl );
		decl->parent = GET_ACCEPT_1(parent, AggregateDecl);
		decl->body   = old->body;
		decl->params = GET_ACCEPT_V(parameters, TypeDecl);
		decl->members    = GET_ACCEPT_V(members, Decl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;

		if ( Validate::dtorStruct == old ) {
			dtorStruct = decl;
		}
	}

	virtual void visit( UnionDecl * old ) override final {
		if ( inCache( old ) ) return;
		auto decl = new ast::UnionDecl(
			old->location,
			old->name,
			GET_ACCEPT_V(attributes, Attribute),
			{ old->linkage.val }
		);
		cache.emplace( old, decl );
		decl->parent = GET_ACCEPT_1(parent, AggregateDecl);
		decl->body   = old->body;
		decl->params = GET_ACCEPT_V(parameters, TypeDecl);
		decl->members    = GET_ACCEPT_V(members, Decl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( EnumDecl * old ) override final {
		if ( inCache( old ) ) return;
		auto decl = new ast::EnumDecl(
			old->location,
			old->name,
			GET_ACCEPT_V(attributes, Attribute),
			{ old->linkage.val }
		);
		cache.emplace( old, decl );
		decl->parent = GET_ACCEPT_1(parent, AggregateDecl);
		decl->body   = old->body;
		decl->params = GET_ACCEPT_V(parameters, TypeDecl);
		decl->members    = GET_ACCEPT_V(members, Decl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( TraitDecl * old ) override final {
		if ( inCache( old ) ) return;
		auto decl = new ast::TraitDecl(
			old->location,
			old->name,
			GET_ACCEPT_V(attributes, Attribute),
			{ old->linkage.val }
		);
		cache.emplace( old, decl );
		decl->parent = GET_ACCEPT_1(parent, AggregateDecl);
		decl->body   = old->body;
		decl->params = GET_ACCEPT_V(parameters, TypeDecl);
		decl->members    = GET_ACCEPT_V(members, Decl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( TypeDecl * old ) override final {
		if ( inCache( old ) ) return;
		auto decl = new ast::TypeDecl{
			old->location,
			old->name,
			{ old->storageClasses.val },
			GET_ACCEPT_1(base, Type),
			(ast::TypeVar::Kind)(unsigned)old->kind,
			old->sized,
			GET_ACCEPT_1(init, Type)
		};
		cache.emplace( old, decl );
		decl->assertions = GET_ACCEPT_V(assertions, DeclWithType);
		decl->params     = GET_ACCEPT_V(parameters, TypeDecl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;

		this->node = decl;
	}

	virtual void visit( TypedefDecl * old ) override final {
		auto decl = new ast::TypedefDecl(
			old->location,
			old->name,
			{ old->storageClasses.val },
			GET_ACCEPT_1(base, Type),
			{ old->linkage.val }
		);
		decl->assertions = GET_ACCEPT_V(assertions, DeclWithType);
		decl->params     = GET_ACCEPT_V(parameters, TypeDecl);
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( AsmDecl * old ) override final {
		auto decl = new ast::AsmDecl{
			old->location,
			GET_ACCEPT_1(stmt, AsmStmt)
		};
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( StaticAssertDecl * old ) override final {
		auto decl = new ast::StaticAssertDecl{
			old->location,
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_1(message, ConstantExpr)
		};
		decl->extension  = old->extension;
		decl->uniqueId   = old->uniqueId;
		decl->storage    = { old->storageClasses.val };

		this->node = decl;
	}

	virtual void visit( CompoundStmt * old ) override final {
		if ( inCache( old ) ) return;
		auto stmt = new ast::CompoundStmt(
			old->location,
			to<std::list>::from( GET_ACCEPT_V(kids, Stmt) ),
			GET_LABELS_V(old->labels)
		);

		this->node = stmt;
		cache.emplace( old, this->node );
	}

	virtual void visit( ExprStmt * old ) override final {
		if ( inCache( old ) ) return;
		this->node = new ast::ExprStmt(
			old->location,
			GET_ACCEPT_1(expr, Expr),
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( AsmStmt * old ) override final {
		if ( inCache( old ) ) return;
		this->node = new ast::AsmStmt(
			old->location,
			old->voltile,
			GET_ACCEPT_1(instruction, Expr),
			GET_ACCEPT_V(output, Expr),
			GET_ACCEPT_V(input, Expr),
			GET_ACCEPT_V(clobber, ConstantExpr),
			GET_LABELS_V(old->gotolabels),
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( DirectiveStmt * old ) override final {
		if ( inCache( old ) ) return;
		this->node = new ast::DirectiveStmt(
			old->location,
			old->directive,
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( IfStmt * old ) override final {
		if ( inCache( old ) ) return;
		this->node = new ast::IfStmt(
			old->location,
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_1(thenPart, Stmt),
			GET_ACCEPT_1(elsePart, Stmt),
			GET_ACCEPT_V(initialization, Stmt),
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( SwitchStmt * old ) override final {
		if ( inCache( old ) ) return;
		this->node = new ast::SwitchStmt(
			old->location,
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_V(statements, Stmt),
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( CaseStmt * old ) override final {
		if ( inCache( old ) ) return;
		this->node = new ast::CaseStmt(
			old->location,
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_V(stmts, Stmt),
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( WhileStmt * old ) override final {
		if ( inCache( old ) ) return;
		this->node = new ast::WhileStmt(
			old->location,
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_1(body, Stmt),
			GET_ACCEPT_V(initialization, Stmt),
			old->isDoWhile,
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( ForStmt * old ) override final {
		if ( inCache( old ) ) return;
		this->node = new ast::ForStmt(
			old->location,
			GET_ACCEPT_V(initialization, Stmt),
			GET_ACCEPT_1(condition, Expr),
			GET_ACCEPT_1(increment, Expr),
			GET_ACCEPT_1(body, Stmt),
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( BranchStmt * old ) override final {
		if ( inCache( old ) ) return;
		if (old->computedTarget) {
			this->node = new ast::BranchStmt(
				old->location,
				GET_ACCEPT_1(computedTarget, Expr),
				GET_LABELS_V(old->labels)
			);
		} else {
			ast::BranchStmt::Kind kind;
			switch (old->type) {
			#define CASE(n) \
			case BranchStmt::n: \
				kind = ast::BranchStmt::n; \
				break
			CASE(Goto);
			CASE(Break);
			CASE(Continue);
			CASE(FallThrough);
			CASE(FallThroughDefault);
			#undef CASE
			default:
				assertf(false, "Invalid BranchStmt::Type %d\n", old->type);
			}

			auto stmt = new ast::BranchStmt(
				old->location,
				kind,
				make_label(&old->originalTarget),
				GET_LABELS_V(old->labels)
			);
			stmt->target = make_label(&old->target);
			this->node = stmt;
		}
		cache.emplace( old, this->node );
	}

	virtual void visit( ReturnStmt * old ) override final {
		if ( inCache( old ) ) return;
		this->node = new ast::ReturnStmt(
			old->location,
			GET_ACCEPT_1(expr, Expr),
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( ThrowStmt * old ) override final {
		if ( inCache( old ) ) return;
		ast::ExceptionKind kind;
		switch (old->kind) {
		case ThrowStmt::Terminate:
			kind = ast::ExceptionKind::Terminate;
			break;
		case ThrowStmt::Resume:
			kind = ast::ExceptionKind::Resume;
			break;
		default:
			assertf(false, "Invalid ThrowStmt::Kind %d\n", old->kind);
		}

		this->node = new ast::ThrowStmt(
			old->location,
			kind,
			GET_ACCEPT_1(expr, Expr),
			GET_ACCEPT_1(target, Expr),
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( TryStmt * old ) override final {
		if ( inCache( old ) ) return;
		this->node = new ast::TryStmt(
			old->location,
			GET_ACCEPT_1(block, CompoundStmt),
			GET_ACCEPT_V(handlers, CatchStmt),
			GET_ACCEPT_1(finallyBlock, FinallyStmt),
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( CatchStmt * old ) override final {
		if ( inCache( old ) ) return;
		ast::ExceptionKind kind;
		switch (old->kind) {
		case CatchStmt::Terminate:
			kind = ast::ExceptionKind::Terminate;
			break;
		case CatchStmt::Resume:
			kind = ast::ExceptionKind::Resume;
			break;
		default:
			assertf(false, "Invalid CatchStmt::Kind %d\n", old->kind);
		}

		this->node = new ast::CatchStmt(
			old->location,
			kind,
			GET_ACCEPT_1(decl, Decl),
			GET_ACCEPT_1(cond, Expr),
			GET_ACCEPT_1(body, Stmt),
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( FinallyStmt * old ) override final {
		if ( inCache( old ) ) return;
		this->node = new ast::FinallyStmt(
			old->location,
			GET_ACCEPT_1(block, CompoundStmt),
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( WaitForStmt * old ) override final {
		if ( inCache( old ) ) return;
		ast::WaitForStmt * stmt = new ast::WaitForStmt(
			old->location,
			GET_LABELS_V(old->labels)
		);

		stmt->clauses.reserve( old->clauses.size() );
		for (size_t i = 0 ; i < old->clauses.size() ; ++i) {
			stmt->clauses.push_back({
				ast::WaitForStmt::Target{
					GET_ACCEPT_1(clauses[i].target.function, Expr),
					GET_ACCEPT_V(clauses[i].target.arguments, Expr)
				},
				GET_ACCEPT_1(clauses[i].statement, Stmt),
				GET_ACCEPT_1(clauses[i].condition, Expr)
			});
		}
		stmt->timeout = {
			GET_ACCEPT_1(timeout.time, Expr),
			GET_ACCEPT_1(timeout.statement, Stmt),
			GET_ACCEPT_1(timeout.condition, Expr),
		};
		stmt->orElse = {
			GET_ACCEPT_1(timeout.statement, Stmt),
			GET_ACCEPT_1(timeout.condition, Expr),
		};

		this->node = stmt;
		cache.emplace( old, this->node );
	}

	virtual void visit( WithStmt * old ) override final {
		if ( inCache( old ) ) return;
		this->node = new ast::WithStmt(
			old->location,
			GET_ACCEPT_V(exprs, Expr),
			GET_ACCEPT_1(stmt, Stmt),
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( NullStmt * old ) override final {
		if ( inCache( old ) ) return;
		this->node = new ast::NullStmt(
			old->location,
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( DeclStmt * old ) override final {
		if ( inCache( old ) ) return;
		this->node = new ast::DeclStmt(
			old->location,
			GET_ACCEPT_1(decl, Decl),
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, this->node );
	}

	virtual void visit( ImplicitCtorDtorStmt * old ) override final {
		if ( inCache( old ) ) return;
		auto stmt = new ast::ImplicitCtorDtorStmt(
			old->location,
			nullptr,
			GET_LABELS_V(old->labels)
		);
		cache.emplace( old, stmt );
		stmt->callStmt = GET_ACCEPT_1(callStmt, Stmt);
		this->node = stmt;
	}

	ast::TypeSubstitution * convertTypeSubstitution(const TypeSubstitution * old) {

		if (!old) return nullptr;

		ast::TypeSubstitution *rslt = new ast::TypeSubstitution();

		for (decltype(old->begin()) old_i = old->begin(); old_i != old->end(); old_i++) {
			rslt->add( old_i->first,
			           getAccept1<ast::Type>(old_i->second) );
		}

		for (decltype(old->beginVar()) old_i = old->beginVar(); old_i != old->endVar(); old_i++) {
			rslt->addVar( old_i->first,
			              getAccept1<ast::Expr>(old_i->second) );
		}

		return rslt;
	}

	void convertInferUnion(ast::Expr::InferUnion               &newInferred,
						   const std::map<UniqueId,ParamEntry> &oldInferParams,
						   const std::vector<UniqueId>         &oldResnSlots) {

		assert( oldInferParams.empty() || oldResnSlots.empty() );
		assert( newInferred.mode == ast::Expr::InferUnion::Empty );

		if ( !oldInferParams.empty() ) {
			ast::InferredParams &tgt = newInferred.inferParams();
			for (auto & old : oldInferParams) {
				tgt[old.first] = ast::ParamEntry(
					old.second.decl,
					getAccept1<ast::Decl>(old.second.declptr),
					getAccept1<ast::Type>(old.second.actualType),
					getAccept1<ast::Type>(old.second.formalType),
					getAccept1<ast::Expr>(old.second.expr)
				);
			}
		} else if ( !oldResnSlots.empty() ) {
			ast::ResnSlots &tgt = newInferred.resnSlots();
			for (auto old : oldResnSlots) {
				tgt.push_back(old);
			}
		}
	}

	ast::Expr * visitBaseExpr_SkipResultType(Expression * old, ast::Expr * nw) {

		nw->env    = convertTypeSubstitution(old->env);

		nw->extension = old->extension;
		convertInferUnion(nw->inferred, old->inferParams, old->resnSlots);

		return nw;
	}

	ast::Expr * visitBaseExpr(Expression * old, ast::Expr * nw) {

		nw->result = GET_ACCEPT_1(result, Type);
		return visitBaseExpr_SkipResultType(old, nw);;
	}

	virtual void visit( ApplicationExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::ApplicationExpr(
				old->location,
				GET_ACCEPT_1(function, Expr),
				GET_ACCEPT_V(args, Expr)
			)
		);
	}

	virtual void visit( UntypedExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::UntypedExpr(
				old->location,
				GET_ACCEPT_1(function, Expr),
				GET_ACCEPT_V(args, Expr)
			)
		);
	}

	virtual void visit( NameExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::NameExpr(
				old->location,
				old->get_name()
			)
		);
	}

	virtual void visit( CastExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::CastExpr(
				old->location,
				GET_ACCEPT_1(arg, Expr),
				old->isGenerated ? ast::GeneratedCast : ast::ExplicitCast
			)
		);
	}

	virtual void visit( KeywordCastExpr * old) override final {
		ast::KeywordCastExpr::Target castTarget = ast::KeywordCastExpr::NUMBER_OF_TARGETS;
		switch (old->target) {
			case KeywordCastExpr::Coroutine:
				castTarget = ast::KeywordCastExpr::Coroutine;
				break;
			case KeywordCastExpr::Thread:
				castTarget = ast::KeywordCastExpr::Thread;
				break;
			case KeywordCastExpr::Monitor:
				castTarget = ast::KeywordCastExpr::Monitor;
				break;
			default:
				break;
		}
		assert ( castTarget < ast::KeywordCastExpr::NUMBER_OF_TARGETS );
		this->node = visitBaseExpr( old,
			new ast::KeywordCastExpr(
				old->location,
				GET_ACCEPT_1(arg, Expr),
				castTarget
			)
		);
	}

	virtual void visit( VirtualCastExpr * old ) override final {
		this->node = visitBaseExpr_SkipResultType( old,
			new ast::VirtualCastExpr(
				old->location,
				GET_ACCEPT_1(arg, Expr),
				GET_ACCEPT_1(result, Type)
			)
		);
	}

	virtual void visit( AddressExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::AddressExpr(
				old->location,
				GET_ACCEPT_1(arg, Expr)
			)
		);
	}

	virtual void visit( LabelAddressExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::LabelAddressExpr(
				old->location,
				make_label(&old->arg)
			)
		);
	}

	virtual void visit( UntypedMemberExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::UntypedMemberExpr(
				old->location,
				GET_ACCEPT_1(member, Expr),
				GET_ACCEPT_1(aggregate, Expr)
			)
		);
	}

	virtual void visit( MemberExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::MemberExpr(
				old->location,
				GET_ACCEPT_1(member, DeclWithType),
				GET_ACCEPT_1(aggregate, Expr)
			)
		);
	}

	virtual void visit( VariableExpr * old ) override final {
		auto expr = new ast::VariableExpr(
			old->location
		);

		visitBaseExpr_SkipResultType( old,
			expr
		);

		expr->var = GET_ACCEPT_1(var, DeclWithType);
		expr->result = expr->var->get_type();
		if(const ast::FunctionType * ft = expr->result.as<ast::FunctionType>()) {
			if(dynamic_cast<PointerType *>(old->result)) {
				expr->result = new ast::PointerType(ft);
			}
		}
		add_qualifiers( expr->result, ast::CV::Lvalue );
		this->node = expr;
	}

	bool isIntlikeConstantType(const Type *t) {
		if ( const BasicType * basicType = dynamic_cast< const BasicType * >( t ) ) {
			if ( basicType->isInteger() ) {
				return true;
			}
		} else if ( dynamic_cast< const OneType * >( t ) ) {
			return true;
		} else if ( dynamic_cast< const ZeroType * >( t ) ) {
			return true;
		} else if ( dynamic_cast< const PointerType * >( t ) ) {
			// null pointer constants, with zero int-values
			return true;
		}
		return false;
	}

	int isFloatlikeConstantType(const Type *t) {
		if ( const BasicType * bty = dynamic_cast< const BasicType * >( t ) ) {
			if ( ! bty->isInteger() ) {
				return true;
			}
		}
		return false;
	}

	int isStringlikeConstantType(const Type *t) {
		const Type *referentType = nullptr;
		if ( const ArrayType * aty = dynamic_cast< const ArrayType * >( t ) ) {
			referentType = aty->base;
		} else if ( const PointerType * pty = dynamic_cast< const PointerType * >( t ) ) {
			referentType = pty->base;
		}
		if (referentType) {
			if ( const BasicType * bty = dynamic_cast< const BasicType * >( referentType ) ) {
			   if ( bty->kind == BasicType::Kind::Char ) {
				   return true;
			   }
			}
		}
		return false;
	}

	virtual void visit( ConstantExpr * old ) override final {
		ast::ConstantExpr *rslt = nullptr;
		if (isStringlikeConstantType(old->result)) {
			rslt = new ast::ConstantExpr(
				old->location,
				GET_ACCEPT_1(result, Type),
				old->constant.get_value(),
				0,
				ast::ConstantExpr::Kind::String
			);
		} else if (isIntlikeConstantType(old->result)) {
			rslt = new ast::ConstantExpr(
				old->location,
				GET_ACCEPT_1(result, Type),
				old->constant.get_value(),
				(unsigned long long) old->intValue(),
				ast::ConstantExpr::Kind::Integer
			);
		} else if (isFloatlikeConstantType(old->result)) {
			rslt = new ast::ConstantExpr(
				old->location,
				GET_ACCEPT_1(result, Type),
				old->constant.get_value(),
				(double) old->constant.get_dval()
			);
		}
		assert(rslt);
		this->node = visitBaseExpr( old, rslt );
	}

	virtual void visit( SizeofExpr * old ) override final {
		assert (old->expr || old->type);
		assert (! (old->expr && old->type));
		ast::SizeofExpr *rslt;
		if (old->expr) {
			assert(!old->isType);
			rslt = new ast::SizeofExpr(
				old->location,
				GET_ACCEPT_1(expr, Expr)
			);
		}
		if (old->type) {
			assert(old->isType);
			rslt = new ast::SizeofExpr(
				old->location,
				GET_ACCEPT_1(type, Type)
			);
		}
		this->node = visitBaseExpr( old, rslt );
	}

	virtual void visit( AlignofExpr * old ) override final {
		assert (old->expr || old->type);
		assert (! (old->expr && old->type));
		ast::AlignofExpr *rslt;
		if (old->expr) {
			assert(!old->isType);
			rslt = new ast::AlignofExpr(
				old->location,
				GET_ACCEPT_1(expr, Expr)
			);
		}
		if (old->type) {
			assert(old->isType);
			rslt = new ast::AlignofExpr(
				old->location,
				GET_ACCEPT_1(type, Type)
			);
		}
		this->node = visitBaseExpr( old, rslt );
	}

	virtual void visit( UntypedOffsetofExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::UntypedOffsetofExpr(
				old->location,
				GET_ACCEPT_1(type, Type),
				old->member
			)
		);
	}

	virtual void visit( OffsetofExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::OffsetofExpr(
				old->location,
				GET_ACCEPT_1(type, Type),
				GET_ACCEPT_1(member, DeclWithType)
			)
		);
	}

	virtual void visit( OffsetPackExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::OffsetPackExpr(
				old->location,
				GET_ACCEPT_1(type, StructInstType)
			)
		);
	}

	virtual void visit( LogicalExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::LogicalExpr(
				old->location,
				GET_ACCEPT_1(arg1, Expr),
				GET_ACCEPT_1(arg2, Expr),
				old->get_isAnd() ?
					ast::LogicalFlag::AndExpr :
					ast::LogicalFlag::OrExpr
			)
		);
	}

	virtual void visit( ConditionalExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::ConditionalExpr(
				old->location,
				GET_ACCEPT_1(arg1, Expr),
				GET_ACCEPT_1(arg2, Expr),
				GET_ACCEPT_1(arg3, Expr)
			)
		);
	}

	virtual void visit( CommaExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::CommaExpr(
				old->location,
				GET_ACCEPT_1(arg1, Expr),
				GET_ACCEPT_1(arg2, Expr)
			)
		);
	}

	virtual void visit( TypeExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::TypeExpr(
				old->location,
				GET_ACCEPT_1(type, Type)
			)
		);
	}

	virtual void visit( AsmExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::AsmExpr(
				old->location,
				GET_ACCEPT_1(inout, Expr),
				GET_ACCEPT_1(constraint, Expr),
				GET_ACCEPT_1(operand, Expr)
			)
		);
	}

	virtual void visit( ImplicitCopyCtorExpr * old ) override final {
		auto rslt = new ast::ImplicitCopyCtorExpr(
			old->location,
			GET_ACCEPT_1(callExpr, ApplicationExpr)
		);

		this->node = visitBaseExpr( old, rslt );
	}

	virtual void visit( ConstructorExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::ConstructorExpr(
				old->location,
				GET_ACCEPT_1(callExpr, Expr)
			)
		);
	}

	virtual void visit( CompoundLiteralExpr * old ) override final {
		this->node = visitBaseExpr_SkipResultType( old,
			new ast::CompoundLiteralExpr(
				old->location,
				GET_ACCEPT_1(result, Type),
				GET_ACCEPT_1(initializer, Init)
			)
		);
	}

	virtual void visit( RangeExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::RangeExpr(
				old->location,
				GET_ACCEPT_1(low, Expr),
				GET_ACCEPT_1(high, Expr)
			)
		);
	}

	virtual void visit( UntypedTupleExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::UntypedTupleExpr(
				old->location,
				GET_ACCEPT_V(exprs, Expr)
			)
		);
	}

	virtual void visit( TupleExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::TupleExpr(
				old->location,
				GET_ACCEPT_V(exprs, Expr)
			)
		);
	}

	virtual void visit( TupleIndexExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::TupleIndexExpr(
				old->location,
				GET_ACCEPT_1(tuple, Expr),
				old->index
			)
		);
	}

	virtual void visit( TupleAssignExpr * old ) override final {
		this->node = visitBaseExpr_SkipResultType( old,
			new ast::TupleAssignExpr(
				old->location,
				GET_ACCEPT_1(result, Type),
				GET_ACCEPT_1(stmtExpr, StmtExpr)
			)
		);
	}

	virtual void visit( StmtExpr * old ) override final {
		auto rslt = new ast::StmtExpr(
			old->location,
			GET_ACCEPT_1(statements, CompoundStmt)
		);
		rslt->returnDecls = GET_ACCEPT_V(returnDecls, ObjectDecl);
		rslt->dtors       = GET_ACCEPT_V(dtors      , Expr);

		this->node = visitBaseExpr_SkipResultType( old, rslt );
	}

	virtual void visit( UniqueExpr * old ) override final {
		auto rslt = new ast::UniqueExpr(
			old->location,
			GET_ACCEPT_1(expr, Expr)
		);
		rslt->object = GET_ACCEPT_1(object, ObjectDecl);
		rslt->var    = GET_ACCEPT_1(var   , VariableExpr);

		this->node = visitBaseExpr( old, rslt );
	}

	virtual void visit( UntypedInitExpr * old ) override final {
		std::deque<ast::InitAlternative> initAlts;
		for (auto ia : old->initAlts) {
			initAlts.push_back(ast::InitAlternative(
				getAccept1< ast::Type, Type * >( ia.type ),
				getAccept1< ast::Designation, Designation * >( ia.designation )
			));
		}
		this->node = visitBaseExpr( old,
			new ast::UntypedInitExpr(
				old->location,
				GET_ACCEPT_1(expr, Expr),
				std::move(initAlts)
			)
		);
	}

	virtual void visit( InitExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::InitExpr(
				old->location,
				GET_ACCEPT_1(expr, Expr),
				GET_ACCEPT_1(designation, Designation)
			)
		);
	}

	virtual void visit( DeletedExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::DeletedExpr(
				old->location,
				GET_ACCEPT_1(expr, Expr),
				inCache(old->deleteStmt) ?
					this->node :
					GET_ACCEPT_1(deleteStmt, Node)
			)
		);
	}

	virtual void visit( DefaultArgExpr * old ) override final {
		this->node = visitBaseExpr( old,
			new ast::DefaultArgExpr(
				old->location,
				GET_ACCEPT_1(expr, Expr)
			)
		);
	}

	virtual void visit( GenericExpr * old ) override final {
		std::vector<ast::GenericExpr::Association> associations;
		for (auto association : old->associations) {
			associations.push_back(ast::GenericExpr::Association(
				getAccept1< ast::Type, Type * >( association.type ),
				getAccept1< ast::Expr, Expression * >( association.expr )
			));
		}
		this->node = visitBaseExpr( old,
			new ast::GenericExpr(
				old->location,
				GET_ACCEPT_1(control, Expr),
				std::move(associations)
			)
		);
	}

	virtual void visit( VoidType * old ) override final {
		this->node = new ast::VoidType{ cv( old ) };
	}

	virtual void visit( BasicType * old ) override final {
		auto type = new ast::BasicType{ (ast::BasicType::Kind)(unsigned)old->kind, cv( old ) };
		// I believe this should always be a BasicType.
		if ( Validate::SizeType == old ) {
			sizeType = type;
		}
		this->node = type;
	}

	virtual void visit( PointerType * old ) override final {
		this->node = new ast::PointerType{
			GET_ACCEPT_1( base, Type ),
			GET_ACCEPT_1( dimension, Expr ),
			(ast::LengthFlag)old->isVarLen,
			(ast::DimensionFlag)old->isStatic,
			cv( old )
		};
	}

	virtual void visit( ArrayType * old ) override final {
		this->node = new ast::ArrayType{
			GET_ACCEPT_1( base, Type ),
			GET_ACCEPT_1( dimension, Expr ),
			(ast::LengthFlag)old->isVarLen,
			(ast::DimensionFlag)old->isStatic,
			cv( old )
		};
	}

	virtual void visit( ReferenceType * old ) override final {
		this->node = new ast::ReferenceType{
			GET_ACCEPT_1( base, Type ),
			cv( old )
		};
	}

	virtual void visit( QualifiedType * old ) override final {
		this->node = new ast::QualifiedType{
			GET_ACCEPT_1( parent, Type ),
			GET_ACCEPT_1( child, Type ),
			cv( old )
		};
	}

	virtual void visit( FunctionType * old ) override final {
		auto ty = new ast::FunctionType {
			(ast::ArgumentFlag)old->isVarArgs,
			cv( old )
		};
		ty->returns = GET_ACCEPT_V( returnVals, DeclWithType );
		ty->params = GET_ACCEPT_V( parameters, DeclWithType );
		ty->forall = GET_ACCEPT_V( forall, TypeDecl );
		this->node = ty;
	}

	void postvisit( ReferenceToType * old, ast::ReferenceToType * ty ) {
		ty->forall = GET_ACCEPT_V( forall, TypeDecl );
		ty->params = GET_ACCEPT_V( parameters, Expr );
		ty->hoistType = old->hoistType;
	}

	virtual void visit( StructInstType * old ) override final {
		ast::StructInstType * ty;
		if ( old->baseStruct ) {
			ty = new ast::StructInstType{
				GET_ACCEPT_1( baseStruct, StructDecl ),
				cv( old ),
				GET_ACCEPT_V( attributes, Attribute )
			};
		} else {
			ty = new ast::StructInstType{
				old->name,
				cv( old ),
				GET_ACCEPT_V( attributes, Attribute )
			};
		}
		postvisit( old, ty );
		this->node = ty;
	}

	virtual void visit( UnionInstType * old ) override final {
		ast::UnionInstType * ty;
		if ( old->baseUnion ) {
			ty = new ast::UnionInstType{
				GET_ACCEPT_1( baseUnion, UnionDecl ),
				cv( old ),
				GET_ACCEPT_V( attributes, Attribute )
			};
		} else {
			ty = new ast::UnionInstType{
				old->name,
				cv( old ),
				GET_ACCEPT_V( attributes, Attribute )
			};
		}
		postvisit( old, ty );
		this->node = ty;
	}

	virtual void visit( EnumInstType * old ) override final {
		ast::EnumInstType * ty;
		if ( old->baseEnum ) {
			ty = new ast::EnumInstType{
				GET_ACCEPT_1( baseEnum, EnumDecl ),
				cv( old ),
				GET_ACCEPT_V( attributes, Attribute )
			};
		} else {
			ty = new ast::EnumInstType{
				old->name,
				cv( old ),
				GET_ACCEPT_V( attributes, Attribute )
			};
		}
		postvisit( old, ty );
		this->node = ty;
	}

	virtual void visit( TraitInstType * old ) override final {
		ast::TraitInstType * ty;
		if ( old->baseTrait ) {
			ty = new ast::TraitInstType{
				GET_ACCEPT_1( baseTrait, TraitDecl ),
				cv( old ),
				GET_ACCEPT_V( attributes, Attribute )
			};
		} else {
			ty = new ast::TraitInstType{
				old->name,
				cv( old ),
				GET_ACCEPT_V( attributes, Attribute )
			};
		}
		postvisit( old, ty );
		this->node = ty;
	}

	virtual void visit( TypeInstType * old ) override final {
		ast::TypeInstType * ty;
		if ( old->baseType ) {
			ty = new ast::TypeInstType{
				old->name,
				GET_ACCEPT_1( baseType, TypeDecl ),
				cv( old ),
				GET_ACCEPT_V( attributes, Attribute )
			};
		} else {
			ty = new ast::TypeInstType{
				old->name,
				old->isFtype ? ast::TypeVar::Ftype : ast::TypeVar::Dtype,
				cv( old ),
				GET_ACCEPT_V( attributes, Attribute )
			};
		}
		postvisit( old, ty );
		this->node = ty;
	}

	virtual void visit( TupleType * old ) override final {
		this->node = new ast::TupleType{
			GET_ACCEPT_V( types, Type ),
			// members generated by TupleType c'tor
			cv( old )
		};
	}

	virtual void visit( TypeofType * old ) override final {
		this->node = new ast::TypeofType{
			GET_ACCEPT_1( expr, Expr ),
			(ast::TypeofType::Kind)old->is_basetypeof,
			cv( old )
		};
	}

	virtual void visit( AttrType * ) override final {
		assertf( false, "AttrType deprecated in new AST." );
	}

	virtual void visit( VarArgsType * old ) override final {
		this->node = new ast::VarArgsType{ cv( old ) };
	}

	virtual void visit( ZeroType * old ) override final {
		this->node = new ast::ZeroType{ cv( old ) };
	}

	virtual void visit( OneType * old ) override final {
		this->node = new ast::OneType{ cv( old ) };
	}

	virtual void visit( GlobalScopeType * ) override final {
		this->node = new ast::GlobalScopeType{};
	}

	virtual void visit( Designation * old ) override final {
		this->node = new ast::Designation(
			old->location,
			GET_ACCEPT_D(designators, Expr)
		);
	}

	virtual void visit( SingleInit * old ) override final {
		this->node = new ast::SingleInit(
			old->location,
			GET_ACCEPT_1(value, Expr),
			(old->get_maybeConstructed()) ? ast::MaybeConstruct : ast::DoConstruct
		);
	}

	virtual void visit( ListInit * old ) override final {
		this->node = new ast::ListInit(
			old->location,
			GET_ACCEPT_V(initializers, Init),
			GET_ACCEPT_V(designations, Designation),
			(old->get_maybeConstructed()) ? ast::MaybeConstruct : ast::DoConstruct
		);
	}

	virtual void visit( ConstructorInit * old ) override final {
		this->node = new ast::ConstructorInit(
			old->location,
			GET_ACCEPT_1(ctor, Stmt),
			GET_ACCEPT_1(dtor, Stmt),
			GET_ACCEPT_1(init, Init)
		);
	}

	virtual void visit( Constant * ) override final {
		// Handled in visit( ConstantEpxr * ).
		// In the new tree, Constant fields are inlined into containing ConstantExpression.
		assert( 0 );
	}

	virtual void visit( Attribute * old ) override final {
		this->node = new ast::Attribute(
			old->name,
			GET_ACCEPT_V( parameters, Expr )
		);
	}

	virtual void visit( AttrExpr * ) override final {
		assertf( false, "AttrExpr deprecated in new AST." );
	}
};

#undef GET_LABELS_V
#undef GET_ACCEPT_V
#undef GET_ACCEPT_1

std::list< ast::ptr< ast::Decl > > convert( const std::list< Declaration * > && translationUnit ) {
	ConverterOldToNew c;
	std::list< ast::ptr< ast::Decl > > decls;
	for(auto d : translationUnit) {
		d->accept( c );
		decls.emplace_back( c.decl() );
	}
	deleteAll(translationUnit);
	return decls;
}
