//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// CodeGenerator.cc -- 
//
// Author           : Richard C. Bilson
// Created On       : Mon May 18 07:44:20 2015
// Last Modified By : Rob Schluntz
// Last Modified On : Wed Jul 15 14:47:42 2015
// Update Count     : 177
//

#include <algorithm>
#include <iostream>
#include <cassert>
#include <list>

#include "Parser/ParseNode.h"

#include "SynTree/Type.h"
#include "SynTree/Expression.h"
#include "SynTree/Initializer.h"
#include "SynTree/Statement.h"

#include "utility.h"
#include "UnimplementedError.h"

#include "CodeGenerator.h"
#include "OperatorTable.h"
#include "GenType.h"

using namespace std;

namespace CodeGen {
	int CodeGenerator::tabsize = 4;

	// the kinds of statements that would ideally be followed by whitespace
	bool wantSpacing( Statement * stmt) {
		return dynamic_cast< IfStmt * >( stmt ) || dynamic_cast< CompoundStmt * >( stmt ) ||
			dynamic_cast< WhileStmt * >( stmt ) || dynamic_cast< ForStmt * > ( stmt ) || dynamic_cast< SwitchStmt *>( stmt );
	}

	ostream & CodeGenerator::Indenter::operator()( ostream & output ) {
	  return output << string( cg.cur_indent, ' ' );
	}

	ostream & operator<<( ostream & output, CodeGenerator::Indenter &indent ) {
		return indent( output );
	}

	CodeGenerator::CodeGenerator( std::ostream &os ) : indent(*this), cur_indent( 0 ), insideFunction( false ), output( os ) { }

	CodeGenerator::CodeGenerator( std::ostream &os, std::string init, int indentation, bool infunp )
			: indent(*this), cur_indent( indentation ), insideFunction( infunp ), output( os ) {
		//output << std::string( init );
	}

	CodeGenerator::CodeGenerator( std::ostream &os, char *init, int indentation, bool infunp )
			: indent(*this), cur_indent( indentation ), insideFunction( infunp ), output( os ) {
		//output << std::string( init );
	}

	string mangleName( DeclarationWithType *decl ) {
		if ( decl->get_mangleName() != "" ) {
			return decl->get_mangleName();
		} else {
			return decl->get_name();
		} // if
	}

	//*** Declarations
	void CodeGenerator::visit( FunctionDecl *functionDecl ) {
		handleStorageClass( functionDecl );
		if ( functionDecl->get_isInline() ) {
			output << "inline ";
		} // if
		if ( functionDecl->get_isNoreturn() ) {
			output << "_Noreturn ";
		} // if
		output << genType( functionDecl->get_functionType(), mangleName( functionDecl ) );

		// how to get this to the Functype?
		std::list< Declaration * > olds = functionDecl->get_oldDecls();
		if ( ! olds.empty() ) {
			output << " /* function has old declaration */";
		} // if

		// acceptAll( functionDecl->get_oldDecls(), *this );
		if ( functionDecl->get_statements() ) {
			functionDecl->get_statements()->accept(*this );
		} // if
	}

	void CodeGenerator::visit( ObjectDecl *objectDecl ) {
		handleStorageClass( objectDecl );
		output << genType( objectDecl->get_type(), mangleName( objectDecl ) );
	
		if ( objectDecl->get_init() ) {
			output << " = ";
			objectDecl->get_init()->accept( *this );
		} // if
		if ( objectDecl->get_bitfieldWidth() ) {
			output << ":";
			objectDecl->get_bitfieldWidth()->accept( *this );
		} // if
	}

	void CodeGenerator::handleAggregate( AggregateDecl *aggDecl ) {
		if ( aggDecl->get_name() != "" )
			output << aggDecl->get_name();
	
		std::list< Declaration * > &memb = aggDecl->get_members();

		if ( ! memb.empty() ) {
			output << " {" << endl;

			cur_indent += CodeGenerator::tabsize; 
			for ( std::list< Declaration* >::iterator i = memb.begin(); i != memb.end();  i++) {
				output << indent; 
				(*i)->accept(*this );
				output << ";" << endl;
			}

			cur_indent -= CodeGenerator::tabsize; 

			output << indent << "}";
		} // if
	}

	void CodeGenerator::visit( StructDecl *structDecl ) {
		output << "struct ";
		handleAggregate( structDecl );
	}

	void CodeGenerator::visit( UnionDecl *aggregateDecl ) {
		output << "union ";
		handleAggregate( aggregateDecl );
	}
  
	void CodeGenerator::visit( EnumDecl *aggDecl ) {
		output << "enum ";

		if ( aggDecl->get_name() != "" )
			output << aggDecl->get_name();
	
		std::list< Declaration* > &memb = aggDecl->get_members();

		if ( ! memb.empty() ) {
			output << " {" << endl;

			cur_indent += CodeGenerator::tabsize; 
			for ( std::list< Declaration* >::iterator i = memb.begin(); i != memb.end();  i++) {
				ObjectDecl *obj = dynamic_cast< ObjectDecl* >( *i );
				assert( obj );
				output << indent << mangleName( obj ); 
				if ( obj->get_init() ) {
					output << " = ";
					obj->get_init()->accept(*this );
				} // if
				output << "," << endl;
			} // for

			cur_indent -= CodeGenerator::tabsize; 

			output << indent << "}";
		} // if
	}
  
	void CodeGenerator::visit( ContextDecl *aggregateDecl ) {}
  
	void CodeGenerator::visit( TypedefDecl *typeDecl ) {
		output << "typedef ";
		output << genType( typeDecl->get_base(), typeDecl->get_name() );
	}
  
	void CodeGenerator::visit( TypeDecl *typeDecl ) {
		// really, we should mutate this into something that isn't a TypeDecl but that requires large-scale changes,
		// still to be done
		output << "extern unsigned long " << typeDecl->get_name();
		if ( typeDecl->get_base() ) {
			output << " = sizeof( " << genType( typeDecl->get_base(), "" ) << " )";
		} // if
	}

	void CodeGenerator::visit( SingleInit *init ) {
		init->get_value()->accept( *this );
	}

	void CodeGenerator::visit( ListInit *init ) {
		output << "{ ";
		genCommaList( init->begin_initializers(), init->end_initializers() );
		output << " }";
	}

	void CodeGenerator::visit( Constant *constant ) { 
		output << constant->get_value() ;
	}

	//*** Expressions
	void CodeGenerator::visit( ApplicationExpr *applicationExpr ) {
		if ( VariableExpr *varExpr = dynamic_cast< VariableExpr* >( applicationExpr->get_function() ) ) {
			OperatorInfo opInfo;
			if ( varExpr->get_var()->get_linkage() == LinkageSpec::Intrinsic && operatorLookup( varExpr->get_var()->get_name(), opInfo ) ) {
				std::list< Expression* >::iterator arg = applicationExpr->get_args().begin();
				switch ( opInfo.type ) {
				  case OT_PREFIXASSIGN:
				  case OT_POSTFIXASSIGN:
				  case OT_INFIXASSIGN:
					{
						assert( arg != applicationExpr->get_args().end() );
						if ( AddressExpr *addrExpr = dynamic_cast< AddressExpr * >( *arg ) ) {
	        
							*arg = addrExpr->get_arg();
						} else {
							UntypedExpr *newExpr = new UntypedExpr( new NameExpr( "*?" ) );
							newExpr->get_args().push_back( *arg );
							*arg = newExpr;
						} // if
						break;
					}
	      
				  default:
					// do nothing
					;
				}
	    
				switch ( opInfo.type ) {
				  case OT_INDEX:
					assert( applicationExpr->get_args().size() == 2 );
					(*arg++)->accept( *this );
					output << "[";
					(*arg)->accept( *this );
					output << "]";
					break;
	      
				  case OT_CALL:
				  case OT_CTOR:
				  case OT_DTOR:
					// there are no intrinsic definitions of the function call operator or constructors or destructors
					assert( false );
					break;
	      
				  case OT_PREFIX:
				  case OT_PREFIXASSIGN:
					assert( applicationExpr->get_args().size() == 1 );
					output << "(";
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;
	      
				  case OT_POSTFIX:
				  case OT_POSTFIXASSIGN:
					assert( applicationExpr->get_args().size() == 1 );
					(*arg)->accept( *this );
					output << opInfo.symbol;
					break;

				  case OT_INFIX:
				  case OT_INFIXASSIGN:
					assert( applicationExpr->get_args().size() == 2 );
					output << "(";
					(*arg++)->accept( *this );
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;
	      
				  case OT_CONSTANT:
				  case OT_LABELADDRESS:
					// there are no intrinsic definitions of 0/1 or label addresses as functions
					assert( false );
				}
			} else {
				varExpr->accept( *this );
				output << "(";
				genCommaList( applicationExpr->get_args().begin(), applicationExpr->get_args().end() );
				output << ")";
			} // if
		} else {
			applicationExpr->get_function()->accept( *this );
			output << "(";
			genCommaList( applicationExpr->get_args().begin(), applicationExpr->get_args().end() );
			output << ")";
		} // if
	}
  
	void CodeGenerator::visit( UntypedExpr *untypedExpr ) {
		if ( NameExpr *nameExpr = dynamic_cast< NameExpr* >( untypedExpr->get_function() ) ) {
			OperatorInfo opInfo;
			if ( operatorLookup( nameExpr->get_name(), opInfo ) ) {
				std::list< Expression* >::iterator arg = untypedExpr->get_args().begin();
				switch ( opInfo.type ) {
				  case OT_INDEX:
					assert( untypedExpr->get_args().size() == 2 );
					(*arg++)->accept( *this );
					output << "[";
					(*arg)->accept( *this );
					output << "]";
					break;
	      
				  case OT_CALL:
					assert( false );
					break;
	      
				  case OT_PREFIX:
				  case OT_PREFIXASSIGN:
				  case OT_LABELADDRESS:
					assert( untypedExpr->get_args().size() == 1 );
					output << "(";
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;
	      
				  case OT_POSTFIX:
				  case OT_POSTFIXASSIGN:
					assert( untypedExpr->get_args().size() == 1 );
					(*arg)->accept( *this );
					output << opInfo.symbol;
					break;
  
				  case OT_INFIX:
				  case OT_INFIXASSIGN:
					assert( untypedExpr->get_args().size() == 2 );
					output << "(";
					(*arg++)->accept( *this );
					output << opInfo.symbol;
					(*arg)->accept( *this );
					output << ")";
					break;
					
				  case OT_CONSTANT:
					// there are no intrinsic definitions of 0 or 1 as functions
					assert( false );
				}
			} else {
				nameExpr->accept( *this );
				output << "(";
				genCommaList( untypedExpr->get_args().begin(), untypedExpr->get_args().end() );
				output << ")";
			} // if
		} else {
			untypedExpr->get_function()->accept( *this );
			output << "(";
			genCommaList( untypedExpr->get_args().begin(), untypedExpr->get_args().end() );
			output << ")";
		} // if
	}
  
	void CodeGenerator::visit( NameExpr *nameExpr ) {
		OperatorInfo opInfo;
		if ( operatorLookup( nameExpr->get_name(), opInfo ) ) {
			assert( opInfo.type == OT_CONSTANT );
			output << opInfo.symbol;
		} else {
			output << nameExpr->get_name();
		} // if
	}
  
	void CodeGenerator::visit( AddressExpr *addressExpr ) {
		output << "(&";
		// this hack makes sure that we don't convert "constant_zero" to "0" if we're taking its address
		if ( VariableExpr *variableExpr = dynamic_cast< VariableExpr* >( addressExpr->get_arg() ) ) {
			output << mangleName( variableExpr->get_var() );
		} else {
			addressExpr->get_arg()->accept( *this );
		} // if
		output << ")";
	}

	void CodeGenerator::visit( CastExpr *castExpr ) {
		output << "((";
		if ( castExpr->get_results().empty() ) {
			output << "void" ;
		} else {
			output << genType( castExpr->get_results().front(), "" );
		} // if
		output << ")";
		castExpr->get_arg()->accept( *this );
		output << ")";
	}
  
	void CodeGenerator::visit( UntypedMemberExpr *memberExpr ) {
		assert( false );
	}
  
	void CodeGenerator::visit( MemberExpr *memberExpr ) {
		memberExpr->get_aggregate()->accept( *this );
		output << "." << mangleName( memberExpr->get_member() );
	}
  
	void CodeGenerator::visit( VariableExpr *variableExpr ) {
		OperatorInfo opInfo;
		if ( variableExpr->get_var()->get_linkage() == LinkageSpec::Intrinsic && operatorLookup( variableExpr->get_var()->get_name(), opInfo ) && opInfo.type == OT_CONSTANT ) {
			output << opInfo.symbol;
		} else {
			output << mangleName( variableExpr->get_var() );
		} // if
	}
  
	void CodeGenerator::visit( ConstantExpr *constantExpr ) {
		assert( constantExpr->get_constant() );
		constantExpr->get_constant()->accept( *this );
	}
  
	void CodeGenerator::visit( SizeofExpr *sizeofExpr ) {
		output << "sizeof(";
		if ( sizeofExpr->get_isType() ) {
			output << genType( sizeofExpr->get_type(), "" );
		} else {
			sizeofExpr->get_expr()->accept( *this );
		} // if
		output << ")";
	}
  
	void CodeGenerator::visit( LogicalExpr *logicalExpr ) {
		output << "(";
		logicalExpr->get_arg1()->accept( *this );
		if ( logicalExpr->get_isAnd() ) {
			output << " && ";
		} else {
			output << " || ";
		} // if
		logicalExpr->get_arg2()->accept( *this );
		output << ")";
	}
  
	void CodeGenerator::visit( ConditionalExpr *conditionalExpr ) {
		output << "(";
		conditionalExpr->get_arg1()->accept( *this );
		output << " ? ";
		conditionalExpr->get_arg2()->accept( *this );
		output << " : ";
		conditionalExpr->get_arg3()->accept( *this );
		output << ")";
	}
  
	void CodeGenerator::visit( CommaExpr *commaExpr ) {
		output << "(";
		commaExpr->get_arg1()->accept( *this );
		output << " , ";
		commaExpr->get_arg2()->accept( *this );
		output << ")";
	}
  
	void CodeGenerator::visit( TupleExpr *tupleExpr ) {}
  
	void CodeGenerator::visit( TypeExpr *typeExpr ) {}

	//*** Statements
	void CodeGenerator::visit( CompoundStmt *compoundStmt ) {
		std::list<Statement*> ks = compoundStmt->get_kids();
		output << "{" << endl;

		cur_indent += CodeGenerator::tabsize;

		for ( std::list<Statement *>::iterator i = ks.begin(); i != ks.end();  i++) {
			output << indent << printLabels( (*i)->get_labels() );
			(*i)->accept(*this );

			output << endl;
			if ( wantSpacing( *i ) ) {
				output << endl;
			}
		}
		cur_indent -= CodeGenerator::tabsize; 

		output << indent << "}";
	}

	void CodeGenerator::visit( ExprStmt *exprStmt ) {
		// I don't see why this check is necessary. 
		// If this starts to cause problems then put it back in, 
		// with an explanation
		assert( exprStmt );

		// if ( exprStmt != 0 ) {
		exprStmt->get_expr()->accept( *this );
		output << ";" ;
		// } // if
	}

	void CodeGenerator::visit( IfStmt *ifStmt ) {
		output << "if (";
		ifStmt->get_condition()->accept(*this );
		output << ") ";

		ifStmt->get_thenPart()->accept(*this );

		if ( ifStmt->get_elsePart() != 0) {
			output << " else ";
			ifStmt->get_elsePart()->accept(*this );
		} // if
	}

	void CodeGenerator::visit( SwitchStmt *switchStmt ) {
		output << "switch (" ;
		switchStmt->get_condition()->accept(*this );
		output << ") ";
		
		output << "{" << std::endl;
		cur_indent += CodeGenerator::tabsize;

		acceptAll( switchStmt->get_branches(), *this );

		cur_indent -= CodeGenerator::tabsize;

		output << indent << "}";
	}

	void CodeGenerator::visit( CaseStmt *caseStmt ) {
		output << indent;
		if ( caseStmt->isDefault()) {
			output << "default";
		} else {
			output << "case ";
			caseStmt->get_condition()->accept(*this );
		} // if
		output << ":\n";
		
		std::list<Statement *> sts = caseStmt->get_statements();

		cur_indent += CodeGenerator::tabsize;
		for ( std::list<Statement *>::iterator i = sts.begin(); i != sts.end();  i++) {
			output << indent << printLabels( (*i)->get_labels() )  ;
			(*i)->accept(*this );
			output << endl;
		}
		cur_indent -= CodeGenerator::tabsize;
	}

	void CodeGenerator::visit( BranchStmt *branchStmt ) {
		switch ( branchStmt->get_type()) {
		  case BranchStmt::Goto:
			if ( ! branchStmt->get_target().empty() )
				output << "goto " << branchStmt->get_target();
			else { 
				if ( branchStmt->get_computedTarget() != 0 ) {
					output << "goto *";
					branchStmt->get_computedTarget()->accept( *this );
				} // if
			} // if
			break;
		  case BranchStmt::Break:
			output << "break";
			break;
		  case BranchStmt::Continue:
			output << "continue";
			break;
		}
		output << ";";
	}


	void CodeGenerator::visit( ReturnStmt *returnStmt ) {
		output << "return ";

		// xxx -- check for null expression;
		if ( returnStmt->get_expr() ) {
			returnStmt->get_expr()->accept( *this );
		} // if
		output << ";";
	}

	void CodeGenerator::visit( WhileStmt *whileStmt ) {
		if ( whileStmt->get_isDoWhile() )
			output << "do" ;
		else {
			output << "while (" ;
			whileStmt->get_condition()->accept(*this );
			output << ")";
		} // if
		output << " ";

		output << CodeGenerator::printLabels( whileStmt->get_body()->get_labels() );
		whileStmt->get_body()->accept( *this );

		output << indent;

		if ( whileStmt->get_isDoWhile() ) {
			output << " while (" ;
			whileStmt->get_condition()->accept(*this );
			output << ");";
		} // if
	}

	void CodeGenerator::visit( ForStmt *forStmt ) {
		// initialization is always hoisted, so don't 
		// bother doing anything with that 
		output << "for (;";

		if ( forStmt->get_condition() != 0 )
			forStmt->get_condition()->accept( *this );
		output << ";";

		if ( forStmt->get_increment() != 0 )
			forStmt->get_increment()->accept( *this );
		output << ") ";

		if ( forStmt->get_body() != 0 ) {
			output << CodeGenerator::printLabels( forStmt->get_body()->get_labels() );
			forStmt->get_body()->accept( *this );
		} // if
	}

	void CodeGenerator::visit( NullStmt *nullStmt ) {
		//output << indent << CodeGenerator::printLabels( nullStmt->get_labels() );
		output << "/* null statement */ ;";
	}

	void CodeGenerator::visit( DeclStmt *declStmt ) {
		declStmt->get_decl()->accept( *this );
	
		if ( doSemicolon( declStmt->get_decl() ) ) {
			output << ";";
		} // if
	}

	std::string CodeGenerator::printLabels( std::list< Label > &l ) {
		std::string str( "" );
		l.unique(); // assumes a sorted list. Why not use set?

		for ( std::list< Label >::iterator i = l.begin(); i != l.end(); i++ )
			str += *i + ": ";

		return str;
	}

	void CodeGenerator::handleStorageClass( Declaration *decl ) {
		switch ( decl->get_storageClass() ) {
		  case DeclarationNode::Extern:
			output << "extern ";
			break;
		  case DeclarationNode::Static:
			output << "static ";
			break;
		  case DeclarationNode::Auto:
			// silently drop storage class
			break;
		  case DeclarationNode::Register:
			output << "register ";
			break;
		  case DeclarationNode::Inline:
			output << "inline ";
			break;
		  case DeclarationNode::Fortran:
			output << "fortran ";
			break;
		  case DeclarationNode::Noreturn:
			output << "_Noreturn ";
			break;
		  case DeclarationNode::Threadlocal:
			output << "_Thread_local ";
			break;
		  case DeclarationNode::NoStorageClass:
			break;
		} // switch
	}
} // namespace CodeGen

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
