//
// Cforall Version 1.0.0 Copyright (C) 2015 University of Waterloo
//
// The contents of this file are covered under the licence agreement in the
// file "LICENCE" distributed with Cforall.
//
// Expr.hpp --
//
// Author           : Aaron B. Moss
// Created On       : Fri May 10 10:30:00 2019
// Last Modified By : Aaron B. Moss
// Created On       : Fri May 10 10:30:00 2019
// Update Count     : 1
//

#pragma once

#include <cassert>
#include <map>
#include <string>
#include <utility>        // for move
#include <vector>

#include "Fwd.hpp"        // for UniqueId
#include "Label.hpp"
#include "ParseNode.hpp"
#include "Visitor.hpp"

namespace ast {

/// Contains the ID of a declaration and a type that is derived from that declaration,
/// but subject to decay-to-pointer and type parameter renaming
struct ParamEntry {
	UniqueId decl;
	ptr<Type> actualType;
	ptr<Type> formalType;
	ptr<Expr> expr;

	ParamEntry() : decl( 0 ), actualType( nullptr ), formalType( nullptr ), expr( nullptr ) {}
	ParamEntry( UniqueId id, Type* actual, Type* formal, Expr* e )
	: decl( id ), actualType( actual ), formalType( formal ), expr( e ) {}
};

/// Pre-resolution list of parameters to infer
using ResnSlots = std::vector<UniqueId>;
/// Post-resolution map of inferred parameters
using InferredParams = std::map< UniqueId, ParamEntry >;

/// Base node for expressions
class Expr : public ParseNode {
public:
	/// Saves space (~16 bytes) by combining ResnSlots and InferredParams
	struct InferUnion {
		enum { Empty, Slots, Params } mode;
		union data_t {
			char def;
			ResnSlots resnSlots;
			InferredParams inferParams;

			data_t() : def('\0') {}
			~data_t() {}
		} data;

		/// initializes from other InferUnion
		void init_from( const InferUnion& o ) {
			switch ( o.mode ) {
			case Empty:  return;
			case Slots:  new(&data.resnSlots) ResnSlots{ o.data.resnSlots }; return;
			case Params: new(&data.inferParams) InferredParams{ o.data.inferParams }; return;
			}
		}

		/// initializes from other InferUnion (move semantics)
		void init_from( InferUnion&& o ) {
			switch ( o.mode ) {
			case Empty:  return;
			case Slots:  new(&data.resnSlots) ResnSlots{ std::move(o.data.resnSlots) }; return;
			case Params:
				new(&data.inferParams) InferredParams{ std::move(o.data.inferParams) }; return;
			}
		}

		/// clears variant fields
		void reset() {
			switch( mode ) {
			case Empty:  return;
			case Slots:  data.resnSlots.~ResnSlots(); return;
			case Params: data.inferParams.~InferredParams(); return;
			}
		}

		InferUnion() : mode(Empty), data() {}
		InferUnion( const InferUnion& o ) : mode( o.mode ), data() { init_from( o ); }
		InferUnion( InferUnion&& o ) : mode( o.mode ), data() { init_from( std::move(o) ); }
		InferUnion& operator= ( const InferUnion& ) = delete;
		InferUnion& operator= ( InferUnion&& ) = delete;
		~InferUnion() { reset(); }

		ResnSlots& resnSlots() {
			switch (mode) {
			case Empty: new(&data.resnSlots) ResnSlots{}; mode = Slots; // fallthrough
			case Slots: return data.resnSlots;
			case Params: assert(!"Cannot return to resnSlots from Params");
			}
		}

		InferredParams& inferParams() {
			switch (mode) {
			case Slots: data.resnSlots.~ResnSlots(); // fallthrough
			case Empty: new(&data.inferParams) InferredParams{}; mode = Params; // fallthrough
			case Params: return data.inferParams;
			}
		}
	};

	ptr<Type> result;
	ptr<TypeSubstitution> env;
	InferUnion inferred;
	bool extension = false;

	Expr( const CodeLocation & loc, const Type * res = nullptr )
	: ParseNode( loc ), result( res ), env(), inferred() {}

	Expr * set_extension( bool ex ) { extension = ex; return this; }

	virtual const Expr * accept( Visitor & v ) const override = 0;
private:
	Expr * clone() const override = 0;
};

/// The application of a function to a set of parameters.
/// Post-resolver form of `UntypedExpr`
class ApplicationExpr final : public Expr {
public:
	ptr<Expr> func;
	std::vector<ptr<Expr>> args;

	ApplicationExpr( const CodeLocation & loc, const Expr * f, std::vector<ptr<Expr>> && as = {} );

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	ApplicationExpr * clone() const override { return new ApplicationExpr{ *this }; }
};

/// The application of a function to a set of parameters, pre-overload resolution.
class UntypedExpr final : public Expr {
public:
	ptr<Expr> func;
	std::vector<ptr<Expr>> args;

	UntypedExpr( const CodeLocation & loc, const Expr * f, std::vector<ptr<Expr>> && as = {} )
	: Expr( loc ), func( f ), args( std::move(as) ) {}

	/// Creates a new dereference expression
	static UntypedExpr * createDeref( const CodeLocation & loc, Expr * arg );
	/// Creates a new assignment expression
	static UntypedExpr * createAssign( const CodeLocation & loc, Expr * lhs, Expr * rhs );

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	UntypedExpr * clone() const override { return new UntypedExpr{ *this }; }
};

/// A name whose name is as-yet undetermined.
/// May also be used to avoid name mangling in codegen phase.
class NameExpr final : public Expr {
public:
	std::string name;

	NameExpr( const CodeLocation & loc, const std::string & n ) : Expr( loc ), name( n ) {}

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	NameExpr * clone() const override { return new NameExpr{ *this }; }
};

/// Address-of expression `&e`
class AddressExpr final : public Expr {
public:
	ptr<Expr> arg;

	AddressExpr( const CodeLocation & loc, const Expr * a );

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	AddressExpr * clone() const override { return new AddressExpr{ *this }; }
};

/// GCC &&label
/// https://gcc.gnu.org/onlinedocs/gcc-3.4.2/gcc/Labels-as-Values.html
class LabelAddressExpr final : public Expr {
public:
	Label arg;

	LabelAddressExpr( const CodeLocation & loc, Label && a );

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	LabelAddressExpr * clone() const override { return new LabelAddressExpr{ *this }; }
};

/// Whether a cast existed in the program source or not
enum GeneratedFlag { ExplicitCast, GeneratedCast };

/// A type cast, e.g. `(int)e`
class CastExpr final : public Expr {
public:
	ptr<Expr> arg;
	GeneratedFlag isGenerated;

	CastExpr( const CodeLocation & loc, const Expr * a, const Type * to,
		GeneratedFlag g = GeneratedCast ) : Expr( loc, to ), arg( a ), isGenerated( g ) {}
	/// Cast-to-void
	CastExpr( const CodeLocation & loc, const Expr * a, GeneratedFlag g = GeneratedCast );

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	CastExpr * clone() const override { return new CastExpr{ *this }; }
};

/// A cast to "keyword types", e.g. `(thread &)t`
class KeywordCastExpr final : public Expr {
public:
	ptr<Expr> arg;
	enum Target { Coroutine, Thread, Monitor, NUMBER_OF_TARGETS } target;

	KeywordCastExpr( const CodeLocation & loc, const Expr * a, Target t )
	: Expr( loc ), arg( a ), target( t ) {}

	/// Get a name for the target type
	const std::string& targetString() const;

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	KeywordCastExpr * clone() const override { return new KeywordCastExpr{ *this }; }
};

/// A virtual dynamic cast, e.g. `(virtual exception)e`
class VirtualCastExpr final : public Expr {
public:
	ptr<Expr> arg;

	VirtualCastExpr( const CodeLocation & loc, const Expr * a, const Type * to )
	: Expr( loc, to ), arg( a ) {}

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	VirtualCastExpr * clone() const override { return new VirtualCastExpr{ *this }; }
};

/// A member selection operation before expression resolution, e.g. `q.p`
class UntypedMemberExpr final : public Expr {
public:
	ptr<Expr> member;
	ptr<Expr> aggregate;

	UntypedMemberExpr( const CodeLocation & loc, const Expr * mem, const Expr * agg )
	: Expr( loc ), member( mem ), aggregate( agg ) { assert( aggregate ); }

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	UntypedMemberExpr * clone() const override { return new UntypedMemberExpr{ *this }; }
};

/// A member selection operation after expression resolution, e.g. `q.p`
class MemberExpr final : public Expr {
public:
	readonly<DeclWithType> member;
	ptr<Expr> aggregate;

	MemberExpr( const CodeLocation & loc, const DeclWithType * mem, const Expr * agg );

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	MemberExpr * clone() const override { return new MemberExpr{ *this }; }
};

/// A reference to a named variable.
class VariableExpr final : public Expr {
public:
	readonly<DeclWithType> var;

	VariableExpr( const CodeLocation & loc, const DeclWithType * v );

	/// generates a function pointer for a given function
	static VariableExpr * functionPointer( const CodeLocation & loc, const FunctionDecl * decl );

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	VariableExpr * clone() const override { return new VariableExpr{ *this }; }
};

/// A compile-time constant
class ConstantExpr final : public Expr {
	union Val {
		unsigned long long ival;
		double dval;

		Val( unsigned long long i ) : ival( i ) {}
		Val( double d ) : dval( d ) {}
	} val;
public:
	std::string rep;

	ConstantExpr(
		const CodeLocation & loc, const Type * ty, const std::string & r, unsigned long long v )
	: Expr( loc, ty ), val( v ), rep( r ) {}
	ConstantExpr( const CodeLocation & loc, const Type * ty, const std::string & r, double v )
	: Expr( loc, ty ), val( v ), rep( r ) {}

	/// Gets the value of this constant as an integer
	long long int intValue() const;
	/// Gets the value of this constant as floating point
	double floatValue() const;

	/// generates a boolean constant of the given bool
	static ConstantExpr * from_bool( const CodeLocation & loc, bool b );
	/// generates a char constant of the given char
	static ConstantExpr * from_char( const CodeLocation & loc, char c );
	/// generates an integer constant of the given int
	static ConstantExpr * from_int( const CodeLocation & loc, int i );
	/// generates an integer constant of the given unsigned long int
	static ConstantExpr * from_ulong( const CodeLocation & loc, unsigned long i );
	/// generates a floating point constant of the given double
	static ConstantExpr * from_double( const CodeLocation & loc, double d );
	/// generates an array of chars constant of the given string
	static ConstantExpr * from_string( const CodeLocation & loc, const std::string & s );
	/// generates a null pointer value for the given type. void * if omitted.
	static ConstantExpr * null( const CodeLocation & loc, const Type * ptrType = nullptr );

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	ConstantExpr * clone() const override { return new ConstantExpr{ *this }; }
};

/// sizeof expression, e.g. `sizeof(int)`, `sizeof 3+4`
class SizeofExpr final : public Expr {
public:
	ptr<Expr> expr;
	ptr<Type> type;

	SizeofExpr( const CodeLocation & loc, const Expr * e );
	SizeofExpr( const CodeLocation & loc, const Type * t );
	// deliberately no disambiguating overload for nullptr_t

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	SizeofExpr * clone() const override { return new SizeofExpr{ *this }; }
};

/// alignof expression, e.g. `alignof(int)`, `alignof 3+4`
class AlignofExpr final : public Expr {
public:
	ptr<Expr> expr;
	ptr<Type> type;

	AlignofExpr( const CodeLocation & loc, const Expr * e );
	AlignofExpr( const CodeLocation & loc, const Type * t );
	// deliberately no disambiguating overload for nullptr_t

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	AlignofExpr * clone() const override { return new AlignofExpr{ *this }; }
};

/// offsetof expression before resolver determines field, e.g. `offsetof(MyStruct, myfield)`
class UntypedOffsetofExpr final : public Expr {
public:
	ptr<Type> type;
	std::string member;

	UntypedOffsetofExpr( const CodeLocation & loc, const Type * ty, const std::string & mem )
	: Expr( loc ), type( ty ), member( mem ) {}

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	UntypedOffsetofExpr * clone() const override { return new UntypedOffsetofExpr{ *this }; }
};

/// offsetof expression after resolver determines field, e.g. `offsetof(MyStruct, myfield)`
class OffsetofExpr final : public Expr {
public:
	ptr<Type> type;
	readonly<DeclWithType> member;

	OffsetofExpr( const CodeLocation & loc, const Type * ty, const DeclWithType * mem );

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	OffsetofExpr * clone() const override { return new OffsetofExpr{ *this }; }
};

/// a pack of field-offsets for a generic type
class OffsetPackExpr final : public Expr {
public:
	ptr<StructInstType> type;

	OffsetPackExpr( const CodeLocation & loc, const StructInstType * ty );

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	OffsetPackExpr * clone() const override { return new OffsetPackExpr{ *this }; }
};

/// Variants of short-circuiting logical expression
enum LogicalFlag { OrExpr, AndExpr };

/// Short-circuiting boolean expression (`&&` or `||`)
class LogicalExpr final : public Expr {
public:
	ptr<Expr> arg1;
	ptr<Expr> arg2;
	LogicalFlag isAnd;

	LogicalExpr( const CodeLocation & loc, const Expr * a1, const Expr * a2, LogicalFlag ia );

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	LogicalExpr * clone() const override { return new LogicalExpr{ *this }; }
};

/// Three-argument conditional e.g. `p ? a : b`
class ConditionalExpr final : public Expr {
public:
	ptr<Expr> arg1;
	ptr<Expr> arg2;
	ptr<Expr> arg3;

	ConditionalExpr( const CodeLocation & loc, const Expr * a1, const Expr * a2, const Expr * a3 )
	: Expr( loc ), arg1( a1 ), arg2( a2 ), arg3( a3 ) {}

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	ConditionalExpr * clone() const override { return new ConditionalExpr{ *this }; }
};

/// Comma expression e.g. `( a , b )`
class CommaExpr final : public Expr {
public:
	ptr<Expr> arg1;
	ptr<Expr> arg2;

	CommaExpr( const CodeLocation & loc, const Expr * a1, const Expr * a2 )
	: Expr( loc ), arg1( a1 ), arg2( a2 ) {}

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	CommaExpr * clone() const override { return new CommaExpr{ *this }; }
};

/// A type used as an expression (e.g. a type generator parameter)
class TypeExpr final : public Expr {
public:
	ptr<Type> type;

	TypeExpr( const CodeLocation & loc, const Type * t ) : Expr(loc), type(t) {}

	const Expr * accept( Visitor & v ) const override { return v.visit( this ); }
private:
	TypeExpr * clone() const override { return new TypeExpr{ *this }; }
};

/// A GCC "asm constraint operand" used in an asm statement, e.g. `[output] "=f" (result)`.
/// https://gcc.gnu.org/onlinedocs/gcc-4.7.1/gcc/Machine-Constraints.html#Machine-Constraints
class AsmExpr final : public Expr {
public:
	ptr<Expr> inout;
	ptr<Expr> constraint;
};

}

// Local Variables: //
// tab-width: 4 //
// mode: c++ //
// compile-command: "make install" //
// End: //
